/* 29 march 2001: missing 'n' option test corrected by D. Taupin */
/*
   Copyright (C) 1991-1995 Eberhard Mattes 

   dvispell.c - part of dvispell.
   dvispell converts DVI files into readable text files

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
*/

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h> 
#include <getopt.h> 
#include <errno.h>
#include <math.h>
#include <kpathsea/progname.h>
#include <kpathsea/tex-file.h>
#include <kpathsea/pathsearch.h>
#include "dvispell.h"
//#include "ext.h"

#define DVI_set_char_0    0
#define DVI_set1        128
#define DVI_set2        129
#define DVI_set3        130
#define DVI_set4        131
#define DVI_set_rule    132
#define DVI_put1        133
#define DVI_put2        134
#define DVI_put3        135
#define DVI_put4        136
#define DVI_put_rule    137
#define DVI_nop         138
#define DVI_bop         139
#define DVI_eop         140
#define DVI_push        141
#define DVI_pop         142
#define DVI_right1      143
#define DVI_right2      144
#define DVI_right3      145
#define DVI_right4      146
#define DVI_w0          147
#define DVI_w1          148
#define DVI_w2          149
#define DVI_w3          150
#define DVI_w4          151
#define DVI_x0          152
#define DVI_x1          153
#define DVI_x2          154
#define DVI_x3          155
#define DVI_x4          156
#define DVI_down1       157
#define DVI_down2       158
#define DVI_down3       159
#define DVI_down4       160
#define DVI_y0          161
#define DVI_y1          162
#define DVI_y2          163
#define DVI_y3          164
#define DVI_y4          165
#define DVI_z0          166
#define DVI_z1          167
#define DVI_z2          168
#define DVI_z3          169
#define DVI_z4          170
#define DVI_fnt_num_0   171
#define DVI_fnt1        235
#define DVI_fnt2        236
#define DVI_fnt3        237
#define DVI_fnt4        238
#define DVI_xxx1        239
#define DVI_xxx2        240
#define DVI_xxx3        241
#define DVI_xxx4        242
#define DVI_fnt_def1    243
#define DVI_fnt_def2    244
#define DVI_fnt_def3    245
#define DVI_fnt_def4    246
#define DVI_pre         247
#define DVI_post        248
#define DVI_post_post   249
#define IVD_beg_reflect 250
#define IVD_end_reflect 251

#define DVI_EOF         223
#define DVI_id_byte       2

#define DVI_STACK_SIZE   64


struct dvi_stack
{
  long h, v, w, x, y, z;
};

struct font
{
  struct font *next;
  struct encoding *encoding;
  long number;
  long width[256];
  long word_space;
  long back_space;
  long vert_space;
};

struct encoding
{
  unsigned *symbols[CHAR_CODES];
  unsigned lengths[CHAR_CODES];
};

struct font_or_layout
{
  struct encoding *encoding;
  unsigned name;
};

struct conv
{
  unsigned *in_table;
  unsigned *out_table;
  unsigned in_count;
  unsigned out_count;
};

struct conversion
{
  struct conv *table;
  unsigned count;
  unsigned name;
};

struct prod
{
  unsigned *table;
  unsigned count;
};

struct transition
{
  unsigned state;               /* State number */
  unsigned input;               /* Input symbol */
  unsigned output;              /* New state number */
};

static FILE * dvi_file;
static FILE * tfm_file;
static FILE * output_file;

static int console;
static int column;

static struct dvi_stack  position;

static char dat_fname[260] = "plain";
static FILE *  dat_file;

static struct dvi_stack  dvi_stack[DVI_STACK_SIZE];
static int  dvi_sp;

static long dvi_num;
static long dvi_den;

static char ** str_pool;
static unsigned * str_length;
static unsigned str_count;

static struct prod * prod_table;

static struct transition * trans_table;
static unsigned * def_table;
static unsigned  dfa_hash[DFA_HASH_SIZE+1];

static char *  letters_table;

static struct font_or_layout *  font_table;
static unsigned  font_count;

static struct font_or_layout *  layout_table;
static unsigned  layout_count;

static unsigned *  output_table;
static unsigned  output_count;
static unsigned  output_hash[OUTPUT_HASH_SIZE+1];

static struct font *  fonts;
static struct font *  cur_font;
static struct font *  last_font;
static double  tfm_conv;
static double  word_space_adjust;
static double  vert_space_adjust;
static long  last_h;
static long  last_v;

static const char * output_name;
static const char * conversion_name;
static int  start_of_line;
static char page_number_str[1025] = {'\0'};

//static struct emtex_dir  ed_tfm;
//static struct emtex_dir  ed_data;

static char  omit_unprintable;
static char  debugging;
static char  letters_only;
static char  page_numbers;
static char  logo;
static char  verbose;

#define MAX_LOOKAHEAD  32

struct automaton
{
  struct automaton *next;
  unsigned len;
  unsigned read;
  unsigned ptr;
  unsigned input[MAX_LOOKAHEAD];
  unsigned state;
  unsigned level;
  int more_flag;
};

static struct automaton *  auto0;

static int  automatons;

#if defined (__GNUC__)
#define NORETURN2 __attribute__ ((noreturn))
static void error (int rc, char *fmt, ...) NORETURN2;
static void output_error (void) NORETURN2;
static void internal (int n) NORETURN2;
#endif


static void banner (void)
{
  fputs ("GNU dvispell 1.0c \n"
	 "kpathsea version 3.3.1\n"
	 "converts dvi files into readable text files\n"
	 "Copyright (c) 1995 by Eberhard Mattes\n"
  	 "GNU dvispell comes with ABSOLUTELY NO WARRANTY\n"
	 "This is free software. You may redistribute copies of dvispell\n"
	 "under the terms of the GNU General Public License.\n"
	 "For more information about these matters, see the file named COPYING.\n\n"  ,stderr);
}


static void error (int rc, char *fmt, ...)
{
  va_list arg_ptr;

  fflush (stdout);
  fprintf (stderr, "\n*** ");
  va_start (arg_ptr, fmt);
  vfprintf (stderr, fmt, arg_ptr);
  va_end (arg_ptr);
  fprintf (stderr, "\n");
  exit (rc);
}


static void warning (char *fmt, ...)
{
  va_list arg_ptr;

  fflush (stdout);
  fprintf (stderr, "\nWarning: ");
  va_start (arg_ptr, fmt);
  vfprintf (stderr, fmt, arg_ptr);
  va_end (arg_ptr);
  fprintf (stderr, "\n");
}


static void output_error (void)
{
  error (EXIT_FILE, "Error while writing to output file: %s",
         strerror (errno));
}


static void *xmalloc (size_t n)
{
  void *p;

  p = malloc (n);
  if (p == NULL)
    error (EXIT_MEM, "Out of memory");
  return (p);
}


static void xfree (void *p)
{
  if (p != NULL)
    free (p);
}


static void usage (void)
{
  banner ();
  fputs ("\nUsage: dvispell [<options>] <dvi_file> [<output_file>]\n", stderr);
  fputs ("\nOptions:\n", stderr);
  fputs ("  -h, --help                        Give this help\n", stderr);
  fputs ("  -V, --version                     Display version number\n", stderr);
  fputs ("  -c NAME, --conversion-table=NAME  Use conversion table <NAME>, \n",stderr);
  fputs ("                                    default is <DEFAULT>\n", stderr);
  fputs ("  -d FILE, --parameter-file=FILE    Name of parameter file, default is plain.dsb\n", stderr);
  fputs ("  -l, --letters-only                Letters only\n", stderr);
  fputs ("  -n, --page-number                 Insert page number at start of each line\n", stderr);
  fputs ("  -o NAME, --output-table=NAME      Use output table <NAME>, default is <DEFAULT>\n", stderr);
  fputs ("  -q, --quiet                       Don't display logo\n", stderr);
  fputs ("  -u, --omit-unprintable            Suppress output of names of unprintable characters\n", stderr);
  fputs ("  -v  --verbose                     Be verbose: display page numbers\n", stderr);
  fputs ("  -w FACTOR, --word-space=FACTOR    Set factor for word space, default is 1.0\n", stderr);
  fputs ("  -y FACTOR, --vert-space=FACTOR    Set factor for vertical space, default is 0.9\n", stderr);
  fputs ("\nExample: dvispell myfile.dvi\n", stderr);
  exit (EXIT_ARG);
}


static void internal (int n)
{
  error (EXIT_INT, "Internal error, case %d", n);
}


static void get_args (int argc, char *argv[])
{
  int quit=FALSE;
  int c;
  char dvi_fname[260];
  char *end;
  int option_index = 0;
  extern int optopt;
  struct option long_options[] = 
  {
    {"help",0,0,'h'},
    {"version",0,0,'V'},
    {"conversion-table",1,0,'c'},
    {"parameter-file",1,0,'d'},
    {"letters-only",0,0,'l'},
    {"page-number",0,0,'n'},
    {"output-table",1,0,'o'},
    {"quiet",0,0,'q'},
    {"omit-unprintable",0,0,'u'},
    {"verbose",0,0,'v'},
    {"word-space",1,0,'w'},
    {"vert-space",1,0,'y'},
    {"debugging",0,0,'D'},
    {0,0,0,0}
  };
  
  opterr = FALSE;
  //optswchar = "-/";
  optind = 0;
  output_file = stdout;
  output_name = "DEFAULT";
  conversion_name = "DEFAULT";
  omit_unprintable = FALSE;
  letters_only = FALSE;
  debugging = FALSE;
  page_numbers = FALSE;
  logo = TRUE; verbose = FALSE;
  word_space_adjust = 1.0;
  vert_space_adjust = 0.9;

  while ((c = getopt_long (argc, argv, "Vhc:d:lno:quvw:y:",
			  long_options, &option_index)) !=EOF && !quit)
    {
      switch(c)
	{
	case 'V':
	  logo=TRUE;
	  quit=TRUE;
	  break;
	case 'h':
	  usage();
	  quit=TRUE;
	  break;
	case 'c':
	  conversion_name = optarg;
          break;
	case 'd':
          strcpy (dat_fname,  optarg);
          break; 
	case 'l':
          letters_only = TRUE;
          break; 
	case 'n':
          page_numbers = TRUE;
          break; 
	case 'o':
	  output_name = optarg;
	  break;
	case 'q':
	  logo=FALSE;
	  break;
	case 'u':
	  omit_unprintable = TRUE;
          break;
	case 'v':
	  verbose=TRUE;
	  break;
	case 'w':
          word_space_adjust = strtod (optarg, &end);
          if (word_space_adjust < 0.01 || word_space_adjust > 10.0
              || *end != 0)
            usage ();
          break;
	case 'y':
          vert_space_adjust = strtod (optarg, &end);
          if (vert_space_adjust < 0.01 || vert_space_adjust > 10.0
              || *end != 0)
            usage ();
          break;
	case 'D':
	  debugging=TRUE;
	  break;
	default:
	  quit=TRUE;
	  fprintf(stderr,"Unknown option -%c\n",optopt);
	  usage();
	  break;
	}
    }

  if (logo)
    banner ();
  if (quit)
    exit(0);
  if (argc - optind < 1 || argc - optind > 2)
    usage ();
 
  strcpy (dvi_fname, argv[optind]);
  fndefext (dvi_fname, "dvi");
  dvi_file = fopen (dvi_fname, "rb");
  if (dvi_file == NULL)
    error (EXIT_FILE, "Cannot open input file `%s'", dvi_fname);
  ++optind;
  if (optind < argc)
    {
      output_file = fopen (argv[optind], "wt");
      if (output_file == NULL)
        error (EXIT_FILE, "Cannot open output file `%s'", argv[optind]);
    }
}


static unsigned find_stringn (const char *str, size_t len)
{
  unsigned i;

  for (i = 0; i < str_count; ++i)
    if (str_length[i] == len && memcmp (str_pool[i], str, len) == 0)
      return (i);
  return (NIL16);
}


static unsigned find_string (const char *str)
{
  return (find_stringn (str, strlen (str)));
}


static unsigned find_icase_stringn (const char *str, size_t len)
{
  unsigned i;
  size_t j;

  for (i = 0; i < str_count; ++i)
    if (str_length[i] == len)
      {
        for (j = 0; j < len; ++j)
          if (tolower (str_pool[i][j]) != tolower (str[j]))
            break;
        if (j == len)
          return (i);
      }
  return (NIL16);
}


static unsigned find_icase_string (const char *str)
{
  return (find_icase_stringn (str, strlen (str)));
}


static struct automaton *create_automaton (unsigned level)
{
  struct automaton *ap;

  ++automatons;
  ALLOC (ap);
  ap->read = 0; ap->ptr = 0; ap->len = 0;
  ap->state = 0;
  ap->level = level;
  ap->more_flag = FALSE;
  ap->next = NULL;
  return (ap);
}


static void print_page_number (void)
{
//  if (page_number_str != NULL)
    if (fprintf (output_file, "%s:", page_number_str) < 0)
      output_error ();
}


static void output_char (unsigned c)
{
  if (start_of_line && page_numbers)
    print_page_number ();
  if (putc (c, output_file) == EOF)
    output_error ();
  start_of_line = (c == '\n');

  /* _IOLBF mode seems to be broken in MS C 6.00a.  We use _IOFBF
     instead and flush the buffer here. */

  if (c == '\n' && console)
    fflush (output_file);
}


static void output (unsigned n)
{
  unsigned i, h, end;

  if (n == NIL16)
    return;
  h = n % OUTPUT_HASH_SIZE;
  end = output_hash[h+1];
  for (i = output_hash[h]; i < end; ++i)
    if (output_table[2*i+0] == n)
      {
        n = output_table[2*i+1];
        for (i = 0; i < str_length[n]; ++i)
          output_char (str_pool[n][i]);
        return;
      }
  if (!omit_unprintable)
    {
      if (start_of_line && page_numbers)
        print_page_number ();
      if (fprintf (output_file, "<%.*s>",
                   (int)str_length[n], str_pool[n])< 0)
        output_error ();
      start_of_line = FALSE;
    }
}


static struct automaton * ins_ap;

static void dfa_insert (unsigned n)
{
  ins_ap->input[ins_ap->ptr++] = n;
}


static void automaton (struct automaton *ap, unsigned input)
{
  unsigned insert, loops, i, end, st, inp, h;
  unsigned result;
  char more;

  if (ap->read >= MAX_LOOKAHEAD) error (EXIT_MEM, "DFA buffer overflow");
  ap->input[ap->read++] = input;
  loops = 0;
  while (ap->ptr < ap->read)
    {
      ++loops;
      if (loops > 100)
        error (EXIT_SYNTAX, "Loop in conversion table?");
      more = FALSE; result = NIL16;
      st = ap->state;
      inp = ap->input[ap->ptr];
      h = DFA_HASH (st, inp);
      end = dfa_hash[h+1];
      for (i = dfa_hash[h]; i < end; ++i)
        if (trans_table[i].state == ap->state && trans_table[i].input == inp)
          {
            ap->state = trans_table[i].output;
            more = TRUE;
            break;
          }
      if (!more && def_table[st] != NIL16)
        {
          ap->len = ap->ptr;
          result = def_table[st];
        }
      ++ap->ptr;
      if (!more)
        {
          if (ap->len == 0)
            {
              if (letters_only && ap->input[0] != S_NEWLINE
                  && ap->input[0] != S_NEWPAGE && ap->input[0] != S_WORDSPACE
                  && ap->input[0] != S_END && !letters_table[ap->input[0]])
                {
                  ap->len = 1; insert = 1;
                  result = NIL16;
                }
              else if (ap->more_flag)
                {
                  if (ap->next == NULL)
                    ap->next = create_automaton (ap->level + 1);
                  automaton (ap->next, ap->input[0]);
                  ap->len = 1; insert = 0;
                }
              else
                {
                  output (ap->input[0]);
                  ap->len = 1; insert = 0;
                }
            }
          else
            {
              insert = prod_table[result].count;
              ap->more_flag = TRUE;
            }
          ap->read -= ap->len;
          if (insert + ap->read > MAX_LOOKAHEAD)
            error (EXIT_MEM, "DFA buffer overflow");
          memmove (ap->input + insert, ap->input + ap->len,
                   ap->read * sizeof (ap->input[0]));
          ap->state = 0; /* Start state */
          ap->len = 0;
          if (insert > 0)
            {
              ap->ptr = 0;
              ins_ap = ap;
              if (result == NIL16)
                dfa_insert (S_NONLETTER);
              else
                for (i = 0; i < prod_table[result].count; ++i)
                  dfa_insert (prod_table[result].table[i]);
              ap->read += insert;
            }
          ap->ptr = 0;
        }
    }
}


static void convert_code (unsigned n)
{
  automaton (auto0, n);
}


static void dvi_eof (void)
{
  error (EXIT_DVI, "EOF reached on DVI file");
}


static void tfm_eof (void)
{
  error (EXIT_TFM, "EOF reached on TFM file");
}


#define DVI_GET (byte)getc (dvi_file)
#define DVI_EOF_CHECK {if (feof (dvi_file)) dvi_eof ();}

#define TFM_GET (byte)getc (tfm_file)
#define TFM_EOF_CHECK {if (feof (tfm_file)) tfm_eof ();}

static unsigned dvi_get1u (void)
{
  unsigned x;

  x = DVI_GET;
  DVI_EOF_CHECK;
  return (x);
}


static int dvi_get1s (void)
{
  unsigned n;

  n = DVI_GET;
  DVI_EOF_CHECK;
  if (n >= 128)
    return (n-256);
  else
    return (n);
}


static unsigned dvi_get2u (void)
{
  unsigned a, b;

  a = DVI_GET;
  b = DVI_GET;
  DVI_EOF_CHECK;
  return (a*256+b);
}


static int dvi_get2s (void)
{
  unsigned a, b;

  a = DVI_GET;
  b = DVI_GET;
  DVI_EOF_CHECK;
  if (a >= 128)
    return ((a-256)*256+b);
  else
    return (a*256+b);
}


static long dvi_get3u (void)
{
  unsigned a, b, c;

  a = DVI_GET;
  b = DVI_GET;
  c = DVI_GET;
  DVI_EOF_CHECK;
  return (a*65536L+b*256L+c);
}



static long dvi_get3s (void)
{
  unsigned a, b, c;

  a = DVI_GET;
  b = DVI_GET;
  c = DVI_GET;
  DVI_EOF_CHECK;
  if (a >= 128)
    return ((a-256)*65536L+b*256L+c);
  else
    return (a*65536L+b*256L+c);
}


static long dvi_get4 (void)
{
  unsigned a, b, c, d;

  a = DVI_GET;
  b = DVI_GET;
  c = DVI_GET;
  d = DVI_GET;
  DVI_EOF_CHECK;
  if (a >= 128)
    return ((a-256)*16777216L+b*65536L+c*256L+d);
  else
    return (a*16777216L+b*65536L+c*256L+d);
}


static long dvi_get_u (int l)
{
  switch (l)
    {
    case 0:
      return ((long)dvi_get1u ());
    case 1:
      return ((long)dvi_get2u ());
    case 2:
      return (dvi_get3u ());
    case 3:
      return (dvi_get4 ());
    default:
      internal (1);
    }
}


static long dvi_get_s (int l)
{
  switch (l)
    {
    case 0:
      return ((long)dvi_get1s ());
    case 1:
      return ((long)dvi_get2s ());
    case 2:
      return (dvi_get3s ());
    case 3:
      return (dvi_get4 ());
    default:
      internal (2);
    }
}


static unsigned tfm_get1u (void)
{
  int x;

  x = TFM_GET;
  TFM_EOF_CHECK;
  return (x);
}


static unsigned tfm_get2u (void)
{
  unsigned a, b;

  a = (unsigned)TFM_GET;
  b = (unsigned)TFM_GET;
  TFM_EOF_CHECK;
  return (a*256+b);
}


static long tfm_get4 (void)
{
  int a, b, c, d;

  a = TFM_GET;
  b = TFM_GET;
  c = TFM_GET;
  d = TFM_GET;
  TFM_EOF_CHECK;
  if (a >= 128)
    return ((a-256)*16777216L+b*65536L+c*256L+d);
  else
    return (a*16777216L+b*65536L+c*256L+d);
}


static void push_position (void)
{
  if (dvi_sp >= DVI_STACK_SIZE)
    error (EXIT_DVI, "Position stack overflow");
  dvi_stack[dvi_sp++] = position;
}


static void pop_position (void)
{
  if (dvi_sp <= 0)
    error (EXIT_DVI, "Position stack underflow");
  position = dvi_stack[--dvi_sp];
}


static struct encoding *find_font_encoding (unsigned idx)
{
  unsigned i;

  if (idx != NIL16)
    for (i = 0; i < font_count; ++i)
      if (font_table[i].name == idx)
        return (font_table[i].encoding);
  return (NULL);
}


static struct encoding *find_layout_encoding (unsigned idx)
{
  unsigned i;

  if (idx != NIL16)
    for (i = 0; i < layout_count; ++i)
      if (layout_table[i].name == idx)
        return (layout_table[i].encoding);
  return (NULL);
}


#define TFM_LF  0
#define TFM_LH  1
#define TFM_BC  2
#define TFM_EC  3
#define TFM_NW  4
#define TFM_NH  5
#define TFM_ND  6
#define TFM_NI  7
#define TFM_NL  8
#define TFM_NK  9
#define TFM_NE 10
#define TFM_NP 11

static void dvi_fnt_def (int cmd)
{
  char font_name[255+1], tfm_name[255+1+4], fname[255+1+100];
  char coding_scheme[39+1];
  long ext_num, dvi_check_sum, dvi_scaled_size, dvi_design_size;
  long tfm_check_sum, tfm_design_size;
  long width[256], param[7];
  unsigned tfm_len[12], i, tmp, cn, area_len, name_len;
  struct font *fp;
  double factor, d_dvi, d_tfm;
  char *full_tfm_name;

  ext_num = dvi_get_u (cmd-DVI_fnt_def1);
  dvi_check_sum = dvi_get4 ();
  dvi_scaled_size = dvi_get4 ();
  dvi_design_size = dvi_get4 ();
  area_len = dvi_get1u ();
  name_len = dvi_get1u ();
  for (i = 0; i < area_len; ++i)
    dvi_get1u ();
  for (i = 0; i < name_len; ++i)
    font_name[i] = (char)dvi_get1u ();
  font_name[name_len] = 0;
  factor = tfm_conv * (double)dvi_scaled_size / (double)65536.0;
  ALLOC (fp);
  strcpy (tfm_name, font_name);
  strcat (tfm_name, ".tfm");
 full_tfm_name = kpse_find_tfm(tfm_name);
 if(full_tfm_name) 
    tfm_file = fopen (full_tfm_name, "rb");
  else
    tfm_file = NULL;
  if (tfm_file == NULL)
    error (EXIT_FILE, "TFM file for font `%s' not found", font_name);
  for (i = 0; i < 12; ++i)
    tfm_len[i] = tfm_get2u ();
  tmp = 6 + tfm_len[TFM_LH] + tfm_len[TFM_EC] - tfm_len[TFM_BC] + 1;
  for (i = TFM_NW; i <= TFM_NP; ++i)
    tmp += tfm_len[i];
  if (tfm_len[TFM_LF] != tmp || tfm_len[TFM_EC] > 255 || tfm_len[TFM_NW] > 256
      || tfm_len[TFM_NH] > 16 || tfm_len[TFM_ND] > 16)
    error (EXIT_TFM, "Invalid TFM file `%s' (lengths)", fname);
  cn = tfm_len[TFM_EC] - tfm_len[TFM_BC] + 1;

  tfm_check_sum = tfm_get4 ();
  tfm_design_size = tfm_get4 ();

  if (tfm_check_sum != dvi_check_sum)
    warning ("Wrong checksum (TFM file `%s')", fname);

  d_dvi = ((double)dvi_design_size * (double)dvi_num / (double)dvi_den
           * 72.27 / 254000.0);
  d_tfm = ldexp ((double)tfm_design_size, -20);
  if (fabs (d_tfm - d_dvi) >= 0.1)
    warning ("Wrong design size (TFM file `%s')", fname);

  tmp = tfm_get1u ();
  if (tmp > 39)
    error (EXIT_TFM, "Invalid TFM file `%s' (coding scheme)", fname);
  for (i = 0; i < tmp; ++i)
    coding_scheme[i] = (char)tfm_get1u ();
  coding_scheme[tmp] = 0;

  fp->number = ext_num;
  for (i = 0; i < 256; ++i)
    fp->width[i] = 0;
  fseek (tfm_file, (6 + tfm_len[TFM_LH] + cn) * 4, SEEK_SET);
  for (i = 0; i < tfm_len[TFM_NW]; ++i)
    width[i] = tfm_get4 ();
  tmp = 6 + tfm_len[TFM_LH] + cn;
  for (i = TFM_NW; i <= TFM_NE; ++i)
    tmp += tfm_len[i];
  fseek (tfm_file, (long)tmp * 4, SEEK_SET);
  for (i = 0; i < 7; ++i)
    param[i] = 0;
  tmp = tfm_len[TFM_NP];
  if (tmp > 7)
    tmp = 7;
  for (i = 0; i < tmp; ++i)
    param[i] = tfm_get4 ();

  /* The word space is NOT rounded up; it's better to make this value
     too small than too big. */

  fp->word_space = (long)((param[2-1] - param[4-1]) * factor
                          * word_space_adjust);

  fp->back_space = (long)(0.9 * param[6-1] + 0.5);
  fp->vert_space = (long)(dvi_scaled_size * vert_space_adjust);
  fseek (tfm_file, (6 + tfm_len[TFM_LH]) * 4, SEEK_SET);
  for (i = tfm_len[TFM_BC]; i <= tfm_len[TFM_EC]; ++i)
    {
      tmp = tfm_get1u ();
      if (tmp >= tfm_len[TFM_NW])
        error (EXIT_TFM, "Invalid TFM file `%s' (width)", fname);
      fp->width[i] = (long)(width[tmp] * factor + 0.5);
      tfm_get1u ();             /* height, depth           */
      tfm_get1u ();             /* Italics correction, tag */
      tfm_get1u ();             /* Remainder               */
    }
  fclose (tfm_file);

  /* Find the encoding table for this font. */

  fp->encoding = find_font_encoding (find_string (font_name));
  if (fp->encoding == NULL)
    fp->encoding = find_layout_encoding (find_string (coding_scheme));

  /* If no encoding table is found, try again, ignoring case. */

  if (fp->encoding == NULL)
    fp->encoding = find_font_encoding (find_icase_string (font_name));
  if (fp->encoding == NULL)
    fp->encoding = find_layout_encoding (find_icase_string (coding_scheme));

  /* Add this font to the list of DVI fonts. */

  fp->next = fonts;
  fonts = fp;
}


static void dvi_set_char (long n, int move)
{
  long h_diff;
  unsigned i;
    
  h_diff = position.h - last_h;
  if ((cur_font != NULL && labs (position.v - last_v) >= cur_font->vert_space)
      || (last_font != NULL
          && labs (position.v - last_v) >= last_font->vert_space))
    convert_code (S_NEWLINE);
  else if ((cur_font != NULL
            && (h_diff >= cur_font->word_space
                || h_diff <= -cur_font->back_space))
           || (last_font != NULL && (h_diff >= last_font->word_space
                                     || h_diff <= -last_font->back_space)))
    convert_code (S_WORDSPACE);
  if (n >= 0 && n < CHAR_CODES)
    {
      if (cur_font->encoding != NULL)
        {
          unsigned count, *symbols;

          count = cur_font->encoding->lengths[n];
          symbols = cur_font->encoding->symbols[n];
          for (i = 0; i < count; ++i)
            convert_code (symbols[i]);
        }
      if (move && cur_font != NULL)
        position.h += cur_font->width[n];
    }
  last_h = position.h;
  last_v = position.v;
  last_font = cur_font;
}


static void dvi_fnt_num (long n)
{
  struct font *fp;

  for (fp = fonts; fp != NULL; fp = fp->next)
    if (fp->number == n)
      {
        cur_font = fp;
        return;
      }
  error (EXIT_DVI, "Font %ld not defined", n);
}


static void dvi_start_page (void)
{
  int i, j, kkk, add;
  long page_number[10], back_pointer;
  char *p;
  static char buf[120];

  for (i = 0; i < 10; ++i)
    page_number[i] = dvi_get4 ();

  back_pointer = dvi_get4 ();
  j = 0;
  for (i = 0; i < 10; ++i)
  {
    if (page_number[i] != 0L)
      j = i;
  }
  page_number_str[0] = '\0';
  for (i = 0; i <= j; ++i)
    {
      p = strchr (page_number_str,'\0');
      if (i == j)
      {
        sprintf(p,"%d",page_number[i]);
      } else
      {
        sprintf(p,"%d.",page_number[i]);
      }
    }

  if (verbose)
    {
      add = strlen (page_number_str) + 2;
      if (column != 0)
        {
          if (column + 1 + add >= 80)
            {
              fputc ('\n', stderr);
              column = 0;
            }
          else
            {
              fputc (' ', stderr);
              ++column;
            }
        }
      fprintf (stderr, "[%s]", page_number_str);
      column += add;
    }
}


static void dvi_end_page (void)
{
  convert_code (S_NEWPAGE);
}


static void dvi_rule (int move)
{
  long dvi_height, dvi_width;

  dvi_height = dvi_get4 ();
  dvi_width = dvi_get4 ();
  if (move)
    position.h += dvi_width;
}


static void dvi_right (long n)
{
  position.h += n;
}


static void dvi_down (long n)
{
  position.v += n;
}


static void dvi_special (long n)
{
  while (n > 0)
    {
      --n;
      dvi_get1u ();
    }
}


static void dvi_read (void)
{
  int b1, b2;
  long mag, n4;

  automatons = 0;
  auto0 = create_automaton (0);
  b1 = dvi_get1u ();
  b2 = dvi_get1u ();
  if (b1 != DVI_pre || b2 != DVI_id_byte)
    error (EXIT_DVI, "Input file is not a DVI file");
  dvi_num = dvi_get4 ();
  dvi_den = dvi_get4 ();
  mag = dvi_get4 ();
  b1 = dvi_get1u ();
  while (b1 > 0)
    {
      --b1;
      dvi_get1u ();
    }
  tfm_conv = ((25400000.0 / (double)dvi_num)
              * ((double)dvi_den / 473628672.0) / 16.0);
  dvi_sp = 0;
  convert_code (S_START);
  for (;;)
    {
      b1 = dvi_get1u ();
      if (b1 >= DVI_set_char_0 && b1 < DVI_set_char_0+128)
        dvi_set_char ((long)b1-DVI_set_char_0, TRUE);
      else if (b1 >= DVI_fnt_num_0 && b1 < DVI_fnt_num_0+64)
        dvi_fnt_num ((long)(b1-DVI_fnt_num_0));
      else
        switch (b1)
          {
          case DVI_set1:
          case DVI_set2:
          case DVI_set3:
          case DVI_set4:
            n4 = dvi_get_u (b1-DVI_set1);
            dvi_set_char (n4, TRUE);
            break;
          case DVI_put1:
          case DVI_put2:
          case DVI_put3:
          case DVI_put4:
            n4 = dvi_get_u (b1-DVI_put1);
            dvi_set_char (n4, FALSE);
            break;
          case DVI_set_rule:
            dvi_rule (TRUE);
            break;
          case DVI_put_rule:
            dvi_rule (FALSE);
            break;
          case DVI_nop:
            break;
          case DVI_bop:
            dvi_start_page ();
            break;
          case DVI_eop:
            dvi_end_page ();
            break;
          case DVI_push:
            push_position ();
            break;
          case DVI_pop:
            pop_position ();
            break;
          case DVI_right1:
          case DVI_right2:
          case DVI_right3:
          case DVI_right4:
            n4 = dvi_get_s (b1-DVI_right1);
            dvi_right (n4);
            break;
          case DVI_down1:
          case DVI_down2:
          case DVI_down3:
          case DVI_down4:
            n4 = dvi_get_s (b1-DVI_down1);
            dvi_down (n4);
            break;
          case DVI_w0:
            dvi_right (position.w);
            break;
          case DVI_w1:
          case DVI_w2:
          case DVI_w3:
          case DVI_w4:
            position.w = dvi_get_s (b1-DVI_w1);
            dvi_right (position.w);
            break;
          case DVI_x0:
            dvi_right (position.x);
            break;
          case DVI_x1:
          case DVI_x2:
          case DVI_x3:
          case DVI_x4:
            position.x = dvi_get_s (b1-DVI_x1);
            dvi_right (position.x);
            break;
          case DVI_y0:
            dvi_down (position.y);
            break;
          case DVI_y1:
          case DVI_y2:
          case DVI_y3:
          case DVI_y4:
            position.y = dvi_get_s (b1-DVI_y1);
            dvi_down (position.y);
            break;
          case DVI_z0:
            dvi_down (position.z);
            break;
          case DVI_z1:
          case DVI_z2:
          case DVI_z3:
          case DVI_z4:
            position.z = dvi_get_s (b1-DVI_z1);
            dvi_down (position.z);
            break;
          case DVI_fnt1:
          case DVI_fnt2:
          case DVI_fnt3:
          case DVI_fnt4:
            n4 = dvi_get_u (b1-DVI_fnt1);
            dvi_fnt_num (n4);
            break;
          case DVI_xxx1:
          case DVI_xxx2:
          case DVI_xxx3:
          case DVI_xxx4:
            n4 = dvi_get_u (b1-DVI_xxx1);
            dvi_special (n4);
            break;
          case DVI_fnt_def1:
          case DVI_fnt_def2:
          case DVI_fnt_def3:
          case DVI_fnt_def4:
            dvi_fnt_def (b1);
            break;
          case DVI_post:
            goto done;
          default:
            error (EXIT_DVI, "Invalid DVI file");
          }
    }
done:
  while (auto0->read > 0 && auto0->input[0] != S_END)
    convert_code (S_END);
  if (debugging)
    printf ("%d automatons\n", automatons);
}


static void start (char * progname)
{
  fonts = NULL;
  cur_font = NULL;
  last_font = NULL;
  last_h = last_v = 0;
  start_of_line = TRUE; page_number_str[0] = '\0';
  console = isatty (fileno (output_file));
  column = 0;
  kpse_set_progname(progname);
  setvbuf (output_file, NULL, _IOFBF, BUFSIZ);
}


static void read_bytes (void *dst, size_t n)
{
  if (fread (dst, 1, n, dat_file) != n)
    error (EXIT_FILE, "Cannot read from input file `%s'", dat_fname);
}


static unsigned read_u16 (void)
{
  unsigned char b[2];

  read_bytes (b, 2);
  return (((unsigned)b[0] << 8) | b[1]);
}


static u32 read_u32 (void)
{
  unsigned char b[4];

  read_bytes (b, 4);
  return (((u32)b[0] << 24) | ((u32)b[1] << 16) | ((u32)b[2] << 8) | b[3]);
}


/* Load the string table. */

static void load_str (long pos)
{
  unsigned str_size, *str_off;
  char *str_buffer;
  unsigned i;

  fseek (dat_file, pos, SEEK_SET);
  str_count = read_u16 ();
  str_size = read_u16 ();

  ALLOCARRAY (str_off, str_count + 1);
  ALLOCARRAY (str_buffer, str_size);
  ALLOCARRAY (str_pool, str_count);
  ALLOCARRAY (str_length, str_count);
  for (i = 0; i <= str_count; ++i)
    str_off[i] = read_u16 ();
  read_bytes (str_buffer, str_size);
  for (i = 0; i < str_count; ++i)
    str_length[i] = str_off[i+1] - str_off[i];
  for (i = 0; i < str_count; ++i)
    str_pool[i] = str_buffer + str_off[i];
  xfree (str_off);
}


static void load_output (long pos)
{
  unsigned count, name, i, target;
  u32 pos_out;

  target = find_string (output_name);
  fseek (dat_file, pos, SEEK_SET);
  count = read_u16 ();
  for (i = 0; i < count; ++i)
    {
      name = read_u16 ();
      if (name == target)
        break;
    }
  if (i >= count)
    error (EXIT_ARG, "Output table <%s> not defined", output_name);
  fseek (dat_file, pos + 2 + 2 * count + 4 * i, SEEK_SET);
  pos_out = read_u32 ();
  fseek (dat_file, pos_out, SEEK_SET);
  output_count = read_u16 ();
  ALLOCARRAY (output_table, output_count * 2);
  for (i = 0; i < output_count * 2; ++i)
    output_table[i] = read_u16 ();
  for (i = 0; i <= OUTPUT_HASH_SIZE; ++i)
    output_hash[i] = read_u16 ();
}


static void load_conv (long pos)
{
  unsigned count, name, i, target, prod_size, trans_count, state_count;
  unsigned prod_count, *prod_mem, *prod_offsets;
  u32 pos_dfa;

  target = find_string (conversion_name);
  fseek (dat_file, pos, SEEK_SET);
  count = read_u16 ();
  for (i = 0; i < count; ++i)
    {
      name = read_u16 ();
      if (name == target)
        break;
    }
  if (i >= count)
    error (EXIT_ARG, "Conversion table <%s> not defined", conversion_name);
  fseek (dat_file, pos + 2 + 2 * count + 4 * i, SEEK_SET);
  pos_dfa = read_u32 ();
  fseek (dat_file, pos_dfa, SEEK_SET);
  state_count = read_u16 ();
  prod_count = read_u16 ();
  prod_size = read_u16 ();
  trans_count = read_u16 ();
  ALLOCARRAY (prod_offsets, prod_count + 1);
  ALLOCARRAY (prod_mem, prod_size);
  ALLOCARRAY (prod_table, prod_count);
  ALLOCARRAY (trans_table, trans_count);
  ALLOCARRAY (def_table, state_count);
  for (i = 0; i <= prod_count; ++i)
    prod_offsets[i] = read_u16 ();
  for (i = 0; i <= state_count; ++i)
    read_u16 ();                /* trans_off[] -- not hashed*/
  for (i = 0; i < prod_size; ++i)
    prod_mem[i] = read_u16 ();
  for (i = 0; i < state_count; ++i)
    def_table[i] = read_u16 ();
  for (i = 0; i < trans_count; ++i)
    {
      read_u16 ();              /* trans_table[].input */
      read_u16 ();              /* trans_table[].output */
    }
  for (i = 0; i < trans_count; ++i)
    {
      trans_table[i].state = read_u16 ();
      trans_table[i].input = read_u16 ();
      trans_table[i].output = read_u16 ();
    }
  for (i = 0; i <= DFA_HASH_SIZE; ++i)
    dfa_hash[i] = read_u16 ();
  for (i = 0; i < prod_count; ++i)
    {
      prod_table[i].table = &prod_mem[prod_offsets[i]];
      prod_table[i].count = prod_offsets[i+1] - prod_offsets[i];
    }
  xfree (prod_offsets);
}


static void load_letters (long pos)
{
  unsigned i, count, letter;

  ALLOCARRAY (letters_table, str_count);
  memset (letters_table, 0, str_count);
  fseek (dat_file, pos, SEEK_SET);
  count = read_u16 ();
  for (i = 0; i < count; ++i)
    {
      letter = read_u16 ();
      if (letter >= str_count)
        error (EXIT_FILE, "Invalid letters table");
      letters_table[letter] = 1;
    }
}


static struct encoding *load_encoding (long pos)
{
  unsigned chars, size, i;
  unsigned *offsets, *symbols;
  struct encoding *enc;

  ALLOC (enc);
  for (i = 0; i < CHAR_CODES; ++i)
    {
      enc->lengths[i] = 0;
      enc->symbols[i] = NULL;
    }
  fseek (dat_file, pos, SEEK_SET);
  chars = read_u16 ();
  size = read_u16 ();
  ALLOCARRAY (offsets, chars + 1);
  ALLOCARRAY (symbols, size);
  for (i = 0; i <= chars; ++i)
    offsets[i] = read_u16 ();
  for (i = 0; i < size; ++i)
    symbols[i] = read_u16 ();
  for (i = 0; i < chars; ++i)
    {
      enc->symbols[i] = &symbols[offsets[i]];
      enc->lengths[i] = offsets[i+1] - offsets[i];
    }
  xfree (offsets);
  return (enc);
}


static void load_layout (long pos, struct font_or_layout **ptable,
                         unsigned *pcount)
{
  unsigned i, count;
  u32 *pos_table;
  struct font_or_layout *table;

  fseek (dat_file, pos, SEEK_SET);
  count = read_u16 ();
  ALLOCARRAY (table, count);
  ALLOCARRAY (pos_table, count);
  for (i = 0; i < count; ++i)
    table[i].name = read_u16 ();
  for (i = 0; i < count; ++i)
    pos_table[i] = read_u32 ();
  for (i = 0; i < count; ++i)
    table[i].encoding = load_encoding (pos_table[i]);
  xfree (pos_table);
  *ptable = table;
  *pcount = count;
}


static void load_data_file (void)
{
  static char header[] = {'d', 's', 'b', 1};
  char tmp[4];
  char *dat_path;
  char *dir;
  char emtex_dir[DNAME_SIZE];
  u32 pos_str, pos_conv, pos_font, pos_layout, pos_output, pos_letters;

  //dat_path=xmalloc(DNAME_SIZE);
  kpse_path_element(dat_fname);
  if (!kpse_path_element(NULL))
  {
    dir=getenv("EMTEXDIR");
    if (dir)
    {
      if (strlen(dir)>DNAME_SIZE-6)
      {
	error(EXIT_FILE,"Path to dsb file is too long");
      }
      sprintf(emtex_dir,"%s/data",dir);
    }
    else
    {
      sprintf(emtex_dir,"/emtex/data");
    }
    fndefext (dat_fname, "dsb");
    dat_path=kpse_path_search(emtex_dir,dat_fname,FALSE);
    if (!dat_path)
    {
      dat_path=(char*)xmalloc(strlen(dat_fname));
      strcpy(dat_path,dat_fname);
    }
  }
  else
  {
    dat_path=(char*)xmalloc(strlen(dat_fname));
    strcpy(dat_path,dat_fname);
    fndefext (dat_path, "dsb");
  }

  dat_file=kpse_open_file(dat_path,kpse_base_format);
  if (dat_file == NULL)
    error (EXIT_FILE, "Cannot open data file `%s'", dat_fname);
  read_bytes (tmp, 4); 
  if (memcmp (tmp, header, 3) != 0)
    error (EXIT_FILE, "`%s' is not a binary dvispell parameter file",
           dat_fname);
  if (tmp[3] != header[3])
    error (EXIT_FILE, "Unsupported data file version");

  pos_str = read_u32 ();
  pos_conv = read_u32 ();
  pos_font = read_u32 ();
  pos_layout = read_u32 ();
  pos_output = read_u32 ();
  pos_letters = read_u32 ();

  load_str (pos_str);
  load_output (pos_output);
  load_conv (pos_conv);
  load_letters (pos_letters);
  load_layout (pos_font, &font_table, &font_count);
  load_layout (pos_layout, &layout_table, &layout_count);

  fclose (dat_file);
}


int main (int argc, char *argv[])
{
  /* Parse the command line arguments. */

  get_args (argc, argv);

  /* Initialize variables. */

  start (argv[0]);

  /* Read the binary parameter file. */

  load_data_file ();

  /* Read and convert the DVI file. */

  dvi_read ();

  /* Terminate the last line. */

  if (verbose && column != 0)
    {
      fputc ('\n', stderr);
      column = 0;
    }

  /* Close the files. */

  fclose (dvi_file);
  fflush (output_file);
  if (ferror (output_file))
    output_error ();
  fclose (output_file);

  /* Done. */

  return (EXIT_OK);
}
