/*
  This file is part of the CJK package for LaTeX2e.
 
  This file is part of the NTU2CJK package.
 
  $Id: mkmap.c,v 1.10 1997/12/28 02:25:50 d791013 Exp $
 
  Generate the Fontmap and the psfonts.map typed files. Fontmap-like output
  is temporarily used with Ghostscript, while NTU2CJK is making PostScript
  fonts for CJK package. psfonts.map-like output is used when invoking
  DVIPS. Alternatively, the generated file can be appended to the official
  released Fontmap and psfonts.map to simulate printer resident PS fonts.
 
  Note: you'd better backup the original Fontmap and psfonts.map.
*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <regex.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>

#define BUFLEN (7*1024)		/* reading buffer for matching */
#define RALEN  1024		/* buffer size for reading ahead */
#define NMATCH 1		/* we expect at most one set of match only */

int pmc = 0, download = 0;
char *xfontfile = NULL;
char *fontname = NULL;
char *fontfile = NULL;
char *psfontmap = NULL;
char *gsfontmap = NULL;
char *prog;

int
Usage () 
{
  fprintf(stderr, "
  Usage:
    %s [-p] [-d] [-x extend]
        [-F fontname-prefix] [-f fontfile-prefix]
        [-m psfontmap] [-F gsfontmap]

    -p  font sequential number in hexidecimal format (pmC font encoding).
    -d  (for psfonts.map) setup as downloadable fonts.
    -x  fontfile prefix of extended font.
    -f  fontfile prefix
    -F  fontname prefix
    -m  psfont.map-like output
    -M  Fontmap-like output

  Example:
    %s -F FunSong -f b5fs -m CJK.map -M Fontmap\n",
          prog, prog);
  exit (1);
}

void
fatal (const char *msg, ...) 
{
  va_list ap;
  va_start (ap, msg);
  vfprintf (stderr, msg, ap);
  va_end (ap);
  exit (-1);
}

void *
xmalloc (size_t size)
{
  register void *value = malloc (size);
  if (value == 0)
    fatal ("virtual memory exhausted");
  return value;
}

void *
xrealloc (void *ptr, size_t size)
{
  register void *value = realloc (ptr, size);
  if (value == 0)
    fatal ("Virtual memory exhausted");
  return value;
}

int
my_asprintf (char **buffer, const char *tmpl, ...)
{
  int nchars;
  size_t size = 16;
  va_list args;

  va_start (args, tmpl);
  while (1) {
    *buffer = (char *) xrealloc (*buffer, size);
    /* Try to print in the allocated space. */
    nchars = vsnprintf (*buffer, size, tmpl, args);
    /* If that worked, return the string. */
    if (nchars < size) {
      va_end (args);
      return nchars;
    }
    /* Else try again with twice as much space. */
    size *= 2;
  }
}

int 
erase_match (mapfile, patt_bgn, patt_end, check_only, cflags)
     const char *mapfile;
     char *patt_bgn; /* = "/KaiSuA1 " , e.g. */
     char *patt_end; /* = "\\(b5kaFE.pfb\\).*;" */
     int check_only;
     int cflags;
{
  FILE       *in, *out = NULL;
  regex_t    *cpatt_bgn = xmalloc (sizeof(regex_t));
  regex_t    *cpatt_end = xmalloc (sizeof(regex_t));
  regex_t    *cpatt = cpatt_bgn;
  size_t      buflen = BUFLEN + RALEN;
  char       *buffer = xmalloc (buflen);
  size_t      nmatch = NMATCH;
  regmatch_t *matchptr = xmalloc (nmatch * sizeof(regmatch_t));
  size_t      nread = 0;
  ssize_t     pos = 0;
  int errcode, hitted = -1, found = 0;

  /* compile pattern */
  errcode = regcomp (cpatt_bgn, patt_bgn, cflags);
  if (errcode!=0) {
    (void) regerror (errcode, cpatt_bgn, buffer, BUFLEN);
    printf ("%s\n",buffer);
    return 1;
  }
  errcode = regcomp (cpatt_end, patt_end, cflags);
  if (errcode!=0) {
    (void) regerror (errcode, cpatt_end, buffer, BUFLEN);
    printf ("%s\n",buffer);
    return 1;
  }
  
  if ((in = fopen (mapfile, "r")) == NULL) {
    fprintf (stderr, "Warning: %s: doesn't exist or open error!\n", mapfile);
    return 1;
  }
  if (check_only == 0) 
    if ((out = fopen (mapfile, "r+")) == NULL) 
      fatal ("Error: %s: writing error!\n", mapfile);

  /* initial value */
  memset (buffer, 0, buflen);

  while (1) {
    errcode = regexec (cpatt, buffer+pos, nmatch, matchptr, 0);
    if (pos+matchptr->rm_so < BUFLEN && errcode == 0) {
      if (check_only == 0) {
	if (hitted < 0) {
	  fwrite (buffer+pos, sizeof(char), matchptr->rm_so, out);
	  cpatt = cpatt_end;
	} else {
	  cpatt = cpatt_bgn;
	  found = 1;
	}
      } else 
	if (hitted > 0) {
	  found = 1;
	  break;
	}
      pos += hitted<0 ? matchptr->rm_so : matchptr->rm_eo+1;
      if (pos > BUFLEN) pos = BUFLEN;
      hitted = -hitted;
    } else {
      if (check_only == 0 && hitted < 0)
	fwrite (buffer+pos, sizeof(char), nread-pos, out);
      if (feof(in) != 0) break;
      memset (buffer, 0, buflen);
      nread = fread(buffer, sizeof(char), buflen, in);
      if (nread > BUFLEN) {
	fseek (in, (long)(BUFLEN-nread), SEEK_CUR);
	nread = BUFLEN;
      }
      *(buffer+buflen-1) = '\0';
      pos = 0;
    }
  }
  
  fclose (in);
  if (check_only == 0) {
    if (found) ftruncate (fileno(out), ftell(out));
    fclose (out);
  }
  
  regfree (cpatt_bgn);
  regfree (cpatt_end);

  free (cpatt_bgn);
  free (cpatt_end);
  free (buffer);
  free (matchptr);

  return found;
}

char *
Fontmap_patt (int plane) 
{
  char *str = NULL;
  char *fmt = (pmc==0? "/%s%02d( |\t)+\\([^ ]*\\)( |\t)+;" :
                       "/%s%02X( |\t)+\\([^ ]*\\)( |\t)+;");
  my_asprintf(&str, fmt, fontname, plane);
  return str;
}

int 
Fontmap_lines (const char *mapfile) 
{
  FILE *map = fopen (mapfile, "a");
  char *fmt = (pmc==0 ? 
               "/%s%02d (%s%02d.pfb) ;\n" :
               "/%s%02X (%s%02x.pfb) ;\n");
  int plane0 = (pmc==0 ? 1 : 0xA1);
  int plane1 = (pmc==0 ? 55: 0xFE);
  int iplane;

  if (map == NULL) 
    fatal ("Open error: %s!", mapfile);

  for (iplane=plane0; iplane <= plane1; iplane++)
    fprintf (map, fmt, fontname, iplane, fontfile, iplane);

  fclose (map);
  return 0;
}

char *
psmap_patt (const char *font, const char *file, int plane) 
{
  char *str = NULL;
  char *fmt = (pmc==0 ? "^%s%02d( |\t)+%s%02d.*$" :
                        "^%s(%02x|%02X)( |\t)+%s%02X.*$");
  if (pmc==0)
    my_asprintf(&str, fmt, file, plane, font, plane);
  else
    my_asprintf(&str, fmt, file, plane, plane, font, plane);
  return str;
}

void
psfonts_line (FILE *map, char *fmt, int plane) {
  fprintf (map, fmt, fontfile, plane, fontname, plane); 
}

void
partial_psfonts_line (FILE *map, char *fmt, int plane) {
  fprintf (map, fmt, fontfile, plane, fontname, plane, fontfile, plane); 
}

void
extend_psfonts_line (FILE *map, char *fmt, int plane) { 
  fprintf (map, fmt, xfontfile, plane, fontname, plane); 
}

void
extend_partial_psfonts_line (FILE *map, char *fmt, int plane) {
  fprintf (map, fmt, xfontfile, plane, fontname, plane, fontfile, plane); 
}

int
psmap_lines (const char *mapfile)
{
  FILE *map = fopen (mapfile, "a");
  char *fmt = NULL, *tfmt = "%%s%%02%c %%s%%02%c%s%c%s";
  int plane0, plane1, iplane, pfmt, Pfmt;
  void (*line) (FILE *, char *, int);
  
  if (map == NULL) 
    fatal ("Open error: %s!", mapfile);

  if (pmc == 0) {
    plane0 = 1;
    plane1 = 55;
    pfmt = 'd';
    Pfmt = 'd';
  } else {
    plane0 = 0xA1;
    plane1 = 0xFE;
    pfmt = 'x';
    Pfmt = 'X';
  }
  if (download == 1) {
    line = partial_psfonts_line;
    my_asprintf (&fmt, tfmt, pfmt, Pfmt, " <%s%02", pfmt, ".pfb\n");
  } else {
    line = psfonts_line;
    my_asprintf (&fmt, tfmt, pfmt, Pfmt, "\n", 0, 0);
  }

  for (iplane=plane0; iplane<=plane1; iplane++)
    (*line) (map, fmt, iplane);
  
  if (xfontfile != NULL) {
    if (download == 1) {
      line = extend_partial_psfonts_line;
      my_asprintf (&fmt, tfmt, pfmt, Pfmt,
                " \"1.2 ExtendFont\" <%s%02", pfmt, ".pfb\n");
    } else {
      line = extend_psfonts_line;
      my_asprintf (&fmt, tfmt, pfmt, Pfmt,
                " \"1.2 ExtendFont\"\n", 0, 0);
    }

    for (iplane=plane0; iplane <= plane1; iplane++)
      (*line) (map, fmt, iplane);
  }

  fclose (map);
  return 0;
}

int
main (int argc, char * argv[]) 
  {
   
  if ((prog = strrchr(argv[0], '/')))
    prog++;
  else
    prog = argv[0];

  if (argc <= 1) Usage ();
  while (1)
    {
    char opt;

    opt = getopt(argc,argv,"pdx:f:F:m:M:");
    if (opt==EOF) break;

    switch (opt) 
      {
      case 'p': pmc = 1; break;
      case 'd': download = 1; break;
      case 'x': xfontfile = optarg; break;
      case 'f': fontfile = optarg; break;
      case 'F': fontname = optarg; break;
      case 'm': psfontmap = optarg; break;
      case 'M': gsfontmap = optarg; break;
      case '?': Usage();
      case ':': Usage();
      default: abort();
      }
  }

  /* generate Fontmap-like map */
  if (gsfontmap != NULL) {
    erase_match(gsfontmap,
                Fontmap_patt(pmc==0?1:0xA1),
                Fontmap_patt(pmc==0?55:0xFE),
                0, REG_EXTENDED);
    Fontmap_lines (gsfontmap);
  }
  
  /* psfonts.map-like map */
  if (psfontmap != NULL) {
    erase_match (psfontmap, 
                 psmap_patt(fontname, fontfile, pmc==0?1:0xA1),
                 psmap_patt(fontname, fontfile, pmc==0?55:0xFE),
                 0, REG_NEWLINE|REG_EXTENDED);
    if (xfontfile!=NULL) 
      erase_match (psfontmap, 
                   psmap_patt(fontname, xfontfile, pmc==0?1:0xA1),
                   psmap_patt(fontname, xfontfile, pmc==0?55:0xFE),
                   0, REG_NEWLINE|REG_EXTENDED);
    psmap_lines (psfontmap);
  }

  return 0;
}
