/* Copyright (C) 2005-2008 Damien Stehle.
Copyright (C) 2007 David Cade.

This file is part of the fplll Library.

The fplll Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version.

The fplll Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with the fplll Library; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA. */


#ifndef UTIL_H
#define UTIL_H

#include <cstdio>
#include <gmp.h>
#include <mpfr.h>

#include "defs.h"
#include "nr.h"
#include "vect.h"
#include "matrix.h"

#ifdef VERBOSE
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#endif

/* this trick will not work on 16-bit machines*/
#if (LONG_MAX==2147483647L)
#define CPU_32
#define CPU_SIZE 32
#define CPU_SIZE_1 31
#define MAX_LONG 0x1p30
#define MAX_LONG_FAST 0x1p31
#define EXPO_MAX 30
#define EXPO_MAX2 30
#define MAX_EXP_1 30
#define MAX_EXP_2 15
#else
#define CPU_64
#define CPU_SIZE 64
#define MAX_LONG 0x1p53
#define MAX_LONG_FAST 0x1p53
#define EXPO_MAX 53
#define CPU_SIZE_1 53
#define EXPO_MAX2 62
#define MAX_EXP_1 62
#define MAX_EXP_2 31
#endif
#define MAX_DIM_DOUBLE 106
#define MAX_EXP_DOUBLE 500
#define PREC_DOUBLE 53

#define SIZE_RED_FAILURE_THRESH 5

// Default floating-point and integer types

#ifndef FLOAT_T_IS_DOUBLE
  typedef mpfr_t FloatT;
#else
  typedef double FloatT;
#endif
typedef FP_NR<FloatT> Float;
typedef Vect<Float> FloatVect;
typedef FP_mat<FloatT> FloatMatrix;

#ifndef INTEGER_T_IS_DOUBLE
  typedef mpz_t IntegerT;
#else
  typedef double IntegerT;
#endif
typedef Z_NR<IntegerT> Integer;
typedef Vect<Integer> IntVect;
typedef ZZ_mat<IntegerT> IntMatrix;

/*vectors*/

template<class ZT> inline void ScalarProduct(Z_NR<ZT>& s,Z_NR<ZT>*vec1,Z_NR<ZT>*vec2,int n)
{
  Z_NR<ZT> tmp;
  s.mul(vec1[0],vec2[0]);

  for (int i=1;i<n;i++)
    {
      tmp.mul(vec1[i],vec2[i]);
      s.add(s,tmp);
    }
}

/* computes the prec MSBs of the inner product of the approximate 
   vec1 and vec2.
   If too many bits are lost, then it returns 0, otherwise 1.
 */


template<class FT> inline FT 
fpScalarProduct (FT *vec1, FT *vec2, int n)
{
  int i;
  FT sum;

  sum = vec1[0] * vec2[0];
  for (i=1; i<n; i++)
    sum += vec1[i] * vec2[i];

  return sum;
} 


template<class FT> inline FT 
fpNorm (FT *vec, int n)
{
  int i;
  FT sum;

  sum = vec[0] * vec[0];
  for (i = 1 ; i < n ; i++)
    sum += vec[i]*vec[i];

  return sum;

} 


template<class FT> inline int fpScalarProduct (FP_NR<FT>& s, FP_NR<FT> *vec1, FP_NR<FT> *vec2, int n)
{
  int i, res;
  FP_NR<FT> tmp;
#ifdef CANCELLATIONS
  FP_NR<FT> tmp2;
#endif

  s.mul(vec1[0], vec2[0]);

  for (i=1; i<n; i++)
    {
      tmp.mul(vec1[i], vec2[i]);
      s.add(s, tmp);
    }

#ifdef CANCELLATIONS
  fpNorm (tmp, vec1, n);
  fpNorm (tmp2, vec2, n);
  tmp.mul(tmp, tmp2);
  tmp.div_2ui(tmp, 70);
  tmp2.mul(s, s);

  if (tmp2.cmp(tmp)<=0)
    res = 0;
  else
#endif
    res = 1;

  return res;
} 


template<class FT> inline void fpNorm (FP_NR<FT>& s, FP_NR<FT> *vec, int n)
{
  int i;
  FP_NR<FT> tmp;
  
  s.mul(vec[0], vec[0]);
  
  for (i=1; i<n; i++)
    {
      tmp.mul(vec[i], vec[i]);
      s.add(s, tmp);
    }
} 




/*template<class ZT,class FT> inline void set_fz(FP_NR<FT>& a,Z_NR<ZT>& b)
{
  a=static_cast<FP_NR<FT> >(b);
} 
inline void set_fz(FP_NR<double>& a,Z_NR<mpz_t>& b)
{
  a.set(mpz_get_d(b.GetData()));
}
inline void set_fz(FP_NR<dpe_t>& a,Z_NR<mpz_t>& b)
{
  dpe_set_z(a.GetData(),b.GetData());
}
inline void set_fz(FP_NR<mpfr_t>& a,Z_NR<mpz_t>& b)
{
  mpfr_set_z(a.GetData(),b.GetData(),GMP_RNDN);
}
inline void set_fz(FP_NR<double>& a,Z_NR<long int>& b)
{
  a.set((double)b.GetData());
}
inline void set_fz(FP_NR<dpe_t>& a,Z_NR<long int>& b)
{
  dpe_set_si(a.GetData(),b.GetData());
}
inline void set_fz(FP_NR<mpfr_t>& a,Z_NR<long int>& b)
{
  mpfr_set_si(a.GetData(),b.GetData(),GMP_RNDN);
}
inline void set_fz(FP_NR<double>& a,Z_NR<double>& b)
{
  a.set(b.GetData());
}
inline void set_fz(FP_NR<dpe_t>& a,Z_NR<double>& b)
{
  dpe_set_d(a.GetData(),b.GetData());
}
inline void set_fz(FP_NR<mpfr_t>& a,Z_NR<double>& b)
{
  mpfr_set_d(a.GetData(),b.GetData(),GMP_RNDN);
}*/


inline long int get_z_exp(FP_NR<double>::assoc_int& a,FP_NR<double>& b)
{
  long int expo = ilogb (b.GetData());
  
  if (expo < EXPO_MAX)
    {
      expo = 0;
      a.set(static_cast<long int>(b.GetData()));
    }
  else
    {
      expo -= EXPO_MAX;
      a.set(static_cast<long int>(ldexp(b.GetData(),-expo)));
    }
  return expo;
}
inline long int get_z_exp(FP_NR<dpe_t>::assoc_int&a,FP_NR<dpe_t>& b)
{
  int expo = dpe_get_si_exp(&(a.GetData()),b.GetData());
  if (expo<0)
    {
      a.GetData() = (long int) ldexp ((double)a.GetData(), expo);
      return 0;
    }
  return expo;
}
inline long int get_z_exp(FP_NR<mpfr_t>::assoc_int& a,FP_NR<mpfr_t>& b)
{
  int expo = mpfr_get_z_exp(a.GetData(),b.GetData());
  if (expo<0)
    {
      mpz_div_2exp(a.GetData(),a.GetData(),-expo);
      return 0;
    }
  return expo;


}




/* Converts an FP_NR into a Z_NR */
/*template<class FT, class ZT> inline void set_zf(Z_NR<ZT>& a, FP_NR<FT>& b)
{
  a=static_cast<Z_NR<ZT> >(b);
} 
inline void set_zf(Z_NR<mpz_t>& a, FP_NR<double>& b)
{
  mpz_t ztmp;
  mpz_set_d (ztmp, b.GetData());
  a.set (ztmp);
  mpz_clear (ztmp);
}
inline void set_zf(Z_NR<mpz_t>& a, FP_NR<dpe_t>& b)
{
  dpe_get_z(a.GetData(), b.GetData());
}
inline void set_zf(Z_NR<mpz_t>& a, FP_NR<mpfr_t>& b)
{
  mpfr_get_z(a.GetData(), b.GetData(),GMP_RNDN);
}
inline void set_zf(Z_NR<long int>& a, FP_NR<double>& b)
{
  a.set((long int) b.GetData());
}
inline void set_zf(Z_NR<long int>& a, FP_NR<dpe_t>& b)
{
  a.set(dpe_get_si(b.GetData()));
}
inline void set_zf(Z_NR<long int>& a, FP_NR<mpfr_t>& b)
{
  a.set(mpfr_get_si(b.GetData(),GMP_RNDN));
}*/





template<class FT> inline
void
Print_matf (FT **B, int *expo, int d, int n)
{
  int i, j;

  printf("[");
  for (i=0;i<d;i++) {
    printf("[");
    for (j=0;j<n;j++) {
#ifdef FORCE_LONGDOUBLE
      printf("%1.5Le", B[i][j]);
#else
      printf("%E", B[i][j]);
#endif
      if (j < n-1) printf(" ");
    }
    printf("] * 2^%d\n", expo[i]);
  }
  printf("]\n");
}

template<class ZT> inline int
set_line (double *appv, Z_NR<ZT> *v, int n)
{
  signed long int *exp, i, maxexp = 0;
  exp = new signed long int[n];


  for (i=0; i<n; i++)
    {
      appv[i]=v[i].get_d_2exp(exp+i);
      if (exp[i]>maxexp) maxexp=exp[i];
    }

  for (i=0; i<n; i++)
    appv[i] = ldexp( appv[i], exp[i]-maxexp);

  delete[] exp;
  return maxexp;
}


template <class FT> inline
void
clear_matrixf (FT **B, int d)
{
  int i;
  for (i=0; i<d; i++)
    delete[] B[i];
  delete[] B;
}



#ifdef VERBOSE
inline int
cputime ()
{
  struct rusage rus;

  getrusage (RUSAGE_SELF, &rus);
  return rus.ru_utime.tv_sec * 1000 + rus.ru_utime.tv_usec / 1000;
}
#endif


#endif
