/* Copyright (C) 2005-2008 Damien Stehle.
Copyright (C) 2007 David Cade.

This file is part of the fplll Library.

The fplll Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version.

The fplll Library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with the fplll Library; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA. */


#ifndef HEURISTIC_CPP
#define HEURISTIC_CPP

#if 0
#define MUDISTRIB
#else
#define MUDISTRIB_LOVASZ
#endif

/************************************/
/* ******************************** */
/* Babai's Nearest Plane algorithm  */
/* ******************************** */
/************************************/

/* Size-reduces b_kappa using mu_ij and r_ij for j<=i <kappa
updates B (kappa)
computes mu_kappaj, r_kappaj for j<=kappa, and s(kappa) 
The algorithm is the iterative Babai algorithm of the paper.
*/

template <class ZT,class FT>
inline int heuristic<ZT,FT>::BabaiCall(int* alpha, int zeros, int kappamax, int var_k,
				       Z_NR<ZT>& ztmp, FP_NR<FT>& tmp, 
				       FP_NR<FT>& rtmp, FP_NR<FT>&max,
				       FP_NR<FT>& max2, FP_NR<FT>&max3,
				       int& newvec,int& newvecmax,int n)
{
  return Babai (alpha[kappa], zeros, kappamax, var_k<=n?var_k:n, 
		  ztmp, tmp, rtmp,max,max2,max3,kappa);
}

template <class ZT,class FT>
inline void heuristic<ZT,FT>::GSO(int a, int zeros, int kappamax, int n,
				  Z_NR<ZT>& ztmp, FP_NR<FT>& tmp, 
				  FP_NR<FT>& rtmp,FP_NR<FT>& max, int aa,int red)
{
  max.set(0.0);
  for (int j=aa; j<kappa; j++)
    {
      if (appSP->Get(kappa,j).is_nan()) 
	{
	  if (!(fpScalarProduct (appSP->Get(kappa,j), 
				 appB->GetVec(kappa), 
				 appB->GetVec(j), n)))
	    {
	      ScalarProduct (ztmp, B->GetVec(kappa), B->GetVec(j), n);
	      appSP->Get(kappa,j).set_z(ztmp);
	    }
	}
      
  
#ifdef DEBUG
      if ((kappa==3)&&(j==2)){
	  printf("\n          j is %d\n", j);         
	  printf("          appSP[%d][%d] is: ", kappa, j); 
	  appSP->Get(kappa,j).print(); 
	  printf(", which approximates ");
	  ScalarProduct (ztmp, B->GetVec(kappa), B->GetVec(j), n);
	  printf("\n");
	  B->print( kappa+1, n);
	  ztmp.print(); 
	  printf("\n          Norm of B[%d]^2: ", j);
	  ScalarProduct (ztmp, B->GetVec(j), B->GetVec(j), n);
	  ztmp.print();  
	  printf("\n          Norm of B[%d]^2: ",kappa);
	  ScalarProduct (ztmp, B->GetVec(kappa), B->GetVec(kappa), n);
	  ztmp.print(); 
      }
#endif
      
      if (j > zeros+2)
	{
	  tmp.mul(mu->Get(j,zeros+1), r->Get(kappa,zeros+1));
	  rtmp.sub(appSP->Get(kappa,j), tmp);

	  for (int k=zeros+2; k<j-1; k++)
	    {
	      tmp.mul(mu->Get(j,k), r->Get(kappa,k));
	      rtmp.sub(rtmp, tmp);
	    }

	  tmp.mul(mu->Get(j,j-1), r->Get(kappa,j-1));


	  r->Get(kappa,j).sub(rtmp, tmp);

#if 0	  
	  if ((kappa==3)&&(j==2)){ 
	    rtmp.print(); cout <<" - ";
	    tmp.print(); cout <<" = ";
	    r->Get(kappa,j).print(); cout << "\n";
	  }
#endif

	}
      else if (j==zeros+2)
	{         
	  tmp.mul(mu->Get(j,zeros+1), r->Get(kappa,zeros+1));
	  r->Get(kappa,j).sub(appSP->Get(kappa,j), tmp);
	}
      else r->Get(kappa,j).set(appSP->Get(kappa,j));
      
      mu->Get(kappa,j).div( r->Get(kappa,j), r->Get(j,j));
      rtmp.abs(mu->Get(kappa,j));

      if (max.cmp(rtmp)<0)  
	max.set(rtmp);
    }
}

template<class ZT,class FT> int 
heuristic<ZT,FT>::Babai (int a, int zeros, int kappamax, int n,
			 Z_NR<ZT>& ztmp, FP_NR<FT>& tmp, FP_NR<FT>& rtmp,
			 FP_NR<FT>&max,FP_NR<FT>& max2,FP_NR<FT>&max3,int red)

{
  int i, j, k, test, aa, sg, ll;
  signed long xx;
  long int expo;

#ifdef DEBUG
  int loops=0;
#endif 

  typename FP_NR<FT>::assoc_int X;


  aa = (a > zeros) ? a : zeros+1;

#ifdef DEBUG
  printf("\nappSP: \n");
  appSP->print(kappamax+1, kappamax+1);
  printf("\nr: \n");
  r->print(kappamax+1, kappamax+1);
  printf("\n          STARTING BABAI WITH k=%d\n", kappa);
  printf("\nappB: \n");
  appB->print(kappamax+1, n);
  loops = 0;
  printf("\nmu: \n");
  mu->print(kappamax+1, kappamax+1);
  printf ("\n\na is %d, zeros is %d, aa is %d\n", a, zeros, aa);
  B->print(kappamax+1, n);
#endif

  ll=0;
  do
    {
      ll++;
      test=0;

#ifdef DEBUG      
      if (loops++ > LOOPS_BABAI) 
        {         
          printf("INFINITE LOOP?\n");
          abort();
        }
#endif 


      /* ************************************** */
      /* Step2: compute the GSO for stage kappa */
      /* ************************************** */

      max3.set(max2);
      max2.set(max);

      GSO(a,zeros,kappamax,n,ztmp,tmp,rtmp,max,aa,red);

      if(ll>=3)
	{
	  rtmp.mul_2ui(max2, SIZE_RED_FAILURE_THRESH);

#ifdef DEBUG
	  cout << "\nrtmp="; rtmp.print();
	  cout<<"\nmax2="; max2.print();
	  cout<<"\nmax3="; max3.print();
	  cout<<"\n";
#endif
	  if( max3.is_nan() || max3.cmp(rtmp)<=0)
	    {
#ifdef VERBOSE
	      cerr << "unexpected behaviour -> exit\n";
#endif
	      return kappa;
	    }
	}

#ifdef DEBUG
      if (loops <=LOOPS_BABAI)
        {
          printf("\nmu :\n");
          mu->print(kappa+1, kappa+1);
          printf("\nr :\n");
          r->print(kappa+1, kappa+1);
        }
#endif

      /* **************************** */
      /* Step3--5: compute the X_j's  */
      /* **************************** */
      
      for (j=kappa-1; j>zeros; j--)
        {
          /* test of the relaxed size-reduction condition */
          
          tmp.abs(mu->Get(kappa,j));

#ifdef DEBUG
          if (loops<=LOOPS_BABAI)
            {
	      fflush(stdout);
	      fflush(stderr);
              printf( "tmp is : "); 
              tmp.print(); printf( "\n");
	      fflush(stdout);
	      fflush(stderr);
            }
#endif

          if ((tmp.cmp(halfplus)) > 0) 
            {
              test = 1; 
              /* we consider separately the case X = +-1 */     
              if ((tmp.cmp(onedotfive))<=0 )   
                {
#ifdef DEBUG
                  printf(" X is pm1\n");
#endif

                  sg =  mu->Get(kappa,j).sgn();
                  if ( sg >=0 )   /* in this case, X is 1 */
                    {
                      for (k=zeros+1; k<j; k++)
                        mu->Get(kappa,k).sub( 
                             mu->Get(kappa,k), 
                             mu->Get(j,k));
                      
                      for (i=0; i<n; i++)    
                        B->Get(kappa,i).sub( 
                              B->Get(kappa,i), 
                              B->Get(j,i));      
                    }
                  
                  else          /* otherwise X is -1 */ 
                    {
                      for (k=zeros+1; k<j; k++)
                        mu->Get(kappa,k).add( 
                             mu->Get(kappa,k), 
                             mu->Get(j,k));
                      
                      for (i=0; i<n; i++)    
                        B->Get(kappa,i).add(
                              B->Get(kappa,i), 
                              B->Get(j,i));
                    }
                }
              
              else   /* we must have |X| >= 2 */
                {
                  tmp.rnd(mu->Get(kappa,j));
                  
                  for (k=zeros+1; k<j; k++)
                    {
                      rtmp.mul(tmp, mu->Get(j,k));
                      mu->Get(kappa,k).sub( mu->Get(kappa,k), rtmp);
                    }
                  

                  if (tmp.exp() < CPU_SIZE-2)   
                    /* X is stored in a long signed int */                

                    {                 
                      xx=tmp.get_si();
                      
#ifdef DEBUG
                      if (loops<=LOOPS_BABAI)
                        {
                          printf("          xx[%d] is %ld\n", j, xx);
                          printf("          and tmp was ");
                          tmp.print(); printf("\n");
                        }
#endif
                      
                      for (i=0; i<n; i++)  
                        {
                          if (xx > 0)
                            {
                              B->Get(kappa,i).submul_ui(
                                          B->Get(j,i), 
                                          (unsigned long int) xx);
                            }
                          else
                            {
                              B->Get(kappa,i).addmul_ui( 
                                          B->Get(j,i), 
                                          (unsigned long int) -xx);
                            }
                        }
                    }


                  
                  else
                    {
                      expo = get_z_exp (X, tmp);
                      for (i=0; i<n; i++)  
                        {
			  ztmp.mul_2exp(B->Get(j,i), expo); 
			  B->Get(kappa,i).submul(ztmp,X);
			}
                    }
                }
            }
        }
      
      
      if (test)   /* Anything happened? */
        {
          for (i=0 ; i<n; i++)     /* update appB[kappa] */
            appB->Get(kappa,i).set_z(B->Get(kappa,i));      
          aa = zeros+1;
          for (i=zeros+1; i<=kappa; i++) 
            appSP->Get(kappa,i).set_nan();
          for (i=kappa+1; i<=kappamax; i++) 
            appSP->Get(i,kappa).set_nan();
        }

#ifdef DEBUG
      if (loops<=LOOPS_BABAI)
        {
          printf("          test is %d\n", test);
          printf("\nmu: \n");
          mu->print(kappa+1, kappa+1);
          printf("\nr: \n");
          r->print(kappa+1, kappa+1);
        }
#endif

    }
  while (test);



  if (appSP->Get(kappa,kappa).is_nan()) 
    {
      fpNorm (appSP->Get(kappa,kappa), appB->GetVec(kappa), n);
    }
  s[zeros+1].set(appSP->Get(kappa,kappa));
  

  /* the last s[kappa]=r[kappa,kappa] is computed only if kappa increases */
  for (k=zeros+1; k<kappa-1; k++)
    {
      tmp.mul( mu->Get(kappa,k), r->Get(kappa,k));
      s[k+1].sub(s[k], tmp);
    }
  r->Get(kappa,kappa).set(s[kappa-1]);



#ifdef DEBUG
  printf("          Number of loops is %d\n", loops);
#endif

  return 0;
}







/* ****************** */
/* ****************** */
/* ****************** */
/* The LLL Algorithm  */
/* ****************** */
/* ****************** */
/* ****************** */

/* LLL-reduces the integer matrix B "in place" */


template <class ZT, class FT>
int heuristic<ZT,FT>::LLL ()
{
  int kappa2, d, n, i, j, zeros, kappamax, test;
  FP_NR<FT>  *mutmp, *appBtmp, *appSPtmp;
  FP_NR<FT> tmp, rtmp;
  Z_NR<ZT> ztmp;
  Z_NR<ZT> *Btmp;
  FP_NR<FT> max,max2,max3;
  int newvec=0,newvecmax=1;
  int *alpha;
#ifdef VERBOSE
  int newkappa, start, lovasz, loops;
#endif
#ifdef USE_DPE
#ifdef FORCE_LONGDOUBLE
  long double tmp2, y;
#endif
#endif

  d = B->GetNumRows();
  n = B->GetNumCols();

  unsigned long long int iter=0;
  unsigned long long int maxIter=(unsigned long long)(d - 2*d*(d+1)*((B->getMaxExp()+3)/log (delta)));
 

#ifdef DEBUG
  cerr<<"d = "<<d<<", n="<<n<<"\n";
#endif


#ifdef VERBOSE
  cerr<<   "\nEntering fpLLL:\nLLL-reduction factors (";
  ctt.printerr();
  cerr<< ", ";
  halfplus.printerr(); 
  cerr<< ")\n";
  if (siegel) cerr <<"Using Siegel's condition\n";
  else cerr <<"Using Lovasz's condition\n";
#ifdef USE_MPFR
  fprintf (stderr, 
           "Working precision set to %d bits\n", 
           (int) mpfr_get_default_prec());
  if (mpfr_get_default_prec()<53) fprintf(stderr, "Be careful, your precision is particularly low");
#else
#ifdef USE_DPE
#ifdef FORCE_LONGDOUBLE
  tmp2 = 1.0;
  for (j = 0; (y = tmp2 + 1.0) != tmp2; j++, tmp2 = 2.0 * tmp2);
  fprintf ( stderr, 
            "Working precision set to %d bits.\n", j);
#else
  fprintf ( stderr, 
            "Working precision set to 53 bits.\n");
#endif
#endif
#endif
  fprintf (stderr, 
           "No efficiency nor correctness guarantee here.\nIf you want to be sure, use the guaranteed version.\n\n");
#endif

  alpha = new int[d]; 
 
  mu=new FP_mat<FT>(d, d);
  r =new FP_mat<FT>(d, d);
  
  appB = new FP_mat<FT>(d, n);
  appSP= new FP_mat<FT>(d, d);
  
  s = new FP_NR<FT>[d];
  appSPtmp = new FP_NR<FT>[d];
  
  for (i=0; i<d; i++)
    for (j=0; j<d; j++) 
      appSP->Get(i,j).set_nan();
  
  
  
  /* ************************** */
  /* Step1: Initialization Step */
  /* ************************** */     

#ifdef VERBOSE  
  start = cputime();
#endif
  

  for (i=0; i<d; i++)
    for (j=0; j<n; j++)
      appB->Get(i,j).set_z(B->Get(i,j));


#ifdef DEBUG  
  B->print(d, n);
  appB->print(d, n);
  printf("Step 1 is over\n");
#endif  
  

  /* ********************************* */
  /* Step2: Initializing the main loop */
  /* ********************************* */   


#ifdef VERBOSE  
  newkappa = 0;
  loops = lovasz = 0;
#endif

  kappamax = 0;
  i = 0; 
  
  do
    fpNorm (appSP->Get(i,i), appB->GetVec(i), n); 
  while (appSP->Get(i,i).sgn() == 0  && ++i <d);
  zeros = i-1; /* all vectors B[i] with i <= zeros are zero vectors */
  kappa = i+1;

  if (zeros < d-1) r->Get(i,i).set(appSP->Get(i,i));

  for (i=zeros+1; i<d; i++)
    alpha[i]=0;
  
#ifdef DEBUG
  printf("Step 2 is over\n"); 
#endif  
  
  while (kappa < d)
    {
      
      if (kappa>kappamax) { kappamax++;newvec++;}
      
#ifdef VERBOSE
      loops++;
      if (kappa>newkappa)
        {
          newkappa++;
          fprintf (stderr, 
                   "Discovering vector k = %d, iterations = %d ", 
                   kappa+1, loops);
          fprintf(stderr, "cputime is %d ms\n", (cputime()-start));
        }
#endif
      

#ifdef DEBUG
      printf("alpha= ");
      for (i=zeros+1; i<d; i++) 
        printf("%d ",alpha[i]);
      printf("\n");
      printf("entering the while loop with k=%d\n", kappa);
      B->print(d, n);
#endif

    
      /* ********************************** */
      /* Step3: Call to the Babai algorithm */
      /* ********************************** */   


      if(BabaiCall (alpha, zeros, kappamax, kappamax+1+shift, 
		    ztmp, tmp, rtmp,max,max2,max3,newvec,newvecmax,n))
	{
	  delete[] alpha;
	  
	  delete mu;
	  delete r;
	  delete appB;
	  delete appSP;
	  
	  delete[] s;
	  delete[] appSPtmp;
	  
	  return kappa;
	}
      

#ifdef DEBUG      
      printf("Step 3 is over\n");   
      B->print(kappamax+1, n);
      appB->print(kappamax+1, n);
      printf("\nappSP: \n");
      appSP->print(kappamax+1, kappamax+1);
      printf("\nr: \n");
      r->print(kappamax+1, kappamax+1);
#endif
      
      /* ************************************ */
      /* Step4: Success of Lovasz's condition */
      /* ************************************ */  
      /* ctt * r.Get(kappa-1,kappa-1] <= s[kappa-1] ?? */
      
      tmp.mul(r->Get(kappa-1,kappa-1), ctt);
      
#ifdef DEBUG
      printf("s[%d] is ", kappa-1);
      s[kappa-1].print(); printf("\n");
      r->Get(kappa-1,kappa-1).print(); 
#endif
      
#ifdef VERBOSE
      lovasz++;
#endif

      if (siegel){
#ifdef MUDISTRIB_LOVASZ
	cout <<"[" << kappa << ",";
	mu->Get(kappa,kappa-1).print(); cout <<"],\n";
#endif
	tmp.mul(mu->Get(kappa,kappa-1), r->Get(kappa,kappa-1));
	r->Get(kappa,kappa).sub(s[kappa-1], tmp);
	if (tmp.cmp(r->Get(kappa,kappa)) <=0) 
	  test =1;
	else
	  test = 0;
      }

      if (!siegel){
	if( tmp.cmp(s[kappa-1]) <=0 ){
          tmp.mul(mu->Get(kappa,kappa-1), r->Get(kappa,kappa-1));
          r->Get(kappa,kappa).sub( s[kappa-1], tmp);
	  test = 1;
        } 
	else
	  test = 0;
      }


      if (test == 1){
	alpha[kappa] = kappa;
	kappa++;
      }
      else
        {
          
          /* ******************************************* */
          /* Step5: Find the right insertion index kappa */
          /* kappa2 remains the initial kappa            */
          /* ******************************************* */  
          
          
          kappa2 = kappa;

	  if (siegel){
	    do
	      {
#ifdef VERBOSE
		lovasz++;
#endif
		kappa--;
		if (kappa>zeros+1) tmp.mul(r->Get(kappa-1,kappa-1), ctt);
	      }
	    while (kappa>=zeros+2 && s[kappa].cmp(tmp) <=0 );
	  }
	  else{
	    do
	      {
#ifdef VERBOSE
		lovasz++;
#endif
		kappa--;
		if (kappa>zeros+1) tmp.mul(r->Get(kappa-1,kappa-1), ctt);
	      }
	    while (kappa>=zeros+2 && s[kappa-1].cmp(tmp) <=0 );
	  }

#ifdef DEBUG
          printf( "Index of insertion: %d \n", kappa);
          printf("Step 5 is over\n");
          printf("alpha= ");
          for (i=0; i<=kappamax; i++) 
            printf("%d ", alpha[i]);
          printf("\n");
#endif

          for (i=kappa; i<kappa2; i++)
            if (kappa <= alpha[i]) alpha[i] = kappa;

          for (i=kappa2; i>kappa; i--)
            alpha[i] = alpha[i-1];
          
          for (i=kappa2+1; i<=kappamax; i++)
            if (kappa < alpha[i]) alpha[i] = kappa;
          
          alpha[kappa] = kappa;
          
#ifdef DEBUG
          printf("alpha= ");
          for (i=0; i<=kappamax; i++) 
            printf("%d ", alpha[i]);
          printf("\n");
#endif
          
          /* ****************************** */
          /* Step6: Update the mu's and r's */
          /* ****************************** */  

          mutmp = mu->GetVec(kappa2);
          for (i=kappa2; i>kappa; i--)
            mu->GetVec(i) = mu->GetVec(i-1);
          mu->GetVec(kappa) = mutmp;

          mutmp = r->GetVec(kappa2);
          for (i=kappa2; i>kappa; i--)
            r->GetVec(i) = r->GetVec(i-1);
          r->GetVec(kappa) = mutmp;

          r->Get(kappa,kappa).set(s[kappa]);
          
#ifdef DEBUG 
          printf("Step 6 is over\n");
#endif
          
          /* ************************ */
          /* Step7: Update B and appB */
          /* ************************ */          
          
          Btmp = B->GetVec(kappa2);
          for (i=kappa2; i>kappa; i--)
            B->GetVec(i) = B->GetVec(i-1);
          B->GetVec(kappa) = Btmp;
          
          appBtmp = appB->GetVec(kappa2);
          for (i=kappa2; i>kappa; i--)
            appB->GetVec(i) = appB->GetVec(i-1);
          appB->GetVec(kappa) = appBtmp;
          
#ifdef DEBUG
          B->print(d, n);
          appB->print( d, n);
          printf("Step 7 is over\n");
#endif
          
          /* *************************** */
          /* Step8: Update appSP: tricky */
          /* *************************** */      
          
          for (i=0; i<=kappa2; i++)
            appSPtmp[i].set(appSP->Get(kappa2,i));
          
          for (i=kappa2+1; i<=kappamax; i++)
            appSPtmp[i].set(appSP->Get(i,kappa2));
          
          for (i=kappa2; i>kappa; i--)
            {
              for (j=0; j<kappa; j++)
                appSP->Get(i,j).set(appSP->Get(i-1,j));
              
              appSP->Get(i,kappa).set(appSPtmp[i-1]);
              
              for (j=kappa+1; j<=i; j++)
                appSP->Get(i,j).set(appSP->Get(i-1,j-1));
              
              for (j=kappa2+1; j<=kappamax; j++)
                appSP->Get(j,i).set(appSP->Get(j,i-1));     
            }
          
          for (i=0; i<kappa; i++)
            appSP->Get(kappa,i).set(appSPtmp[i]);
          
          appSP->Get(kappa,kappa).set(appSPtmp[kappa2]);
          
          for (i=kappa2+1; i<=kappamax; i++)
            appSP->Get(i,kappa).set(appSPtmp[i]);
          

          if (r->Get(kappa,kappa).sgn() <= 0)
            {
              zeros++;
              kappa++;
              fpNorm (appSP->Get(kappa,kappa), appB->GetVec(kappa), n);
              r->Get(kappa,kappa).set(appSP->Get(kappa,kappa));
            }
          
          kappa++;
          
#ifdef DEBUG      
          appB->print(kappamax+1, n);
          appSP->print( kappamax+1, kappamax+1);
          printf("Step 8 is over, new kappa=%d\n",kappa);
#endif
          
        }      
      iter++;
      if (iter> maxIter)
	{
	  cerr<<"Too many loop iterations";

	  delete[] alpha;
	  
	  delete mu;
	  delete r;
	  delete appB;
	  delete appSP;
	  
	  delete[] s;
	  delete[] appSPtmp;
	  return -1;
	}

    };  
  
#ifdef VERBOSE

  tmp.set(1.0);
  for (i = zeros+1; i<d; i++)
    tmp.mul(tmp, r->Get(i,i));
  tmp.sqrt(tmp);



  fprintf (stderr, "\nLoop iterations = %d \n", loops);
  fprintf (stderr, "Lovasz tests = %d \n", lovasz);
  fprintf (stderr, "Cputime is %d ms\n", (cputime()-start));
  if (zeros < d-1)
    {
      fprintf (stderr, "Vol(L) is "); tmp.printerr(); fprintf(stderr, "\n");
      tmp.sqrt(r->Get(zeros+1,zeros+1));
      fprintf (stderr, "||b_1|| is "); 
      tmp.printerr(); fprintf(stderr, "\n\n");
    }
#endif



#ifdef MUDISTRIB
  for (i=0; i<d; i++){
    for (j=0; j<i; j++){
      cout << "[" << i << "," << j << ",";
      mu->Get(i,j).print();
      cout << "],\n";
    }
  }
#endif


  delete[] alpha;
 
  delete mu;
  delete r;
  delete appB;
  delete appSP;
  
  delete[] s;
  delete[] appSPtmp;
  
  return 0;

}



template<class ZT,class FT>
heuristic<ZT,FT>::heuristic(ZZ_mat<ZT>* B_,int preci,double et,double delt, int siege)
{
  siegel = siege;
  prec=preci;
  eta=et;
  delta=delt;

  halfplus.set(eta);
  onedotfive.set(1.5);
  if (!siegel)
    ctt.set(delta);
  else
    ctt.set(delta-eta*eta);

  B=B_;

  double eps = 0.01;
  double epstmp = 1.0 - delta;
  if (epstmp>eps) eps = epstmp;
  epstmp = eta-0.5; 
  if (epstmp>eps) eps = epstmp;


  double rho = ((1.0+eta)*(1.0+eta)+eps)/(delta-eta*eta) + 0.2;
  eps -= 0.00001;

  unsigned int d= B->GetNumRows();
  unsigned int goodprec = (unsigned int) (2.1 * log ((double) d)/log(2.0)  + d* log(rho) / log(2.0) + 11.0
					  - log(eps)/log(2.0));


#ifdef VERBOSE
  cerr << "Setting precision at "<< (!prec?goodprec:prec) <<" if the type allows it."<<endl;
#endif

  
  if (prec!=0)
    FP_NR<FT>::setprec(prec);
  else
    FP_NR<FT>::setprec(goodprec);


  G=new ZZ_mat<ZT>(d,d);

#ifdef WITH_TRANSFORM
  U=new ZZ_mat<ZT>(d,d);
  for (int i=0;i<d;i++)
    for (int j=0;j<d;j++)
      if (i==j) U->Get(i,i).set(1);
      else U->Get(i,j).set(0);
#else
  U=new ZZ_mat<ZT>(0,0); 
#endif

  shift=B->getShift();
}

template<class ZT,class FT>
heuristic<ZT,FT>::~heuristic ()
{
  if (U) delete U;
  if (G) delete G;
}



template <class ZT,class FT> inline ZZ_mat<ZT>* heuristic<ZT,FT>::GetBase()
{
  return B;
}


#endif
