#############################################################################
##
#A  semisimple.g           CHEVIE library      Cdric Bonnaf and Jean Michel
##
#Y  Copyright (C) 2004 - 2010  Lehrstuhl D fur Mathematik, RWTH Aachen,
#Y  University  Paris VII and Universit de Franche-Comt
##
##  This file contains functions dealing with semisimple elements of algebraic
##  groups.
##
## A reductive group G of rank n is represented in CHEVIE by
## CoxeterGroup(R,C)  where R  is the  inregral matrix  whose lines are the
## roots expressed in the canonical basis of X(T) and C the integral matrix
## whose lines are the coroots expressed in the canonical basis of Y(T).
##
## A finite order semisimple element is represented as an element of
## Y(T)\otimes Q/Z=(Q/Z)^n, that is a list of length n of rationals r
## such that 0<=r<1.
##
##  A semisimple element record contains the following fields:
##  .v     The defining list of rationals
##  .group The parent of the defining group

#############################################################################
##
#F  Mod1( <r> )  . . .  Reduction mod 1 of a Rational or a list
#
Mod1:=function(x)
  if IsList(x) then return List(x,Mod1);
  else return (Numerator(x) mod Denominator(x))/Denominator(x);
  fi;
end;

#############################################################################
##
#F  SemisimpleElementOps . . . . .  operations record for semisimple elements
##  
##  
SemisimpleElementOps:=OperationsRecord("SemisimpleElementOps");

# semisimple elements have a domain to be usable as group elements
SemisimpleElement:=function(W,v) 
  if IsRec(W) then W:=Parent(W);fi;
  if ForAll(v,IsRat) then v:=Mod1(v);fi;
  return rec(v:=v,group:=W, operations:=SemisimpleElementOps,
  domain:=rec(operations:=rec(
    \in:=function(a,b)return a.operations=SemisimpleElementOps;end,
    Group:=function(D,gens,id) return rec(isDomain:=true,isGroup:=true,
      identity:=id,generators:=gens,operations:=GroupOps,isFinite:=true);
    end)));end;

IsSemisimpleElement:=x->IsRec(x) and IsBound(x.operations) and x.operations=
  SemisimpleElementOps;

SemisimpleElementOps.\*:=function(a,b)
  if IsList(a) then return List(a,x->x*b);
  elif IsList(b) then return List(b,x->a*x);
  elif ForAll(a.v,IsRat) then return SemisimpleElement(a.group,a.v+b.v);
  else return SemisimpleElement(a.group,
                 Zip(a.v,b.v,function(x,y)return x*y;end));
  fi;
end;

SemisimpleElementOps.\/:=function(a,b)return a*b^-1;end;

SemisimpleElementOps.\=:=function(a,b)return a.v=b.v;end;

SemisimpleElementOps.\<:=function(a,b)return a.v<b.v;end;

SemisimpleElementOps.\^:=function(s,n)local QZ;
  if not IsSemisimpleElement(s) then return s.operations.\^(s,n);fi;
  QZ:=ForAll(s.v,IsRat);
  if IsInt(n) then # raise to a power
    if QZ then return SemisimpleElement(s.group,n*s.v);
          else return SemisimpleElement(s.group,List(s.v,x->x^n));
    fi;
  elif IsPerm(n) then # act by an element of W or WF
    return s^MatYPerm(s.group,n);
  elif IsMat(n) then # act by an element of GL(X(T))
    if QZ then return SemisimpleElement(s.group,s.v*n);
          else return SemisimpleElement(s.group,List(TransposedMat(n),
             v->Product(Zip(v,s.v,function(a,b)return b^a;end))));
    fi;
  elif IsSemisimpleElement(n) then return s;
  elif IsList(n) then # act by an element of the root lattice
    if QZ then return Mod1(n*s.group.simpleRoots*s.v);
          else return Product(Zip(n*s.group.simpleRoots,s.v,
                            function(a,b)return b^a;end));
    fi;
  else Error("does not know how to make ",n," act on a semisimple element");
  fi;
end;

SemisimpleElementOps.Comm:=function(a,b)return a/a;end;

SemisimpleElementOps.Frobenius:=function(W,s,i)return s^(W.phi^i);end;

SemisimpleElementOps.String:=x->SPrint("<",Join(x.v),">");

SemisimpleElementOps.Print:=function(x)Print(String(x));end;

#############################################################################
##
#F  AlgebraicCentre( <W> )  . . . centre of algebraic group W
##  
##  <W>  should be a Weyl group record  (or an extended Weyl group record).
##  The  function returns information  about the centre  Z of the algebraic
##  group defined by <W> as a record with fields:
##   Z0:         basis of Y(Z^0) (wrt the canonical basis of Y(T))
##   complement: a basis of Y(S) where S=complement torus of Z0 in T
##   AZ:         representatives of Z/Z^0 given as subgroup of Y(S)
##   [implemented only for connected groups 18/1/2010]
##   [I added something hopefully correct in general. JM 22/3/2010]
##
AlgebraicCentre:=function(W)local V,res,g,auts,w,r,o,complfix,Yfix;
  if IsExtendedGroup(W) then 
    if Size(W.permauts)<>1 then auts:=W.permauts;fi;
    W:=W.group;
  fi;
  if W.simpleRoots=[] then V:=IdentityMat(W.rank);
  else V:=NullspaceIntMat(TransposedMat(W.simpleRoots));
  fi;
  if IsBound(auts) then # compute fixed space of auts in Y(T)
    Yfix:=IdentityMat(W.rank);
    for w in auts.generators do
      Yfix:=BaseIntersectionIntMats(Yfix,
         NullspaceIntMat(MatYPerm(Parent(W),w)-IdentityMat(W.rank)));
    od;
    V:=BaseIntersectionIntMats(V,Yfix);
  fi;
  r:=ComplementIntMat(IdentityMat(W.rank),V);
  res:=rec(Z0:=Zip(r.sub,r.moduli,function(x,y)return x/y;end),
    complement:=r.complement);
  if res.complement=[] then res.AZ:=[[1..W.rank]*0];
  elif IsBound(auts) then  
    complfix:=ComplementIntMat(Yfix,V).complement;
    o:=Orbits(auts,W.rootInclusion{W.generatingReflections});
    o:=List(o,x->Sum(W.simpleRoots{W.rootRestriction{x}}));
    if complfix=[] then res.AZ:=[[1..W.rank]*0];
    else res.AZ:=(complfix*TransposedMat(o))^-1*complfix;
    fi;
  else 
    res.AZ:=(res.complement*TransposedMat(W.simpleRoots))^-1*res.complement;
  fi;
  res.AZ:=ApplyFunc(Group,AbelianGenerators(
     List(res.AZ,x->SemisimpleElement(W,x))));
  return res;
end;

# describe  Z/Z0(W) as  quotient of  the centre  Pi of the simply connected
# goup  with same isogeny type. Return words  in the generators of Pi which
# generate the kernel.
DescAZ:=function(W)local V,l,toAZ,z,hom,e,AZ;
  AZ:=List(Concatenation(List(ReflectionType(W),t->
    List(ReflTypeOps.CenterSimplyConnected(t),function(l)local v;
      v:=[1..W.rank]*0;v{t.indices}:=l;return v;end))),
    x->SemisimpleElement(W,x)); # centre of simply connected group W'
  if Length(AZ)=0 then return AZ;else AZ:=ApplyFunc(Group,AZ);fi;
  z:=AlgebraicCentre(W);
  toAZ:=function(W,s) 
    s:=SolutionMat(Concatenation(z.complement,z.Z0),s.v);
    return SemisimpleElement(W,s{[1..Length(z.complement)]}*z.complement);
  end;
  l:=List(AZ.generators,x->x.v);
  if W.simpleRoots=[] then V:=IdentityMat(W.rank);
  else V:=NullspaceIntMat(TransposedMat(W.simpleRoots));
  fi;
  hom:=List(l*Concatenation(W.simpleCoroots,V),s->
   toAZ(W,SemisimpleElement(W,s))); # map of root data Y(W')->Y(W)
  hom:=GroupHomomorphismByImages(AZ,z.AZ,AZ.generators,hom);
  return List(Kernel(hom).generators,x->GetWord(AZ,x));
end;

#############################################################################
##
#F  SemisimpleSubgroup( <W>, <V>, <n> )  . . . n-torsion of V
##
##  Returns the subgroup of semisimple elements of order dividing n in the 
##  subtorus of T represented by V, an integral basis of a sublattice of Y(T).
#
SemisimpleSubgroup:=function(W,V,n)
  return ApplyFunc(Group,List(V,v->SemisimpleElement(W,v/n)));
end;

#############################################################################
##
#F  SemisimpleCentralizer( <W>, <s>)  . . . Stabilizer of <s> in <W>
##  
##  Returns  the stabilizer  of the  semisimple element  <s> in  <W>, which
##  describes  also C_G(s),  if G  is the  algebraic group described by the
##  Weyl  group record W.  The stabilizer is  an extended reflection group,
##  with the reflection group part equal to the Weyl group of C_G^0(s), and
##  the diagram automorphism part being those induced by C_G(s)/C_G^0(s) on
##  C_G^0(s). It is accepted that <W> itself is an extended group.
##
SemisimpleCentralizer:=function(W,s)local p,W0s,N,totalW,QZ;
  if IsExtendedGroup(W) then 
    totalW:=Subgroup(Parent(W.group),Concatenation(W.group.generators,
      W.permauts.generators));
    W:=W.group;
  else totalW:=W;
  fi;
  QZ:=ForAll(s.v,IsRat);
  if QZ then p:=Filtered(W.rootInclusion{[1..W.N]},i->s^Parent(W).roots[i]=0);
  else p:=Filtered(W.rootInclusion{[1..W.N]},i->s^Parent(W).roots[i]=s.v[1]^0);
  fi;
  W0s:=ReflectionSubgroup(W,p);
  N:=Normalizer(totalW,W0s);
  N:=Filtered(List(LeftCosets(N,W0s),Representative),w->s=s^w);
  N:=List(N,x->ReducedInRightCoset(W0s,x));
  N:=Subgroup(W,AbelianGenerators(N));
  if W.rank<>W.semisimpleRank then
    if Length(N.generators)=0 then N:=Group(W.matgens[1]^0);
    else N:=ApplyFunc(Group,List(N.generators,x->MatXPerm(W,x)));
    fi;
  fi;
  return ExtendedReflectionGroup(W0s,N);
end;

#############################################################################
##
#F  IsQuasiIsolated( <W>, <s>)  . . . . whether <s> is quasi-isolated in <W>
##
IsQuasiIsolated:=function(W,s)
  return Length(AlgebraicCentre(SemisimpleCentralizer(W,s)).Z0)=
    W.rank-W.semisimpleRank;
end;

#############################################################################
##
#F  IsIsolated( <W>, <s>)  . . . . whether <s> is isolated in <W>
IsIsolated:=function(W,s)
  return Length(AlgebraicCentre(SemisimpleCentralizer(W,s).group).Z0)=
    W.rank-W.semisimpleRank;
end;

#############################################################################
##
#F  FundamentalGroup(<W>)  fundamental group of algebraic group <W>
##
## This  function  returns  the  fundamental  group  of the algebraic group
## corresponding  to the Weyl group record <W> as the diagram automorphisms
## of  the  corresponding  affine  Weyl  group  induced  by  <W>, thus as a
## permutation group on the simple roots and the negative longest roots
## of each irreducible component.
#
CoxeterGroupOps.FundamentalGroup:=function(W)local n,l,res,omega,w0,m,r;
  if W.cartan=[] then return Group(());fi;
  omega:=W.cartan^-1*W.simpleCoroots;
  res:=[];
  for n in List(W.type,x->x.indices) do
    m:=Minimum(List(W.roots,x->Sum(x{n})));
    r:=PositionProperty(W.roots,x->Sum(x{n})=m);
    l:=Filtered(n,i->W.roots[r][i]=-1 and ForAll(omega[i],IsInt));
    w0:=LongestCoxeterElement(W,W.rootInclusion{n});
    Append(res,List(w0*List(l,
       i->LongestCoxeterElement(W,W.rootInclusion{Difference(n,[i])})),
       p->RestrictedPerm(p,W.rootInclusion{Concatenation(n,[r])})));
  od;
  return Group(AbelianGenerators(res),());
end;

#############################################################################
##
#F  QuasiIsolatedRepresentatives(<W>)  . . representatives of W-orbits of 
##       quasi-isolated semisimple elements.
##
##   This function only works for a Weyl group record representing an 
##  irreducible and semisimple algebraic group $G$. It follows the algorithm
##  given by C. Bonnafe ``Quasi-Isolated Elements in Reductive Groups 1''
##  Communications in Algebra, Volume 33, Issue 7 June 2005 , pages 2315 - 2337
##
QuasiIsolatedRepresentatives:=function(W)local H,iso,d,p,res;
  if W.rank<>W.semisimpleRank or Length(ReflectionType(W))>1 then
    Error(W," should be irreducible and semi-simple");
  fi;
  d:=Concatenation(W.generatingReflections,[2*W.N]);
  H:=FundamentalGroup(W);
  p:=Concatenation(List([1..Size(H)],i->Combinations(d,i)));
  res:=List(Orbits(H,p,OnSets),i->i[1]);
  res:=Filtered(res, P->Length(Orbit(Stabilizer(H,P,OnSets),
    [P[1]],OnSets))=Length(P));
  res:=List(res,i->List(i,j->Position(d,j)));
  iso:=Zip(W.cartan^-1*W.simpleCoroots,W.roots[W.N],
     function(x,y)return x/y;end);
  Add(iso,0*iso[1]);
  res:=List(res,i->Sum(iso{i})/Length(i));
  return Set(List(res,s->SemisimpleElement(W,s)));
end;

#############################################################################
##
#F  StructureRationalPointsConnectedCenter(<MF>,q)  . . Structure of Z^0(M)^F
#
#   MF is a spets. Gives the abelian invariants of Z^0(M)^F(q)
#
StructureRationalPointsConnectedCentre:=function(MF,q)local M,W,Z0,Phi,Z0F;
  if IsSpets(MF) then M:=Group(MF);
  else M:=MF;MF:=Spets(M);
  fi;
  W:=Parent(M);
  Z0:=AlgebraicCentre(M).Z0;
  Phi:=MatYPerm(W,MF.phi);
  Z0F:=Z0*(Phi*q-Phi^0);
  Z0F:=List(Z0F,x->SolutionIntMat(Z0,x));
  Z0F:=DiagonalOfMat(DiagonalizeIntMat(Z0F).normal);
  return Filtered(Z0F,x->x<>1);
end;

IntermediateGroup:=function(W,I)local C,C1,d,R;
  C:=CartanMat(W);C1:=C^-1;R:=C^0;
  if I<>[] then Append(R,C1{I});
    d:=Lcm(List(Flat(C1{I}),Denominator));R:=BaseIntMat(d*R)/d;
  fi;
  return CoxeterGroup(R^-1,C*TransposedMat(R));
end;

RootDatum:=function(type,rank)local i,R,R1,res,p;
  if type="gl" then R:=List([1..rank-1],function(i)local v;
               v:=[1..rank]*0;v{[i,i+1]}:=[1,-1];return v;end);
    res:=CoxeterGroup(R,R);
  elif type="u" then R:=List([1..rank-1],function(i)local v;
               v:=[1..rank]*0;v{[i,i+1]}:=[1,-1];return v;end);
    res:=CoxeterGroup(R,R);
    res:=CoxeterCoset(res,Product([1..QuoInt(rank-1,2)],i->(i,rank-i)));
  elif type="sl" then res:=CoxeterGroup("A",rank-1,"sc");
  elif type="su" then res:=CoxeterCoset(CoxeterGroup("A",rank-1,"sc"),
     Product([1..QuoInt(rank-1,2)],i->(i,rank-i)));
  elif type="pgl" then res:=CoxeterGroup("A",rank-1);
  elif type="psu" then res:=CoxeterCoset(CoxeterGroup("A",rank-1),
     Product([1..QuoInt(rank-1,2)],i->(i,rank-i)));
  elif type="sp" then rank:=rank/2;
    R:=IdentityMat(rank); for i in [2..rank] do R[i][i-1]:=-1;od;
    R1:=Copy(R);R1[1]:=2*R1[1];
    res:=CoxeterGroup(R1,R);
  elif type="so" then p:=rank mod 2;rank:=QuoInt(rank,2);
    R:=IdentityMat(rank); for i in [2..rank] do R[i][i-1]:=-1;od;
    if p=1 then R1:=Copy(R);R1[1]:=R1[1]*2; res:=CoxeterGroup(R,R1);
    else R[1][2]:=1; res:=CoxeterGroup(R,R);
    fi;
  elif type="psp" then res:=CoxeterGroup("C",rank/2);
  elif type="pso" then 
    if rank mod 2=1 then res:=CoxeterGroup("B",QuoInt(rank,2));
    else res:=CoxeterGroup("D",rank/2);
    fi;
  elif type="spin" then res:=CoxeterGroup("D",rank/2,"sc");
  elif type="halfspin" then
    R:=IdentityMat(rank/2);R1:=CartanMat("D",rank/2);p:=R1^-1;
    R[rank/2]:=Concatenation([-rank,4-rank]/4,[2-rank/2,3-rank/2..-2],[2]);
    res:=CoxeterGroup(R,R1*TransposedMat(R^-1));
  fi;
  p:=["gl","u","sl","su","pgl","psu","sp","psp","so","pso","spin","halfspin"];
  if not IsBound(res) then Error("accepted types are:\n",
    Join(List(p,FormatGAP)),"\n");
  else res.name:=SPrint("RootDatum(",FormatGAP(type),",",rank,")");
    return res;
  fi;
end;
