#############################################################################
##
#A  permroot.g            CHEVIE library             Jean Michel 
##
#Y  Copyright (C) 1999   University Paris VII.
##
# "PermRootGroup"s are (possibly complex)  reflection groups acting on a
# vector space V (in GAP, the matrices act on the *right*), generated by
# reflections with respect to a set of 'roots' in V.
# They are represented as permutation groups on the orbit under W of the
# initial roots.
#  The data  used to  generate them  is a list  of roots  in V,  and the
# eigenvalue of  the corresponding  (pseudo-)reflection (contained  in a
# list EigenvaluesGeneratingReflections,  by default assumed  to contain
# only -1, i.e. true reflections).
#
# They are represented as Permutation group records with the following
# additional fields:
#
# .nbGeneratingReflections
#      equal to the number of initial roots given to generate W
# .roots -- a set of roots V (on which G has a faithful permutation
#     representation). roots{generatingReflections} is the
#     initially given roots. The roots are usually chosen to be in minimal
#     number but this may be equal to Size(G) for some complex groups!
# .EigenvaluesGeneratingReflections -- the orders of the generators.
# .matgens -- the generators (matrices of the reflections wrt the initial roots)
# .rank -- the dimension of V
# .semisimpleRank  -- the dimension of the subspace of V generated by
#     the roots (for an irreducible parent group, equal to the
#     dimension of V). For the case of real initial roots, equal to the
#     number of generators; but may be smaller than the number of
#     generators for complex groups.
# .orbitRepresentative -- for each root the smallest index of
#                         a root in same orbit.
#
#  A reflection subgroup has record of the connection with its parent
#  via:
#  .rootInclusion    the indices of the roots of W in its parent
#  .rootRestriction  where bound the indices of the roots of the parent in W

if not IsBound(PermGroupOps.PositionClass) then ReadChv("lib/classinv");fi;
PermRootOps:=OperationsRecord("PermRootOps",PermGroupOps);

PermRootOps.Format:=function(W,opt)
  if IsBound(opt.GAP) then return W.name;
  else return ReflectionName(W,opt);
  fi;
end;

PermRootOps.Display:=function(W,opt)Print(Format(W,opt),"\n");end;

PermRootOps.String:=FormatGAP;

#############################################################################
##
#F  Reflection( <root>,<coroot>)  .  .  .  . returns  matrix  of complex
##  reflection with respect to <root> and given <coroot>. The reflection
##  s with root r and coroot r^  is defined by s(x)=x-r^(x)r where r^ is
##  the coroot  corresponding to r  ; the  root and coroot  are uniquely
##  determined by the  reflection up to multiplication of the  root by a
##  scalar and of the coroot by the inverse scalar.
##
#F  Reflection( <root>[, <eigenvalue>]) . . . returns matrix of complex
##  reflection  when  the  vector  space and  its  dual  are  identified
##  via  a  scalar  product  <,>;  r^   is  equal  to  the  linear  form
##  x->(1-zeta)<x,r>/<r,r> where zeta is the nontrivial eigenvalue.
##
#F  Reflection(  <W>, <n> )  . . . returns the permutation of  the roots
##  induced by the n-th reflection in group W
#
Reflection:=function(arg)local g,r,rbar;g:=arg[1];
  if IsGroup(g) then r:=arg[2];
    if not IsBound(g.reflections[r]) then
      g.reflections[r]:=g.operations.Reflection(g,r);
    fi;
    return g.reflections[r];
  fi;
  if Length(arg)=1 then r:=-1; else r:=arg[2]; fi;
  if not IsList(r) then
    rbar:=List(g,x->GaloisCyc(x,-1));r:=(1-r)*rbar/(rbar*g);
#   InfoChevie("# root:",g,"  norm:",rbar*g,"\n");
  fi;
  return IdentityMat(Length(g))-List(r,i->i*g);
end;

#############################################################################
##
#F  AsReflection( <m> [,<r>]) . . . determines if m is the matrix of a
##   reflection [with root r]. Returns false if not. Otherwise returns
##   root,  coroot,  eigenvalue,  isOrthogonal  [if  the  reflection  is
##   orthogonal  with  respect  to  the  usual  scalar  product,  so  is
##   determined by only the root and the eigenvalue].
##
AsReflection:=function(arg)local e,rc,j,r,zeta,norm,orth,m;
  m:=arg[1];
  if IsBound(arg[2]) then r:=arg[2];fi;
  norm:=x->x*ComplexConjugate(x);
  e:=m^0-m;
  rc:=[];
  for j in [1..Length(m)] do
    if not IsBound(r)  then
      if not ForAll(e[j],x->x=0*x) then
         r:=e[j]; 
# if IsRat(norm(r)) then r:=r/GetRoot(norm(r),2);fi;
         rc[j]:=ProportionalityCoefficient(e[j],r);
      else
         rc[j]:=0;
      fi;
    else
      rc[j]:=ProportionalityCoefficient(e[j],r);
      if rc[j]=false then 
        InfoChevie2("# WARNING: ",m," is not a reflection\n");
	return false;
      fi;
    fi;
  od;
  if not IsBound(r) then 
    InfoChevie2("# WARNING: ",m," is not a reflection\n");
    return false;
  fi;
  zeta:=ProportionalityCoefficient(r*m,r);
  if zeta=false or AsRootOfUnity(zeta)=false then 
    InfoChevie2("# WARNING: ",m," is not a reflection of root ",r,"\n");
    return false;
  fi;
  orth:=(rc*norm(r)=(1-zeta)*ComplexConjugate(r));
# Print("root vector=",Format(r,"GAP")," norm=",norm(r),"\n");
# Print("coroot vector=",Format(rc,"GAP")," norm=",norm(rc),"\n");
# Print("eigenvalue=",zeta,"\n");
# if orth then
#   Print("reflection is orthogonal\n");
# else
#   Print("r^*(r,r)=",Format(rc*norm(r))," but (1-zeta)r=",
#                     Format((1-zeta)*ComplexConjugate(r)),"\n");
# fi;
  return rec(root:=r,coroot:=rc,eigenvalue:=zeta,isOrthogonal:=orth);
end;

#############################################################################
##
#F  PermRootGroup( <roots> [,<eigenvalues>])
#F  or PermRootGroup( <roots>, <coroots>)
#F  or PermRootGroup(<record g>) . . . 
#   Makes a group permuting roots from record g.
# g should have fields roots and EigenvaluesGeneratingReflections bound
#   -- if EigenvaluesGeneratingReflections are omitted they are assumed to be
#      all -1's.
#   -- if coroots are omitted reflections are computed using standard
#      scalar product
#
#  In the first 2 forms these 3 fields are built from the arguments
#
PermRootGroupNC:=function(arg)local g,n,r,j,i,p,G,new,newi,newr,rbar,indices;
  if IsRec(arg[1]) then g:=arg[1];
  else g:=rec(roots:=ShallowCopy(arg[1]));
    if Length(arg)=2 then 
      if Length(arg[2])>0 and IsList(arg[2][1]) then g.simpleCoroots:=arg[2];
      else g.EigenvaluesGeneratingReflections:=arg[2];
      fi;
    fi;
  fi;
  if not IsBound(g.nbGeneratingReflections) then 
     g.nbGeneratingReflections:=Length(g.roots);
  fi;
  g.generatingReflections:=[1..g.nbGeneratingReflections];
  if not IsBound(g.EigenvaluesGeneratingReflections) then 
    if IsBound(g.simpleCoroots) then
      g.EigenvaluesGeneratingReflections:=List([1..Length(g.roots)],
	   i->AsRootOfUnity(1-g.simpleCoroots[i]*g.roots[i]));
    else 
      g.EigenvaluesGeneratingReflections:=List(g.generatingReflections,x->1/2);
    fi;
  fi;
  g.OrdersGeneratingReflections:=List(g.EigenvaluesGeneratingReflections,
    Denominator);
  if not IsBound(g.simpleCoroots) then
    g.standardScalarProduct:=true;
    g.simpleCoroots:=List(g.generatingReflections,function(i)local rbar;
	rbar:=List(g.roots[i],x->GaloisCyc(x,-1));
	return (1-E(g.OrdersGeneratingReflections[i])^
        Numerator(g.EigenvaluesGeneratingReflections[i]))*rbar/(rbar*g.roots[i]);end);
  fi;
  g.matgens:=List(g.generatingReflections,
                  j->Reflection(g.roots[j],g.simpleCoroots[j]));
  if Length(g.roots)>0 
  then g.rank:=Length(g.roots[1]);
       g.semisimpleRank:=RankMat(g.roots{g.generatingReflections});
  else g.semisimpleRank:=0;g.rank:=0;
  fi;
  g.reflections:=List(g.generatingReflections,x->[]);

  # the following section is quite subtle: it has the (essential -- this
  # is what allows to construct reflexion subgroups in a consistent way)
  # property that the order of the constructed roots depends only on the
  # Cartan matrix of g, not on the actual root values.

  indices:=ShallowCopy(g.generatingReflections);
  InfoChevie2("# roots: ");
  repeat
    n:=Length(g.roots);
    for j in [1..Length(g.matgens)] do
      SortParallel(g.roots,indices);
      r:=Length(g.reflections[j]);
      newi:=Filtered([1..Length(indices)],i->indices[i]>r);
      newr:=g.roots{newi}*g.matgens[j];
      newi:=indices{newi};
      SortParallel(newi,newr);
      new:=[];
      r:=Length(g.roots);
      for i in [1..Length(newr)] do
	p:=Position(g.roots,newr[i]); # Position is faster than PositionSorted!
#	p:=PositionSorted(g.roots,newr[i]);
	if p>r or g.roots[p]<>newr[i] 
	then Add(new,newr[i]);p:=r+Length(new);
	else p:=indices[p];
	fi;
	g.reflections[j][newi[i]]:=p;
      od;
      Append(indices,r+[1..Length(new)]);
      Append(g.roots,new);
    od;
    InfoChevie2(" ",Length(g.roots),"\c");
  until Length(g.roots)=n;
  InfoChevie2("\n");
  SortParallel(indices,g.roots);

  g.reflections:=List(g.reflections,PermList);
  G:=Group(g.reflections,());Inherit(G,g);
  G.rootRestriction:=[1..Length(G.roots)];
  G.rootInclusion:=[1..Length(G.roots)];
  G.orbitRepresentative:=[];
  for r in Orbits(G,G.rootInclusion) do
    G.orbitRepresentative{G.rootRestriction{r}}:=[1..Length(r)]*0+Minimum(r);
  od;
  G.reflectionsLabels:=G.generatingReflections;
  G.operations:=PermRootOps;
  G.name:=SPrint("PermRootGroup(",G.generatingReflections,",");
  if IsBound(G.standardScalarProduct) then
       Append(G.name,String(G.EigenvaluesGeneratingReflections)); 
  else 
    Append(G.name,"***");
  fi;
  Append(G.name,")");
  return G;
end;

PermRootGroup:=function(arg)local G;
  G:=ApplyFunc(PermRootGroupNC,arg);
  ReflectionType(G); # this may change the generating roots
  return G;
end;

# tests if w is a word for W
IsWordFor:=function(W,w)
  return IsList(w) and IsSubset(W.reflectionsLabels,Set(w));
end;

#############################################################################
##
#F  Reflections(W) . . . returns the list of reflections of W
##
PermRootOps.Reflections:=W->List([1..Length(W.roots)],r->Reflection(W,r));
  # JM: should be fixed to not return too many reflections
     
#############################################################################
##
#F  ReflectionsSubgroup(W,refs) . . . returns the subgroup of W
#F    generated by Reflections(Parent(W)){refs} as a PermRootGroup
##
PermRootOps.ReflectionSubgroupNC:=function(W,refs)local res,i,r;
  if not IsSubset(W.rootInclusion,refs) then
    Error("second argument should be a subset of rootInclusion");
  fi;
  if IsBound(W.parent) then W:=W.parent;fi;


  r:=PermRootGroupNC(W.roots{refs},List(refs,i->PermRootOps.Coroot(W,i)));
  r.rootInclusion:=List(r.roots,x->Position(W.roots,x));
  r.rank:=W.rank; # otherwise when refs=[] gets it wrong

  if r.rootInclusion{r.generatingReflections}=W.generatingReflections
  then return W;fi;
  # We want to have ReflectionSubgroups which just re-order the generators.
  # But the command 'Subgroup' returns W, if Size(res)=Size(W).
  # To avoid this we substitute it by some more lines.
  res:=rec(isDomain:=true,isGroup:=true,parent:=W,identity:=(),
    isPermGroup:=true,
    generators := List(r.rootInclusion{r.generatingReflections},
                          i->Reflection(W,i)));
  for i in [1..Length(res.generators)] do res.(i):=res.generators[i];od;

  for i in [ "roots", "simpleCoroots", "nbGeneratingReflections", 
    "generatingReflections", "EigenvaluesGeneratingReflections", 
    "OrdersGeneratingReflections", "matgens", "rank", "semisimpleRank",
    "rootInclusion","operations"]
  do res.(i):=ShallowCopy(r.(i));
  od;
  res.rootRestriction:=[];
  res.rootRestriction{res.rootInclusion}:=[1..Length(res.roots)];
  res.orbitRepresentative:=res.rootInclusion{r.orbitRepresentative};
  res.reflections:=ShallowCopy(res.generators);
  res.reflectionsLabels:=res.rootInclusion{[1..res.nbGeneratingReflections]};
  res.name:=SPrint("ReflectionSubgroup(",res.parent,", ",
	      res.rootInclusion{res.generatingReflections},")");
  return res;
end;

PermRootOps.ReflectionSubgroup:=function(W,refs)local res;
  res:=PermRootOps.ReflectionSubgroupNC(W,refs);
  ReflectionType(res);
  return res;
end;

PermRootOps.ProductRootEmbed:=function(W1,W2)return
  [Concatenation(W1.generatingReflections,
  W2.nbGeneratingReflections+[1+W1.nbGeneratingReflections..Length(W1.roots)]),
  Concatenation(Length(W1.generatingReflections)+W2.generatingReflections,
   Length(W1.roots)+[1+W2.nbGeneratingReflections..Length(W2.roots)])];
end;
#############################################################################
##
#F  W1*W2 . . . returns the product of PermRootGroups W1 and W2
#F      acting on V1+V2
##
PermRootOps.\*:=function(W1,W2)local g,t,t1,t2,r1,r2,maps,embed,maketype;
  if IsGroup(W2) then
  embed:=function(W,x,map)return
     RestrictedPerm(x,W.rootInclusion)^MappingPermListList(W.rootInclusion,map);
  end;
  if not (IsPermGroup(W1) and IsPermGroup(W2) and 
          IsBound(W1.roots) and IsBound(W2.roots)) then 
     Error("only implemented for permutation groups of the roots");fi;
  if Length(W1.roots)=0 then r1:=0;else r1:=Length(W1.roots[1]);fi;
  if Length(W2.roots)=0 then r2:=0;else r2:=Length(W2.roots[1]);fi;
  if IsCoxeterGroup(W2) then maps:=W1.operations.ProductRootEmbed(W1,W2);
  else                       maps:=W2.operations.ProductRootEmbed(W1,W2);
  fi;
  g:=Group(Concatenation(List(W1.generators,x->embed(W1,x,maps[1])),
                         List(W2.generators,x->embed(W2,x,maps[2]))),());
  g.EigenvaluesGeneratingReflections:=
       Concatenation(W1.EigenvaluesGeneratingReflections,
                     W2.EigenvaluesGeneratingReflections);
  g.OrdersGeneratingReflections:=
       Concatenation(W1.OrdersGeneratingReflections,
                     W2.OrdersGeneratingReflections);
  g.matgens:=Concatenation(
     List(W1.matgens,x->DiagonalMat(x,IdentityMat(r2))),
     List(W2.matgens,x->DiagonalMat(IdentityMat(r1),x)));
  g.generatingReflections:=Concatenation(W1.generatingReflections,
        Length(W1.generatingReflections)+W2.generatingReflections);
  g.reflectionsLabels:=g.generatingReflections; # default choice
  g.reflections:=g.generators;
  g.roots:=[];
  g.roots{maps[1]}:=List(W1.roots,x->Concatenation(x,[1..r2]*0));
  g.roots{maps[2]}:=List(W2.roots,x->Concatenation([1..r1]*0,x));
  g.simpleCoroots:=[];
  g.simpleCoroots{maps[1]{W1.generatingReflections}}:=
    List(W1.simpleCoroots,x->Concatenation(x,[1..r2]*0));
  g.simpleCoroots{maps[2]{W2.generatingReflections}}:=
    List(W2.simpleCoroots,x->Concatenation([1..r1]*0,x));
  g.rank:=W1.rank+W2.rank;
  g.semisimpleRank:=W1.semisimpleRank+W2.semisimpleRank;
  g.nbGeneratingReflections:=W1.nbGeneratingReflections+
                             W2.nbGeneratingReflections;
  if IsBound(W1.degrees) and IsBound(W2.degrees) then
    g.degrees:=Concatenation(W1.degrees,W2.degrees);
  fi;
  g.rootRestriction:=[1..Length(g.roots)];                                    
  g.rootInclusion:=[1..Length(g.roots)];                                      
  g.orbitRepresentative:=[];                                                   
  for t in Orbits(g,g.rootInclusion) do                                        
     g.orbitRepresentative{t}:=[1..Length(t)]*0+Minimum(t); 
  od;                                                                          
  g.operations:=PermRootOps;                                                   
  maketype:=function(W,t,map)local res;
    res:=ShallowCopy(t);res.indices:=map{res.indices};
    if not IsBound(res.group) then res.group:=W;fi;
    if not IsBound(res.embed) then res.embed:=[1..Length(W.roots)];fi;
    res.embed:=map{res.embed};
    return res;
  end;
  if IsBound(W1.type) then t1:=List(W1.type,t->maketype(W1,t,maps[1]));
  else t1:=[rec(group:=W1,indices:=maps[1]{W1.generatingReflections},embed:=maps[1])];
  fi;
  if IsBound(W2.type) then t2:=List(W2.type,t->maketype(W2,t,maps[2]));
  else t2:=[rec(group:=W2,indices:=maps[2]{W2.generatingReflections},embed:=maps[2])];
  fi;
  g.type:=Concatenation(t1,t2);
  return g;
  else return ReflectionCoset(W1,W2);
  fi;
end;

# first set of independant roots in W.generatingReflections
PermRootOps.IndependentRoots:=function(W)local r;
  if not IsBound(W.independentRoots) then
    r:=IndependentLines(W.roots{W.generatingReflections});
    W.independentRoots:=W.generatingReflections{r};
    if r<>[1..Length(r)] then InfoChevie("# independent roots are not ",
      W.generatingReflections{[1..Length(r)]}," but ",W.independentRoots,"\n");
    fi;
  fi;
  return W.independentRoots;
end;

PermRootOps.ConjugateSubgroup:=function(W,w)
  return ReflectionSubgroup(Parent(W),
    OnTuples(W.rootInclusion{W.generatingReflections},w));
end;

# base of X made up of independent roots, then vectors orthogonal to these
PermRootOps.BaseX:=function(W)local ir;
  if not IsBound(W.forMatX) then 
    ir:=PermRootOps.IndependentRoots(W);
    W.forMatX:=W.roots{ir};
    if Length(W.forMatX)=0 then W.forMatX:=IdentityMat(W.rank);
    else Append(W.forMatX,NullspaceMat(TransposedMat(
      List(ir,i->PermRootOps.Coroot(W,i)))));
    fi;
  fi;
  return W.forMatX;
end;

# matrix of w on X. Only the effect of w on IndependentRoots(W) is used:
# w should send these to roots of W.parent
PermRootOps.MatXPerm:=function(W,w)local OR,proots;
  OR:=PermRootOps.BaseX(W);
  if OR=[] then return [];fi;
  if IsBound(W.parent) then proots:=W.parent.roots;
  else proots:=W.roots;fi;
  return # operation is trivial on space orthogonal to W.roots
  OR^-1*Concatenation(proots{OnTuples(W.rootInclusion{
   PermRootOps.IndependentRoots(W)},w)},OR{[W.semisimpleRank+1..W.rank]});
end;

PermRootOps.Coroot:=function(W,i)local m;
  if i in W.generatingReflections then
    if not IsBound(W.simpleCoroots[i]) then 
      m:=MatXPerm(W,Reflection(W,i));
      W.simpleCoroots[i]:=List(m^0-m,v->ProportionalityCoefficient(v,W.roots[i]));
    fi;
    return W.simpleCoroots[i];
  fi;
  m:=MatXPerm(W,Reflection(W,i));
  return List(m^0-m,v->ProportionalityCoefficient(v,W.roots[i]));
end;

# returns reflection corresponding to r-th root
PermRootOps.Reflection:=function(g,r)local o;
  o:=g.orbitRepresentative[r];
  return g.reflections[g.rootRestriction[o]]^
   RepresentativeOperation(g,o,g.rootInclusion[r]);
end;

CartanCoefficient:=function(W,i,j)local R,v;R:=Parent(W).roots;
  v:=1;while R[i][v]=0 do v:=v+1;od;
  j:=R[j]-R[j^Reflection(W,W.rootRestriction[i])];
  return j[v]/R[i][v];
end;

# CartanMat(W [,l])
# The Cartan coefficient between two roots a and b with associated reflections
# r and s is the number x such that r(b)=b-xa
# l omitted is taken as  W.rootInclusion{W.generatingReflections}
PermRootOps.CartanMat:=function(arg)local W,cartan;
  cartan:=l->List(l,i->List(l,j->CartanCoefficient(W,i,j)));
  W:=arg[1];
  if Length(arg)=1 then
    if not IsBound(W.cartan) then 
      W.cartan:=cartan(W.rootInclusion{W.generatingReflections});fi;
    return W.cartan;
  else return cartan(arg[2]);
  fi;
end;

# Let M be a matrix; return the permutation of the roots of W
# it effects if any, otherwise false
PermRootOps.PermMatX:=function(W, M) local tmp;
  # we always have to look at the parent group:
  if IsBound(W.parent) then W:=W.parent; fi;
  if M=[] then return ();fi;
  return PermListList(W.roots,W.roots*M);
end;

# for now only accept those F which send the generators of W to a conjugate set
PermRootOps.ReducedInRightCoset:=function(W,F)local w,base,ir,a,t;
  ir:=PermRootOps.IndependentRoots(W);
  if IsSubset(W.rootInclusion,OnTuples(W.rootInclusion{ir},F)) then
    w:=PermMatX(W,MatXPerm(W,F));
    if w<>false and w in W then return w^-1*F;fi;
  fi;
  base:=List(W.generatingReflections,i->Reflection(W,i));
  w:=RepresentativeOperation(W,base,OnTuples(base,F),OnTuples);
  if w<>false then return w^-1*F;fi;
  ir:=OnSets(Set(base),F);
  w:=RepresentativeOperation(W,Set(base),ir,OnSets);
  if w<>false then return F*w^-1;fi;
  # use the classification
  t:=List(ReflectionType(W),ShallowCopy);
  for a in t do 
    a.subgroup:=ReflectionSubgroup(W,W.rootInclusion{a.indices});
    base:=OnTuples(a.subgroup.generators,F);
    if not ForAll(base,x->x in W) then Error("F must normalize W\n");
    elif not ForAll(base,x->x in a.subgroup) then Error("case not handled\n");
    fi;
    if not (a.series="ST" and IsBound(a.p) and [a.p,a.q,a.rank]=[3,3,3])
    then  
#     if Length(t)=1 then Error("recursive");fi;
      F:=ReducedInRightCoset(a.subgroup,F);
      if F=false then Error("should not happen\n");fi;
    else base:=CHEVIE.RawData("ReducedInRightCoset","timp")(a.subgroup,F);
      if base=false then Error("should not happen");
      else a.indices:=W.rootRestriction{a.subgroup.rootInclusion{base[1]}};
           F:=base[2];
      fi;
    fi;
  od;
  return rec(phi:=F,reflectionGroup:=ReflectionSubgroup(W,
    Concatenation(List(t,x->W.rootInclusion{x.indices}))));
end;

#############################################################################
##
#F  ReflectionCharvalue(W,w) ..... Reflection character of W on w
##
PermRootOps.ReflectionCharValue:=function(W,w)
  return TraceMat(MatXPerm(W,w));
end;

#############################################################################
##
#F  ReflectionCharacter(W) ..... All values of Reflection character of W
##
ReflectionCharacter:=function(W)
  if not IsBound(W.reflectionCharacter) then
    W.reflectionCharacter:=List(ConjugacyClasses(W),
       c->ReflectionCharValue(W,Representative(c)));
  fi;
  return W.reflectionCharacter;
end;

#############################################################################
##
#F  ReflectionEigenvalues(W [,c]) ..... Eigenvalues [of cth] class in
#F        reflection representation
##
## The eigenvalue E(n)^i is represented by i/n
##
PermRootOps.ReflectionEigenvalues:=function(arg)local W,refchar,p,n,ct;
  W:=arg[1];
  if not IsBound(W.eigenvalues) then
    ct:=CharTable(W);
    refchar:=ReflectionCharacter(W);
    W.eigenvalues:=[];
    for p in List([1..Length(ct.irreducibles)],i->Eigenvalues(ct,refchar,i)) do
      n:=Length(p);
      Add(W.eigenvalues,Concatenation(List([1..n],i->List([1..p[i]],
        function(j)if i=n then return 0;else return i/n;fi;end))));
    od;
  fi;
  if Length(arg)>1 then return W.eigenvalues[arg[2]];
  else return W.eigenvalues;
  fi;
end;

#############################################################################
##
#F  ReflectionDegrees(W) ..... General method for reflection degrees
##
##   the function below uses the formula
##   prod_{g\in W}det(g-t)=prod_i(t^{d_i}-1)^{|W|/d_i}
##
PermRootOps.ReflectionDegrees:=function(W)local l,c,res,i,e,p,d,j,eig,mul,n;
  c:=CharTable(W).classes;
  l:=Concatenation(List([1..Length(c)],
    i->List(ReflectionEigenvalues(W,i),e->[e,c[i]])));
  e:=CollectBy(l,x->x[1]);e:=List(e,x->[x[1][1],Sum(x,y->y[2])]);
  e:=TransposedMat(e);
  if e=[] then return [];fi;
  eig:=e[1];mul:=e[2];
  res:=[];
  for d in Reversed(Set(List(eig,Denominator))) do
   p:=Position(eig,Mod1(1/d));
   if p<>false and mul[p]>0 then
     for i in [1..d*mul[p]/Size(W)] do Add(res,d);od;
     n:=mul[p];
     for j in [0..d-1] do p:=Position(eig,j/d); mul[p]:=mul[p]-n;od;
   fi;
  od;
  W.degrees:=Reversed(res);return W.degrees;
end;

#############################################################################
##
#F  GenericOrder(W,q) . . . The (compact) generic polynomial order of W
##      q^NHyp(\prod(q^{d_i}-1) over the degrees)
##
PermRootOps.GenericOrder:=function(W,q)
  return q^(Sum(1+ReflectionCoDegrees(W)))*
     Product(ReflectionDegrees(W),d->q^d-1);
end;

#############################################################################
##
#F  HyperplaneOrbits(W) . . . Information about orbits of reflecting hyperplanes
##
##  returns a list of records, one for each hyperplane orbit of W, 
##  containing the following fields for each orbit:
##   .s        smallest index of a generating reflection in orbit
##   .e_s      order of s
##   .classno  for i in [1..e_s-1], PositionClass(W,s^i)
##   .N_s      Size of orbit
##   .det_s    for i in [1..e_s-1], position in CharTable of (det_s)^i
##
HyperplaneOrbits:=function(W)local ct,o,s;
  if not IsBound(W.orbits) then
    ct:=CharTable(W);
    W.orbits:=List(Orbits(W,Reflections(W)),
      x->rec(s:=W.rootInclusion[PositionProperty(W.reflections,s->s in x)]));
    for o in W.orbits do
      s:=W.reflections[W.rootRestriction[o.s]];
      o.e_s:=OrderPerm(s);
      o.classno:=List([1..o.e_s-1],j->PositionClass(W,s^j));
      o.N_s:=ct.classes[o.classno[1]];
    od;
    for o in W.orbits do
      o.det_s:=List([1..o.e_s-1],j->PositionProperty(ct.irreducibles,
        chi->chi[1]=1 and chi[o.classno[1]]=E(o.e_s)^j and 
	ForAll(Filtered(W.orbits,o1->o1.s<>o.s),o1->chi[o1.classno[1]]=1)));
    od;
    SortBy(W.orbits,x->x.s);
  fi;
  return W.orbits;
end;

#############################################################################
##
#F  TorusOrder(W,q,i) . . . The generic order torus for ith class of W
##
PermRootOps.TorusOrder:=function(W,i,q) #torus order of ith class
  return Product(ReflectionEigenvalues(W,i),
                 l->q-E(Denominator(l))^Numerator(l));
end;

#############################################################################
##
#F  Fakedegrees(W,q) . . . General method for Fake degrees
##
PermRootOps.FakeDegrees:=function(W,q)local P,qq,ct,to;
  qq:=X(Cyclotomics);
  P:=GenericOrder(W,qq);P.valuation:=0;
  ct:=CharTable(W);
  to:=i->P/(ct.centralizers[i]*Product(ReflectionEigenvalues(W,i),
                 l->(qq*E(Denominator(l))^-Numerator(l)-1)));
  return List(ct.irreducibles*List([1..Length(ct.classes)],to),x->Value(x,q));
end;

PermRootOps.LowestPowerFakeDegrees:=function(W)local q;
  q:=X(Cyclotomics);
  return List(FakeDegrees(W,q),x->x.valuation);
end;

PermRootOps.HighestPowerFakeDegrees:=function(W)local q;
  q:=X(Cyclotomics);
  return List(FakeDegrees(W,q),Degree);
end;

############################################################################
##
#F  jInductionTable( <subgroup>, <group> ) . . j-induction of characters
##  
##  This    function  works   like  'InductionTable'   but   computes  the
##  j-induction, as defined in [Lusztig-book, p.78].
##  
jInductionTable:=function(u,g)local it,i,j,bu,bg;
  it:=InductionTable(u,g);
  it.head:=function(t,option)
    if IsBound(option.TeX) then return SPrint("$j$-Induction from $",
      ReflectionName(t.u,option),"$ to $",ReflectionName(t.g,option),"$");
    else return SPrint("j-Induction from ",
      ReflectionName(t.u,option)," to ",ReflectionName(t.g,option));fi;end;
  it.what:="jInductionTable";
  bu:=LowestPowerFakeDegrees(u);bg:=LowestPowerFakeDegrees(g);
  for i in [1..Length(bu)] do
    for j in [1..Length(bg)] do
      if bg[j]<>bu[i] then it.scalar[j][i]:=0;fi;
    od;
  od;
  return it;
end;

#############################################################################
##
#F  EltWord(W,q) . . . General method for Element from word in the generators
##
PermRootOps.EltWord:=function(W,w)
  if w=[] then return W.identity;fi;
  return ApplyWord(w,Parent(W).reflections);
end;

PermRootOps.ReflectionName:=function(W,opts)return
  ReflectionName(ReflectionType(W),opts);end;

# W is an irreducible CRG of known rank r. We try to classify W by:
#  1- size
#  2- o:={the set of divisors of the orders of generating reflections}  
#  3- reflection degrees
#
# G(de,e,r)  has  :
#  size=Factorial(r)*d^r*e^(r-1)  
#  o=divisors{2,d}
#  degrees=Concatenation(e*d*[1..r-1],[r*d])
#
# One  can  thus  check  that  (r,size,o)  are  sufficient to find a unique
# matching   G(de,e,r)  excepted  for   ambiguity  between  G(4e,4e,2)  and
# G(2e,e,2),  which is then resolved  by degrees [remaining ambiguities are
# isomorphisms].  The  possible  conflicts  ---  same  (r,size,o)  --- with
# primitive    groups   are:   G9/G(24,6,2)   ;   G12/G(12,6,2)/G(24,24,2);
# G13/G(24,12,2)/G(48,48,2)  G22/G(60,30,2)/G(120,120,2). They are resolved
# by ReflectionDegrees and AbelianInvariants.
PermRootOps.typeirred:=function(W)local s,list,r,o,n,de,ST,d,e;
  list:=
  [ rec( deg := [ 4, 6 ], o := [ 3 ] , series:="ST"),   #4
    rec( deg := [ 6, 12 ], o := [ 3 ] , series:="ST"),
    rec( deg := [ 4, 12 ], o := [ 2, 3 ] , series:="ST"), 
    rec( deg := [ 12, 12 ], o := [ 2, 3 ] , series:="ST"), 
    rec( deg := [ 8, 12 ], o := [ 4 ] , series:="ST"),
    rec( deg := [ 8, 24 ], o := [ 2, 4 ], series:="ST"),#9
    rec( deg := [ 12, 24 ], o := [ 3, 4 ], series:="ST"),
    rec( deg := [ 24, 24 ], o := [ 2, 3, 4 ], series:="ST"),
    rec( deg := [ 6, 8 ], o := [ 2 ], series:="ST"),    #12
    rec( deg := [ 8, 12 ], o := [ 2 ], series:="ST"),   #13
    rec( deg := [ 6, 24 ], o := [ 2, 3 ], series:="ST"),#14
    rec( deg := [ 12, 24 ], o := [ 2, 3 ], series:="ST"),
    rec( deg := [ 20, 30 ], o := [ 5 ], series:="ST"),
    rec( deg := [ 20, 60 ], o := [ 2, 5 ], series:="ST"),
    rec( deg := [ 30, 60 ], o := [ 3, 5 ], series:="ST"),
    rec( deg := [ 60, 60 ], o := [ 2, 3, 5 ], series:="ST"),
    rec( deg := [ 12, 30 ], o := [ 3 ], series:="ST"), 
    rec( deg := [ 12, 60 ], o := [ 2, 3 ], series:="ST"),
    rec( deg := [ 12, 20 ], o := [ 2 ], series:="ST"),  #22
    rec( deg := [ 2, 6, 10 ], o := [ 2 ], series:="H"),
    rec( deg := [ 4, 6, 14 ], o := [ 2 ], series:="ST"), 
    rec( deg := [ 6, 9, 12 ], o := [ 3 ], series:="ST"),
    rec( deg := [ 6, 12, 18 ], o := [ 2, 3 ], series:="ST"),
    rec( deg := [ 6, 12, 30 ], o := [ 2 ], series:="ST"), 
    rec( deg := [ 2, 6, 8, 12 ], o := [ 2 ], series:="F"), 
    rec( deg := [ 4, 8, 12, 20 ], o := [ 2 ], series:="ST"),
    rec( deg := [ 2, 12, 20, 30 ], o := [ 2 ], series:="H"),
    rec( deg := [ 8, 12, 20, 24 ], o := [ 2 ], series:="ST"),
    rec( deg := [ 12, 18, 24, 30 ], o := [ 3 ], series:="ST"),
    rec( deg := [ 4, 6, 10, 12, 18 ], o := [ 2 ], series:="ST"),
    rec( deg := [ 6, 12, 18, 24, 30, 42 ], o := [ 2 ], series:="ST"),
    rec( deg := [ 2, 5, 6, 8, 9, 12 ], o := [ 2 ], series:="E"),
    rec( deg := [ 2, 6, 8, 10, 12, 14, 18 ], o := [ 2 ], series:="E"),
    rec( deg := [ 2, 8, 12, 14, 18, 20, 24, 30 ], o := [ 2 ], series:="E") ];
  r:=RankMat(W.roots{W.generatingReflections});
  s:=Size(W);
  # Now find de=possible [d,e] (using s and r only)
  d:=s/Factorial(r);
  if not IsInt(d) then de:=[];
  elif d=r+1 then return rec(series:="A",rank:=r);
  elif r=1 then return rec(series:="ST",p:=d,q:=1,rank:=1);
  else de:=List(Cartesian(List(Collected(Factors(d)),function(x)local u,a,res;
      u:=x[2]; a:=QuoInt(u+r-1,r); res:=[];
      # find list of possible [a,b] such that u=ar-b with b<=a 
      while a*r-u<=a do Add(res,[x[1]^(a+u-a*r),x[1]^(a*r-u)]);a:=a+1; od;
      return res;
    end)),p->[Product(p,x->x[1]),Product(p,x->x[2])]);
  fi;
  # now use o
  o:=Set(Concatenation(List(W.OrdersGeneratingReflections,DivisorsInt)));
  de:=Filtered(de,x->o=Set(Concatenation(List([2,x[1]],DivisorsInt))));
  #Print("r=",r," o=",o," s=",s,"\n");
  #Print("de=",de,"\n");
  if Length(de)>1 then # we have G(2e,e,2),G(4e,4e,2) and maybe G12,G13,G22
    if Length(de)<>2 then Error("theory");fi;
    Sort(de);
    d:=Minimum(Filtered(ReflectionDegrees(W),x->x<>1));
    if d=4 then de:=[de[2]];elif d=2 then de:=[de[1]];else de:=[];fi;
  fi;
  ST:=Filtered([1..Length(list)],i->r=Length(list[i].deg) and
    s=Product(list[i].deg) and 
    o=Set(Concatenation(List(list[i].o,DivisorsInt))))+3;
  if Length(de)>0 and Length(ST)>0 then # [G(24,8,2),G7,G14]
  # [G(12,3,2),G8] [G(24,6,2),G9] [G(48,16,2),G15] [G(120,24,2),G17]
  # [G(120,40,2),G21]
    d:=Filtered(ReflectionDegrees(W),x->x<>1);
    if Gcd(d)>=12 then de:=[];if 14 in ST then ST:=[7];fi;
    else # [G(24,8,2),G14] [G(12,3,2),G8] [G(24,6,2),G9]
      if 14 in ST then ST:=[14];fi;
      d:=AbelianInvariants(W);
      if Length(d)=3 or (8 in ST and Length(d)=2) then ST:=[];
      else de:=[];
      fi;
    fi;
  fi;
  if Length(de)=0 then
    if Length(ST)=0 then Error("theory");
    elif list[ST[1]-3].series="ST" then 
	  return rec(series:="ST",ST:=ST[1],rank:=r);
    else  return rec(series:=list[ST[1]-3].series,rank:=r);
    fi;
  fi;
  d:=de[1][1];e:=de[1][2];
  if d=2 and e=1 then return rec(series:="B",rank:=r);fi;
  if d=1 then
    if e=2 then return rec(series:="D",rank:=r);
    elif r=2 then 
      if e=4 then return rec(series:="B",rank:=2);
      elif e=6 then return rec(series:="G",rank:=2);
      else return rec(series:="I",rank:=2,bond:=e);
      fi;
    fi;
  fi;
  return rec(series:="ST",p:=d*e,q:=e,rank:=r);
end;

PermRootOps.ReflectionType:=function(W)
  local type,l,f,t,d,findgoodgens,findgensCartan,findgensSymCartan,
    findgensDiagCartan,fixCartan,representativediagconjugation;

# find elements of H satisfying braid/order relations of type t in 
# sublist g of H.rootInclusion
  findgoodgens:=function(H,g,t)local findarr,rels,j,mj,orders;
    rels:=[];
    for j in BraidRelations(t) do 
      mj:=Maximum(Concatenation(j));
      if not IsBound(rels[mj]) then rels[mj]:=[];fi;
      Add(rels[mj],j);
    od;
    orders:=List(ReflTypeOps.EigenvaluesGeneratingReflections(t),x->1/x);
    findarr:=function(gens,rest)local i,e,res,newgens,r;
      # check gens satisfy relation rel
      if Length(gens)=Length(orders) then 
        if Size(Subgroup(H,gens))=Size(H) then return [];
	else return false;
	fi;
      fi;
      i:=Length(gens)+1;
      for e in rest do
#       Print(e,"(",Length(gens),")\c");
        r:=Reflection(H,H.rootRestriction[e]);
	if Order(H,r)=orders[i] then
	  newgens:=Concatenation(gens,[r]);
	  if  not IsBound(rels[i]) or 
	      ForAll(rels[i],r->CheckRelation(newgens,r)) then
	    res:=findarr(newgens,Difference(rest,[e]));
	    if res<>false then return Concatenation([e],res); 
#           else Print("R\c");
            fi;
#         else Print("r\c");
	  fi;
#       else Print("o\c");
	fi;
      od;
#     Print(rest,"\n");
      return false;
    end;
    return findarr([],g);
  end;

  # find a tuple of roots of H with sym(CartanMat) equal to sym(C)
  # sym is obtained by replacing C[i][j] by C[i][j]*C[j][i] if i<>j
  findgensSymCartan:=function(H,C)local complete,orders,refs,sym;
#   Print("findgensCartan \c");Stime();
    orders:=List(1-DiagonalOfMat(C),x->Denominator(AsRootOfUnity(x)));
    refs:=List(orders,o->Filtered(H.rootInclusion,
        i->OrderPerm(Reflection(H,H.rootRestriction[i]))=o));
    complete:=function(l)local r,c,n;n:=Length(l);
      if n=Length(C) then return l;fi;
      for r in refs[n+1] do
        if ForAll([1..n],i->CartanCoefficient(H,l[i],r)*
	   CartanCoefficient(H,r,l[i])=C[i][n+1]*C[n+1][i])
	then c:=complete(Concatenation(l,[r]));if c<>false then return c;fi;
        fi;
      od;
      return false;
    end;
    C:=complete([]);
#   Print("takes ",Stime(),"\n");
    return C;
  end;

  # find a tuple of roots of H with CartanMat equal to C
  findgensCartan:=function(H,C)local complete,diag;
    diag:=List(H.rootInclusion,r->CartanCoefficient(H,r,r));
    complete:=function(l)local r,c,L;L:=Length(l);
      if L=Length(C) then return l;fi;
      for r in H.rootInclusion do if C[L+1][L+1]=diag[H.rootRestriction[r]] and 
        ForAll([1..L],i->CartanCoefficient(H,l[i],r)=C[i][L+1]) and
        ForAll([1..L],i->CartanCoefficient(H,r,l[i])=C[L+1][i])
	then c:=complete(Concatenation(l,[r]));
	     if c<>false then return c;fi;
        fi;
      od;
      return false;
    end;
    return complete([]);
  end;

  # find a tuple of roots of H with CartanMat equal to C^diagonal matrix
  findgensDiagCartan:=function(H,C)local complete,f;
    f:=function(x,y)
      if y=0 then if x=0 then return 0;else return false;fi;else return x/y;fi;
    end;
    complete:=function(l,d)local r,c,cc,n;
      n:=Length(l)+1;
      if Length(l)=Length(C) then return [l,d];fi;
      for r in H.rootInclusion do
	cc:=List([1..n-1],i->f(d[i]*C[i][n],CartanCoefficient(H,l[i],r)));
       Append(cc,List([1..n-1],i->f(d[i]*CartanCoefficient(H,r,l[i]),C[n][i])));
        cc:=Set(cc);cc:=Difference(cc,[0]);
	if Length(cc)<=1 and not false in cc then
	   if Length(cc)=0 then cc:=[1];fi;
	   c:=complete(Concatenation(l,[r]),Concatenation(d,[cc[1]]));
	   if c<>false then return c;fi;
        fi;
      od;
      return false;
    end;
    return complete([],[]);
  end;

  # make CartanMat(H,p) like C by rotating roots
  fixCartan:=function(H,C,p)local i,j,r,seen,CH;
    CH:=CartanMat(H,p);
    r:=FiniteCoxeterTypeFromCartanMat(CH)[1];
    if r<>false then return [r,p];fi;
    seen:=[Length(CH)];
    for i in [Length(CH),Length(CH)-1..1] do 
    # go reverse for better luck in type B?
      for j in [i-1,i-2..1] do
	if CH[i][j]<>C[i][j] then
	  if CH[i][j]=0 or j in seen then return false;fi;
	  r:=C[i][j]/CH[i][j];
	  r:=Position(H.roots,H.roots[H.rootRestriction[p[j]]]*r);
	  if r=false then return false;fi;
	  p:=ShallowCopy(p);p[j]:=H.rootInclusion[r]; 
	  return fixCartan(H,C,p);
	fi;
	if C[i][j]<>0 then Add(seen,j);fi;
      od;
    od;
    return [r,p];
  end;

  type:=List(DecomposedMat(CartanMat(W)),function(I)
    local msg,t,H,C,CH,p,i,j,res,ret;
    msg:=function(l,a)
      InfoChevie("#I ",l," ",a," for ",ReflectionName(t),"\n");end;
    ret:=function()local perm,t1;
      CH:=CartanMat(H,p);
      t1:=FiniteCoxeterTypeFromCartanMat(CH)[1];
      if t1<>false then t1.indices:=W.rootRestriction{p{t1.indices}};return t1;fi;
      perm:=PermMatMat(CH,C);
      if perm<>false then p:=Permuted(p,perm);
        if CartanMat(H,p)<>C then Error("E1");fi;
      else t1:=fixCartan(H,C,p);
        if t1<>false then p:=t1[2];
	  if t1[1]<>false then t:=t1[1];fi;
	fi;
      fi;
      t.indices:=W.rootRestriction{p};
      if C=CartanMat(H,p) then return t;fi;
      t1:=findgensSymCartan(H,C);
      if t1<>false then
        p:=t1;
        t.indices:=W.rootRestriction{p};
        if C=CartanMat(H,p) then return t;fi;
      fi;
      t1:=findgensCartan(H,C);
      if t1<>false then t.indices:=W.rootRestriction{t1};
        return t;fi;
      t1:=RepresentativeDiagonalConjugation(C,
                                    CartanMat(H,H.rootInclusion{t.indices}));
      if t1<>false then t.cartanType:=t1[2];
# this is unsufficient for a group like W(6,3,2): it could be W.roots[3]
# is not in the orbit of W.roots[2] ....
        return t;fi;
      p:=findgensDiagCartan(H,C);
      if p=false then Error("E3");fi;
      t.indices:=W.rootRestriction{p[1]};t.cartanType:=p[2];return t;
#     ChevieErr("# Cartan(",ReflectionName(t),
# ") not matched by\n",Format(CartanMat(H,H.rootInclusion{t.indices})),
# "\n");
    end;
    I:=W.generatingReflections{I};
    CH:=CartanMat(W,W.rootInclusion{I});
    t:=FiniteCoxeterTypeFromCartanMat(CH)[1];
    if t<>false then t.indices:=I{t.indices};return t;fi;
    if I=W.generatingReflections then H:=ShallowCopy(W);
    else H:=PermRootOps.ReflectionSubgroupNC(W,W.rootInclusion{I});
    fi;
    t:=PermRootOps.typeirred(H);t.operations:=ReflTypeOps;
    C:=CartanMat(t);
    if Length(CH)=Length(C) then 
      p:=PermMatMat(CH,C);
      if p<>false then t.indices:=Permuted(I,p);return t;fi;
      msg(W.rootInclusion{I},"non-standard roots");
      p:=findgoodgens(H,W.rootInclusion{I},t);
      if p<>false then msg(p,"satisfy relations");return ret();fi;
    fi;
    if Length(I)=1+t.rank then 
       msg(W.rootInclusion{I},"non-standard generators");
    fi;
    p:=[1..Length(H.roots)];
    # try to get roots in same orbit by sorting as follows
    SortBy(p,x->[H.orbitRepresentative[x],H.rootInclusion[x]]);
    p:=List([1..Length(p)],i->First(p,j->Reflection(H,j)=Reflection(H,i)));
    p:=findgoodgens(H,Set(H.rootInclusion{p}),t);
    if p=false then Error("could not find generators");fi;
    # try to adjust the roots to fix the Cartan mat
    return ret();
  end);
  l:=Concatenation(List(type,x->x.indices));
  W.type:=type;
  if PermListList(W.generatingReflections,l)=false then
    if Length(W.generatingReflections)>2*Length(l) then
      InfoChevie("# changing generatingReflections to <",
        Join(W.rootInclusion{l}),"> for ",ReflectionName(W.type),"<",
       Length(W.generatingReflections)," refs>\n");
    else InfoChevie("# changing generatingReflections to <",
      Join(W.rootInclusion{l}),"> for ",ReflectionName(W.type),"<",
       Join(W.rootInclusion{W.generatingReflections}),">\n");
    fi;
    if IsParent(W) then
      t:=PermRootGroupNC(W.roots{l},List(l,i->PermRootOps.Coroot(W,i)));
    else
      t:=PermRootOps.ReflectionSubgroupNC(Parent(W),W.rootInclusion{l});
    fi;
    Inherit(W,t);
    for t in type do
      t.indices:=List(t.indices,x->Position(l,x));
      for d in ["charparam","classInfo","reflectionName"] do Unbind(t.(d));od;
      d:=Difference(RecFields(t),["series","rank","indices","ST","operations",
       "ReflectionCoDegrees","degrees","p","q","bond","cartanType"]);
      if Length(d)>1 then Error(d);fi;
    od;
    # the next fields if computed are now incorrect (they should be
    # recomputed as HasTypeOps)
    for f in ["cartan","independentRoots","forMatX","conjugacyClasses",
	 "charTable","reflectionCharacter","eigenvalues","trivialSubgroup",
	 "orbit","transversal","stabilizer","stabChainOptions","stabChain",
	 "dixon","derivedSubgroup","commutatorFactorGroup","elements"]
    do Unbind(W.(f));od;
#   if Length(Union(List(ConjugacyClasses(W),Elements)))<>Size(W) then Error();fi;
  fi;
  W.operations:=ComplexGroupOps;
  return type;
end;

PermRootOps.FieldOfDefinition:=W->NF(Flat(CartanMat(W)));

PermRootOps.ReflectionLength:=function(W,w)
  if not IsBound(W.reflectionLengths) then
    W.reflectionLengths:=List(ReflectionEigenvalues(W),x->Number(x,y->y<>0));
  fi;
  return W.reflectionLengths[PositionClass(W,w)];
end;

# ReflectionWord(W,w[,list of reflections])
ReflectionWord:=function(arg)local W,w,ref,res,ref,l,r;
  W:=arg[1];w:=arg[2];res:=[];
  if Length(arg)=3 then ref:=arg[3];else ref:=Reflections(W);fi;
  l:=ReflectionLength(W,w);
  while l>0 do
    r:=PositionProperty(ref,x->ReflectionLength(W,x^-1*w)<l);
    if r=false then Error(w," has no reduced expression in the given reflections");fi;
    Add(res,r);
    w:=ref[r]^-1*w;
    l:=l-1;
  od;
  return res;
end;

ReadChv("prg/eigenspaces"); # to have some more PermRootOps methods defined

# detects if H is a parabolic subgroup  of Parent(H)
IsParabolic:=function(H)local v,gens,W,setr;W:=Parent(H);
  setr:=s->Set(List(s,x->Reflection(W,x)));
  if H.nbGeneratingReflections=0 then return true;fi;
  v:=VectorSpace(W.roots{H.rootInclusion{H.generatingReflections}},Cyclotomics);
  gens:=Filtered([1..Length(W.roots)],i->W.roots[i] in v);
  return setr(gens)=setr(H.rootInclusion);
end;

PermRootOps.ParabolicRepresentatives:=function(W)
  local by,l,i,v,S,c,O,new,stoi,ref,rr,cand;
  if Length(W.type)>1 then Error("need irreducible W");fi;
  by:=1+W.nbGeneratingReflections-W.semisimpleRank;
  stoi:=v->List(v,s->W.rootInclusion[Position(Reflections(W),s)]);
  l:=[List(Set(W.orbitRepresentative),x->ReflectionSubgroup(W,[x]))];
  for i in [2..W.semisimpleRank-1] do
    new:=[];ref:=[];
    for v in l[i-1] do
      InfoChevie("# Extending ",ReflectionName(v),"\c");
      S:=Normalizer(W,v);
      if v.nbGeneratingReflections=i-1 then
        c:=Union(List([1..by],i->Combinations(Set(stoi(Reflections(W))),i)));
      else c:=Combinations(Set(stoi(Reflections(W))),1);
      fi;
      c:=List(c,x->Union(x,v.rootInclusion{v.generatingReflections}));
      c:=Filtered(c,x->RankMat(W.roots{W.rootRestriction{x}})=i);
      InfoChevie(" ",Length(c)," new subgroups\c");
      c:=List(c,function(x)Print("*\c");return ReflectionSubgroup(W,x);end);
      c:=Filtered(c,IsParabolic);
      c:=Set(List(c,x->Set(Reflections(x))));
      O:=Orbits(S,c,OnSets);
      O:=List(O,function(o)local m;m:=Minimum(List(o,x->Sum(stoi(x))));
           return First(o,x->Sum(stoi(x))=m);end);
      O:=List(O,x->ReflectionSubgroup(W,stoi(x)));
      InfoChevie("# ",Length(O)," to go\n");
      for c in O do
        rr:=Collected(ReflectionDegrees(c));
	cand:=Filtered([1..Length(ref)],i->ref[i]=rr);
	InfoChevie("# candidates for ",c," to be conjugate:",new{cand},"\n");
	if ForAll(new{cand},v->RepresentativeOperation(W,v,c)=false)
	then InfoChevie("# new:",ReflectionName(c),"\n");Add(new,c);Add(ref,rr);
	fi;
      od;
    od;
    InfoChevie("# i=",i," found:",new,"\n");
    Add(l,new);
  od;
  l:=List(Concatenation(l),function(v)local p;
    p:=StandardParabolic(W,v);if p<>false then v:=v^p;fi;return v;end);
  l:=List(l,x->x.rootInclusion{x.generatingReflections});
  Add(l,[]);Add(l,W.rootInclusion{W.generatingReflections});
  l:=List(l,x->List(x,y->W.rootInclusion[Position(Reflections(W),
    Reflection(W,W.rootRestriction[y]))]));
  l:=List(l,Set);
  SortBy(l,x->[Length(x),x]);
  return l;
end;
