#############################################################################
##
#A  wclassinv.g               CHEVIE library                     Frank Luebeck
##
#A  $Id: wclsinv.g,v 1.1 1997/01/21 13:46:37 gap Exp $
##  
#Y  Copyright (C) 1992 - 1996  Lehrstuhl D f\"ur Mathematik, RWTH Aachen, IWR
#Y  der Universit\"at Heidelberg, University of St. Andrews, and   University 
#Y  Paris VII.
##  
##  This  file contains   special  functions for  computing invariants  of
##  conjugacy classes in Coxeter groups and Coxeter cosets.
##  
##  The main difference   to the general functions   'ClassInvariants' and
##  'PositionClass' for permutation groups is the use of the 'ClassParamX'
##  functions defined  for the  different  types of simple  groups. If the
##  class of  an element cannot be identified  cheaply then the element is
##  transformed into a   word and the CHEVIE  label  for its corresponding
##  conjugacy class is computed by quite efficient functions.
##  

#############################################################################
##
#F  ComponentWordsPerm( <W>, <w> ) . . . . . . . 
#F  ComponentWordsPermCoset( <WF>, <w> ). . . . . similar to 'CoxeterWord'
#F  but giving lists of  words in standard generators of simple components
##    
ComponentWordsPerm:=function(W,w)local i,l,cw;
  if IsBound(W.forComponentWords) then cw:=W.forComponentWords;
  else
    cw:=[[],[]];
    for i in [1..Length(W.type)] do
      l:=W.type[i].indices;
      cw[1]{l}:=l*0+i;
      cw[2]{l}:=[1..Length(l)];
    od;
    W.forComponentWords:=cw;
  fi;
  
  l := List(W.type,i->[]);
  while true  do
    i := FirstLeftDescending(W,w);
    if i=false then return l;fi;
    Add( l[cw[1][i]], cw[2][i]);
    w := W.reflections[i] * w;
  od;
end;

ComponentWordsPermCoset:=function( WF, w ) local  W, i, j, l, cw, typ;

  W:=Group(WF); w:=w/WF.phi;

  if IsBound(WF.forComponentWordsCoset) then cw:=WF.forComponentWordsCoset;
  else
    cw:=[];
    typ:=ReflectionType(WF);
    for i in [1..Length(typ)] do
      for j in [1..Length(typ[i].orbit)] do
        for l in [1..Length(typ[i].orbit[j].indices)] do
          cw[typ[i].orbit[j].indices[l]]:=[i,j,l];
        od;
      od;
    od;
    WF.forComponentWordsCoset:=cw;
  fi;

  l := List(WF.type,i->List(i.orbit,j->[]));
  while true  do
    i := FirstLeftDescending(W,w);
    if i=false then return l;fi;
    Add( l[cw[i][1]][cw[i][2]], cw[i][3]);
    w := W.reflections[i] * w;
  od;
end;

#############################################################################
##
#F  CoxeterClassParamCheckFunction( <r>, <l> ) . . . . . . . 
#F  CoxeterCosetClassParamCheckFunction( <r>, <l> ) . . . . returns function
#F  which computes a label for the (F-)conjugacy class of a given element 
##  
##  This function  can be used as argument  for 'ClassInvariants' for Coxeter
##  groups or Coxeter cosets, respectively.   They return a function which is
##  always  able to distinguish all   classes (and this quite  effectively
##  compared to backtrack conjugacy tests).
##  
##  It uses the classification.
##  
CoxeterClassParamCheckFunction:=function(r,l)
  return function(x) local cw, typ; 
    cw:=ComponentWordsPerm(r.g,x);
    typ:=r.g.type;
    return List([1..Length(typ)],i->CHEVIE.Data("ClassParameter",typ[i],cw[i]));
  end;
end;
       
CoxeterCosetClassParamCheckFunction:=function(r,l)
  return function(x) local i, j, h, cw, typ; 
    cw:=ComponentWordsPermCoset(r.g.coset,x);
    
    # change to other representative in same F-conjugacy class,
    # which is of form (x,1,...,1) on F orbits of simple
    # components:
    for i in [1..Length(cw)] do
      h:=[];
      for j in [2..Length(cw[i])] do
	h:=Concatenation(h,Reversed(cw[i][j]));
      od;
      cw[i]:=Concatenation(cw[i][1],Reversed(h));
    od;

    typ:=r.g.coset.type;
    return List([1..Length(typ)],i->CHEVIE.Data("ClassParameter",typ[i],cw[i]));
  end;
end;

#############################################################################
##
#F  CoxeterGroupOps.ClassInvariants(  ... )  . . . . . 'ClassInvariants' for
#F  Coxeter groups
#F  CoxeterGroupOps.PositionClass( ... ) . . . . . . . . . 'PositionClass' for
#F  Coxeter groups
##
##  Here we use the  ClassParam for irreducible types to  distinguish classes
##  which are not distinguished by cycle  type (and cycle type of elements
##  multiplied by center elements and not lying in small classes).
##  
# the command 
#     CoxeterGroupOps.ClassInvariants:=CoxeterGroupOpsClassInvariants;
# is in coxeter.g
CoxeterGroupOpsClassInvariants:=function(arg)
  if IsList(arg[1]) then
    arg:=arg[1];
  fi;
  if Length(arg)=1 then
    return PermGroupOps.ClassInvariants(arg[1],
                   CentreMultFunction,
                   ShortClassListFunction,
                   CoxeterClassParamCheckFunction);
  else
    return PermGroupOps.ClassInvariants(arg);
  fi;
end;

#############################################################################
##
#F  CoxeterCosetOps.ClassInvariants(  ... )  . . . . . 'ClassInvariants' for
#F  Coxeter cosets
#F  CoxeterCosetOps.PositionClass( ... ) . . . . . . . 'PositionClass' for
#F  Coxeter cosets
##
##  Here we use the   CoxeterClassParam function to  distinguish classes
##  which are not distinguished by cycle  type (and cycle type of elements
##  multiplied by center elements and not lying in small classes).
##  
# the command 
#     CoxeterCosetOps.ClassInvariants:=CoxeterCosetOpsClassInvariants;
# is in coset.g

CoxeterCosetOpsClassInvariants:=function(WF)local tmp;
  if IsList(WF) then WF:=WF[1];fi;
  
  # in non twisted case use the function for Coxeter groups:
  if WF.phi=() then return ClassInvariants(Group(WF));fi;
  
  tmp:=Copy(Group(WF).generators);Add(tmp,WF.phi);
  tmp:=Group(tmp,());
  tmp.conjugacyClasses:=ConjugacyClasses(WF);
  tmp.size:=Size(WF);
  tmp.coset:=WF;
  
  return PermGroupOps.ClassInvariants(tmp,
    ShortClassListFunction,
    CoxeterCosetClassParamCheckFunction);
end;
