#############################################################################
##
#A  ucl.g      Unipotent Classes    Jean Michel
##
#Y  Copyright 2008-2010, Univ. Paris VII
##  
##  This file contains basic functions for unipotent classes of
##  reductive groups.
##  

UnipotentClassesOps:=OperationsRecord("UnipotentClassesOps");

UnipotentClassesOps.String:=function(r)local res;
  res:=SPrint("UnipotentClasses( ",ReflectionName(r.group));
  if r.p<>0 then PrintToString(res,",",r.p);fi;
  Append(res," )");return res;
end;

UnipotentClassesOps.Print:=function(r)Print(String(r));end;

# chars is a list of indices of characters of the group Au.
# If  k is the common kernel of chars, we return [Au/k,chars1] where chars1
# is the index of chars as characters of Au/k.
# Since  GAP3  has  many  problems  with  quotient groups, we are forced to
# program an ad hoc solution which works only for Au occuring as A_G(u) for
# unipotent classes of a reductive group G.
QuotientAu:=function(Au,chars)local ct,cl,k,f,q,n,d,g,AbGens,finish,e,Z,h;
  AbGens:=function(g)local l,res,o;
    res:=[];l:=g.generators;
    while l<>[] do
      o:=List(l,x->Order(g,x));
      Add(res,l[Position(o,Maximum(o))]);
      l:=Difference(l,Elements(Subgroup(g,res)));
    od;
    return res;
  end;
  # q=Au/k,  ww=words in q images of Au.generators
  finish:=function(q,ww)local h,fusion,ctu,cth;
    h:=GroupHomomorphismByImages(Au,q,Au.generators,List(ww,x->EltWord(q,x)));
    fusion:=List([1..NrConjugacyClasses(Au)],i->PositionClass(q,
      Image(h,EltWord(Au,ChevieClassInfo(Au).classtext[i]))));
    ctu:=CharTable(Au).irreducibles;
    cth:=CharTable(q).irreducibles;
    return rec(Au:=q,chars:=List(chars,
      c->Position(cth,List([1..NrConjugacyClasses(q)],
      j->ctu[c][Position(fusion,j)]))));
  end;
  Z:=n->ComplexReflectionGroup(n,1,1);
  ct:=TransposedMat(CharTable(Au).irreducibles{chars});
  cl:=Filtered([1..Length(ct)],i->ct[i]=ct[1]);
  if Length(cl)=1 then return rec(Au:=Au,chars:=chars);fi;
  ct:=TransposedMat(Set(ct));
  k:=Subgroup(Au,Union(List(cl,i->Elements(ConjugacyClasses(Au)[i]))));
  if Size(k)=Size(Au) then
    return rec(Au:=CoxeterGroup(),chars:=[1]);fi;
  if Length(ReflectionType(Au))=1 and Au.type[1].rank=1 then 
    return finish(Z(Size(Au)/Size(k)),[[1]]);
  elif IsAbelian(Au/k) then
    q:=Au/k;q.generators:=AbGens(q);h:=NaturalHomomorphism(Au,q);
    f:=List(Au.generators,x->GetWord(q,x^h));
 #  Print(Product(List(q.generators,x->Z(Order(q,x))))," ",f,"\n");
    return finish(Product(List(q.generators,x->Z(Order(q,x)))),f);
  elif ReflectionName(Au)="A2xA1" and Size(k)=2 and Au.3 in k.generators 
  then return finish(CoxeterGroup("A",2),[[1],[2],[]]); # chars=2,4,6
  elif ReflectionName(Au)="A1xB2" and Size(k)=2 
    and LongestCoxeterElement(Au) in k
  then return finish(CoxeterGroup("B",2),[[1,2,1,2],[1],[2]]);#chars=2,6,8,9,10
  fi;
# e:=WordEnumerator(Au,rec(monoid:=true));
# Print(" Au=",ReflectionName(Au)," sub=",List(k.generators,e.Get),"\n");
  Error();
# q:=Au/k; f:=FusionConjugacyClasses(Au,q); Print(" quot=",q," fusion=",f,"\n");
# return rec(Au:=Au,chars:=chars);
end;

# UnipotentClasses(W [,characteristic])
UnipotentClasses:=function(arg)local obj,f;obj:=arg[1];
  if not IsRec(obj) then Error(obj," has no method for UnipotentClasses");fi;
  if Length(arg)=1 then arg:=[arg[1],0];fi;
  f:=SPrint("unipotentclasses",arg[2]);
  if not IsBound(obj.(f)) then obj.(f):=Dispatcher("UnipotentClasses",arg);fi;
  return obj.(f);
end;

ReflTypeOps.UnipotentClasses:=function(t,p)local s;
  s:=["UnipotentClasses",t];
  if IsBound(t.cartanType) then Add(s,t.cartanType);fi;
  Add(s,p);
  return ApplyFunc(CHEVIE.Data,s);
end;

CoxeterGroupOps.UnipotentClasses:=function(W,p)
  local t,u,uc,ucl,ll,i,j,k,getf,f,p,g,l,chars,Au;
  t:=ReflectionType(W); uc:=List(t,x->UnipotentClasses(x,p));
  if false in uc then return false;fi;
  getf:=function(r,n,f)
    if IsBound(r.(n)) then return r.(n);else return f(r);fi;end;
  ucl:=rec(classes:=List(Cartesian(List(uc,x->x.classes)),function(v)local u;
    if Length(v)=1 then u:=Copy(v[1]);else u:=rec();fi;
    u.name:=Join(List(v,x->x.name));
    if Length(v)=0 then u.Au:=CoxeterGroup();else u.Au:=Product(v,x->x.Au);fi;
    u.parameter:=List(v,x->getf(x,"parameter",x->x.name));
    u.dimBu:=Sum(v,x->getf(x,"dimBu",x->0));
    if ForAll(v,x->IsBound(x.dynkin)) then
      u.dynkin:=[];
      for i in [1..Length(t)] do u.dynkin{t[i].indices}:=v[i].dynkin;od;
      if W.N=0 then u.dimBu:=0;
      else u.dimBu:=W.N-Number(W.roots*u.dynkin,x->not x in [0,1])/2;
      fi;
    fi;
    return u;end));
  ucl.size:=Length(ucl.classes);
  ll:=List(uc,x->Length(x.classes));
  ucl.orderClasses:=List(Cartesian(List(ll,x->[1..x])),function(v)local o;
    o:=Cartesian(List([1..Length(v)],j->
      Concatenation(uc[j].orderClasses[v[j]],[v[j]])));
    o:=List(o,x->PositionCartesian(ll,x));
    return Difference(o,[PositionCartesian(ll,v)]);
    end);
  ucl.springerSeries:=List(Cartesian(List(uc,x->x.springerSeries)),
    function(v)local s;
      if v=[] then 
        return rec(relgroup:=CoxeterGroup(),Z:=[],levi:=[],locsys:=[[1,1]]);
      fi;
      s:=rec( 
	relgroup:=Product(List(v,x->x.relgroup)),
	levi:=Concatenation(List([1..Length(v)],i->t[i].indices{v[i].levi})),
	Z:=Concatenation(List(v,x->x.Z)),
	locsys:=List(Cartesian(List(v,x->x.locsys)),function(v)
	  v:=TransposedMat(v);
	  v[2]:=PositionCartesian(List([1..Length(v[1])],i->
	    NrConjugacyClasses(uc[i].classes[v[1][i]].Au)),v[2]);
	  v[1]:=PositionCartesian(ll,v[1]);
	  return v;end));
      if ForAll(v,x->IsBound(x.parameter)) then
        s.parameter:=List(v,x->x.parameter);
      fi;
      if Length(v)=1 then Inherit(s,v[1],Difference(RecFields(v[1]),
	["relgroup","levi","Z","locsys","parameter"]));fi;
      return s;end);
  ucl.operations:=UnipotentClassesOps;
  ucl.group:=W;
  ucl.p:=p;
  if Length(uc)=1 then Inherit(ucl,uc[1],Difference(RecFields(uc[1]),
    ["group","operations","springerSeries","classes","orderClasses"]));fi;
# To deal with a general group intermediate between Gad and Gsc, we discard
# the  Springer series  corresponding to  a central  character which is not
# trivial on the fundamental group (seen as a subgroup of ZGsc)
# We  quotient the Aus by the common  kernel of the remaining characters of
# Au. In the following l is the list of kept SpringerSeries
  g:=DescAZ(W);
  l:=Filtered([1..Length(ucl.springerSeries)],i->
    ForAll(g,y->Product(ucl.springerSeries[i].Z{y})=1));
  ucl:=ShallowCopy(ucl); ucl.springerSeries:=ucl.springerSeries{l};
  for i in [1..Size(ucl)] do
    l:=List(ucl.springerSeries,s->
       Filtered([1..Length(s.locsys)],k->s.locsys[k][1]=i));
    chars:=Concatenation(List([1..Length(l)],j->
      List(ucl.springerSeries[j].locsys{l[j]},x->x[2])));
    Au:=ucl.classes[i].Au;
    f:=QuotientAu(Au,chars);
#   if Size(Au)<>Size(f.Au) then
#     Print("class ",i,"=",ucl.classes[i].name," ",[Au,chars],"=>",f,"\n");
#   fi;
    ucl.classes[i].Au:=f.Au;chars:=f.chars;
    k:=1;
    for j in [1..Length(l)] do 
      ucl.springerSeries[j].locsys:=Copy(ucl.springerSeries[j].locsys);
      for f in l[j] do ucl.springerSeries[j].locsys[f][2]:=chars[k];k:=k+1;od;
    od;
  od;
  return ucl;
end;

# compute name of i-th class according to options opt
UnipotentClassesOps.ClassName:=function(uc,i,opt)local n,cl;
  cl:=uc.classes[i];
  if IsBound(opt.mizuno) and IsBound(cl.mizuno) then n:=cl.mizuno;
  elif IsBound(opt.shoji) and IsBound(cl.shoji) then n:=cl.shoji;
  else n:=cl.name;fi;
  if not IsBound(opt.TeX) then 
    n:=String(Replace(n,"_","","\\tilde ","~","{","","}",""));fi;
  if IsBound(opt.locsys) then
    if opt.locsys=PositionId(cl.Au) then return n;fi;
    cl:=SPrint("(",CharNames(cl.Au,opt)[opt.locsys],")");
    if IsBound(opt.TeX) then return SPrint(n,"^{",cl,"}");
    else return SPrint(n,cl);fi;
  elif IsBound(opt.class) then
    if opt.class=PositionId(cl.Au) then return n;fi;
    cl:=ChevieClassInfo(cl.Au).classnames[opt.class];
    if IsBound(opt.TeX) then return SPrint("\\hbox{$",n,"$}_{(",cl,")}");
    else return SPrint(n,"_",cl);fi;
  else return n;
  fi;
end;

UnipotentClassesOps.Poset:=function(uc)local res;
  res:=Poset(uc.orderClasses); res.uc:=uc;
  res.label:=function(p,n,opt)return ClassName(p.uc,n,opt);end;
  return res;
end;

UnipotentClassesOps.DisplayOptions:=rec(
 order:=true);

UnipotentClassesOps.Format:=function(uc,opt)local W,sp,tbl,res,p,ds;
  p:=ShallowCopy(UnipotentClassesOps.DisplayOptions);
  Inherit(p,opt);
  opt:=p;
  opt.rowLabels:=List([1..Size(uc)],i->ClassName(uc,i,opt));
  if opt.order then res:=Format(Poset(uc),opt);else res:="";fi;
  sp:=List(uc.springerSeries,ShallowCopy);
  if IsBound(opt.fourier) then
    for p in sp do p.locsys:=p.locsys{DetPerm(p.relgroup)};od;
  fi;
  W:=uc.group;
  tbl:=List(uc.classes,function(u)local i,res;
    i:=Position(uc.classes,u);
    if uc.p=0 then res:=[IntListToString(u.dynkin)];else res:=[];fi;
    Append(res,[u.dimBu,ReflectionName(u.Au,opt)]);
    if IsBound(opt.TeX) then p:="\\kern 0.8em ";else p:=" ";fi;
    Append(res,List(sp,ss->Join(List(PositionsProperty(ss.locsys,y->y[1]=i),
      function(i)local c1,c2;
        c1:=CharNames(u.Au,opt)[ss.locsys[i][2]];
	c2:=CharNames(ss.relgroup,opt)[i];
	if c1="" then return c2;else return SPrint(c1,":",c2);fi;
	end),p)));return res;end);
  opt.rowsLabel:="u";
  if uc.p=0 then 
    if IsBound(opt.TeX) then opt.columnLabels:=["\\hbox{diagram}"];
    else opt.columnLabels:=["diagram"];fi;
  else opt.columnLabels:=[];fi;
  if IsBound(opt.TeX) then Add(opt.columnLabels,"\\dim{\\cal B}_u");
                      else Add(opt.columnLabels,"Dim Bu"); fi;
  Add(opt.columnLabels,"A(u)");
  ds:=function(ss)local res;res:=SPrint(ReflectionName(ss.relgroup,opt),
   "(",IsomorphismType(ReflectionSubgroup(W,W.rootInclusion{ss.levi}),opt),")");
    if not ForAll(ss.Z,x->x=1) then PrintToString(res,"/",
        Join(List(ss.Z,x->Format(x,opt))));fi;return res;end;
  Append(opt.columnLabels,List(sp,ds));
  if not IsBound(opt.rows) then
    p:=SortingPerm(List(uc.classes,x->x.dimBu));
    tbl:=Permuted(tbl,p);opt.rowLabels:=Permuted(opt.rowLabels,p);
  fi;
  Append(res,FormatTable(tbl,opt));
  return res;
end;

UnipotentClassesOps.Display:=function(uc,opt)
  opt:=ShallowCopy(opt);opt.screenColumns:=SizeScreen()[1];
  Print(Format(uc,opt));
end;

# coefficients of R_\chi on unipotently supported local systems 
# ICCTable(uc[,seriesno[,variable]]) eg (uc,1,X(Rationals))
# Works for G split.
ICCTable:=function(arg)local W,i,q,tbl,o,res,q,uc,ss,b,f,k,R,n;
  uc:=arg[1];W:=uc.group;
  if Length(arg)<2 then i:=1;else i:=arg[2];fi;
  q:=Indeterminate(Rationals);
  ss:=uc.springerSeries[i];
  res:=rec(group:=W,relgroup:=ss.relgroup,series:=i,q:=q,p:=uc.p);
  if IsBound(ss.warning) then Print("# ",ss.warning,"\n");
    res.warning:=ss.warning;fi;
# We are going to solve the equation in "unipotent support", page 151
# ${}^tP\Lambda P=\omega$
# where $\Lambda_{i,j}$ is  $\sum_{g\in G^F} Y_i(g)\overline{Y_j(g)}$
# and $\Omega_{i,j}$ is equal to
# $|Z^0(G^F)|q^{-b_i-b_j-\text{semisimple rank}L} 
#         FakeDegree(\chi_i\otimes\chi_j\otimes\sgn) |G^F|/P(W_G(L))$
# and  where $P(W_G(L))$ is the Poincare polynomial $\prod_i(q^{d_i}-1)$
# where $d_i$ are the reflection degrees of $W_G(L)$
# res.scalar below is thus the $P_{i,j}$
  R:=ss.relgroup; f:=FakeDegrees(R,q);k:=PositionDet(R);
  n:=Length(f);
# Partition on characters of ss.relgroup induced by poset of unipotent classes
  res.dimBu:=List(ss.locsys,x->uc.classes[x[1]].dimBu);
  res.blocks:=CollectBy([1..Length(ss.locsys)],-res.dimBu);
  tbl:=BigCellDecomposition(List([1..n],i->List([1..n],
  # q^{-b_i-b_j}*matrix of FakeDegrees(chi_i tensor chi_j tensor sgn)
     j->q^(-res.dimBu[i]-res.dimBu[j])*f*DecomposeTensor(R,i,j,k))),
   res.blocks);
  res.scalar:=tbl[1];
  res.locsys:=ss.locsys;
  res.L:=tbl[2]*GenericOrder(W,q)/Product(ReflectionDegrees(R),d->q^d-1)/
    q^(W.semisimpleRank-R.semisimpleRank);
  res.uc:=uc;
  if IsBound(ss.parameter) then res.parameter:=ss.parameter;
  else res.parameter:=[1..Length(ss.locsys)];
  fi;
  if Length(arg)=3 and q<>arg[3] then
    q:=arg[3];
    res.scalar:=List(res.scalar,x->List(x,y->Value(y,q)));
    res.L:=List(res.L,x->List(x,y->Value(y,q)));
  fi;
  res.operations:=rec(
    String:=x->SPrint("ICCTable(",W,",",i,",",q,")"),
    Print:=function(x)Print(String(x));end,
    Format:=function(x,opt)local tbl,res;
     if IsBound(opt.TeX) then
     res:=SPrint("Coefficients of $X_\\phi$ on $Y_\\psi$ for $",
        ReflectionName(x.relgroup,opt),"$\n\\medskip\n\n");
     else
     res:=SPrint("Coefficients of X_phi on Y_psi for ",
        ReflectionName(x.relgroup,opt),"\n\n");
     fi;
     if not IsBound(opt.columns) and not IsBound(opt.rows) then 
       opt.rows:=[1..Length(x.dimBu)];SortParallel(-x.dimBu,opt.rows);
       opt.columns:=opt.rows;
     fi;
     tbl:=Copy(x.scalar);
     if not IsBound(opt.CycPol) then opt.CycPol:=true;fi;
     if opt.CycPol then tbl:=List(tbl,x->List(x,
       function(p)p:=CycPol(p);p.vname:="q";return p;end));fi;
     opt.columnLabels:=List(x.locsys,p->ClassName(x.uc,p[1],
         Inherit(rec(locsys:=p[2]),opt)));
     opt.rowLabels:=List(CharNames(x.relgroup,opt),
       function(x)if IsBound(opt.TeX) then return SPrint("X_{",x,"}");
         else return SPrint("X",x);fi;end);
     PrintToString(res,FormatTable(TransposedMat(tbl),opt));
     return String(res);
     end,
    Display:=function(x,opt)opt.screenColumns:=SizeScreen()[1];
               Print(Format(x,opt));end);
  return res;
end;

# Green functions: Green(uc[,opt]) values on unipotent classes or local systems
# opt: variable (default X(Cyclotomics))
#
# Formatting: options of FormatTable + [.classes, .CycPol]
GreenTable:=function(arg)local opt,uc,W,res,pieces,l,m,n,p,q;
  uc:=arg[1];W:=uc.group;
  if Length(arg)=1 then opt:=rec();else opt:=arg[2];fi;
  if not IsBound(opt.variable) then opt.variable:=X(Cyclotomics);fi;
  q:=opt.variable;
  pieces:=List([1..Length(uc.springerSeries)],i->ICCTable(uc,i,q));
  m:=ApplyFunc(DiagonalMat,List(pieces,x->List(x.scalar,
   l->Zip(l,x.dimBu,function(x,y)return x*q^y;end))))*q^0;
  l:=Concatenation(List(pieces,x->x.locsys));
  p:=SortingPerm(l);
  res:=rec(scalar:=TransposedMat(Permuted(m,p)),uc:=uc,locsys:=Permuted(l,p),
    parameter:=Concatenation(List(pieces,x->x.parameter)),
    relgroups:=List(uc.springerSeries,x->x.relgroup));
  n:=Length(res.locsys);
  res.operations:=rec();
  res.operations.String:=x->SPrint("GreenTable(",W,",rec(variable:=",q,"))");
  res.operations.Format:=function(x,opt)local res,tbl,i,b;
    res:=SPrint("Values of character sheaves on");
    opt.rowLabels:=Concatenation(List(x.relgroups,g->
      List(CharNames(g,opt),n->SPrint("Q^{",ReflectionName(g),"}_{",n,"}"))));
    opt.rowsLabel:="Q";
    tbl:=Copy(x.scalar);
    if IsBound(opt.classes) then
      PrintToString(res," unipotent classes\n");
      for i in [1..Length(x.uc.classes)] do
	b:=Filtered([1..Length(x.locsys)],j->x.locsys[j][1]=i);
	tbl{[1..Length(tbl)]}{b}:=tbl{[1..Length(tbl)]}{b}*
	   CharTable(x.uc.classes[i].Au).irreducibles;
      od;
      opt.columnLabels:=List(x.locsys,p->ClassName(x.uc,p[1],
	  Inherit(rec(class:=p[2]),opt)));
    else PrintToString(res," local systems\n");
      opt.columnLabels:=List(x.locsys,p->ClassName(x.uc,p[1],
        Inherit(rec(locsys:=p[2]),opt)));
    fi;
    if not IsBound(opt.CycPol) then opt.CycPol:=true;fi;
    if opt.CycPol then tbl:=List(tbl,y->List(y,CycPol));fi;
    PrintToString(res,FormatTable(tbl,opt));
    return String(res);
  end;
  res.operations.Print:=function(x)Print(String(x));end;
  res.operations.Display:=function(x,opt)opt.screenColumns:=SizeScreen()[1];
               Print(Format(x,opt));end;
  return res;
end;

# values of unipotent characters
# UnipotentValues(uc[,opt]) values on unipotent classes or local systems
UnipotentValues:=function(arg)local uc,opt,res,q;
  uc:=arg[1];
  if Length(arg)=1 then opt:=rec();else opt:=arg[2];fi;
  if not IsBound(opt.variable) then opt.variable:=X(Cyclotomics);fi;
  q:=opt.variable;
  res:=Copy(GreenTable(uc,opt));
  uc:=UnipotentCharacters(uc.group);
  res.scalar:=TransposedMat(uc.operations.Fourier(uc){res.parameter})
    *res.scalar;
  res.operations.String:=x->SPrint("UnipotentValues(",uc.group,",",q,")");
  res.operations.Format:=function(x,opt)local s,i,b,tbl;
    s:="Values of unipotent characters for ";
    if IsBound(opt.TeX) then PrintToString(s,"$",
      ReflectionName(uc.group,opt),"$\\par");
    else PrintToString(s,ReflectionName(uc.group,opt));
    fi;
    tbl:=Copy(x.scalar);
    if IsBound(opt.classes) then
      PrintToString(s," on unipotent classes\n");
      for i in [1..Length(x.uc.classes)] do
	b:=Filtered([1..Length(x.locsys)],j->x.locsys[j][1]=i);
	tbl{[1..Length(tbl)]}{b}:=tbl{[1..Length(tbl)]}{b}*
	   CharTable(x.uc.classes[i].Au).irreducibles;
      od;
      opt.columnLabels:=List(x.locsys,p->ClassName(x.uc,p[1],
	  Inherit(rec(class:=p[2]),opt)));
    else opt.columnLabels:=List(x.locsys,p->ClassName(x.uc,p[1],
        Inherit(rec(locsys:=p[2]),opt)));
      PrintToString(s," on local systems\n");
    fi;
    opt.rowLabels:=CharNames(uc,opt);
    if not IsBound(opt.CycPol) then opt.CycPol:=true;fi;
    if opt.CycPol then tbl:=List(tbl,y->List(y,CycPol));fi;
    return Concatenation(s,FormatTable(tbl,opt));
  end;
  return res;
end;

# values of mellin  on unipotent classes
MellinValues:=function(arg)local uc,q,res,b,tbl,f,labels;
  uc:=arg[1];if Length(arg)=1 then q:=X(Cyclotomics); else q:=arg[2];fi;
  res:=Copy(GreenTable(uc,q));
  uc:=UnipotentCharacters(uc.group);
  uc.mellin:=IdentityMat(Size(uc));
  labels:=[];
  for f in uc.families do
    uc.mellin{f.charNumbers}{f.charNumbers}:=f.mellin;
    labels{f.charNumbers}:=List(f.mellinLabels,
      x->SPrint(Position(uc.families,f),x));
  od;
  res.scalar:=TransposedMat(uc.operations.Fourier(uc){res.parameter})
    *res.scalar;
  res.scalar:=uc.mellin*res.scalar;
  res.operations.String:=x->SPrint("MellinValues(",uc.group,",",q,")");
  res.operations.Format:=function(x,opt)local s,i,b,tbl;
    s:="Values of Mellin of unipotent characters for ";
    if IsBound(opt.TeX) then PrintToString(s,"$",ReflectionName(uc.group,opt),
      "$\\par");
    else PrintToString(s,ReflectionName(uc.group,opt));
    fi;
    tbl:=Copy(x.scalar);
    if IsBound(opt.classes) then
      PrintToString(s," on unipotent classes\n");
      for i in [1..Length(x.uc.classes)] do
	b:=Filtered([1..Length(x.locsys)],j->x.locsys[j][1]=i);
	tbl{[1..Length(tbl)]}{b}:=tbl{[1..Length(tbl)]}{b}*
	   CharTable(x.uc.classes[i].Au).irreducibles;
      od;
      opt.columnLabels:=List(x.locsys,p->ClassName(x.uc,p[1],
	  Inherit(rec(class:=p[2]),opt)));
    else opt.columnLabels:=List(x.locsys,p->ClassName(x.uc,p[1],
        Inherit(rec(locsys:=p[2]),opt)));
      PrintToString(s," on local systems\n");
    fi;
    opt.rowLabels:=labels;
    if not IsBound(opt.CycPol) then opt.CycPol:=true;fi;
    if opt.CycPol then tbl:=List(tbl,y->List(y,CycPol));fi;
    return Concatenation(s,FormatTable(tbl,opt));
  end;
  return res;
end;
