###########################################################################
##
#A  modhom.g                 autag package                 Michael J Smith
##
##  August 1996
##
##  This file forms part of a package for computing decompositions of
##  modules into indecomposable summands, as well as computing generating
##  sets for module automorphisms.
##
##  It is an integral part of the soluble group automorphism package.
##
###########################################################################


# The SpinHom routine in this file was written during August 1996. The
# basic idea comes from a discussion I had with Charles Leedham-Green early
# in 1995. He gave me a rough sketch of the algorithm that he and John
# Cannon developed for Magma. Some details were missing, and this is my
# attempt at filling in some of them. 
#
# Many improvements were made on my earlier version, in large part due to a
# discussion I had with Alice Neimeyer in early 1996. She relayed to me
# some comments of Klaus Lux on my earlier version. This is a combination
# of the suggestions of Klaus and Alice and my own ideas.
#
# Note: This provides an enormous speed-up on the the default GAP routine,
# and on my own naive intertwining routine, especially when the module is
# large enough and/or it is irreducible. However, this routine is nowhere
# near as good as the Magma algorithm, and I do not know how to improve it.
#
# The code is heavily commented, and I appreciate suggestions on how to
# improve it (particularly bits of code).



SpinHom := function (V, W)
    #
    # Compute Hom(V,W) for G-modules <V> and <W>. The algorithm starts with
    # the trivial submodule <U> of <V> for which Hom(U,V) is trivial.  It
    # then computes Hom(U',W) for U' a submodule generated by <U> and a
    # single element <v0> in <V>. This U' becomes the next <U> as the process
    # is iterated, ending when <U'> = <V>. The element <v0> is chosen in a
    # nullspace of a group algebra element in order to restrict it possible
    # images in <W>.

    local nv, nw, F, zeroW, gV, gW, k, U, echu, r, homs, s, work, ans, v0, 
          M, x, pos, z, echm, t, v, echv, a, u, e, start, oldlen, i, j, ag, 
          m, uu, ret, c, l, s1, X, mat, uuc, uic, newhoms, hom, Uhom, imv0, 
          imv0c, image;

    nv := V.dim;
    nw := W.dim;
    
    F := V.field;

    zeroW := [1..nw] * F.zero;

    # group generating sets acting on each module
    gV := V.generators;
    gW := W.generators;

    # <k> is the number of generators of the acting group
    k := Length(gV);

    # <U> is the semi-ech basis for the currently known submodule, of
    # dimension <r>
    U := [];
    echu := [];
    r := 0;

    # <homs> contains a basis for Hom(U,W), of dimension <s>
    homs := [];
    s := 0;

    # define a record which stores information about the modules <V>, <W>
    # and <U> for passing into a routine that selects a new vector <v0>
    # for spinning up to a larger submodule U'.
    work := rec(V := V, W := W, U := U, echu := echu);

    repeat
        #
        # we loop until <U> is the whole of <V>

        ans := GModOps.SpinHomFindVector(work);
        v0 := ans[1];
        M := ans[2];
                
        # find residue of <v0> modulo current submodule <U>
        x := GModOps.EchResidue(U, echu, v0);

        # normalise <x> (ie get a 1 in leading position)
        pos := PositionProperty(x, IsNonZeroElt);
        z := x[pos];
        x := x / z; 
        v0 := v0 / z;

        # we know that <v0> has to map into the subspace <M> of <W>. 
        echm := List(M, y -> PositionProperty(y, IsNonZeroElt));
        t := Length(M);

        # now we start building extension of semi-echelonised basis for
        # the submodule U' generated by <U> and <v0>
        #
        # new elements of semi-ech basis will be stored in <v>, with
        # echelon weights stored in <echv>

        v := [ x ];
        echv := [ pos ];

        # we need to keep track of how each new element of the semi-ech
        # basis was obtained from <v0> --- new basis element <v[i]> will
        # satisfy:
        #
        #     v[i]  =  v0*a[i] + u[i]
        #
        # where <a[i]> is an element of the group algebra FG, and <u[i]> is
        # the element of <U> that was subtracted during semi-ech reduction

        a := [ M ];
        u := [ x - v0 ];

        # we will accumulate the homogeneous linear system in <e> 
        #
        # the first <s> variables are the coefficients of basis elements of
        # Hom(U,W), which describes how a hom of U' acts on submodule <U>
        #
        # the other <t> variables are the coefficients of basis elements of
        # <M>, which describes the image of <v0> under a hom
        #
        e := EqnOps.NewEqns(s + t, F);

        # we will close the submodule by spinning <v0> --- the variable
        # <start> will trim off the elements of <v> that we have already
        # used
        start := 1;

        repeat
            #
            # take an element <v[i]> of <v> and a group generator <g[j]>
            # and check whether <v[i]^g[j]> is a new basis element.
            #
            # if it is, add it to the basis, with its definition.
            #
            # if it isn't, we get an equation which an element of Hom(U',W)
            # must satisfy

            oldlen := Length(v);

            for i in [start..oldlen] do     ### loop on vectors in <v>
                for j in [1..k] do          ### loop on generators of G

                    ag := a[i] * gW[j];

                    # create new element <x>, with its definition as the
                    # difference between <v0^m> and <uu> in <U>.
                    x := v[i] * gV[j];
                    m := ag;
                    uu := u[i] * gV[j];

                    ret := GModOps.EchResidueCoeffs(U, echu, x);
                    x := ret.residue;
                    uu := uu - ret.projection;

                    # reduce modulo the new semi-ech basis elements in <v>,
                    # storing the coefficients in <c>
                    #
                    c := [1..Length(v)] * F.zero;
                    for l in [1..Length(v)] do
                        z := x[echv[l]];
                        if z <> F.zero then
                            x := x - z * v[l];
                            if Length(m) > 0 then
                                m := m - z * a[l];
                            fi;
                            c[l] := c[l] + z;
                            uu := uu - z * u[l];
                        fi;
                    od;

                    # Note: at this point, <x> has been reduced modulo the
                    # semi-ech basis <U> union <v>, and that
                    # 
                    #     x = v0 * a[i] + uu

                    pos := PositionProperty(x, IsNonZeroElt);
                    if pos <> false then
                        #
                        # new semi-ech basis element <x>

                        z := x[pos];
                        Add(v, x/z);
                        Add(echv, pos);
                        Add(a, m/z);
                        Add(u, uu/z);

                    else
                        #
                        # we get some equations !

                        s1 := Sum([1..Length(v)], y -> c[y] * v[y]);
                        uu := v[i] * gV[j] - s1;

                        X := NullMat(t, nw, F);
                        for l in [1..Length(v)] do
                            if c[l] <> F.zero then
                                if Length(X) > 0 then
                                    X := X + c[l] * a[l];
                                fi;
                                uu := uu + c[l] * u[l];
                            fi;
                        od;

                        if Length(X) > 0 then
                            X := X - ag;
                        fi;

                        mat := [];
                        uuc := GModOps.EchCoeffs(U, echu, uu);
                        uic := GModOps.EchCoeffs(U, echu, u[i]);
                        for l in [1..s] do
                            Add(mat, uuc * homs[l]
                                - uic * homs[l] * gW[j]);
                        od;
                        Append(mat, X);
                        EqnOps.AddEqns(e, TransposedMat(mat), zeroW);
                    fi;
                od;
            od;

            start := oldlen+1;

            # exit when no new elements were added --- i.e. the subspace
            # is closed under action of G and is therefore a submodule
            #
        until oldlen = Length(v); 


        # we have the system of equations, so find its solution space
        #
        ans := EqnOps.NullspaceEqns(e);

        # Now build the homomorphisms
        #
        newhoms := [];
        for i in [1..Length(ans)] do
            #
            # Each row of ans is of the form:
            #
            #     [ b_1, b_2, ..., b_s, c_1, c_2, ..., c_t ]
            #
            # where the action of this hom on <U> is as \Sum{b_l homs[l]}
            # and the hom sends <v0> to Sum{c_l M[l]}

            hom := [];
            if r > 0 then
                Uhom := NullMat(r, nw, F);
                for l in [1..s] do
                    if ans[i][l] <> F.zero then
                        Uhom := Uhom + ans[i][l] * homs[l];
                    fi;
                od;
                for l in [1..r] do
                    Add(hom, Uhom[l]);
                od;
            fi;

            imv0 := zeroW * F.zero;
            for l in [1..t] do
                if ans[i][s+l] <> F.zero then
                    imv0 := imv0 + ans[i][s+l] * M[l];
                fi;
            od;
            imv0c := GModOps.EchCoeffs(M, echm, imv0);
            for l in [1..Length(v)] do
                image := imv0c * a[l];
                if r > 0 then
                    image := image + GModOps.EchCoeffs(U, echu, u[l]) * Uhom;
                fi;
                Add(hom, image);
            od;
            Add(newhoms, hom);
        od;

        # now update <U> to be the now larger submodule
        #
        Append(U,v);
        Append(echu, echv);
        homs := newhoms;
        r := Length(U);
        s := Length(homs);

        InfoHom("U is now dimension ", r);
        InfoHom(" and Dim(Hom(U,W)) = ", s, "\n");

    until r = nv; # i.e. <U> = <V>

    # We must change basis on <V> from <U> to the usual one before returning
    #
    return U^-1 * homs;

end;



GModOps.SpinHomFindVector := function (r)
    #
    # <r> contains information about modules <V> and <W>, and a submodule
    # <U> of <V> with semi-ech information <echu>. The routine selects
    # an element of <V> lying outside of <U> that will be used to spin
    # up to a new submodule U'.
    #
    # It returns a list [<v0>, <M>] where <v0> is the element of <V>
    # and <M> is a basis for a submodule of <W> which <v0> must map into
    # under any hom.
    
    local V, nv, W, nw, U, echu, F, max_stack_len, newstack, A, v0, 
          extradim, N, count, look_lim, grpalg, i, j, M, pos, _t, done;

    V := r.V;    nv := V.dim;
    W := r.W;    nw := W.dim;
    U := r.U;    echu := r.echu;
    F := V.field;
    
    if not IsBound(r.setup) then
        r.zeroFG := GModOps.GrpAlg.RecMake(NullMat(nv, nv, F), 
                            NullMat(nw, nw, F));
        GModOps.GrpAlg.Setup(r);

        # we build a stack of good grpalg elements to use for choosing
        # elements <v0> --- an element <A> in <stack> is of the form:
        #   A[1] = v0
        #   A[2] = grpalg element whose nullspace contains v0
        #   A[3] = Dim(<U,v0>^G)-Dim(U) i.e. increase in dim by adding 
        #          <v0> to <U>
        r.stack := [];

        r.setup := true;
    fi;

    max_stack_len := 10;

    # adjust the elements of the stack to account for the larger
    # submodule <U> we now have
    _t := Runtime();
    newstack := [];
    for A in r.stack do
        v0 := A[1];
        extradim := Length(GModOps.SpinSpaceVector(V, U, echu, v0))
                    - Length(U);
        if extradim > 0 then
            Add(newstack, [v0, A[2], extradim]);
        fi;
    od;
    r.stack := newstack;
    InfoStack("stack reduced to length ", Length(r.stack), " (",
            Time(Runtime()-_t), ")\n");

    # <N> contains the nullspace in <V> of a group algebra element ---
    # initialise it to the empty list for the following repeat loop
    N := [];

    count := 0;
    look_lim := 5;  # give up after this many random grpalg elements

    _t := Runtime();

    if Length(r.stack) > 0 then
        # if we have something left, don't bother generating any new
        # grpalg elements (?)
        count := look_lim + 1;
    fi;

    done := false; 
    while count < look_lim and Length(r.stack) < max_stack_len and not done do
        #
        # we look for a while and take the best element found
        #
        # We are looking for an element <v0> of a nullspace that lies
        # outside of <U>

        repeat
            grpalg := GModOps.GrpAlg.Random(r);
            N := NullspaceMat(grpalg.V);
            count := count + 1;
        until Length(N) > 0 or count >= look_lim;

        if Length(N) > 0 then

            # now find best element of <N> for adding to <stack>
            extradim := List(N, y ->
                             Length(GModOps.SpinSpaceVector(V, U, echu, y)) 
                             - Length(U));
            i := 1;
            for j in [2..Length(extradim)] do
                if extradim[j] > extradim[i] then
                    i := j;
                fi;
            od;
            if extradim[i] > 0 then
                # exit early if we have found an element that gets use all
                # of <V> after spinning
                done := extradim[i] = nv - Length(U);
                if done then
                    r.stack := [[N[i], grpalg, extradim[i]]];
                else
                    Add(r.stack, [N[i], grpalg, extradim[i]]);
                fi;
            fi;
        fi;

    od;
    InfoStack("stack loop done, stack now length ", Length(r.stack), " (",
            Time(Runtime()-_t), ")\n");

    if Length(r.stack) > 0 then
        #
        # find best element in r.stack and use it
        i := 1;
        for j in [2..Length(r.stack)] do
            if r.stack[j][3] > r.stack[i][3] then
                i := j;
            fi;
        od;
        v0 := r.stack[i][1];
        M := NullspaceMat(r.stack[i][2].W);
    else
        #
        # we haven't found a good grpalg element, so just choose
        # something outside of <U> and use it

        InfoHom("too many random grpalg elements...\n");
        M := IdentityMat(nw,F);
        pos := Difference([1..nv], echu)[1];
        v0 := [1..nv] * F.zero;
        v0[pos] := F.one;
    fi;
    return [v0, M];

end;



GModOps.Intertwine := function (V, W)
    #
    # This function takes a pair of FG-modules <V> and <W> and returns a
    # basis for Hom(V,W). It works by considering the equations
    #
    #             B_i X - X C_i = 0,   for i = 1..d
    # 
    # where X is a matrix of indeterminates, and B_i and C_i are the
    # generators of G acting on <V> and <W> respectively.

    local B, C, F, d, n, m, e, g, i, j, newrow, r, s, k;

    B := V.generators;
    C := W.generators;
    F := V.field;
    d := Length(B);

    n := Length(B[1]);
    m := Length(C[1]);
    e := EqnOps.NewEqns(n*m, F);

    for g in [1..d] do                      # loop over generators in B,C
        for i in [1..n] do                  # loop over rows of BX-XC
            for j in [1..m] do              # loop over columns of BX-XC

                newrow := [1..n*m]*F.one;

                for r in [1..n] do          # loops for coeffs of entry Xij
                    for s in [1..m] do

                        k := (r-1)*m + s;   # current entry position in row
                        if (s = j) then
                            if (r = i) then
                                newrow[k] := B[g][i][r] - C[g][s][j];
                            else
                                newrow[k] := B[g][i][r];
                            fi;
                        else
                            if (r = i) then
                                newrow[k] := -C[g][s][j];
                            else
                                newrow[k] := 0 * F.one;
                            fi;
                        fi;
                    od;
                od;

                EqnOps.AddEqns(e, [newrow], [F.zero]);

            od;
        od;
    od;

    return List(EqnOps.NullspaceEqns(e),
                x -> GModOps.WrapMat(x,n,m));

end;





GModOps.EchResidueCoeffs := function (base, ech, v)
    # 
    # Take a semi-ech basis <base>, with ech weights <ech>, and a vector
    # <v> in the subspace spanned by <base>. Return a record containing the
    # residue after removing projection of <v> onto subspace spanned by
    # <base>, as well as the coefficients of the linear combination of
    # <base> elements used to obtain the projection. Also return the
    # projection.
    
    # Note that the pivots of <base> must be set to 1.

    local n, coeffs, x, zero, i, z;

    n := Length(base);

    if n = 0 then
        coeffs := [];
        x := v;
    else
        x := v;
        zero := x[1]*0;
        coeffs := [1..n] * zero;
        for i in [1..n] do
            z := x[ech[i]];
            if z <> zero then
                x := x - z * base[i];
                coeffs[i] := z;
            fi;
        od;
    fi;

    return rec(coeffs := coeffs,
               residue := x,
               projection := v - x
               );
end;



GModOps.EchCoeffs := function (base, ech, v)
    return GModOps.EchResidueCoeffs(base, ech, v).coeffs;
end;


GModOps.EchResidue := function (base, ech, v)
    return GModOps.EchResidueCoeffs(base, ech, v).residue;
end;


GModOps.SpinSpaceVector := function (V, _U, _ech, v)
    #
    # Take <U> a semi-ech basis for a submodule of <V>, with ech-weights
    # <ech>, and a vector <v> in <V>. Return a semi-ech basis for the
    # submodule generated by <U> and <v>.

    local U, ech, gens, pos, settled, oldlen, i, j;
    
    U := ShallowCopy(_U);
    ech := ShallowCopy(_ech);
    gens := V.generators;

    v := GModOps.EchResidue(U, ech, v);
    pos := PositionProperty(v, IsNonZeroElt);
    if pos = false then
        return U;
    fi;
    Add(U, v/v[pos]); Add(ech, pos);

    settled := Maximum(Length(_U),1); # <U> is a submodule
    repeat
        oldlen := Length(U);
        for i in [settled+1..Length(U)] do
            for j in [1..Length(gens)] do
                v := GModOps.EchResidue(U, ech, (U[i] * gens[j]));
                pos := PositionProperty(v, IsNonZeroElt);
                if pos <> false then
                    Add(U, v/v[pos]); Add(ech, pos);
                fi;
            od;
        od;
        settled := oldlen;
    until oldlen = Length(U);
    return U;
end;



# ======================================================================


GModOps.GrpAlg := rec();

GModOps.GrpAlg.Setup := function (r)

    local r, matsV, matsW, k, i, g1, g2;

    matsV := GModOps.Generators(r.V);
    matsW := GModOps.Generators(r.W);
    k := Length(matsV);

    r.mats := List([1..k], i -> GModOps.GrpAlg.RecMake(matsV[i], matsW[i]));

    # 
    # do preprocessing to make random matrices list
    #
    for i in [1..10] do
        g1 := Random([1..k]);
        g2 := g1;
        while g2 = g1 and Length(r.mats)>1 do
            g2 := Random([1..k]);
        od;
        Add(r.mats, r.mats[g1]*r.mats[g2]);
        k := k + 1;
    od;

end;
    


GModOps.GrpAlg.Random := function (r)
    # 
    # Take a work record <r> containing the information about the two
    # modules <V> and <W>, and return a random group algebra element
    # record containing its action on each of the modules.

    local V, nv, W, nw, F, k, g1, g2, a;

    V := r.V;  nv := GModOps.DimFlag(V); 
    W := r.W;  nw := GModOps.DimFlag(W);
    F := GModOps.Field(V);
    k := Length(r.mats);

    # first take two elements of the list and multiply them
    # together
    g1 := Random([1..k]);
    repeat
        g2 := Random([1..k]);
    until g2 <> g1 or Length(r.mats)=1;
    Add(r.mats, r.mats[g1]*r.mats[g2]);
    k := k + 1;

    # Now take a random linear sum of the existing generators as new
    # generator.  Record the sum in coefflist
    #
    a := GModOps.GrpAlg.RecMake(NullMat(nv,nv,F), NullMat(nw,nw,F));

    for g1 in [1..k] do
        g2 := Random(F);
        if g2 <> F.zero then
            a := a + g2*r.mats[g1];
        fi;
    od;

    # Add(r.mats, a);

    return a;
end;



##
## Now the group algebra records.
##

GModOps.GrpAlg.RecOps := rec();

GModOps.GrpAlg.RecOps.\* := function (a, b)
    local r;
    if IsFFE(a) then
        r := GModOps.GrpAlg.RecMake(a * b.V, a * b.W);
    elif IsFFE(b) then
        r := GModOps.GrpAlg.RecMake(a.V * b, a.W * b);
    else
        r := GModOps.GrpAlg.RecMake(a.V * b.V, a.W * b.W);
    fi;
    return r;
end;

GModOps.GrpAlg.RecOps.\/ := function (a, b)
    local r;
    r := GModOps.GrpAlg.RecMake(a.V / b, a.W / b);
    return r;
end;

GModOps.GrpAlg.RecOps.\+ := function (a, b)
    local r;
    r := GModOps.GrpAlg.RecMake(a.V + b.V, a.W + b.W);
    return r;
end;

GModOps.GrpAlg.RecOps.\- := function (a, b)
    local r;
    r := GModOps.GrpAlg.RecMake(a.V - b.V, a.W - b.W);
    return r;
end;

GModOps.GrpAlg.RecOps.Print := function (r)
    Print("GrpAlgRec(\n");
    Nice([r.V, r.W]);
    Print("  )");
end;

GModOps.GrpAlg.RecMake := function (gV, gW)
    local r;
    r := rec();
    r.operations := GModOps.GrpAlg.RecOps;
    r.V := gV;
    r.W := gW;
    return r;
end;

