#############################################################################
##
#A  action.g       VKCURVE package         David Bessis 
##
#Y  Copyright (C) 2001 - 2002  University Paris VII, France.
##
##  This file implements BnActsOnFn (Hurwitz action)
##                       VKQuotient (classical Van Kampen presentation)
##                   and DBVKQuotient
##    (Modified Van Kampen technique allowing vertical asymptotes)
## 
#############################################################################
##################################################
# Implements the action of B_n on F_n
#   Input:  element of B_n, ambient free group F_m
#           (with m >= n; when m>n, Hurwitz
#            action is extended trivially to the extra generators)
#   Output: automorphism of F_n
##################################################

BnActsOnFn:=function(b,F) local n,f,g,res,s,phi;
  n:=Length(F.generators);
  f:=F.generators;
  res:=GroupHomomorphismByImages(F,F,f,f);
  res.isMapping:=true;
  for s in AsWord(b) do
    g:=ShallowCopy(f);
    if s>0 then g[s]:=f[s+1]; g[s+1]:=f[s+1]^-1*f[s]*f[s+1];
    else g[-s+1]:=f[-s]; g[-s]:=f[-s]*f[-s+1]/f[-s];
    fi;
    phi:=GroupHomomorphismByImages(F,F,f,g);
    phi.isMapping:=true;
    res:=CompositionMapping(phi,res);
  od;
  return res;
end;

############################################################
# Computes the quotient of the free group by the action
#  of the group generated by a certain list of braids.
# Input:  list of braids living in the same ambient braid group
# Output: the quotient, encoded as an FpGroup
############################################################
# Printing controlled by VKCURVE.showAction
############################################################

VKQuotient:=function(braids) local auts,n,F,f,aut,rels;
  # get the true monodromy braids and the Hurwitz action basic data
  n:=braids[1].monoid.group.nbGeneratingReflections+1;
  F:=FreeGroup(n);
  auts:=List(braids,b->BnActsOnFn(b,F));
  rels:=[];
  for f in F.generators do
    for aut in auts do Add(rels,Image(aut,f)/f); od;
  od;
  return F/rels; 
end;

############################################################
# A variant of the previous function.
# See arXiv:math.GR/0301327 for more mathematical details.
# Input: global VKCURVE record
# Output: the quotient, encoded as an FpGroup
############################################################
# Printing controlled by VKCURVE.showAction
############################################################

DBVKQuotient:=function(r) local braids,z,b,auts,n,F,seg,
                               zero,basestring,dist,k,height,
                               fbase,aut,correctaut,ifbase,f,
                               choices,conjugator,conj,rels,g;
  # get the true monodromy braids and the Hurwitz action basic data
  n:=r.braids[1].monoid.group.nbGeneratingReflections+1;
  braids:=r.braids;#{List(r.trueroots,z->Position(r.roots,z))};
  F:=FreeGroup(n+Length(r.verticallines));
  auts:=List(braids,b->BnActsOnFn(b,F));
  # above the basepoint for the loops, locate the position of the string
  # corresponding to the trivializing horizontal line
  zero:=r.zeros[r.basepoint];
  dist:=List(zero,z->(z-r.height)*ComplexConjugate(z-r.height));
  height:=zero[Position(dist,Minimum(dist))];
  basestring:=Length(Filtered(zero,z->(z.r < height.r)
	    or ((z.r = height.r) and (z.i <= height.i))));
  fbase:=F.generators[basestring];
  rels:=[];
  for aut in auts do
      # Find an element conjugator such that
      #    conjugator*aut(fbase)/conjugator = fbase
      ifbase:=Image(aut,fbase);
      conjugator:=F.1/F.1;
      choices:=Concatenation(List(F.generators{[1..n]},f->[f,f^-1]));
      repeat
	k:=0;
	repeat k := k+1;
	      until LengthWord(choices[k]*ifbase) < LengthWord(ifbase);
	ifbase:=choices[k] * ifbase /choices[k];
	conjugator:= choices[k] * conjugator;
      until
	LengthWord(ifbase)=1;
      # Replacing aut by  correctaut:= Conj(conjugator)*aut
      conj:=GroupHomomorphismByImages(F,F,
	 F.generators,conjugator*F.generators/conjugator);
      conj.isMapping:=true;
      correctaut:=CompositionMapping(conj,aut);
      if Position(auts,aut) > Length(r.verticallines)
      then
	  Append(rels,List(F.generators{[1..n]},f->Image(correctaut,f)/f));
      else
	  g:=F.generators[Position(auts,aut)+n];
	  Append(rels,List(F.generators{[1..n]},
			       f-> Image(correctaut,f)*g/(g*f)));
      fi;
  od;
  Add(rels,fbase);
  return F/rels;
end;
