//   Copyright (c) 2010-2017 Giovanni Lagorio, John Abbott, Anna M. Bigatti
//   Authors: 2010-2011 Giovanni Lagorio
//   Authors: 2011-2017 John Abbott, Anna M. Bigatti
//
//   This file is part of the source of CoCoALib, the CoCoA Library.
//
//   CoCoALib is free software: you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.
//
//   CoCoALib is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with CoCoALib.  If not, see <http://www.gnu.org/licenses/>.

#include "BuiltInFunctions.H"
#include <boost/version.hpp>

using namespace std;
using namespace boost;
using namespace boost::iostreams;
using namespace CoCoA::AST;
using namespace CoCoA::LexerNS;
using namespace CoCoA::ParserNS;

namespace CoCoA {
namespace InterpreterNS {

//----------------------------------------------------------------------
// builtIns is essentially a global variable
//   but with better controlled initialization.
std::vector<NameFunPair>& builtIns()
{
  static std::vector<NameFunPair> GlobalVar;
  return GlobalVar;
}
//---- initialization ----
void RuntimeEnvironment::initBuiltInFunctions()
{
  BOOST_FOREACH(NameFunPair &p, builtIns())
		this->setTopLevelVar(p.first, p.second, VariableSlot::VSF_SystemProtected);
}
//----------------------------------------------------------------------

//---- BOOL

DECLARE_STD_BUILTIN_FUNCTION(not, 1) {
  intrusive_ptr<BOOL> arg = runtimeEnv->evalArgAs<BOOL>(ARG(0));
  return Value::from(!(arg->theBool));
}
END_STD_BUILTIN_FUNCTION

//---- ZIP
#if 0  // 2 functions for accessing zip files, currently disabled
DECLARE_STD_BUILTIN_FUNCTION(ZipFileList, 1) {
	const Argument &arg = ARG(0);
	const string filename = runtimeEnv->evalArgAs<STRING>(arg)->theString;
	int err;
	struct zip *zf = ::zip_open(filename.c_str(), ZIP_CHECKCONS, &err);
	if (!zf)
		throw RuntimeException("Cannot read Zip file \""+filename+"\"", arg.exp);
	intrusive_ptr<LIST> result(new LIST);
	const int numEntries = zip_get_num_files(zf);
	for(int a=0; a<numEntries; ++a)
		result->addValue(new STRING(zip_get_name(zf, a, ZIP_FL_UNCHANGED)));
	::zip_close(zf);
	return result;
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(ZipRead, 2) {
	const Argument &argFilename = ARG(0);
	const Argument &argEntryName = ARG(1);
	const string filename = runtimeEnv->evalArgAs<STRING>(argFilename)->theString;
	const string entryName = runtimeEnv->evalArgAs<STRING>(argEntryName)->theString;
	int err;
	struct zip *zf = ::zip_open(filename.c_str(), ZIP_CHECKCONS, &err);
	if (!zf)
		throw RuntimeException("Cannot read Zip file \""+filename+"\"", argFilename.exp);
	const int index = zip_name_locate(zf, entryName.c_str(), /* flags */ 0);
	if (index<0) {
		::zip_close(zf);
		throw RuntimeException("Cannot find an entry named \""+entryName+" inside the Zip file \""+filename+"\"", argEntryName.exp);
	}
	struct zip_file *entry = ::zip_fopen_index(zf, index, /* flags */ 0);
	if (!entry) {
		::zip_close(zf);
		throw RuntimeException("Error reading the entry named \""+entryName+" inside the Zip file \""+filename+"\"", argEntryName.exp);
	}
	string result;
	for(;;) {
		char buf[1024];
		int nRead = ::zip_fread(entry, buf, sizeof(buf));
		if (nRead<=0) {
			::zip_fclose(entry);
			::zip_close(zf);
			return new STRING(result);
		}
		result.append(buf, buf+nRead);
	}
}
END_STD_BUILTIN_FUNCTION
#endif


//---- LIST

DECLARE_ARITYCHECK_FUNCTION(NewList) { return (1<=nArg) && (nArg<=2); }
DECLARE_BUILTIN_FUNCTION(NewList) {
	invocationExpression->checkNumberOfArgs(1,2);
  //	intrusive_ptr<INT> N = runtimeEnv->evalArgAs<INT>(ARG(0));
  long n = runtimeEnv->evalArgAsLong(ARG(0));
  if (n<0) throw RuntimeException("invalid length", ARG(0).exp);
	intrusive_ptr<LIST> returnValue = new LIST();
  intrusive_ptr<RightValue> val = INT::zero;
  if (invocationExpression->args.size()==2)
    val = runtimeEnv->evalArgAs<RightValue>(ARG(1));
  for(long i=0; i<n; ++i)  returnValue->addValue(val);
	return returnValue;
}

DECLARE_ARITYCHECK_FUNCTION(first) { return (1<=nArg) && (nArg<=2); }
DECLARE_BUILTIN_FUNCTION(first) {
	invocationExpression->checkNumberOfArgs(1,2);
	intrusive_ptr<LIST> list = runtimeEnv->evalArgAs<LIST>(ARG(0));
  if (invocationExpression->args.size()==1)
  {
    if (list->size() == 0) throw RuntimeException("list is empty", ARG(0).exp);
    return list->getValue(0);
  }
  //	intrusive_ptr<INT> N = runtimeEnv->evalArgAs<INT>(ARG(1));
	long n = runtimeEnv->evalArgAsLong(ARG(1));
  if (n<0) throw RuntimeException("invalid length", ARG(1).exp);
  if ((unsigned long)n > list->size())
    throw RuntimeException("value greater than list length", ARG(1).exp);
	intrusive_ptr<LIST> returnValue = new LIST();
	for(long i=0; i<n; ++i)  returnValue->addValue(list->getValue(i));
	return returnValue;
}

DECLARE_ARITYCHECK_FUNCTION(last) { return (1<=nArg) && (nArg<=2); }
DECLARE_BUILTIN_FUNCTION(last) {
	invocationExpression->checkNumberOfArgs(1,2);
	intrusive_ptr<LIST> list = runtimeEnv->evalArgAs<LIST>(ARG(0));
  ///////// here
  LIST::ContainerType::size_type s = list->size();
  if (invocationExpression->args.size()==1) 
  if (invocationExpression->args.size()==1)
  {
    if (list->size() == 0) throw RuntimeException("list is empty", ARG(0).exp);
    return list->getValue(s-1);
  }
  //	intrusive_ptr<INT> N = runtimeEnv->evalArgAs<INT>(ARG(1));
  long n = runtimeEnv->evalArgAsLong(ARG(1));
  if (n<0) throw RuntimeException("invalid length", ARG(1).exp);
  if ((unsigned long)n>s)
    throw RuntimeException("value greater than list length", ARG(1).exp);
	intrusive_ptr<LIST> returnValue = new LIST();
	for(unsigned long i=s-n; i<s; ++i)  returnValue->addValue(list->getValue(i));
	return returnValue;
}

DECLARE_ARITYCHECK_FUNCTION(sum) { return (1<=nArg) && (nArg<=2); }
DECLARE_BUILTIN_FUNCTION(sum) {
	invocationExpression->checkNumberOfArgs(1,2);
	const Argument &a0 = ARG(0);
	intrusive_ptr<LIST> list = runtimeEnv->evalArgAs<LIST>(a0);
	LIST::ContainerType::size_type size = list->size();
  intrusive_ptr<RightValue> result = INT::zero;
  if (invocationExpression->args.size()==1)
  {
    if (size==0) return result;
    result = list->getValue(--size);
  }
  else
    result = runtimeEnv->evalArgAs<RightValue>(ARG(1));
  for(LIST::ContainerType::size_type a=size; a!=0; /**/)
    try	{
			result = runtimeEnv->binaryOperatorDispatch(list->getValue(--a), result, RuntimeEnvironment::opPlusMap, a0.exp->getBegin(), a0.exp->getEnd());
	} catch (const InterruptException &) {
		throw;
	} catch (const RuntimeException &) {
		throw RuntimeException("Some elements have incompatible types for the sum", a0.exp);
	}
	return result;
}

DECLARE_ARITYCHECK_FUNCTION(product) { return (1<=nArg) && (nArg<=2); }
DECLARE_BUILTIN_FUNCTION(product) {
	invocationExpression->checkNumberOfArgs(1,2);
	const Argument &a0 = ARG(0);
	intrusive_ptr<LIST> list = runtimeEnv->evalArgAs<LIST>(a0);
	LIST::ContainerType::size_type size = list->size();
	intrusive_ptr<RightValue> result = INT::one;
  if (invocationExpression->args.size()==1)
  {
    if (size==0) return result;
    result = list->getValue(--size);
  }
  else
    result = runtimeEnv->evalArgAs<RightValue>(ARG(1));
  for(LIST::ContainerType::size_type a=size; a!=0; /**/)
		try	{
			result = runtimeEnv->binaryOperatorDispatch(list->getValue(--a), result, RuntimeEnvironment::opStarMap, a0.exp->getBegin(), a0.exp->getEnd());
	} catch (const InterruptException &) {
		throw;
	} catch (const RuntimeException &) {
		throw RuntimeException("Some elements have incompatible types for the product", a0.exp);
	}
	return result;
}

DECLARE_ARITYCHECK_FUNCTION(concat) { (void)nArg; return true; }
DECLARE_BUILTIN_FUNCTION(concat) {
	intrusive_ptr<LIST> returnValue = new LIST();
	BOOST_FOREACH(const Argument &arg, invocationExpression->args) {
		intrusive_ptr<LIST> list = runtimeEnv->evalArgAs<LIST>(arg);
		int size = list->size();
		for(int a=0; a<size; ++a)
			returnValue->addValue(list->getValue(a));
	}
	return returnValue;
}

DECLARE_STD_BUILTIN_FUNCTION(ConcatList, 1) {
	intrusive_ptr<LIST> listOfList = runtimeEnv->evalArgAs<LIST>(ARG(0));
	intrusive_ptr<LIST> result(new LIST);
	const LIST::ContainerType::size_type nLists = listOfList->size();
	for(LIST::ContainerType::size_type a=0; a<nLists; ++a) {
		intrusive_ptr<LIST> l = dynamic_pointer_cast<LIST>(listOfList->getValue(a));
		if (!l)
			throw RuntimeException("The argument is not a list of lists", ARG(0).exp);
		const LIST::ContainerType::size_type len = l->size();
		for(LIST::ContainerType::size_type b=0; b<len; ++b)
			result->addValue(l->getValue(b));
	}
	return result;
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(append, 2) {
	intrusive_ptr<RightValue> elem = runtimeEnv->evalArgAs<RightValue>(ARG(1));
	intrusive_ptr<LIST> list = runtimeEnv->obtainUnshared<LIST>(ARG(0));
	list->addValue(elem);
	return VoidValue::theInstance;
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(reversed, 1) {
	intrusive_ptr<LIST> list = runtimeEnv->evalArgAs<LIST>(ARG(0));
	intrusive_ptr<LIST> returnValue = new LIST();
	for(LIST::ContainerType::size_type a=list->size(); a!=0; /**/)
		returnValue->addValue(list->getValue(--a));
	return returnValue;
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(reverse, 1) {
	intrusive_ptr<LIST> list = runtimeEnv->obtainUnshared<LIST>(ARG(0));
	const LIST::ContainerType::size_type size = list->size();
	if (size) {
		LIST::ContainerType::size_type left = 0, right = size-1;
		while(left < right) {
			const intrusive_ptr<RightValue> tmp(list->getValue(left));
			list->setValue(left++, list->getValue(right));
			list->setValue(right--, tmp);
		}
	}
	return VoidValue::theInstance;
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(remove, 2) {
	const Argument indexArg = ARG(1);
	intrusive_ptr<RightValue> shouldBeAnIndex = runtimeEnv->evalArgAs<RightValue>(indexArg);
	intrusive_ptr<LIST> list = runtimeEnv->obtainUnshared<LIST>(ARG(0));
	LIST::ContainerType::size_type index = list->checkIndex(shouldBeAnIndex, indexArg.exp);
	list->removeValue(index);
	return VoidValue::theInstance;
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(insert, 3) {
	const Argument &indexArg = ARG(1);
	intrusive_ptr<RightValue> shouldBeAnIndex = runtimeEnv->evalArgAs<RightValue>(indexArg);
	intrusive_ptr<RightValue> elem = runtimeEnv->evalArgAs<RightValue>(ARG(2));
	intrusive_ptr<LIST> list = runtimeEnv->obtainUnshared<LIST>(ARG(0));
	LIST::ContainerType::size_type index = list->checkIndex(shouldBeAnIndex, indexArg.exp);
	list->insertValue(index, elem);
	return VoidValue::theInstance;
}
END_STD_BUILTIN_FUNCTION


//---- RECORD
DECLARE_STD_BUILTIN_FUNCTION(fields, 1) {
	intrusive_ptr<RECORD> r = runtimeEnv->evalArgAs<RECORD>(ARG(0));
	return r->fieldNames();
}
END_STD_BUILTIN_FUNCTION


//---- I/O
DECLARE_STD_BUILTIN_FUNCTION(sprint, 1) {
	const Argument &arg = ARG(0);
	intrusive_ptr<OutputStringStreamValue> outs(new OutputStringStreamValue);
	outs->print(runtimeEnv, runtimeEnv->evalArgAs<RightValue>(arg), arg.exp);
	return outs->close(runtimeEnv, invocationExpression);
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(GetErrMesg, 1) {
	const Argument &arg = ARG(0);
	intrusive_ptr<ERROR> err = runtimeEnv->evalArgAs<ERROR>(arg);
	return new STRING(err->message);
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(close, 1) {
	const Argument &arg = ARG(0);
	return runtimeEnv->evalArgAs<OSTREAM>(arg)->close(runtimeEnv, invocationExpression);
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(OpenOString, 0) {
	(void)runtimeEnv; // to avoid a compilation warning
	return new OutputStringStreamValue();
}
END_STD_BUILTIN_FUNCTION


DECLARE_STD_BUILTIN_FUNCTION(GetEnv, 1) {
	const Argument &arg = ARG(0);
	intrusive_ptr<STRING> varName = runtimeEnv->evalArgAs<STRING>(arg);
	char *value = ::getenv(varName->theString.c_str());
	if (!value)
		return STRING::empty;
	return new STRING(string(value));
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(OpenOFile, 1) {
	const Argument &arg = ARG(0);
	intrusive_ptr<STRING> filename = runtimeEnv->evalArgAs<STRING>(arg);
	return new OutputFileStreamValue(filename, arg.exp);
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(ContentsOfFile, 1) {
	const Argument &arg = ARG(0);
	intrusive_ptr<STRING> filename = runtimeEnv->evalArgAs<STRING>(arg);
	filtering_istream input;
	input.push(newline_filter(newline::posix));
	const file_source fs(filename->theString.c_str());
	if (!fs.is_open())
		throw RuntimeException("Cannot open file \""+filename->theString+"\" for reading.", invocationExpression);
	input.push(fs);
	string wholeFile;
	for(;;) {
		string line;
		getline(input, line);
		if (input.bad())
			throw RuntimeException("Cannot read from \""+filename->theString+"\".", invocationExpression);
		wholeFile += line;
		if (input.eof())
			break;
		wholeFile += '\n';
	}
	return new STRING(wholeFile);
}
END_STD_BUILTIN_FUNCTION


//---- TAGGED
const string invalidTagError("The tag must be a valid identifier");

DECLARE_STD_BUILTIN_FUNCTION(tagged, 2) {
	const Argument &argValue = ARG(0);
	const Argument &argTag = ARG(1);
	intrusive_ptr<RightValue> v = runtimeEnv->evalArgAs<RightValue>(argValue);
	const string tag = runtimeEnv->evalArgAs<STRING>(argTag)->theString;
	// TODO: must be fixed
	//if (!LexerNS::Lexer::isValidIdentifier(tag))
	//	throw RuntimeException(invalidTagError, argTag.exp);
	return new TaggedValue(v->untagged(), invocationExpression->packageName+"."+tag);
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(untagged, 1) {
	return runtimeEnv->evalArgAs<RightValue>(ARG(0))->untagged();
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(tag, 1) {
	const Argument &argValue = ARG(0);
	intrusive_ptr<RightValue> v = runtimeEnv->evalArgAs<RightValue>(argValue);
	if (intrusive_ptr<TaggedValue> taggedValue = dynamic_pointer_cast<TaggedValue>(v)) {
		const string::size_type l = invocationExpression->packageName.length()+1;
		if (taggedValue->tag.substr(0, l)==invocationExpression->packageName+".")
			return new STRING(taggedValue->tag.substr(l, string::npos));
		return new STRING(taggedValue->tag);
	}
	return STRING::empty;
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(TAGGED, 1) {
	const Argument &arg = ARG(0);
	const string tag = runtimeEnv->evalArgAs<STRING>(arg)->theString;
	if (!LexerNS::Lexer::isValidIdentifier(tag))
		throw RuntimeException(invalidTagError, arg.exp);
	return TYPE::tagType(tag);
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(type, 1) {
	return runtimeEnv->evalArgAs<RightValue>(ARG(0))->getType();
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(types, 0) {
	(void)runtimeEnv; // to avoid a compilation warning
	throw RuntimeException("Cannot return a list of all types because they're infinite (CoCoA 4 lies about this, I don't). You might want to use CurrentTypes() to get a list of currently instantiated types", invocationExpression);
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(CurrentTypes, 0) {
	return runtimeEnv->currentTypes();
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(error, 1) {
	const Argument &arg = ARG(0);
	const string message = runtimeEnv->evalArgAs<STRING>(arg)->theString;
	throw RuntimeException(message, invocationExpression);
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(Error, 1) {
	const Argument &arg = ARG(0);
	const string message = runtimeEnv->evalArgAs<STRING>(arg)->theString;
	throw RuntimeException(message, invocationExpression);
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(ascii, 1) {
  int which;
	intrusive_ptr<RightValue> v = runtimeEnv->evalArgAsT1orT2orT3<STRING, INT, LIST>(ARG(0), which);
  switch (which) {
  case 1: {
    intrusive_ptr<LIST> l(new LIST());
		const string s(RefTo<string>(v));
		BOOST_FOREACH(const char c, s)
      l->addValue(new INT(static_cast<int>(c)));
		return l; }
	case 2: {
		long n;
		if (!IsConvertible(n, RefTo<BigInt>(v)) || n<0 || n>255)
			throw RuntimeException("When the argument is an INT, it must be in the range [0..255]", ARG(0).exp);
		return Value::from(string(1, static_cast<char>(n))); }
	case 3: {
		string retValue;
    intrusive_ptr<LIST> list = dynamic_pointer_cast<LIST>(v);
		LIST::ContainerType::size_type size = list->size();
		for(LIST::ContainerType::size_type a=0; a<size; ++a) {
			intrusive_ptr<INT> N = dynamic_pointer_cast<INT>(list->getValue(a));
			long n;
			if (!N || !IsConvertible(n, N->theBigInt) || n<0 || n>255)
				throw RuntimeException("When the argument is a LIST, it must be a list of INT only in the range [0..255]", ARG(0).exp);
			retValue += static_cast<char>(n);
		}
		return Value::from(retValue);
  }
  default: throw RuntimeException(ERRORMissingCode(v),invocationExpression); 
  }
}
END_STD_BUILTIN_FUNCTION

//---- general ----

DECLARE_STD_BUILTIN_FUNCTION(TopLevelFunctions, 0) { // GL
	return runtimeEnv->topLevelFunctions();
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(CocoaPackagePath, 0) { // GL
	(void)runtimeEnv; // keeps the compiler happy (or, at least, silent ;-) )
	return Value::from(packageDir);
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(PkgName, 0) {
	(void)runtimeEnv; // to avoid a compilation warning
	return new STRING(invocationExpression->packageName);
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(aliases, 0) {
	runtimeEnv->topLevelAliases->dump(runtimeEnv->getOutputStream());
	return VoidValue::theInstance;
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(VersionInfo, 0) { // AMB
	(void)runtimeEnv; // keeps the compiler happy (or, at least, silent ;-) )
  intrusive_ptr<RECORD> rec(new RECORD);
  rec->setFieldNoCheck("CoCoALibVersion", Value::from(BuildInfo::version()));
  rec->setFieldNoCheck("CoCoAVersion", Value::from(CoCoAVersion()));
  rec->setFieldNoCheck("CompilationDate", Value::from(CompilationDate()));
  rec->setFieldNoCheck("CompilationFlags", Value::from(BuildInfo::CompilationFlags()));
  rec->setFieldNoCheck("CompilationDefines", Value::from(BuildInfo::CompilationDefines()));
  rec->setFieldNoCheck("Compiler", Value::from(BuildInfo::compiler()));
  rec->setFieldNoCheck("MachineIntNumBits", new INT(std::numeric_limits<unsigned int>::digits));
  rec->setFieldNoCheck("MachineLongNumBits", new INT(std::numeric_limits<unsigned long>::digits));
  rec->setFieldNoCheck("BoostVersion", new INT(BOOST_VERSION));
  const int NumLibs = len(ExternalLibs());
  intrusive_ptr<LIST> ExtLibList(new LIST);
  for (int i=0; i < NumLibs; ++i)
  {
    intrusive_ptr<RECORD> ExtLib(new RECORD);
    ExtLib->setFieldNoCheck("name", Value::from(ExternalLibs()[i].myName));
    ExtLib->setFieldNoCheck("version", Value::from(ExternalLibs()[i].myVersion));
		ExtLibList->addValue(ExtLib);
  }
  rec->setFieldNoCheck("ExternalLibs", ExtLibList);
  return rec;
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(SetStackSize, 1) {
//   intrusive_ptr<INT> N = runtimeEnv->evalArgAs<INT>(ARG(0));
//   const BigInt NewSize = N->theBigInt;
//   long n;
//   if (NewSize < 2 || !IsConvertible(n, NewSize)) throw RuntimeException("Ridiculous stack size", ARG(0).exp);
//   return new INT(runtimeEnv->ResizeStack(n, ARG(0).exp));
  long NewSize = runtimeEnv->evalArgAsLong(ARG(0));
  if (NewSize < 2) throw RuntimeException("Ridiculous stack size", ARG(0).exp);
  return Value::from(runtimeEnv->ResizeStack(NewSize, ARG(0).exp));
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(pause, 0) {
#ifdef C5IDE
	runtimeEnv->interpreter->singleStepExecution = true;
#else // #ifdef C5IDE
	(void)runtimeEnv; // keeps the compiler happy (or, at least, silent ;-) )
#endif // #ifdef C5IDE
	return VoidValue::theInstance;
}
END_STD_BUILTIN_FUNCTION

#ifdef C5IDE
DECLARE_STD_BUILTIN_FUNCTION(sleep, 1) {
	const Argument &arg = ARG(0);
	intrusive_ptr<INT> N = runtimeEnv->evalArgAs<INT>(arg);
	long l;
	if (!IsConvertible(l, N->theBigInt) || l<0 || l>1000)
		throw RuntimeException("The argument must be in the range [0..1000]", arg.exp);
	this_thread::sleep(boost::posix_time::seconds(static_cast<int>(l)));
	return VoidValue::theInstance;
}
END_STD_BUILTIN_FUNCTION

DECLARE_STD_BUILTIN_FUNCTION(debug, 1) {
	const Argument &arg = ARG(0);
	string filename = runtimeEnv->evalArgAs<STRING>(arg)->theString;
	try {
		runtimeEnv->interpreter->singleStepExecution = true;
		runtimeEnv->interpreter->readAndExecute(filename, false, false);
	} catch (const RuntimeException &) {
		throw;
	} catch (const BaseException &be) {
		throw RuntimeException(be.reason, invocationExpression->getBegin(), invocationExpression->getEnd());
	}
	return VoidValue::theInstance;
}
END_STD_BUILTIN_FUNCTION
#endif // #ifdef C5IDE


//---- manual CoCoAHelp ----

//DECLARE_STD_BUILTIN_FUNCTION(ReloadMan, 0) { // AMB
// variable number of args
DECLARE_ARITYCHECK_FUNCTION(ReloadMan) { return (0<=nArg) && (nArg<=1); }
DECLARE_BUILTIN_FUNCTION(ReloadMan) {  // AMB 2018-03
  invocationExpression->checkNumberOfArgs(0,1);// variable number of args
  std::vector<std::string> FileNames;
  if (invocationExpression->args.size()==1)
  {
    FileNames = runtimeEnv->evalArgAsListOf<STRING>(ARG(0));
  }
	try
  {
    ostringstream os;
    OnlineHelp::ReloadMan(os, FileNames);
    runtimeEnv->getOutputStream()->print(os.str())->flush();
	}
  catch (const std::exception& err)
  {
		throw RuntimeException(err.what(), invocationExpression->getBegin(), invocationExpression->getEnd());
	}
	return VoidValue::theInstance;
} 
// ---^^^--- variable number of args ---^^^---


} // namespace InterpreterNS
} // namespace CoCoA

// RCS header/log in the next few lines
// $Header: /Volumes/Home_1/cocoa/cvs-repository/CoCoALib-0.99/src/CoCoA-5/BuiltInFunctions.C,v 1.288 2018/03/08 17:05:06 bigatti Exp $
// $Log: BuiltInFunctions.C,v $
// Revision 1.288  2018/03/08 17:05:06  bigatti
// -- now ReloadMan can take one arg (list of files)
//
// Revision 1.287  2018/01/09 17:21:51  bigatti
// Summary: added include for boost version number
//
// Revision 1.286  2017/11/29 07:42:17  abbott
// Summary: Added Boost versio to versioninfo
//
// Revision 1.285  2017/04/27 07:07:09  bigatti
// -- added ExternalLibs to VersionInfo
//
// Revision 1.284  2017/04/26 09:05:39  bigatti
// -- updated author
//
// Revision 1.283  2017/03/02 10:04:22  bigatti
// -- modified interface for VerbosityLevel
//
// Revision 1.282  2016/11/23 14:01:16  bigatti
// -- added SetVerbosityLevel, IsVerbosityLevel
//
// Revision 1.281  2015/10/08 13:10:04  bigatti
// -- added revision log at the end
//
