//   Copyright (c)  2017  John Abbott,  Anna M. Bigatti

//   This file is part of the source of CoCoALib, the CoCoA Library.

//   CoCoALib is free software: you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.

//   CoCoALib is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.

//   You should have received a copy of the GNU General Public License
//   along with CoCoALib.  If not, see <http://www.gnu.org/licenses/>.

#include "CoCoA/SparsePolyOps.H"

#include "CoCoA/BigIntOps.H"
#include "CoCoA/PPMonoid.H"
#include "CoCoA/SparsePolyIter.H"
#include "CoCoA/SparsePolyOps-RingElem.H"
//#include "CoCoA/SparsePolyRing.H" // from SparsePolyOps-RingElem.H
#include "CoCoA/ring.H"

#include <vector>
using std::vector;

namespace CoCoA
{

  RingElem CoeffHeight(ConstRefRingElem f)
  {
    const char* const FnName = "CoeffHeight";
    const ring& P = owner(f);
    if (!IsPolyRing(P))
      CoCoA_ERROR(ERR::NotPolyRing, FnName);
    if (!IsOrderedDomain(CoeffRing(P)))
      CoCoA_ERROR(ERR::NotOrdDom, FnName);
    RingElem ans = zero(CoeffRing(P));
    // Loop below makes wasteful copies of the coeffs.
    for (SparsePolyIter it = BeginIter(f); !IsEnded(it); ++it)
    {
      const RingElem c = abs(coeff(it));
      if (c > ans)
        ans = c;
    }
    return ans;
  }


  // Just for univariate -- requires non-zero const term
  // A bit ugly because I wanted to avoid copying any coeffs.
  bool IsPalindromic(ConstRefRingElem f)
  {
    const ring& P = owner(f);
    if (!IsSparsePolyRing(P))
      CoCoA_ERROR(ERR::NotSparsePolyRing, "IsPalindromic");
    if (IsZero(f) || deg(f) == 0) return true;
    if (UnivariateIndetIndex(f) < 0)
      CoCoA_ERROR("Expected univariate poly", "IsPalindromic");
    const long n = NumTerms(f);
    const long degf = deg(f);
    vector<RingElemConstRawPtr> C; C.reserve(1+n/2);
    vector<long> exp(1+n/2);
    long count = 0;
    for (SparsePolyIter it = BeginIter(f); !IsEnded(it); ++it)
    {
      ++count;
      if (count <= n/2)
      {
        C.push_back(raw(coeff(it)));
        exp.push_back(deg(PP(it)));
      }
      else
      {
        if (IsOdd(n) && count == (n+1)/2) continue;
        if (deg(PP(it)) != degf-exp[n-count]) return false;
        if (coeff(it) != RingElemAlias(P, C[n-count])) return false;
      }
    }
    return true;
  }

  // univariate case
  RingElem reverse(ConstRefRingElem f)
  {
    const ring& P = owner(f);
    if (!IsSparsePolyRing(P))
      CoCoA_ERROR(ERR::NotSparsePolyRing, "reverse");
    if (IsZero(f)) return f;
    if (UnivariateIndetIndex(f) < 0)
      CoCoA_ERROR("Expected univariate poly", "reverse");
    return reverse(f, LPP(f));
  }

  RingElem reverse(ConstRefRingElem f, ConstRefPPMonoidElem t)
  {
    const ring& P = owner(f);
    if (!IsSparsePolyRing(P))
      CoCoA_ERROR(ERR::NotSparsePolyRing, "reverse");
    if (IsZero(f)) return f;
    RingElem ans = zero(P);
    for (SparsePolyIter it=BeginIter(f); !IsEnded(it); ++it)
    {
      PushFront(ans, coeff(it), t/PP(it));
    }
    return ans;
  }


  // Standard graeffe transformation (squares the roots)
  RingElem graeffe(ConstRefRingElem f)
  {
    const ring& P = owner(f);
    if (!IsSparsePolyRing(P))
      CoCoA_ERROR(ERR::NotSparsePolyRing, "graeffe");
    if (IsZero(f)) return f;
    if (deg(f) == 0) return one(P); //?????
    const long IndetIndex = UnivariateIndetIndex(f);
    if (IndetIndex < 0)
      CoCoA_ERROR("Expected univariate poly", "graeffe");
    PPMonoidElem pp = indet(PPM(P),IndetIndex);
    RingElem EvenPart(P);
    RingElem OddPart(P);
    for (SparsePolyIter it = BeginIter(f); !IsEnded(it); ++it)
    {
      const int d = deg(PP(it));
      if (IsEven(d))
        PushBack(EvenPart, coeff(it), power(pp,d/2));
      else
        PushBack(OddPart, coeff(it), power(pp,d/2)); // integer division!
    }
    const RingElem& x = indet(P, IndetIndex);
    // Fiddle answer so that LC is positive.
    RingElem ans = power(EvenPart,2) - x*power(OddPart,2);
    if (sign(LC(ans)) < 0) ans *= -1;
    return ans;
  }


  // Cubic graeffe transformation  (cubes the roots)
  // FORMULA:  f0^3 -3*f0*f1*f2*x +f1^3*x +f2^3*x^2
  RingElem graeffe3(ConstRefRingElem f)
  {
    const ring& P = owner(f);
    if (!IsSparsePolyRing(P))
      CoCoA_ERROR(ERR::NotSparsePolyRing, "graeffe");
    if (IsZero(f)) return f;
    const long IndetIndex = UnivariateIndetIndex(f);
    if (IndetIndex < 0)
      CoCoA_ERROR("Expected univariate poly", "graeffe");

    PPMonoidElem pp = indet(PPM(P),IndetIndex);
    vector<RingElem> part(3, zero(P));
    for (SparsePolyIter it = BeginIter(f); !IsEnded(it); ++it)
    {
      const int d = deg(PP(it));
      PushBack(part[d%3], coeff(it), power(pp, d/3)); // integer division!
    }
    const RingElem& x = indet(P, IndetIndex);
    const RingElem part012 = part[0]*part[1]*part[2];
    // Fiddle answer so that LC is positive.
    RingElem ans = power(part[0],3) + x*(power(part[1],3)-3*part012 + x*power(part[2],3));
    if (sign(LC(ans)) < 0) ans *= -1;
    return ans;
  }


} // end of namespace CoCoA


// RCS header/log in the next few lines
// $Header: /Volumes/Home_1/cocoa/cvs-repository/CoCoALib-0.99/src/AlgebraicCore/SparsePolyOps.C,v 1.6 2018/05/18 16:38:51 bigatti Exp $
// $Log: SparsePolyOps.C,v $
// Revision 1.6  2018/05/18 16:38:51  bigatti
// -- added include SparsePolyOps-RingElem.H
//
// Revision 1.5  2018/05/18 12:23:50  bigatti
// -- renamed IntOperations --> BigIntOps
//
// Revision 1.4  2018/05/17 15:50:56  bigatti
// -- added include SparsePolyIter
//
// Revision 1.3  2017/10/17 14:13:37  abbott
// Summary: Corrected silly bug in reverse
//
// Revision 1.2  2017/09/25 14:23:05  abbott
// Summary: Added CoeffHeight
//
// Revision 1.1  2017/09/14 16:22:36  abbott
// Summary: defns of graeffe, graeffe3
//
//
