//   Copyright (c)  2005-2010,2014  John Abbott

//   This file is part of the source of CoCoALib, the CoCoA Library.

//   CoCoALib is free software: you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.

//   CoCoALib is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.

//   You should have received a copy of the GNU General Public License
//   along with CoCoALib.  If not, see <http://www.gnu.org/licenses/>.


// Source code for RingWeylImpl

#include "CoCoA/RingWeyl.H"

#include "CoCoA/BigIntOps.H"
#include "CoCoA/DistrMPolyClean.H"
#include "CoCoA/MatrixForOrdering.H"
#include "CoCoA/MatrixOps.H"
#include "CoCoA/OpenMath.H"
#include "CoCoA/PPMonoid.H"
#include "CoCoA/PPMonoidEv.H"
#include "CoCoA/PPOrdering.H"
#include "CoCoA/RingDistrMPolyClean.H"
#include "CoCoA/RingHom.H"
#include "CoCoA/TmpGReductor.H"
#include "CoCoA/TmpUniversalInvolutiveBasisContainer.H"
#include "CoCoA/assert.H"
#include "CoCoA/convert.H"
#include "CoCoA/ideal.H"
#include "CoCoA/matrix.H"
#include "CoCoA/symbol.H"
#include "CoCoA/utils.H"

#include <memory>
using std::auto_ptr;
#include <iostream>
using std::ostream;
using std::endl;   // just for debugging
//#include <vector>
using std::vector;



namespace CoCoA
{

  class RingWeylImpl: public SparsePolyRingBase  //???? should be NCRingBase  non-commutative!!!
  {

  public:
    //    RingWeylImpl(const ring& CoeffRing, long NumIndets, const std::vector<long>& ElimIndets);
    RingWeylImpl(const ring& CoeffRing, const std::vector<symbol>& names, const std::vector<long>& ElimIndets);
    ~RingWeylImpl();

  private: // data members of RingWeyl
    const SparsePolyRing myReprRing;
    long myNumTrueIndetsValue;
    std::auto_ptr<RingElem> myZeroPtr;  ///< Every ring stores its own zero.
    std::auto_ptr<RingElem> myOnePtr;   ///< Every ring stores its own one.
    std::vector<RingElem> myIndetVector;
    std::vector<RingElem> myDerivationVector;

  public:
    //----------------------------------------------------------------------
    // Functions which every ring must implement:
    //----------------------------------------------------------------------
    virtual bool IamCommutative() const;
    virtual bool3 IamIntegralDomain3(bool) const;
    virtual bool IamTrueGCDDomain() const;
    virtual ConstRefRingElem myZero() const;
    virtual ConstRefRingElem myOne() const;
    using RingBase::myNew;    // disable warnings of overloading
    using RingBase::myAssign; // disable warnings of overloading
    using PolyRingBase::myIndets;    // disable warnings of overloading
    virtual RingElemRawPtr myNew() const;
    virtual RingElemRawPtr myNew(const MachineInt& n) const;
    virtual RingElemRawPtr myNew(const BigInt& N) const;
    virtual RingElemRawPtr myNew(ConstRawPtr rawcopy) const;
    virtual void myDelete(RawPtr rawx) const;                             // destroys x (incl all resources)
    virtual void mySwap(RawPtr rawx, RawPtr rawy) const;                  // swap(x, y)
    virtual void myAssign(RawPtr rawlhs, ConstRawPtr rawx) const;         // lhs = x
    virtual void myAssign(RawPtr rawlhs, const MachineInt& n) const;      // lhs = n
    virtual void myAssign(RawPtr rawlhs, const BigInt& N) const;          // lhs = N
    virtual void myAssignZero(RawPtr rawlhs) const;                       // lhs = 0
    virtual void myRecvTwinFloat(RawPtr rawlhs, ConstRawPtr rawx) const;
    virtual void myNegate(RawPtr rawlhs, ConstRawPtr rawx) const;         // lhs = -x
    virtual void myAdd(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const;        // lhs = x+y
    virtual void mySub(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const;        // lhs = x-y
    //    virtual void myMul(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const;        // lhs = x*y
    //    virtual void myDiv(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const;        // lhs = x/y
    virtual bool myIsDivisible(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const;// lhs = x/y, if divisible
    virtual bool myIsInvertible(ConstRawPtr rawx) const;                                // true iff x is invertible
    //    virtual bool myIsPrintAtom(ConstRawPtr rawx) const;
    virtual void myGcd(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const;  // not a TrueGCDDomain
    virtual void myPowerSmallExp(RawPtr rawlhs, ConstRawPtr rawx, long n) const;// lhs = x^n, n>1, x not -1,0,1
    virtual void mySymbols(std::vector<symbol>& SymList) const;   // appends ring's symbols to SymList
    virtual void myOutput(std::ostream& out, ConstRawPtr rawx) const;      // out << x
    using PolyRingBase::myOutputSelf; // disable warnings of overloading
    //    virtual void myOutputSelf(OpenMathOutput& OMOut) const;                // OMOut << R
    virtual std::string myImplDetails() const {return "RingWeyl";}
    virtual void myOutput(OpenMathOutput& OMOut, ConstRawPtr rawx) const;  // OMOut << x
    virtual bool myIsZero(ConstRawPtr rawx) const;                         // x == 0
    virtual bool myIsOne(ConstRawPtr rawx) const;                          // x == 1
    virtual bool myIsMinusOne(ConstRawPtr rawx) const;                     // x == -1
    //    virtual bool myIsZeroAddMul: use default definition
    virtual bool myIsEqual(ConstRawPtr rawx, ConstRawPtr rawy) const;      // x == y

    virtual ideal myIdealCtor(const std::vector<RingElem>& gens) const;

    virtual RingHom myCompose(const RingHom& phi, const RingHom& theta) const; // phi(theta(...))


    /*----------------------------------------------------------------------
     Member functions every PolyRing must have
     in addition to those of RingBase.
    ----------------------------------------------------------------------*/
    virtual long myNumIndets() const;
    virtual const ring& myCoeffRing() const;
    virtual const std::vector<RingElem>& myIndets() const;
    virtual void myIndetPower(RawPtr rawf, long var, long exp) const;

    ///@name Simple functions on polynomials
    //@{
    virtual long myNumTerms(ConstRawPtr rawf) const;
    virtual bool myIsConstant(ConstRawPtr rawf) const;
    virtual bool myIsIndet(long& index, ConstRawPtr rawf) const;
    virtual bool myIsMonomial(ConstRawPtr rawf) const;
    virtual long myStdDeg(ConstRawPtr rawf) const;
    virtual long myDeg(ConstRawPtr rawf, long var) const;
    virtual RingElemAlias myLC(ConstRawPtr rawf) const;
    virtual void myContent(RawPtr rawcontent, ConstRawPtr rawf) const;
    virtual void myRemoveBigContent(RawPtr rawf) const;
    virtual void myMulByCoeff(RawPtr rawf, ConstRawPtr rawc) const; ///< WEAK EXCEPTION GUARANTEE
    virtual bool myDivByCoeff(RawPtr rawf, ConstRawPtr rawc) const; ///< WEAK EXCEPTION GUARANTEE
    virtual void myDeriv(RawPtr rawlhs, ConstRawPtr rawf, ConstRawPtr rawx) const; ///< lhs = deriv(f, x)
    //@}

//     friend const RingElem& indet(const RingWeyl& RW, long var);
//     friend const RingElem& derivation(const RingWeyl& RW, long var);

    virtual RingHom myHomCtor(const ring& codomain, const RingHom& CoeffHom, const std::vector<RingElem>& IndetImages) const;

    //----------------------------------------------------------------------
    // Functions which every SparsePolyRing must implement:
    //----------------------------------------------------------------------

    virtual const PPMonoid& myPPM() const;

    ///@name   Functions for creating/building polynomials
    //@{
    virtual RingElem myMonomial(ConstRawPtr rawc, PPMonoidElemConstRawPtr rawpp) const;
    virtual SparsePolyIter myBeginIter(ConstRawPtr rawf) const;
    virtual SparsePolyIter myEndIter(ConstRawPtr rawf) const;

    virtual void myPushFront(RawPtr rawf, ConstRawPtr rawc, const std::vector<long>& expv) const;
    virtual void myPushBack(RawPtr rawf, ConstRawPtr rawc, const std::vector<long>& expv) const;
    virtual void myPushFront(RawPtr rawf, ConstRawPtr rawc, PPMonoidElemConstRawPtr rawpp) const;
    virtual void myPushBack(RawPtr rawf, ConstRawPtr rawc, PPMonoidElemConstRawPtr rawpp) const;
    //@}

    ///@name Special functions on polynomials needed for implementing Buchberger's Algorithm
    //@{
//     virtual void myWDeg(degree& d, ConstRawPtr rawf) const;
//     virtual int myCmpWDeg(ConstRawPtr rawf, ConstRawPtr rawg) const;
    virtual ConstRefPPMonoidElem myLPP(ConstRawPtr rawf) const;
    virtual void myMulByPP(RawPtr rawf, PPMonoidElemConstRawPtr rawpp) const;
    virtual bool myIsZeroAddLCs(RawPtr rawf, RawPtr rawg) const; ///< f+=LM(g); g-=LM(g); assumes LPP(f)==LPP(g); returns LC(f)+LC(g)==0
    virtual void myMoveLMToFront(RawPtr rawf, RawPtr rawg) const;
    virtual void myMoveLMToBack(RawPtr rawf, RawPtr rawg) const;
    virtual void myDeleteLM(RawPtr rawf) const; // ????? right interface
    virtual void myDivLM(RawPtr rawlhs, ConstRawPtr rawf, ConstRawPtr rawg) const; ///< lhs=div(LM(f),LM(g)); assumes f!=0,g!=0
    virtual int  myCmpLPP(ConstRawPtr rawf, ConstRawPtr rawg) const; ///< cmp(LPP(f),LPP(g)); assumes f!=0,g!=0
    virtual void myAddClear(RawPtr rawf, RawPtr rawg) const; ///< f+=g; g=0;
    virtual void myAppendClear(RawPtr rawf, RawPtr rawg) const; ///< f+=g; g=0; appends g to f with no checks

    virtual void myAddMulLM(RawPtr rawf, ConstRawPtr rawh, ConstRawPtr rawg) const; ///<  f += LM(h)*g
    virtual void myAddMulLM(RawPtr rawf, ConstRawPtr rawh, ConstRawPtr rawg, SkipLMFlag) const; ///<  f += LM(h)*g
    virtual void myReductionStep(RawPtr rawf, ConstRawPtr rawg) const;
    // ??? aggiungere coefficiente
    virtual void myReductionStepGCD(RawPtr rawf, ConstRawPtr rawg, RingElem& FScale) const;
    // should it all be in ReductionStep ??? ANNA
    //@}


  private: // HomImpl class: overwrite some implementations of SparsePolyRing functions
    class HomImpl: public SparsePolyRingBase::HomImpl
    {
    public:
      HomImpl(const SparsePolyRing& domain, const ring& codomain, const RingHom& CoeffHom, const std::vector<RingElem>& IndetImages);
    };

  private: // Ideal class: overwrite some implementations of SparsePolyRing functions
    class IdealImpl: public SparsePolyRingBase::IdealImpl
    {
    public:
      IdealImpl(const SparsePolyRing& P, const std::vector<RingElem>& gens);
      // default copy ctor is OK
      virtual void myReduceMod(RingElemRawPtr rawr) const; // r elem of R, where I is ideal in R
      virtual bool IhaveElem(RingElemConstRawPtr rawr) const;
      virtual void myIntersect(const ideal&);
      virtual void myColon(const ideal&);
      virtual bool myDivMod(RingElemRawPtr rawlhs, RingElemConstRawPtr rawnum, RingElemConstRawPtr rawden) const; // result is true iff quot exists & is unique; if true set lhs = num/den modulo I

      virtual bool myTestIsMaximal() const;
      virtual bool myTestIsPrimary() const;
      virtual bool myTestIsPrime() const;
      virtual bool myTestIsRadical() const;
      virtual const std::vector<RingElem>& myGBasis(const CpuTimeLimit&) const;
    };
  };

  //----------------------------------------------------------------------

  std::vector<symbol> WANAMES(long NumIndets)
  {
    vector<symbol> ans = SymbolRange("x", 0, NumIndets-1);
    vector<symbol> d   = SymbolRange("d", 0, NumIndets-1);
    ans.insert(ans.end(), d.begin(), d.end());
    return ans;
  }

  std::vector<symbol> WANAMES(const std::vector<symbol>& names)
  {
    vector<symbol> ans=names;
    const long NumNames = len(names);
    ans.reserve(2*NumNames);
    for ( long i=0 ; i < NumNames ; ++i )
    {
      if ( NumSubscripts(names[i])!=0 )
        CoCoA_ERROR("names must have no subscripts",
                    "WANAMES(const vector<symbol>& names)");
      ans.push_back(symbol("d"+head(names[i])));
    }
    return ans;
  }


  RingWeylImpl::RingWeylImpl(const ring& CoeffRing, const std::vector<symbol>& WANames, const std::vector<long>& ElimIndets):
    myReprRing(NewPolyRing(CoeffRing,
                           WANames,
                           NewMatrixOrdering(ElimMat(ElimIndets, len(WANames)),
                                             0))),
    myNumTrueIndetsValue(len(WANames)/2)
  {
    const long NumTrueIndets=len(WANames)/2;
    CoCoA_ASSERT(IsCommutative(CoeffRing));
    myRefCountInc();  // this is needed for exception cleanliness, in case one of the lines below throws
    myZeroPtr.reset(new RingElem(ring(this)));
    myOnePtr.reset(new RingElem(ring(this), 1));
    //    myIndetVector.resize(myNumIndetsValue, *myZeroPtr);
    //    myDerivationVector.resize(myNumIndetsValue, *myZeroPtr);
    //    for (long i=0; i < myNumIndetsValue; ++i)
    myIndetVector.resize(2*NumTrueIndets, *myZeroPtr);
    vector<long> expv(2*NumTrueIndets);

    for (long i=0; i < 2*NumTrueIndets; ++i)
    {
      expv[i] = 1;
      myPushFront(raw(myIndetVector[i]), raw(one(CoeffRing)), expv);
      expv[i] = 0;
    }
    myRefCountZero(); // otherwise it is 2 + NumIndets and won't be destroyed
  }


  RingWeylImpl::~RingWeylImpl()
  {}


//   inline const RingWeylImpl* RingWeyl::operator->() const
//   {
//     return static_cast<const RingWeylImpl*>(myRingPtr());
//   }


//   inline WeylPoly& ring_weyl::AsWeylPoly(RawPtr rawx) const
//   {
//     return *x.WeylPolyPtr;
//     //    return *static_cast<ring_weyl::WeylPoly*>(x.WeylPolyPtr);
//   }


//   inline const WeylPoly& ring_weyl::AsWeylPoly(ConstRawPtr rawx) const
//   {
//     return *x.WeylPolyPtr;
//     //    return *static_cast<ring_weyl::WeylPoly*>(x.WeylPolyPtr);
//   }


//   RingWeyl::RingWeyl(const RingWeylImpl* RingPtr):
//       SparsePolyRing(RingPtr)
//   {}


  //----------------------------------------------------------------------
  // Functions which every ring must implement:
  //----------------------------------------------------------------------

  bool RingWeylImpl::IamCommutative() const
  {
    return myNumIndets() == 0; // we are assuming the coeff ring is commutative
  }


  bool3 RingWeylImpl::IamIntegralDomain3(bool QuickMode) const
  {
    return myReprRing->IamIntegralDomain3(QuickMode); //??? I think this is right
  }


  bool RingWeylImpl::IamTrueGCDDomain() const
  {
    return false; // I have no clue how to compute GCDs even if they do exist
  }


  ConstRefRingElem RingWeylImpl::myZero() const
  {
    return *myZeroPtr;
  }


  ConstRefRingElem RingWeylImpl::myOne() const
  {
    return *myOnePtr;
  }


  RingElemRawPtr RingWeylImpl::myNew() const
  {
    return myReprRing->myNew();
  }


  RingElemRawPtr RingWeylImpl::myNew(const MachineInt& n) const
  {
    return myReprRing->myNew(n);
  }


  RingElemRawPtr RingWeylImpl::myNew(const BigInt& N) const
  {
    return myReprRing->myNew(N);
  }


  RingElemRawPtr RingWeylImpl::myNew(ConstRawPtr rawcopy) const
  {
    return myReprRing->myNew(rawcopy);
  }


  void RingWeylImpl::myDelete(RawPtr rawx) const
  {
    myReprRing->myDelete(rawx);
  }


  void RingWeylImpl::mySwap(RawPtr rawx, RawPtr rawy) const
  {
    myReprRing->mySwap(rawx, rawy);
  }


  void RingWeylImpl::myAssign(RawPtr rawlhs, ConstRawPtr rawx) const
  {
    myReprRing->myAssign(rawlhs, rawx);
  }


  void RingWeylImpl::myAssign(RawPtr rawlhs, const MachineInt& n) const
  {
    myReprRing->myAssign(rawlhs, n);
  }


  void RingWeylImpl::myAssign(RawPtr rawlhs, const BigInt& N) const
  {
    myReprRing->myAssign(rawlhs, N);
  }


  void RingWeylImpl::myAssignZero(RawPtr rawlhs) const
  {
    myReprRing->myAssignZero(rawlhs);
  }


  void RingWeylImpl::myRecvTwinFloat(RawPtr rawlhs, ConstRawPtr rawx) const
  {
    CoCoA_ASSERT(!IsExact(myReprRing));
    myReprRing->myRecvTwinFloat(rawlhs, rawx);
  }

  void RingWeylImpl::myNegate(RawPtr rawlhs, ConstRawPtr rawx) const
  {
    myReprRing->myNegate(rawlhs, rawx);
  }


  void RingWeylImpl::myAdd(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const
  {
    myReprRing->myAdd(rawlhs, rawx, rawy);
  }


  void RingWeylImpl::mySub(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const
  {
    myReprRing->mySub(rawlhs, rawx, rawy);
  }


  //??? ANNA: I believe the general implementation in SparsePolyRing works for RingWeyl
//   void RingWeylImpl::myMul(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const
//   {
//     if (myReprRing->myIsZero(x) || myReprRing->myIsZero(y)) { myReprRing->myAssignZero(rawlhs); return; }
//     RingElem f(myReprRing);
//     myReprRing->myAssign(raw(f), rawx);
//     ConstRefRingElem g(myReprRing, rawy);
// //    std::clog<<"f="<<f<<std::endl;
// //    std::clog<<"g="<<g<<std::endl;
//     RingElem ans(myReprRing); // compute answer in a temporary for exception safety
//     while (!CoCoA::IsZero(f))
//     {
//       RingElem LMf(myReprRing);
//       myMoveLMToFront(raw(LMf), raw(f));
//       PPMonoidElem LPPf = LPP(LMf);
//       RingElem LMfg(g);
//       myMulByPP(raw(LMfg), raw(LPPf));
// //      std::clog<<"LMf="<<LMf<<std::endl;
// //      std::clog<<"LMfg="<<LMfg<<std::endl;
//       myReprRing->myMulByCoeff(raw(LMfg), raw(LC(LMf))); //????UGLY!!!!
//       ans += LMfg;
//     }
//     mySwap(rawlhs, raw(ans));// really an assignment -- is this safe????
//   }


//   void RingWeylImpl::myDiv(RawPtr /*rawlhs*/, ConstRawPtr /*rawx*/, ConstRawPtr /*rawy*/) const
//   {
//     CoCoA_ERROR(ERR::NYI, "RingWeylImpl::myDiv");
//     return;//???
// //     bool OK;                                            // Two lines o/w compiler complains that
// //     OK = CoCoA::WeylDiv(AsDMPI(rawlhs), AsDMPI(x), AsDMPI(y)); // OK is unused when debugging is off.
// //     CoCoA_ASSERT(OK);
//   }


  bool RingWeylImpl::myIsDivisible(RawPtr /*rawlhs*/, ConstRawPtr /*rawx*/, ConstRawPtr /*rawy*/) const
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::myIsDivisible");
    return false;
//    return CoCoA::WeylDiv(AsDMPI(rawlhs), AsDMPI(x), AsDMPI(y));
  }


  bool RingWeylImpl::myIsInvertible(ConstRawPtr /*rawx*/) const
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::myIsInvertible");  //??? d[1]*x[1] = 1
    return false;//???
  }


//   bool RingWeylImpl::myIsPrintAtom(ConstRawPtr rawx) const
//   {
//     return myReprRing->myIsPrintAtom(rawx);  //??? default always false
//   }


  void RingWeylImpl::myGcd(RawPtr /*rawlhs*/, ConstRawPtr /*rawx*/, ConstRawPtr /*rawy*/) const
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::myGcd");
  }


  void RingWeylImpl::myPowerSmallExp(RawPtr rawlhs, ConstRawPtr rawx, long n) const
  {
    // Assert that we have a genuinely non-trivial case.
    CoCoA_ASSERT(n > 1);
    CoCoA_ASSERT(!myIsZero(rawx) && !myIsOne(rawx) && !myIsMinusOne(rawx));

    mySequentialPower(rawlhs, rawx, n); // probably better than myBinaryPower, I guess.
  }


  void RingWeylImpl::mySymbols(std::vector<symbol>& SymList) const
{ //??? ANNA: which symbols are in a RingWeylImpl??
    myReprRing->mySymbols(SymList);
  }


  void RingWeylImpl::myOutput(std::ostream& out, ConstRawPtr rawx) const
  {
    myReprRing->myOutput(out, rawx);
  }


//   void RingWeylImpl::myOutputSelf(std::ostream& out) const
//   {
//     out << "RingWeylImpl(" << CoeffRing(myReprRing) << ", " << NumIndets(myReprRing) << ")";
//     //?????????  WHAT ABOUT THE ORDERING AND GRADING????
//   }


//   void RingWeylImpl::myOutputSelf(OpenMathOutput& OMOut) const
//   {
//     OMOut->mySendApplyStart();
//     OMOut << OpenMathSymbol("polyd", "poly_ring_d"); //??? weyl?
//     OMOut << CoeffRing(myReprRing);
//     OMOut << NumIndets(myReprRing); //???? losing the ordering and grading info here!!!
//     OMOut->mySendApplyEnd();
//   }


  void RingWeylImpl::myOutput(OpenMathOutput& OMOut, ConstRawPtr rawx) const
  {
    myReprRing->myOutput(OMOut, rawx);
  }


  bool RingWeylImpl::myIsZero(ConstRawPtr rawx) const
  {
    return myReprRing->myIsZero(rawx);
  }


  bool RingWeylImpl::myIsOne(ConstRawPtr rawx) const
  {
    return myReprRing->myIsOne(rawx);
  }


  bool RingWeylImpl::myIsMinusOne(ConstRawPtr rawx) const
  {
    return myReprRing->myIsMinusOne(rawx);
  }


  // bool RingWeylImpl::myIsZeroAddMul(RawPtr rawlhs, ConstRawPtr rawy, ConstRawPtr rawz) const; // use default definition


  bool RingWeylImpl::myIsEqual(ConstRawPtr rawx, ConstRawPtr rawy) const
  {
    return myReprRing->myIsEqual(rawx, rawy);
  }


  RingHom RingWeylImpl::myCompose(const RingHom& /*phi*/, const RingHom& theta) const
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::compose");
    return theta; // just to keep compiler quiet
  }

  //----------------------------------------------------------------------

//   long NumIndets(const RingWeyl& RW)
//   {
//     return RW->myNumIndets();
//   }

  //----------------------------------------------------------------------
  // Functions which every PolyRing must implement
  //----------------------------------------------------------------------

  long RingWeylImpl::myNumIndets() const
  {
    return myReprRing->myNumIndets();
  }


  const ring& RingWeylImpl::myCoeffRing() const
  {
    return myReprRing->myCoeffRing();
  }


  const std::vector<RingElem>& RingWeylImpl::myIndets() const
  {
    return myIndetVector;
  }


  void RingWeylImpl::myIndetPower(RawPtr rawf, long var, long exp) const
  {
    myReprRing->myIndetPower(rawf, var, exp);
  }


  long RingWeylImpl::myNumTerms(ConstRawPtr rawx) const
  {
    return myReprRing->myNumTerms(rawx);
  }


  bool RingWeylImpl::myIsConstant(ConstRawPtr rawf) const
  {
    return myReprRing->myIsConstant(rawf);
  }


  bool RingWeylImpl::myIsIndet(long& index, ConstRawPtr rawf) const
  {
    return myReprRing->myIsIndet(index, rawf);
  }


  bool RingWeylImpl::myIsMonomial(ConstRawPtr rawf) const
  {
    return myReprRing->myIsMonomial(rawf);
  }


  long RingWeylImpl::myStdDeg(ConstRawPtr rawf) const
  {
    return myReprRing->myStdDeg(rawf);
  }


  long RingWeylImpl::myDeg(ConstRawPtr rawf, long var) const
  {
    CoCoA_ASSERT(!myIsZero(rawf));
    return myReprRing->myDeg(rawf, var);//????????
  }


  RingElemAlias RingWeylImpl::myLC(ConstRawPtr rawf) const
  {
    CoCoA_ASSERT(!myIsZero(rawf));
    return myReprRing->myLC(rawf);//???????????
  }


  void RingWeylImpl::myContent(RawPtr rawdest, ConstRawPtr rawf) const
  {
    myReprRing->myContent(rawdest, rawf);
  }


  void RingWeylImpl::myRemoveBigContent(RawPtr rawf) const
  {
    myReprRing->myRemoveBigContent(rawf);
  }


  void RingWeylImpl::myMulByCoeff(RawPtr rawf, ConstRawPtr rawc) const // WEAK EXCEPTION GUARANTEE
  {
    myReprRing->myMulByCoeff(rawf, rawc);
  }


  bool RingWeylImpl::myDivByCoeff(RawPtr rawf, ConstRawPtr rawc) const // WEAK EXCEPTION GUARANTEE
  {
    return myReprRing->myDivByCoeff(rawf, rawc);
  }


  void RingWeylImpl::myDeriv(RawPtr /*rawlhs*/, ConstRawPtr /*rawf*/, ConstRawPtr /*rawx*/) const
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::myDeriv");
  }


  RingHom RingWeylImpl::myHomCtor(const ring& /*codomain*/, const RingHom& /*CoeffHom*/, const std::vector<RingElem>& /*IndetImages*/) const
  {
    CoCoA_ERROR("DOES NOT EXIST", "RingWeylImpl::myHomCtor");
    return IdentityHom(myReprRing); // just to keep compiler quiet
  }



  //----------------------------------------------------------------------
  // Functions which every SparsePolyRing must implement:
  //----------------------------------------------------------------------

  const PPMonoid& RingWeylImpl::myPPM() const
  {
    return myReprRing->myPPM();
  }


  RingElem RingWeylImpl::myMonomial(ConstRawPtr rawc, PPMonoidElemConstRawPtr rawpp) const
  {
    RingElem ans(ring(this));
    RingElem m = myReprRing->myMonomial(rawc, rawpp);
    mySwap(raw(ans), raw(m));
    return ans;
  }


  SparsePolyIter RingWeylImpl::myBeginIter(ConstRawPtr rawf) const
  {
    return myReprRing->myBeginIter(rawf);
  }


  SparsePolyIter RingWeylImpl::myEndIter(ConstRawPtr rawf) const
  {
    return myReprRing->myEndIter(rawf);
  }


  void RingWeylImpl::myPushFront(RawPtr rawf, ConstRawPtr rawc, const std::vector<long>& expv) const
  {
    myReprRing->myPushFront(rawf, rawc, expv);//???  how many elements does expv have???
  }


  void RingWeylImpl::myPushBack(RawPtr rawf, ConstRawPtr rawc, const std::vector<long>& expv) const
  {
    myReprRing->myPushBack(rawf, rawc, expv);//???  how many elements does expv have???
  }


  void RingWeylImpl::myPushFront(RawPtr rawf, ConstRawPtr rawc, PPMonoidElemConstRawPtr rawpp) const
  {
    myReprRing->myPushFront(rawf, rawc, rawpp);
  }


  void RingWeylImpl::myPushBack(RawPtr rawf, ConstRawPtr rawc, PPMonoidElemConstRawPtr rawpp) const
  {
    myReprRing->myPushBack(rawf, rawc, rawpp);
  }


  ConstRefPPMonoidElem RingWeylImpl::myLPP(ConstRawPtr rawf) const
  {
    CoCoA_ASSERT(!myIsZero(rawf));
    return myReprRing->myLPP(rawf);
  }


  // f = pp*f  NOTE: pp is on the LEFT and f is on the RIGHT
  void RingWeylImpl::myMulByPP(RawPtr rawf, PPMonoidElemConstRawPtr rawpp) const
  {
    if (myReprRing->myIsZero(rawf)) return;

    RingElem g(myReprRing);
    myReprRing->myAssign(raw(g), rawf);
//???    std::clog<<"Alias(f)="<<g<<std::endl;
//    const long nvars = myNumIndetsValue;
    for (long idx=0; idx < myNumTrueIndetsValue; ++idx)
    {
      const long Didx = idx + myNumTrueIndetsValue;
      const long d = PPM(myReprRing)->myExponent(rawpp, Didx);
//???      std::clog<<"deg of D["<<idx<<"]="<<d<<std::endl;
      RingElem der(g);  // copy of f in myReprRing
      RingElem sum = g*IndetPower(myReprRing, Didx, d);
      for (long i=1; i <= d; ++i)
      {
        der = deriv(der, idx);
        if (IsZero(der)) break;
//        std::clog<<"bin(" << d << ", " << i << ")="<<binomial(d, i)<<std::endl;
        sum += binomial(d, i)*der*IndetPower(myReprRing, Didx, d-i);
//        std::clog<<"summand="<<binomial(d, i)*der*IndetPower(myReprRing, Didx, d-i)<<std::endl;
      }
      g = sum;
//      std::clog<<"g="<<g<<std::endl;
    }
    vector<long> expv(2*myNumTrueIndetsValue);
    PPM(myReprRing)->myExponents(expv, rawpp);
    for (long idx=0; idx < myNumTrueIndetsValue; ++idx)
      expv[idx+myNumTrueIndetsValue] = 0;
    PPMonoidElem pp2(PPM(myReprRing));
    PPM(myReprRing)->myAssign(raw(pp2), expv);
    myReprRing->myMulByPP(raw(g), raw(pp2));
//???    std::clog<<"mul gives "<<g<<std::endl;
    mySwap(rawf, raw(g));// really an assignment -- is this safe????
  }


  bool RingWeylImpl::myIsZeroAddLCs(RawPtr rawf, RawPtr rawg) const
  {
    return myReprRing->myIsZeroAddLCs(rawf, rawg);
  }


  void RingWeylImpl::myMoveLMToFront(RawPtr rawf, RawPtr rawg) const
  {
    CoCoA_ASSERT(!myIsZero(rawg));
    myReprRing->myMoveLMToFront(rawf, rawg);
  }


  void RingWeylImpl::myMoveLMToBack(RawPtr rawf, RawPtr rawg) const
  {
    CoCoA_ASSERT(!myIsZero(rawg));
    myReprRing->myMoveLMToBack(rawf, rawg);
  }


  void RingWeylImpl::myDeleteLM(RawPtr rawf) const
  {
    CoCoA_ASSERT(!myIsZero(rawf));
    myReprRing->myDeleteLM(rawf);
  }


  void RingWeylImpl::myDivLM(RawPtr rawlhs, ConstRawPtr rawf, ConstRawPtr rawg) const
  {
    CoCoA_ASSERT(!myIsZero(rawf) && !myIsZero(rawg));
    myReprRing->myDivLM(rawlhs, rawf, rawg);
  }


  int  RingWeylImpl::myCmpLPP(ConstRawPtr rawf, ConstRawPtr rawg) const
  {
    CoCoA_ASSERT(!myIsZero(rawf) && !myIsZero(rawg));
    return myReprRing->myCmpLPP(rawf, rawg);
  }


  void RingWeylImpl::myAddClear(RawPtr rawf, RawPtr rawg) const
  {
    myReprRing->myAddClear(rawf, rawg);
  }


  void RingWeylImpl::myAppendClear(RawPtr rawf, RawPtr rawg) const
  {
    myReprRing->myAppendClear(rawf, rawg);
  }


  void RingWeylImpl::myAddMulLM(RawPtr rawf, ConstRawPtr rawh, ConstRawPtr rawg) const //???? delete me???
  {
    myAddMulLM(rawf, rawh, rawg, DontSkipLMg);
  }


  void RingWeylImpl::myAddMulLM(RawPtr rawf, ConstRawPtr rawh, ConstRawPtr rawg, SkipLMFlag skip) const
  {
    CoCoA_ASSERT(myNumTerms(rawh)==1);
    RingElem prod(ring(this), myNew(rawg));
    myMulByCoeff(raw(prod), raw(myLC(rawh)));
    myMulByPP(raw(prod), raw(myLPP(rawh)));
    myReprRing->myAddMulLM(rawf, raw(myOne()), raw(prod), skip);
  }


  void RingWeylImpl::myReductionStep(RawPtr rawf, ConstRawPtr rawg) const
  {
    PPMonoidElem q = myLPP(rawf)/myLPP(rawg);
    RingElem c = -myLC(rawf)/myLC(rawg); //    RingElem c = -LC(repf)/LC(repg);
    RingElem qg(myReprRing); myReprRing->myAssign(raw(qg), rawg); // qg is copy of rawg
    myMulByPP(raw(qg), raw(q));  // qg = q * g  //??? should we use myAddMul
    myMulByCoeff(raw(qg), raw(c));
    myAdd(rawf, rawf, raw(qg)); //  repf += qg;


/*
//???    ConstRefRingElem aliasg(ring(this), rawg);
//???    std::clog<<"g="<<aliasg<<std::endl;
    PPMonoidElem LPPf = myReprRing->myLPP(f);
    PPMonoidElem LPPg = myReprRing->myLPP(g);
    PPMonoidElem q = LPPf/LPPg; // quotient exists
    RingElem LCf(CoeffRing(myReprRing));
    CoeffRing(myReprRing)->myAssign(raw(LCf), myReprRing->myRawLC(f));
    RingElem LCg(CoeffRing(myReprRing));
    CoeffRing(myReprRing)->myAssign(raw(LCg), myReprRing->myRawLC(g));
    RingElem c = -LCf/LCg;
///    RingElem c(myCoeffRing());
///    myCoeffRing()->myDiv(raw(c), myReprRing->myRawLC(f), myReprRing->myRawLC(g));
    RingElem g1(myReprRing);
//???    std::clog<<"ReductionStep: q="<<q<<std::endl;
//???    std::clog<<"ReductionStep: c="<<c<<std::endl;
    myReprRing->myAssign(raw(g1), rawg);
//???    std::clog<<"ReductionStep: g="<<g1<<std::endl;
    myMul(raw(g1), raw(q));  // g1 = q*g;
    myReprRing->myMulByCoeff(raw(g1), raw(c));
//???    std::clog<<"ReductionStep: prod="<<g1<<std::endl;
    {
//???      ConstRefRingElem aliasf(ring(this),f);
//???      std::clog<<"REDUCING f="<<aliasf<<std::endl;
//???      std::clog<<"REDN by  g="<<aliasg<<std::endl;
      myAdd(f, f, raw(g1));
//???    std::clog<<"RESULT is  "<<aliasf<<std::endl<<std::endl;
    }
*/
  }


  void RingWeylImpl::myReductionStepGCD(RawPtr /*rawf*/, ConstRawPtr /*rawg*/, RingElem& /*fscale*/) const
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::ReductionStepGCD");
  }


  //---------------------------------------------------------------------------
  // Functions to do with RingWeylImpl::HomImpl


  RingWeylImpl::HomImpl::HomImpl(const SparsePolyRing& domain, const ring& codomain, const RingHom& CoeffHom, const std::vector<RingElem>& IndetImages):
    SparsePolyRingBase::HomImpl(domain, codomain, CoeffHom, IndetImages)
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::HomImpl::HomImpl");
  }



  //---------------------------------------------------------------------------
  // Functions for the class RingWeylImpl::IdealImpl

  // inheritance is delicate here: this function is **necessary**
  ideal RingWeylImpl::myIdealCtor(const std::vector<RingElem>& gens) const
  {
    return ideal(new IdealImpl(SparsePolyRing(this), gens)); //??? ugly ???
  }


  RingWeylImpl::IdealImpl::IdealImpl(const SparsePolyRing& P, const std::vector<RingElem>& gens):
    SparsePolyRingBase::IdealImpl(P, gens)
  {}


  bool RingWeylImpl::IdealImpl::myTestIsMaximal() const
  { CoCoA_ERROR(ERR::NYI, "RingWeylImpl::IdealImpl::myTestIsMaximal()"); return false; /* just to keep compiler quiet */ }


  bool RingWeylImpl::IdealImpl::myTestIsPrimary() const
  { CoCoA_ERROR(ERR::NYI, "RingWeylImpl::IdealImpl::myTestIsPrimary()"); return false; /* just to keep compiler quiet */ }


  bool RingWeylImpl::IdealImpl::myTestIsPrime() const
  { CoCoA_ERROR(ERR::NYI, "RingWeylImpl::IdealImpl::myTestIsPrime()"); return false; /* just to keep compiler quiet */ }


  bool RingWeylImpl::IdealImpl::myTestIsRadical() const
  { CoCoA_ERROR(ERR::NYI, "RingWeylImpl::IdealImpl::myTestIsRadical()"); return false; /* just to keep compiler quiet */ }


  void RingWeylImpl::IdealImpl::myReduceMod(RingElemRawPtr /*rawr*/) const
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::WeylIdeal::IdealImpl::myReduceMod");
  }


  bool RingWeylImpl::IdealImpl::IhaveElem(RingElemConstRawPtr /*rawr*/) const
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::WeylIdeal::IhaveElem");
    return false; /* just to keep compiler quiet */
  }


  void RingWeylImpl::IdealImpl::myIntersect(const ideal& /*J*/)
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::WeylIdeal::myIntersect"); //???
  }


  void RingWeylImpl::IdealImpl::myColon(const ideal& /*J*/)
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::WeylIdeal::myColon"); //???
  }


  bool RingWeylImpl::IdealImpl::myDivMod(RingElemRawPtr /*rawlhs*/, RingElemConstRawPtr /*rawnum*/, RingElemConstRawPtr /*rawden*/) const
  {
    CoCoA_ERROR(ERR::NYI, "RingWeylImpl::WeylIdeal::myDivMod");
    return false; /* just to keep compiler quiet */
  }


  const std::vector<RingElem>& RingWeylImpl::IdealImpl::myGBasis(const CpuTimeLimit& CheckForTimeOut) const
  {
    if (IhaveGBasis()) return myGBasisValue;
    CoCoA_ASSERT(myGBasisValue.empty());
    vector<RingElem> GensList, GBList;
    GensList.insert(GensList.end(), myGensValue.begin(), myGensValue.end());
    bool IsHomog=false;
    bool IsSatAlg=false;
    GRingInfo GRI(myP, IsHomog, IsSatAlg, NewDivMaskEvenPowers());
    GBCriteria criteria(GBCriteria::DontUseCoprime, GBCriteria::UseGM,
                        GBCriteria::UseBack, GBCriteria::DontUseDiv);
    GReductor GR(GRI, GensList,
                 CheckForTimeOut,
                 GReductor::AffineAlg,
                 Reductors::DontUseBorel, GReductor::DontUseDynamicAlg,
                 criteria);
    GR.myDoGBasis();
    //    GR.myDoAFFGBasis();
    GR.myGBasis(GBList);
    myGBasisValue.insert(myGBasisValue.end(), GBList.begin(), GBList.end());
    IhaveGBasisValue = true;
    return myGBasisValue;
  }


  //----------------------------------------------------------------------
  // Pseudo-ctors for (sparse) polynomial rings.


  SparsePolyRing NewWeylAlgebra(const ring& CoeffRing, long NumIndets, const vector<long>& ElimIndets)
  {
    return SparsePolyRing(new RingWeylImpl(CoeffRing, WANAMES(NumIndets), ElimIndets));
  }


  SparsePolyRing NewWeylAlgebra(const ring& CoeffRing, const std::vector<symbol>& names, const vector<long>& ElimIndets)
  {
    return SparsePolyRing(new RingWeylImpl(CoeffRing, WANAMES(names), ElimIndets));
  }


//   const RingElem& indet(const RingWeyl& RW, long var)
//   {
//     if (var > CoCoA::NumIndets(RW)) CoCoA_ERROR("Indeterminate index too large", "indet(RW,var)");
//     return RW->myIndetVector[var];
//   }


//   const RingElem& derivation(const RingWeyl& RW, long var)
//   {
//     if (var > CoCoA::NumIndets(RW)) CoCoA_ERROR("Indeterminate index too large", "derivation(RW,var)");
//     return RW->myDerivationVector[var];
//   }


} // end of namespace CoCoA

// #error "JUNK AFTER THIS LINE"
// ?????????????????????????????????????????????????????????????????????????????
// #include <algorithm>
// #include <vector>
// #include <iostream>

// using std::max;
// using std::swap;
// using std::vector;
// using std::ostream;
// using std::endl;   // just for debugging


// #include "CoCoA/deriv.H"
// #include "CoCoA/ring_weyl.H"

// namespace  // unnamed, for file local functions
// {

//   using namespace CoCoA;


// //----------------------------------------------------------------------


// //  // f = pp*g where the product is in the Weyl algebra
// //  void act(RingElem& f, const RingElem& pp, const RingElem& g)
// //  {
// //    ASSERT(IsPolyRing(owner(f)));
// //    const PolyRing& P = PolyRing::owner(f);
// //    ASSERT(&owner(g) == &P && &owner(pp) == &P);
// //    ASSERT(!IsZero(pp) && NumTerms(pp) == 1);
// //    //  std::clog<<"entered ucha's action" << endl;
// //    //  std::clog << "pp=" << pp << endl;
// //    f = g;
// //    if (IsZero(f)) return;
// //    const long nvars = NumIndets(P);

// //    vector<long> expv(nvars);
// //    PPM(P).exponents(expv, raw(LPP(pp)));

// //    for (long var = nvars/2; var < nvars; ++var)
// //    {
// //      //    std::clog << "doing D var=" << var << endl;
// //      const long n = expv[var];
// //      //    std::clog << "order = " << n << endl;
// //      RingElem derf = f;
// //      f *= P.IndetPower(var, n);
// //      for (long i=1; i <= n; ++i)
// //      {
// //        derf = deriv(derf, var-nvars/2);
// //        f += binomial(n, i)*derf*P.IndetPower(var, n-i); // *IndetPower(h, 2*i); // for homog case
// //        std::clog<<"binomial("<<n<<","<<i<<")="<<binomial(n,i)<<std::endl;
// //      }
// //    }
// //    { // f *= var^deg(pp, var); for the early vars
// //      for (long var = nvars/2; var < nvars; ++var)
// //        expv[var] = 0;
// //      PPMonoid::elem qq(PPM(P), expv);
// //      f *= P.monomial(RingElem(CoeffRing(P), 1), qq);
// //    }
// //  }
// //----------------------------------------------------------------------
// } // end of unnamed namespace


// namespace CoCoA
// {




//   RingWeylImpl::ring_weyl(const AbstractRing& R, const PPMonoid& PPM):
//     myCoeffRing(R),
//     myPPM(PPM),
//     myWeylPolyPool(sizeof(WeylPoly), "RingWeylImpl::myWeylPolyPool"),
//     myNumIndets(NumIndets(PPM)),
//     myOrdvWords(OrdvWords(ordering(PPM))),
//     mySummandSize(sizeof(WeylPoly::summand) + sizeof(int)*(myOrdvWords-1)),
//     mySummandPool(mySummandSize, "RingWeylImpl::mySummandPool")
//   {
//     myNumPolys = 0;
//     myZero = new RingElem(*this);
//     myIndetVector.resize(myNumIndets, RingElem(*this));
//     vector<long> expv(myNumIndets);
//     RingElem one(myCoeffRing, 1);
//     for (long i=0; i < myNumIndets; ++i)
//     {
//       expv[i] = 1;
//       PushFront(raw(myIndetVector[i]), raw(one), expv);
//       expv[i] = 0;
//     }
//   }


//   RingWeylImpl::~ring_weyl()
//   {
//     myIndetVector.clear();
//     delete myZero;
//     ASSERT(myNumPolys == 0);
//   }


//   void RingWeylImpl::MakeWritable(RawPtr rawx) const
//   {
//     if (x.WeylPolyPtr->myRefCount == 1) return;
//     --x.WeylPolyPtr->myRefCount;
//     WeylPoly* copy = static_cast<WeylPoly*>(myWeylPolyPool.alloc(sizeof(WeylPoly)));
//     const WeylPoly* orig = x.WeylPolyPtr;
//     x.WeylPolyPtr = copy;
//     copy->myRefCount = 1;
//     copy->mySummands = 0;
//     copy->myEnd = &copy->mySummands;
//     for (const WeylPoly::summand* it = orig->mySummands; it != 0; it = it->myNext)
//     {
//       // more or less hand inlined body of PushBack -- NB using PushBack is "circular"
//       *copy->myEnd = CopySummand(it);
//       copy->myEnd = &((*copy->myEnd)->myNext);
//     }

//     ///  myCoeffRing.init(copy->myDenom);
// //    new (&copy->myLC) RingElem(myCoeffRing, AbstractRing::elem::ALIAS);
//   }


//   inline WeylPoly::summand* RingWeylImpl::AllocSummand() const
//   {
//     return static_cast<WeylPoly::summand*>(mySummandPool.alloc(mySummandSize));
//   }


//   WeylPoly::summand* RingWeylImpl::InitSummand(WeylPoly::summand* ptr) const
//   {
//     myCoeffRing.init(ptr->myCoeff);
//     ptr->myNext = 0;
//     ptr->myModulePosn = 0;
//     return ptr;
//   }


//   WeylPoly::summand* RingWeylImpl::InitSummand(WeylPoly::summand* ptr, ConstRawPtr rawc, const vector<long>& expv) const
//   {
//     myCoeffRing.init(ptr->myCoeff, c);
//     ptr->myNext = 0;
//     ptr->myModulePosn = 0;
//     ordering(myPPM).ComputeOrdv(ptr->myOrdv, &expv[0]); // &expv[0] converts vector<T> to T*
//     return ptr;
//   }


//   WeylPoly::summand* RingWeylImpl::CopySummand(const WeylPoly::summand* original) const
//   {
//     WeylPoly::summand* copy = AllocSummand();
//     copy->myNext = 0;
//     myCoeffRing.init(copy->myCoeff, original->myCoeff);
//     copy->myModulePosn = original->myModulePosn;
//     for (long i=0; i < myOrdvWords; ++i)
//       copy->myOrdv[i] = original->myOrdv[i];
//     return copy;
//   }


//   void RingWeylImpl::SetSummandMOrdv(WeylPoly::summand* dest, const WeylPoly::summand* src) const
//   {
//     dest->myModulePosn = src->myModulePosn;
//     for (long i=0; i < myOrdvWords; ++i)
//       dest->myOrdv[i] = src->myOrdv[i];
//   }


//   void RingWeylImpl::DeleteSummands(WeylPoly::summand* ptr) const
//   {
//     WeylPoly::summand* next;
//     while (ptr != 0)
//     {
//       next = ptr->myNext;
//       myCoeffRing.kill(ptr->myCoeff);
//       mySummandPool.free(ptr, mySummandSize);
//       ptr = next;
//     }
//   }


//   bool RingWeylImpl::EqualSummands(const WeylPoly::summand& lhs, const WeylPoly::summand& x) const
//   {
//     if (lhs.myModulePosn != x.myModulePosn) return false;
//     const PPOrdering::OrdvElem* const lordv = lhs.myOrdv;
//     const PPOrdering::OrdvElem* const rordv = x.myOrdv;
//     for (long i = 0; i < myOrdvWords; ++i)
//       if (lordv[i] != rordv[i]) return false;
//     return myCoeffRing.IsEqual(lhs.myCoeff, x.myCoeff);
//   }


//   inline void RingWeylImpl::MulOrdv(PPOrdering::OrdvElem* ov, const PPOrdering::OrdvElem* ov1, const PPOrdering::OrdvElem* ov2) const
//   {
//     for (long i=0; i < myOrdvWords; ++i)
//       ov[i] = ov1[i]+ov2[i];
//   }


//   inline void RingWeylImpl::DivOrdv(PPOrdering::OrdvElem* ov, const PPOrdering::OrdvElem* ov1, const PPOrdering::OrdvElem* ov2) const
//   {
//     for (long i=0; i < myOrdvWords; ++i)
//       ov[i] = ov1[i]-ov2[i];
//   }




//   long RingWeylImpl::NumIndets() const
//   {
//     return myNumIndets;
//   }


//   const AbstractRing& RingWeylImpl::CoeffRing() const
//   {
//     return myCoeffRing;
//   }


//   const PPMonoid& RingWeylImpl::PPM() const
//   {
//     return myPPM;
//   }


//   long RingWeylImpl::NumTerms(ConstRawPtr rawx) const
//   {
//     long nsummands = 0;
//     for (const WeylPoly::summand* it = AsWeylPoly(rawx).mySummands; it != 0; it = it->myNext) ++nsummands;
//     return nsummands;
//   }


//   PolyIter RingWeylImpl::BeginIter(AbstractRing::RawPtr rawx) const
//   {
//     return PolyIter(*this, &AsWeylPoly(rawx).mySummands);
//   }


//   PolyIter RingWeylImpl::EndIter(AbstractRing::RawPtr rawx) const
//   {
//     return PolyIter(*this, AsWeylPoly(rawx).myEnd);
//   }


//   const RingElem& RingWeylImpl::indet(long var) const
//   {
//     ASSERT(var < myNumIndets);
//     return myIndetVector[var];
//   }


//   RingElem RingWeylImpl::IndetPower(long var, long n) const
//   {
//     ASSERT(0 <= var && var < myNumIndets);
//     return monomial(CoCoA::IndetPower(myPPM, var, n));
//   }


//   RingElem RingWeylImpl::monomial(const RingElem& c, const PPMonoid::alias& pp) const
//   {
//     vector<long> expv(myNumIndets);
//     myPPM.exponents(expv, raw(pp));
//     RingElem ans(*this);
//     PushFront(raw(ans), raw(c), expv);
//     return ans;
//   }


//   RingElem RingWeylImpl::monomial(const PPMonoid::alias& pp) const
//   {
//     RingElem c(myCoeffRing, 1);
//     vector<long> expv(myNumIndets);
//     myPPM.exponents(expv, raw(pp));
//     RingElem ans(*this);
//     PushFront(raw(ans), raw(c), expv);
//     return ans;
//   }


//   RingElem RingWeylImpl::monomial(const RingElem& c) const
//   {
//     vector<long> expv(myNumIndets);
//     RingElem ans(*this);
//     PushFront(raw(ans), raw(c), expv);
//     return ans;
//   }


//   long RingWeylImpl::deg(ConstRawPtr rawf) const
//   {
//     if (IsZero(f)) CoCoA_ERROR("RingWeylImpl::deg: cannot compute degree of zero polynomial");
//     if (GradingDim(myPPM) > 0)
//       return ordering(myPPM).deg(AsWeylPoly(f).mySummands->myOrdv); //// BUG????  "valid" only if grading dim == 1
//     //    return ???; the vector in Z^0 -- ring not graded!!!
//     return -1;//??????????  BUG BUG INCOMPLETE
//   }


//   int RingWeylImpl::deg(ConstRawPtr rawf, long var) const
//   {
//     if (IsZero(f)) CoCoA_ERROR("RingWeylImpl::deg: cannot compute degree of zero polynomial");
//     long d = 0;
//     for (WeylPoly::summand* it = AsWeylPoly(f).mySummands; it != 0; it = it->myNext)
//       d = max(d, ordering(myPPM).exponent(it->myOrdv, var));
//     return d;
//   }


// //    int RingWeylImpl::deg(ConstRawPtr rawf, const PPGrading& G) const
// //    {
// //      if (IsZero(f)) CoCoA_ERROR("RingWeylImpl::deg: cannot compute degree of zero polynomial");
// //      int d = 0;
// //      for (WeylPoly::summand* it = AsWeylPoly(f).mySummands; it != 0; it = it->myNext)
// //        d = max(d, deg(it, G)); // CANNOT JUST USE MAX HERE!!  MUST USE LEX MAX FOR VECTORS
// //      return d;
// //    }



//   const RingElem RingWeylImpl::LC(ConstRawPtr rawf) const
//   {
//     ASSERT(!IsZero(f));
// //    if (IsZero(f)) return ::zero(myCoeffRing);
//     return RingElem(myCoeffRing, AsWeylPoly(f).mySummands->myCoeff);
// //    AsWeylPoly(f).myLC.reseat(AsWeylPoly(f).mySummands->myCoeff);
// //    return AsWeylPoly(f).myLC;
//   }


//   AbstractRing::RawPtr RingWeylImpl::RawLC(RawPtr rawf) const
//   {
//     MakeWritable(f);//?????????
//     ASSERT(!IsZero(f));
//     return AsWeylPoly(f).mySummands->myCoeff;
//   }


//   const AbstractRing::RawPtr RingWeylImpl::RawLC(ConstRawPtr rawf) const
//   {
//     ASSERT(!IsZero(f));
//     return AsWeylPoly(f).mySummands->myCoeff;
//   }


//   RingElem RingWeylImpl::content(ConstRawPtr rawf) const
//   {
//     ASSERT(::IsTrueGCDDomain(myCoeffRing));
//     ASSERT(!IsZero(f));
//     RingElem cont(myCoeffRing);
//     for (WeylPoly::summand* it = AsWeylPoly(f).mySummands; it != 0; it = it->myNext)
//       myCoeffRing.gcd(raw(cont), raw(cont), it->myCoeff);  // be clever if cont == 1??
//     return cont;
//   }



//   void RingWeylImpl::MoveLMToFront(RawPtr rawf, RawPtr rawg) const
//   {
//     ASSERT(!IsZero(g));
//     ASSERT(f.WeylPolyPtr->myRefCount == 1);
//     //    ASSERT(g.WeylPolyPtr->myRefCount == 1);
//     //    MakeWritable(f);//????????
//     MakeWritable(g);//????????
//     WeylPoly& G = AsWeylPoly(g);
//     WeylPoly::summand* ltg = G.mySummands;
//     G.mySummands = G.mySummands->myNext;
//     if (G.mySummands == 0) G.myEnd = &(G.mySummands);
//     PushFront(f, ltg);
//   }


//   void RingWeylImpl::DeleteLM(RawPtr rawf) const
//   {
//     ASSERT(!IsZero(f));
//     //    ASSERT(f.WeylPolyPtr->myRefCount == 1);
//     MakeWritable(f);//????????
//     WeylPoly& F = AsWeylPoly(f);
//     WeylPoly::summand* old_ltf = F.mySummands;
//     F.mySummands = old_ltf->myNext;
//     if (F.mySummands == 0) F.myEnd = &F.mySummands;
//     old_ltf->myNext = 0;
//     DeleteSummands(old_ltf);
//   }


//   void RingWeylImpl::AddMul(RawPtr rawf, const WeylPoly::summand* s, ConstRawPtr rawg, bool SkipLM) const
//   {
// ////    std::clog << "AddMul: Doing funny product of the following two polys" << endl;
// ////    output(std::clog, rawg);
//     ASSERT(f.WeylPolyPtr->myRefCount == 1);
//     //    MakeWritable(f);

//     RingElem ppg(*this);
//     assign(raw(ppg), rawg);
//     vector<long> expv(myNumIndets);
//     ordering(myPPM).ComputeExpv(&expv[0], s->myOrdv);
// ////    std::clog << "expv: "; for (int i=0; i<myNumIndets;++i) std::clog << expv[i] << "  "; std::clog << endl;
//     for (long var = myNumIndets/2; var < myNumIndets; ++var)
//     {
//       const long n = expv[var];
//       if (n == 0) continue;
// ////      std::clog << "AddMul: doing D variable with index " << n << endl;
//       RingElem der(*this);
//       assign(raw(der), raw(ppg));
// //      ppg *= IndetPower(var, n);  CANNOT DO THIS --> INFINITE RECURSION
//       {
//         PlainMul(raw(ppg),raw(ppg),raw(IndetPower(var, n)));
//       }
// //      mul(raw(ppg), raw(ppg), raw(IndetPower(var, n)));
//       for (long i=1; i <= n; ++i)
//       {
//         deriv(raw(der), raw(der), var-myNumIndets/2);
// ////        std::clog << "der(" << i << ")="; output(std::clog, raw(der)); std::clog << endl;
// //        ppg += binomial(n, i)*der*IndetPower(var, n-i); // *IndetPower(h, 2*i); // for homog case
//         {
//           vector<long> expv2(myNumIndets);
//           expv2[var] = n-i;
//           RingElem shift(*this);
//           PushBack(raw(shift), raw(RingElem(myCoeffRing, binomial(n, i))), expv2);
//           RingElem tmp(*this);
//           PlainMul(raw(tmp), raw(shift), raw(der));
// ////          std::clog << "AddMul: adding "; output(std::clog, raw(tmp)); std::clog << endl;
//           add(raw(ppg),raw(ppg),raw(tmp));
//         }
//       }
//     }
//     { // f *= var^deg(pp, var); for the early vars
//       for (long var = myNumIndets/2; var < myNumIndets; ++var)
//         expv[var] = 0;
//       PPMonoid::elem qq(myPPM, expv);
//       RingElem c(myCoeffRing);
//       myCoeffRing.assign(raw(c), s->myCoeff);
//       PlainMul(raw(ppg), raw(ppg), raw(monomial(c, qq)));
// ///      f *= P.monomial(RingElem(CoeffRing(P), 1), qq);
//     }
// ////    std::clog << "AddMul: OUTPUT "; output(std::clog, raw(ppg)); std::clog << endl;
//     add(f, f, raw(ppg));
//   }


//   void RingWeylImpl::AddMul2(RawPtr rawf, ConstRawPtr rawh, ConstRawPtr rawg, bool SkipLM) const //???? delete me???
//   {                                                 //???
//     AddMul(f, (AsWeylPoly(h).mySummands), rawg, SkipLM);     //???
//   }                                                 //???

//   void RingWeylImpl::PlainMul(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const
//   {
//     if (NumTerms(x) > NumTerms(y)) { PlainMul(lhs, rawy, rawx); return; }

//     RingElem ans(*this);

//     for (const WeylPoly::summand* xterm = AsWeylPoly(x).mySummands; xterm; xterm = xterm->myNext)
//       PlainAddMul(raw(ans), xterm, rawy, false);//?????

//     swap(raw(ans), lhs); // really an assignment
//   }
//   void RingWeylImpl::PlainAddMul(RawPtr rawf, const WeylPoly::summand* s, ConstRawPtr rawg, bool SkipLM) const
//   {
// //    ASSERT(f.DMPIPtr->myRefCount == 1);
//     //    MakeWritable(f);
// //          std::clog << "\nF := "; output(std::clog,f);
// //          std::clog<<";\nG := "; output(std::clog,g);
// //          std::clog<<endl;
// //          std::clog<<"s = ";
// //          myCoeffRing.output(std::clog, s->myCoeff);
// //          std::clog<<"x^(";
// //          for(long i=0;i<myOrdvWords;++i)std::clog<<s->myOrdv[i]<<" ";
// //          std::clog<<")"<<endl;
//     const AbstractRing& R = myCoeffRing;
//     typedef WeylPoly::summand summand;
//     const summand* g_smnd = AsWeylPoly(g).mySummands;
//     if (SkipLM)    g_smnd = g_smnd->myNext;
//     summand** f_prev = &(AsWeylPoly(f).mySummands);
//     summand*  f_smnd = *f_prev;

//     summand* tmp_smnd = InitSummand(AllocSummand()); // just sets coeff = 0

//     int CMP=0;

//     //    bool qIsOne = (myPPM.cmp(q, tmpPP)==0);
//     bool qIsOne = false;

//     for (; f_smnd != 0 && g_smnd != 0; g_smnd = g_smnd->myNext)
//     {
//       if (qIsOne)
//         while (f_smnd != 0 && (CMP=ordering(myPPM).CmpOrdvs(f_smnd->myOrdv,g_smnd->myOrdv)) >0)
//           f_smnd = *(f_prev = &f_smnd->myNext);
//       else
//       {
//         MulOrdv(tmp_smnd->myOrdv, s->myOrdv, g_smnd->myOrdv);
//         while (f_smnd != 0 && (CMP=ordering(myPPM).CmpOrdvs(f_smnd->myOrdv,tmp_smnd->myOrdv)) >0)
//           f_smnd = *(f_prev = &f_smnd->myNext);
//       }
//       if (f_smnd == 0)
//       {
//         R.mul(tmp_smnd->myCoeff, s->myCoeff, g_smnd->myCoeff);
//         PushBack(f, tmp_smnd);
//         tmp_smnd = InitSummand(AllocSummand());
//         g_smnd = g_smnd->myNext;
//         break;
//       }
//       if (CMP == 0)
//       {
//         if (R.IsZeroAddMul(f_smnd->myCoeff, s->myCoeff, g_smnd->myCoeff))
//           RemoveSmnd(f, f_prev);  // f_prev = f_prev;
//         else
//           f_prev = &f_smnd->myNext;
//         f_smnd = *f_prev;
//       }
//       else // (CMP < 0)
//       {
//         R.mul(tmp_smnd->myCoeff, s->myCoeff, g_smnd->myCoeff);
//         InsertSmnd(f, tmp_smnd, f_prev);
//         tmp_smnd = InitSummand(AllocSummand());
//         f_prev = &(*f_prev)->myNext;
//         // f_smnd = f_smnd;
//       }
//     }
//     for (;g_smnd != 0; g_smnd = g_smnd->myNext)
//     {
//       MulOrdv(tmp_smnd->myOrdv, s->myOrdv, g_smnd->myOrdv);
//       R.mul(tmp_smnd->myCoeff, s->myCoeff, g_smnd->myCoeff);
//       PushBack(f, tmp_smnd);
//       tmp_smnd = InitSummand(AllocSummand());
//     }
//     DeleteSummands(tmp_smnd); // next ptr will be zero (set by InitSummand)
// //      std::clog << "AddMul: produced f=";output(std::clog,f);std::clog<<endl;
// //      std::clog << "------------------------------------------------------"<<endl;

//   }





//   void RingWeylImpl::ReductionStep(RawPtr rawf, ConstRawPtr rawg) const
//   {
//  //             std::clog << "\nRingWeylImpl::reduce" << endl;
// //              std::clog << "\nF := "; output(std::clog,f);
// //              std::clog<<";\nG := "; output(std::clog,g);
// //              std::clog<<";"<<endl;
//     ASSERT(&g!=&f);
//     const AbstractRing& R = myCoeffRing;
//     const WeylPoly::summand* g_smnd = AsWeylPoly(g).mySummands;
//     const WeylPoly::summand* f_smnd = AsWeylPoly(f).mySummands;
//     WeylPoly::summand* tmp_smnd = InitSummand(AllocSummand()); // just sets coeff = 0

//     DivOrdv(tmp_smnd->myOrdv, f_smnd->myOrdv, g_smnd->myOrdv);
//     R.div(tmp_smnd->myCoeff, f_smnd->myCoeff, g_smnd->myCoeff);
//     R.negate(tmp_smnd->myCoeff, tmp_smnd->myCoeff);

//  //             std::clog<<"--  S := ";
// //              myCoeffRing.output(std::clog, tmp_smnd->myCoeff);
// //              std::clog<<"x^(";
// //              for(long i=0;i<myOrdvWords;++i)std::clog<<tmp_smnd->myOrdv[i]<<" ";
// //              std::clog<<")"<<endl;

//     DeleteLM(f);
//     AddMul(f, tmp_smnd, g, true /*SkipLM*/);

//     DeleteSummands(tmp_smnd); // next ptr will be zero (set by InitSummand)
//  //             std::clog << "H := "; output(std::clog,f);
// //              std::clog << ";\n--------------------------------------------------"<<endl;
//  }


//   void RingWeylImpl::AddClear(RawPtr rawf, RawPtr rawg) const
//   {
//     ASSERT(f.WeylPolyPtr->myRefCount == 1);
//     //    MakeWritable(f);
//     MakeWritable(g);
//     // input polynomial are copied
//     //if (g.WeylPolyPtr->myRefCount != 1)
//     //      std::clog << "AddClear: g.myRefCount == " << g.WeylPolyPtr->myRefCount << endl;
//     const AbstractRing& R = myCoeffRing;
//     typedef WeylPoly::summand summand;
//     WeylPoly& F = AsWeylPoly(f);
//     WeylPoly& G = AsWeylPoly(g);
//     summand*  g_smnd = G.mySummands;
//     summand** f_prev = &(F.mySummands);
//     summand*  f_smnd = *f_prev;
//     int CMP=0;
//     ASSERT(*(G.myEnd)==0);//BUG HUNTING  ???

//     //    std::clog << "input f = "; output(std::clog, f) ;std::clog << endl;
//     while ( f_smnd!=0 && g_smnd!=0 )
//     {
//       while (f_smnd!=0 &&
//              (CMP=ordering(myPPM).CmpOrdvs(f_smnd->myOrdv,g_smnd->myOrdv)) >0)
//         f_smnd = *(f_prev = &f_smnd->myNext);
//       if (f_smnd == 0)  break;
//       //std::clog <<   "(AddClear error: should never happen for Basic Reduction)" << endl;
//       G.mySummands = G.mySummands->myNext;
//       g_smnd->myNext = 0;
//       if (CMP == 0)
//       {
//         R.add(f_smnd->myCoeff, f_smnd->myCoeff, g_smnd->myCoeff);
//         if (R.IsZero(f_smnd->myCoeff))
//           RemoveSmnd(f, f_prev);
//         DeleteSummands(g_smnd);
//       }
//       else // (CMP < 0)
//       {
//         InsertSmnd(f, g_smnd, f_prev);
//         f_prev = &(*f_prev)->myNext;
//       }
//       f_smnd = *f_prev;
//       g_smnd = G.mySummands;
//     }
//     if (G.mySummands!=0)
//     {
//       *(F.myEnd) = G.mySummands;
//       F.myEnd = G.myEnd;
//       G.mySummands = 0;
//     }
//     G.myEnd = &G.mySummands;
//     //    if (rare) {std::clog << "f2 = "; output(std::clog, f) ;std::clog << endl;}
//   }


//   void RingWeylImpl::AppendClear(RawPtr rawf, RawPtr rawg) const
//   {
//     ASSERT(f.WeylPolyPtr->myRefCount == 1);
//     //    MakeWritable(f);
//     MakeWritable(g);
//     // input polynomial are copied
//     //if (g.WeylPolyPtr->myRefCount != 1)
//     //      std::clog << "AppendClear: g.myRefCount == " << g.WeylPolyPtr->myRefCount << endl;
//     WeylPoly& F = AsWeylPoly(f);
//     WeylPoly& G = AsWeylPoly(g);
//     if (G.mySummands!=0)
//     {
//       *(F.myEnd) = G.mySummands;
//       F.myEnd = G.myEnd;
//       G.mySummands = 0;
//     }
//     G.myEnd = &G.mySummands;
//     //    if (rare) {std::clog << "f2 = "; output(std::clog, f) ;std::clog << endl;}
//   }


//   int  RingWeylImpl::CmpLPP(ConstRawPtr rawf, ConstRawPtr rawg) const
//   {
//     ASSERT(!IsZero(f));
//     ASSERT(!IsZero(g));
//     return ordering(myPPM).CmpOrdvs(AsWeylPoly(f).mySummands->myOrdv,AsWeylPoly(g).mySummands->myOrdv);
//   }


//   void RingWeylImpl::DivLM(RawPtr rawlhs, ConstRawPtr rawf, ConstRawPtr rawg) const
//   {
//     //    std::clog << "DivLM" << endl;
//     const AbstractRing& R = myCoeffRing;
//     typedef WeylPoly::summand summand;
//     const summand* f_smnd = AsWeylPoly(f).mySummands;
//     const summand* g_smnd = AsWeylPoly(g).mySummands;
//     MakeWritable(lhs);
//     assign(lhs,0);
//     summand* SpareSummand = InitSummand(AllocSummand());
//     R.div(SpareSummand->myCoeff, f_smnd->myCoeff, g_smnd->myCoeff);
//     DivOrdv(SpareSummand->myOrdv, f_smnd->myOrdv, g_smnd->myOrdv);
//     PushBack(lhs, SpareSummand);
//   }


//   void RingWeylImpl::mul(RawPtr rawf, const PPMonoid::elem& pp) const
//   {
//     //    bool qIsOne = (myPPM.cmp(q, tmpPP)==0);
//     MakeWritable(f);
//     std::vector<long> v(myNumIndets);
//     myPPM.exponents(v, raw(pp));

//     WeylPoly::summand* f_smnd = AsWeylPoly(f).mySummands;
//     WeylPoly::summand* s = InitSummand(AllocSummand());

//     ordering(myPPM).ComputeOrdv(s->myOrdv, &v[0]); // &v[0] converts vector<T> to T*

//     for (; f_smnd != 0 ; f_smnd = f_smnd->myNext)
//       MulOrdv(f_smnd->myOrdv, f_smnd->myOrdv, s->myOrdv);
//   }


//   void RingWeylImpl::PushFront(RawPtr rawf, ConstRawPtr rawc, const std::vector<long>& expv) const
//   {
//     if (CoeffRing().IsZero(c)) return;
//     MakeWritable(f);
//     WeylPoly::summand* t = AllocSummand();
//     InitSummand(t, c, expv);
//     t->myNext = f.WeylPolyPtr->mySummands;
//     if (f.WeylPolyPtr->mySummands == 0) f.WeylPolyPtr->myEnd = &t->myNext;
//     f.WeylPolyPtr->mySummands = t;
//   }


//   void RingWeylImpl::PushBack(RawPtr rawf, ConstRawPtr rawc, const std::vector<long>& expv) const
//   {
//     if (CoeffRing().IsZero(c)) return;
//     MakeWritable(f);
//     WeylPoly::summand* t = AllocSummand();
//     InitSummand(t, c, expv);
//     *(f.WeylPolyPtr->myEnd) = t;
//     f.WeylPolyPtr->myEnd = &t->myNext;
//   }


//   void RingWeylImpl::PushFront(RawPtr rawx, WeylPoly::summand* t) const
//   {
//     MakeWritable(x);
//     WeylPoly& f = AsWeylPoly(x);
//     t->myNext = f.mySummands;
//     f.mySummands = t;
//     if (f.myEnd == &f.mySummands) f.myEnd = &t->myNext;
//   }


//   void RingWeylImpl::PushBack(RawPtr rawx, WeylPoly::summand* t) const
//   {
//     MakeWritable(x);
//     WeylPoly& f = AsWeylPoly(x);
//     *f.myEnd = t;
//     f.myEnd = &t->myNext;
//   }


//   void RingWeylImpl::RemoveSmnd(RawPtr rawx, WeylPoly::summand** prev_link) const
//   {
//     ASSERT(x.WeylPolyPtr->myRefCount == 1);
//     //    MakeWritable(x);
//     WeylPoly::summand* tmp = *prev_link;
//     ASSERT(tmp != 0);
//     if (tmp->myNext==0) // f.myEnd == &(tmp->myNext)
//     {
//       WeylPoly& f = AsWeylPoly(x);
//       f.myEnd = prev_link;
//     }
//     *prev_link = tmp->myNext;
//     tmp->myNext = 0;
//     DeleteSummands(tmp);
//   }


//   void RingWeylImpl::InsertSmnd(RawPtr rawx, WeylPoly::summand* s, WeylPoly::summand** prev_link) const
//   {
//     ASSERT(x.WeylPolyPtr->myRefCount == 1);
//     //    MakeWritable(x);
//     WeylPoly& f = AsWeylPoly(x);
//     s->myNext = (*prev_link);
//     (*prev_link) = s;
//     if (f.myEnd == prev_link) f.myEnd = &(s->myNext);
//   }


//   PPMonoid::elem RingWeylImpl::LPP(ConstRawPtr rawf) const
//   {
//     ASSERT(!IsZero(f));
//     return PPMonoid::elem(PPM(), PPMonoid::elem::FromOrdv, AsWeylPoly(f).mySummands->myOrdv);
//   }


//   bool RingWeylImpl::IsZeroAddLCs(RawPtr rawf, RawPtr rawg) const
//   {
//     ASSERT(!IsZero(f) && !IsZero(g));
//     ASSERT( CmpLPP(f,g) == 0);
//     WeylPoly& F = AsWeylPoly(f);
//     WeylPoly& G = AsWeylPoly(g);
//     ASSERT(F.myRefCount==1 && G.myRefCount==1);
//     myCoeffRing.add(F.mySummands->myCoeff, F.mySummands->myCoeff, G.mySummands->myCoeff);
//     DeleteLM(g);
//     if (!myCoeffRing.IsZero(F.mySummands->myCoeff)) return false;
//     DeleteLM(f);
//     return true;
//   }


//   void RingWeylImpl::deriv(RawPtr rawdest, ConstRawPtr rawf, long var) const
//   {
//     const WeylPoly& F = AsWeylPoly(f);
//     RingElem ans(*this);
//     vector<long> expv(myNumIndets);
//     for (WeylPoly::summand* i=F.mySummands; i; i = i->myNext)
//     {
//       ordering(myPPM).ComputeExpv(&expv[0], i->myOrdv);
//       if (expv[var] == 0) continue;
//       RingElem c(myCoeffRing, expv[var]);
//       if (CoCoA::IsZero(c)) continue;
//       myCoeffRing.mul(raw(c), raw(c), i->myCoeff);
//       if (CoCoA::IsZero(c)) continue;
//       --expv[var];
//       PushBack(raw(ans), raw(c), expv);
//     }
//     swap(raw(ans), dest);
//   }


//   void RingWeylImpl::negate(RawPtr rawlhs, ConstRawPtr rawx) const
//   {
//     if (lhs.WeylPolyPtr == x.WeylPolyPtr)
//     {
//       MakeWritable(lhs);
//       typedef WeylPoly::summand summand;
//       //      std::clog << "-- negate"; output(std::clog, lhs); std::clog << endl;
//       for (summand* smnd = AsWeylPoly(lhs).mySummands; smnd!=0; smnd=smnd->myNext )
//         myCoeffRing.negate(smnd->myCoeff, smnd->myCoeff);
//       //      std::clog << "-> negate"; output(std::clog, lhs); std::clog << endl;
//     }
//     else
//       std::clog << "RingWeylImpl::negate: not yet implemented" << endl;
//   }


//   void RingWeylImpl::add(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const
//   {
//     const AbstractRing& R = myCoeffRing;
//     RawValue ans;
//     //    ans.WeylPolyPtr = static_cast<WeylPoly*>(myWeylPolyPool.alloc(sizeof(WeylPoly)));
//     init(ans);
//     ///    WeylPoly& sum = AsWeylPoly(ans);
//     typedef WeylPoly::summand summand;
//     const summand* gterm = AsWeylPoly(x).mySummands;
//     const summand* hterm = AsWeylPoly(y).mySummands;
//     summand* SpareSummand = InitSummand(AllocSummand()); // just sets coeff = 0
//     while (gterm != 0 && hterm != 0)
//     {
//       const int cmp = ordering(myPPM).CmpOrdvs(gterm->myOrdv, hterm->myOrdv);

//       if (cmp < 0)
//       {
// 	summand* hcopy = CopySummand(hterm);
// 	PushBack(ans, hcopy);
// 	hterm = hterm->myNext;
// 	continue;
//       }

//       if (cmp > 0)
//       {
// 	summand* gcopy = CopySummand(gterm);
// 	PushBack(ans, gcopy);
// 	gterm = gterm->myNext;
// 	continue;
//       }

//       // Must have cmp == 0 here.
//       // The leading PPs are the same, so we must sum the coeffs.
//       R.add(SpareSummand->myCoeff, gterm->myCoeff, hterm->myCoeff);
//       if (!R.IsZero(SpareSummand->myCoeff))
//       {
// 	SetSummandMOrdv(SpareSummand, gterm); // set module posn and PP
// 	PushBack(ans, SpareSummand);
// 	SpareSummand = InitSummand(AllocSummand());// just sets coeff = 0
//       }
//       gterm = gterm->myNext;
//       hterm = hterm->myNext;
//     }
//     while (gterm != 0)
//     {
//       summand* gcopy = CopySummand(gterm);
//       PushBack(ans, gcopy);
//       gterm = gterm->myNext;
//     }
//     while (hterm != 0)
//     {
//       summand* hcopy = CopySummand(hterm);
//       PushBack(ans, hcopy);
//       hterm = hterm->myNext;
//     }
//     DeleteSummands(SpareSummand); // next ptr will be zero (set by InitSummand)
//     swap(lhs, ans); // really an assignment
//     kill(ans);
//   }


//   void RingWeylImpl::sub(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const
//   {
//     // This code copied from RingWeylImpl::add...

//     const AbstractRing& R = myCoeffRing;
//     RawValue ans;
//     //    ans.WeylPolyPtr = static_cast<WeylPoly*>(myWeylPolyPool.alloc(sizeof(WeylPoly)));
//     init(ans);
//     ///    WeylPoly& sum = AsWeylPoly(ans);
//     typedef WeylPoly::summand summand;
//     const summand* gterm = AsWeylPoly(x).mySummands;
//     const summand* hterm = AsWeylPoly(y).mySummands;
//     summand* SpareSummand = InitSummand(AllocSummand()); // just sets coeff = 0
//     while (gterm != 0 && hterm != 0)
//     {
//       const int ord = ordering(myPPM).CmpOrdvs(gterm->myOrdv, hterm->myOrdv);

//       if (ord < 0)
//       {
// 	summand* hcopy = CopySummand(hterm);
// 	R.negate(hcopy->myCoeff, hcopy->myCoeff);
// 	//	sum.push_front(hcopy);
// 	PushBack(ans, hcopy);
// 	hterm = hterm->myNext;
// 	continue;
//       }

//       if (ord > 0)
//       {
// 	summand* gcopy = CopySummand(gterm);
// 	//	sum.push_front(gcopy);
// 	PushBack(ans, gcopy);
// 	gterm = gterm->myNext;
// 	continue;
//       }

//       // The leading PPs are the same, so we must sum the coeffs.
//       R.sub(SpareSummand->myCoeff, gterm->myCoeff, hterm->myCoeff);
//       if (!R.IsZero(SpareSummand->myCoeff))
//       {
// 	SetSummandMOrdv(SpareSummand, gterm); // set module posn and PP
// 	//	sum.push_front(SpareSummand);
// 	PushBack(ans, SpareSummand);
// 	SpareSummand = InitSummand(AllocSummand());// just sets coeff = 0
//       }
//       gterm = gterm->myNext;
//       hterm = hterm->myNext;
//     }
//     while (gterm != 0)
//     {
//       summand* gcopy = CopySummand(gterm);
//       //      sum.push_front(gcopy);
//       PushBack(ans, gcopy);
//       gterm = gterm->myNext;
//     }
//     while (hterm != 0)
//     {
//       summand* hcopy = CopySummand(hterm);
//       //      sum.push_front(hcopy);
//       R.negate(hcopy->myCoeff, hcopy->myCoeff);
//       PushBack(ans, hcopy);
//       hterm = hterm->myNext;
//     }
//     DeleteSummands(SpareSummand); // next ptr will be zero (set by InitSummand)
//     /// NO LONGER NEEDED    sum.reverse();
//     swap(lhs, ans); // really an assignment
//     kill(ans);
//   }


//   void RingWeylImpl::mul(RawPtr rawlhs, ConstRawPtr rawx, ConstRawPtr rawy) const
//   {
// // NO!! NOT COMMUTATIVE!!    if (NumTerms(x) > NumTerms(y)) { mul(lhs, y, x); return; }

// ////    std::clog << "MUL on "; output(std::clog, x); std::clog << " and "; output(std::clog, y); std::clog << endl;
//     RingElem ans(*this);

//     for (const WeylPoly::summand* xterm = AsWeylPoly(x).mySummands; xterm; xterm = xterm->myNext)
//       AddMul(raw(ans), xterm, y, false);//?????

//     swap(raw(ans), lhs); // really an assignment
//   }





// RCS header/log in the next few lines
// $Header: /Volumes/Home_1/cocoa/cvs-repository/CoCoALib-0.99/src/AlgebraicCore/RingWeyl.C,v 1.60 2018/05/25 09:24:46 abbott Exp $
// $Log: RingWeyl.C,v $
// Revision 1.60  2018/05/25 09:24:46  abbott
// Summary: Major redesign of CpuTimeLimit (many consequences)
//
// Revision 1.59  2018/05/18 12:22:30  bigatti
// -- renamed IntOperations --> BigIntOps
//
// Revision 1.58  2018/05/17 15:59:37  bigatti
// -- renamed MatrixOperations --> MatrixOps
// -- sorted #includes
//
// Revision 1.57  2018/04/18 14:32:03  abbott
// Summary: Added missing return stmts in some NYI fns
//
// Revision 1.56  2018/03/29 09:36:40  bigatti
// -- added member functions myTestIsRadical, myTestIsPrimary and flags
//
// Revision 1.55  2018/03/20 11:38:08  bigatti
// -- changed iAm***Test --> myTestIs***;  and it returns bool
//
// Revision 1.54  2017/04/18 09:55:03  bigatti
// -- removed StatLevel from GBasis
//
// Revision 1.53  2016/11/07 12:22:21  bigatti
// -- changed myGBasisIsValid into IhaveGBasis
//
// Revision 1.52  2015/12/08 14:06:22  abbott
// Summary: Added include for UIBC (otherwise does not compile)
//
// Revision 1.51  2015/12/01 13:34:44  abbott
// Summary: Changed arg order in ElimMat and HomogElimMat; doc is out-of-date!!
//
// Revision 1.50  2015/11/30 21:53:55  abbott
// Summary: Major update to matrices for orderings (not yet complete, some tests fail)
//
// Revision 1.49  2015/07/23 11:58:03  bigatti
// -- removed obsolete doxygen header
//
// Revision 1.48  2015/04/24 15:40:58  bigatti
// -- renamed: myAddMul --> myAddMulLM
// -- renamed: myMoveLM --> myMoveLMToFront
// -- new myMoveLMToBack (used in ReductionCog --> bug in test-TmpMorseGraph??)
//
// Revision 1.47  2014/07/28 15:51:31  abbott
// Summary: Redesign: ringhoms no longer cached in rings (caused ref count trouble)
// Author: JAA
//
// Revision 1.46  2014/07/11 16:00:39  bigatti
// -- commented out myOutputSelf(OpenMathOutput& OMOut)
//
// Revision 1.45  2014/07/11 15:49:25  bigatti
// -- commented out myOutputSelf (default impl) and added myImplDetails
//   (is thins correct for Weyl?)
//
// Revision 1.44  2014/07/09 13:03:25  abbott
// Summary: Removed AsSparsePolyRing from commented out code
// Author: JAA
//
// Revision 1.43  2014/05/14 15:57:15  bigatti
// -- added "using" for clang with superpedantic flag
//
// Revision 1.42  2014/05/06 13:20:41  abbott
// Summary: Changed names (my)MaxExponents into (my)Deg
// Author: JAA
//
// Revision 1.41  2014/04/11 15:44:27  abbott
// Summary: Renamed MatrixArith to MatrixOperations (in includes)
// Author: JAA
//
// Revision 1.40  2014/04/02 10:57:46  abbott
// Summary: Revised design of IamIntegralDomain3
// Author: JAA
//
// Revision 1.39  2013/06/28 17:03:51  abbott
// Modified semantics of IdealBase::myDivMod;
// it now returns a boolean.
// Several consequential changes.
//
// Revision 1.38  2013/02/14 17:34:35  bigatti
// -- cleaned up code for elimination matrices
//
// Revision 1.37  2012/10/24 12:20:43  abbott
// Changed return type of myLC.
//
// Revision 1.36  2012/10/17 09:40:16  abbott
// Replaced  RefRingElem  by  RingElem&
// (plus a few consequential changes)
//
// Revision 1.35  2012/10/11 14:29:41  abbott
// Rewrote  myMulByPP  and  myReductionStep  so that they no longer use RefRingElem.
//
// Revision 1.34  2012/05/28 09:18:20  abbott
// Created IntOperations which gathers together all operations on
// integers (both big and small).  Many consequential changes.
//
// Revision 1.33  2012/05/24 14:49:23  bigatti
// -- changed symbol "index" into "subscripts"
//
// Revision 1.32  2012/05/22 10:02:37  abbott
// Removed IsGCDDomain; substituted by IsTrueGCDDomain.
// Added IsFractionFieldOfGCDDomain.
//
// Revision 1.31  2012/01/26 16:47:00  bigatti
// -- changed back_inserter into insert
//
// Revision 1.30  2011/11/09 14:29:37  bigatti
// -- renamed MachineInteger --> MachineInt
//
// Revision 1.29  2011/08/14 15:52:16  abbott
// Changed ZZ into BigInt (phase 1: just the library sources).
//
// Revision 1.28  2011/06/23 16:04:47  abbott
// Added IamExact mem fn for rings.
// Added myRecvTwinFloat mem fn for rings.
// Added first imple of RingHom from RingTwinFloat to other rings.
//
// Revision 1.27  2011/03/10 16:39:33  abbott
// Replaced (very many) size_t by long in function interfaces (for rings,
// PPMonoids and modules).  Also replaced most size_t inside fn defns.
//
// Revision 1.26  2011/03/08 17:58:29  bigatti
// -- changed: ElimIndets is now of  long  instead of  size_t
//
// Revision 1.25  2010/10/08 11:39:53  abbott
// Renamed DistrMPoly to DistrMPolyClean.
//
// Revision 1.24  2010/10/06 14:10:24  abbott
// Added increments to the ref count in ring and PPMonoid ctors to make
// them exception safe.
//
// Revision 1.23  2010/07/27 07:37:13  bigatti
// -- new class GBCriteria, simplified GReductor ctor
//
// Revision 1.22  2010/06/10 08:00:02  bigatti
// -- fixed naming conventions
//
// Revision 1.21  2010/05/14 09:53:09  bigatti
// -- removed empty ctor for SugarDegree
// -- added marker for SugarDegree(uninitialized)
// -- SugarDegree for GBasis input is initialized by myPrepareGBasis
//
// Revision 1.20  2009/10/02 13:47:07  bigatti
// -- myDivByCoeff now returns bool
// -- unique implementation of myDiv in PolyRing.C
//
// Revision 1.19  2009/09/28 08:39:01  bigatti
// -- fixed (debugging) parameter by M.Caboara
//
// Revision 1.18  2009/09/25 13:02:43  bigatti
// -- just a comment
//
// Revision 1.17  2009/09/22 13:35:55  bigatti
// -- following coding conventions in function names Matrix --> Mat
// -- forced all matrices to be over RingZ
//
// Revision 1.16  2009/01/30 13:41:50  bigatti
// -- enum instead of bool arguments
//
// Revision 1.15  2008/12/17 12:11:52  abbott
// Changed type from long to MachineInt in operations which use a machine integer
// in place of a RingElem.  The change is "superficial" but affects many files.
//
// Revision 1.14  2008/11/18 16:34:46  bigatti
// -- removed debugging printout
//
// Revision 1.13  2008/11/18 15:20:10  bigatti
// -- added const to myGBasis return value
// -- added myIdealCtor to RingWeyl for proper inheritance
//
// Revision 1.12  2008/09/19 13:33:42  bigatti
// -- added: Sat algorithm (M.Caboara)
//
// Revision 1.11  2008/04/21 11:23:11  abbott
// Separated functions dealing with matrices and PPOrderings into a new file.
// Added matrix norms, and completed adjoint.
//
// Revision 1.10  2008/04/10 15:13:21  bigatti
// -- added myPushBack/Front(RawPtr, ConstRawPtr, PPMonoidElemConstRawPtr)
//
// Revision 1.9  2007/12/05 11:06:24  bigatti
// -- changed "size_t StdDeg/myStdDeg(f)" into "long"  (and related functions)
// -- changed "log/myLog(f, i)" into "MaxExponent/myMaxExponent(f, i)"
// -- fixed bug in "IsOne(ideal)" in SparsePolyRing.C
//
// Revision 1.8  2007/12/04 14:27:06  bigatti
// -- changed "log(pp, i)" into "exponent(pp, i)"
//
// Revision 1.7  2007/10/30 17:14:07  abbott
// Changed licence from GPL-2 only to GPL-3 or later.
// New version for such an important change.
//
// Revision 1.6  2007/05/31 16:01:45  bigatti
// -- default implementation for IamField, myCharacteristic  in PolyRing
// -- added !IsCommutative in test
//
// Revision 1.4  2007/05/22 22:45:14  abbott
// Changed fn name IsUnit to IsInvertible.
//
// Revision 1.3  2007/05/21 12:45:13  abbott
// Modified a pointless comment.
//
// Revision 1.2  2007/03/09 18:56:56  bigatti
// -- added Tmp prefix to Groebner related files
//
// Revision 1.1.1.1  2007/03/09 15:16:11  abbott
// Imported files
//
// Revision 1.26  2007/03/08 18:22:29  cocoa
// Just whitespace cleaning.
//
// Revision 1.25  2007/03/08 17:43:10  cocoa
// Swapped order of args to the NewPPMonoid pseudo ctors.
//
// Revision 1.24  2007/03/08 16:55:06  cocoa
// Changed name of "range" function to "SymbolRange".
//
// Revision 1.23  2007/03/08 11:07:12  cocoa
// Made pseudo ctors for polynomial rings more uniform.  This allowed me to
// remove an include of CoCoA/symbol.H  from the RingDistrM*.H files, but then
// I had to put the include in several .C files.
//
// Revision 1.22  2007/03/05 21:06:07  cocoa
// New names for homomorphism pseudo-ctors: removed the "New" prefix.
//
// Revision 1.21  2007/02/10 18:44:03  cocoa
// Added "const" twice to each test and example.
// Eliminated dependency on io.H in several files.
// Improved BuildInfo, and added an example about how to use it.
// Some other minor cleaning.
//
// Revision 1.20  2007/01/20 14:07:25  bigatti
// -- moved code for homomorphism into common implementation in SparsePolyRing
//
// Revision 1.19  2007/01/15 16:15:26  cocoa
// -- added prefix "raw" to RawPtr arguments names
// -- changed rhs into rawx, n, or N
//
// Revision 1.18  2007/01/13 14:14:34  cocoa
// Overhaul of RingHom code: it nows uses SmartPtrIRC, and printing is more logical.
// Have not yet updated the documentation.
//
// Revision 1.17  2006/12/21 13:48:32  cocoa
// Made all increment/decrement calls prefix (except where the must be postfix).
//
// Revision 1.16  2006/12/07 17:23:46  cocoa
// -- for compilation with _Wextra: commented out names of unused arguments
//
// Revision 1.15  2006/12/07 12:17:15  cocoa
// -- style: RawPtr args are now called "raw.."
//
// Revision 1.14  2006/11/24 17:06:10  cocoa
// -- reorganized includes of header files
//
// Revision 1.13  2006/11/21 18:09:23  cocoa
// -- added myIsMonomial
// -- implemented myIsOne, myIsMinusOne, myIsConstant, myIsIndet in SparsePolyRing
// -- removed the 4 functions from DistrMPoly(..) and RingDistrMPoly(..)
// -- changed all names of RawPtr arguments into "raw(..)"
//
// Revision 1.12  2006/11/17 12:04:29  cocoa
// -- added (obvious) constructor for RingWeylImpl::IdealImpl
//
// Revision 1.11  2006/11/14 17:36:49  cocoa
// -- fixed implementation for ideal in RingWeyl
//
// Revision 1.10  2006/11/09 17:46:58  cocoa
// -- version 0.9712:
// --   IdealImpl moved to SparsePolyRing from concrete rings
// -- PolyList in GTypes is now vector<RingElem> (was list)
// -- "my" coding convention applied to DistrMPoly
//
// Revision 1.9  2006/11/08 16:21:59  cocoa
// Structural cleaning of RingHom; many consequential changes.
//
// Revision 1.8  2006/11/02 13:25:43  cocoa
// Simplification of header files: the OpenMath classes have been renamed.
// Many minor consequential changes.
//
// Revision 1.7  2006/10/16 23:18:59  cocoa
// Corrected use of std::swap and various special swap functions.
// Improved myApply memfn for homs of RingDistrMPolyInlPP.
//
// Revision 1.6  2006/10/06 14:04:14  cocoa
// Corrected position of #ifndef in header files.
// Separated CoCoA_ASSERT into assert.H from config.H;
// many minor consequential changes (have to #include assert.H).
// A little tidying of #include directives (esp. in Max's code).
//
// Revision 1.5  2006/08/17 09:41:33  cocoa
// -- changed:  I think it works again
//
// Revision 1.4  2006/08/07 21:23:25  cocoa
// Removed almost all publicly visible references to SmallExponent_t;
// changed to long in all PPMonoid functions and SparsePolyRing functions.
// DivMask remains to sorted out.
//
// Revision 1.3  2006/07/20 14:11:27  cocoa
// -- more stable version (more similar to RingDistrMPoly)
//
// Revision 1.1.1.1  2006/05/30 11:39:37  cocoa
// Imported files
//
// Revision 1.6  2006/04/27 13:45:30  cocoa
// Changed name of NewIdentityRingHom to NewIdentityHom.
// Changed name of member functions which print out their own object
// into myOutputSelf (to distinguish from "transitive" myOutput fns).
//
// Revision 1.5  2006/03/21 09:43:13  cocoa
// Changed names of some member fns of ideals (dealing with setting and testing
// the flags for primeness and maximality).  Hope icc will complain less now.
//
// Revision 1.4  2006/03/14 15:01:49  cocoa
// Improved the implementation of ring member fns for computing powers.
// Should keep Intel C++ compiler quieter too.
//
// Revision 1.3  2006/03/12 21:28:33  cocoa
// Major check in after many changes
//
// Revision 1.2  2006/02/20 22:41:19  cocoa
// All forms of the log function for power products now return SmallExponent_t
// (instead of int).  exponents now resizes the vector rather than requiring
// the user to pass in the correct size.
//
// Revision 1.1.1.1  2005/10/17 10:46:54  cocoa
// Imported files
//
// Revision 1.1.1.1  2005/05/03 15:47:31  cocoa
// Imported files
//
// Revision 1.3  2005/04/20 15:40:48  cocoa
// Major change: modified the standard way errors are to be signalled
// (now via a macro which records filename and line number).  Updated
// documentation in error.txt accordingly.
//
// Improved the documentation in matrix.txt (still more work to be done).
//
// Revision 1.2  2005/04/19 14:06:03  cocoa
// Added GPL and GFDL licence stuff.
//
// Revision 1.1.1.1  2005/01/27 15:12:13  cocoa
// Imported files
//
// Revision 1.16  2004/11/18 18:33:41  cocoa
// Now every ring know its own "one" element (as well as "zero").
// Several consequential changes.
//
// Revision 1.15  2004/11/11 14:28:49  cocoa
// -- minor changes for doxygen
// -- change: cout --> std::clog
//
// Revision 1.14  2004/11/09 16:30:51  cocoa
// Removed references to cout.
//
// Revision 1.13  2004/11/04 18:47:43  cocoa
// (1) Ring member functions which previously expected mpz_t args
//     now expect ZZ args.  Numerous minor consequential changes.
// (2) Renamed function which gives access to the mpz_t value inside
//     a ZZ object: previously was raw(...), now is mpzref(...).
//     Plenty of calls had to be altered.
//
// Revision 1.12  2004/10/21 17:16:37  cocoa
// Fairly major change: new OrdvArith namspace with various members,
//   new global typedef  SmallExponent_t (defined in config.H).
//
// Revision 1.11  2004/07/27 16:03:39  cocoa
// Added IsCommutative test and IamCommutative member function
// to all rings.  Tidied geobuckets a little.
//
// Revision 1.10  2004/07/20 15:37:08  cocoa
// Minor fix for some errors which slipped through the net...
//
// Revision 1.9  2004/07/20 15:04:06  cocoa
// The next step in the new "ring element" conversion process:
// handling the case of creating a "const RefRingElem" object
// (since C++ refuses to do this properly itself).
//
// Revision 1.8  2004/07/20 09:21:33  cocoa
// -- fewer Stats are printed
//
// Revision 1.7  2004/06/17 12:44:58  cocoa
// -- applied new coding conventions: compiles and seems to work
//
// Revision 1.6  2004/05/27 16:14:02  cocoa
// Minor revision for new coding conventions.
//
// Revision 1.5  2004/05/24 15:52:13  cocoa
// Major update:
//   new error mechanism
//   many fixes
//   RingHoms almost work now
//   RingFloat much improved
//
// Revision 1.4  2003/10/17 10:51:06  cocoa
// Major cleaning, and new naming convention.
//
// Revision 1.3  2003/10/09 13:32:16  cocoa
// A few glitches which slipped through the first major merge.
//
// Revision 1.2  2003/10/09 12:16:38  cocoa
// New coding convention for rings.
//
// Revision 1.3  2003/06/23 16:58:43  abbott
// Minor cleaning prior to public release.
// Just consequential changes.
//
// Revision 1.2  2003/05/30 15:24:01  abbott
// Changed beyond all recognition -- completely new version.
//
// Revision 1.1  2003/05/02 13:08:03  abbott
// Initial revision
//
//
