//   Copyright (c)  2016  John Abbott

//   This file is part of the source of CoCoALib, the CoCoA Library.

//   CoCoALib is free software: you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.

//   CoCoALib is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.

//   You should have received a copy of the GNU General Public License
//   along with CoCoALib.  If not, see <http://www.gnu.org/licenses/>.


#include "CoCoA/GCDFreeBasis.H"
#include "CoCoA/error.H"
#include "CoCoA/ring.H"
#include "CoCoA/utils.H"

// #include<vector>
using std::vector;

namespace CoCoA
{

  std::vector<RingElem> GCDFreeBasis(const std::vector<RingElem>& L)
  {
    // assume L is list of elems of ring with GCD
    if (L.empty()) CoCoA_ERROR(ERR::Empty, "GCDFreeBasis");
    const int n = len(L);
    const ring& R = owner(L[0]);
    vector<RingElem> ans;
    ans.push_back(gcd(L[0], zero(R)));
    if (n == 1) return ans;
    for (int i=1; i < n; ++i)
    {
      const RingElem RemainingFactor = RefineGCDFreeBasis(ans, L[i]);
      if (IsInvertible(RemainingFactor)) continue;
      ans.push_back(gcd(RemainingFactor, zero(R)));
    }
    return ans;
  }

  struct GFB2
  {
    GFB2(const RingElem& A, const RingElem& B): Afactor(A), Bfactor(B), GFBCommonFactors(0) { CoCoA_ASSERT(IsOne(gcd(A,B)));}
    RingElem Afactor;
    RingElem Bfactor;
    vector<RingElem> GFBCommonFactors;
  };

  GFB2 GCDFreeBasis2(RingElem A, RingElem B)
  {
    const RingElem g = gcd(A,B);
    if (IsOne(g)) return GFB2(A,B);
    A /= g;
    B /= g;
    if (IsInvertible(A))
    {
      while (true)
      {
        RingElem quot;
        if (!IsDivisible(quot, B, g)) break;
        swap(B,quot); // cheap assignment (maybe std::move?)
      }
      GFB2 ans = GCDFreeBasis2(g, B);
      if (!IsInvertible(ans.Afactor)) ans.GFBCommonFactors.push_back(ans.Afactor);
      ans.Afactor = A;
      return ans;
    }
    if (IsInvertible(B))
    {
      while (true)
      {
        RingElem quot;
        if (!IsDivisible(quot, A,g)) break;
        swap(A,quot); // cheap assignment (maybe std::move?)
      }
      GFB2 ans = GCDFreeBasis2(g, A);
      if (!IsInvertible(ans.Bfactor)) ans.GFBCommonFactors.push_back(ans.Bfactor);
      ans.Bfactor = B;
      return ans;
    }
    // general case: A, B are non-trivial
    GFB2 ans = GCDFreeBasis2(A, g);
    if (IsInvertible(ans.Bfactor)) { ans.Bfactor = B; return ans; }
    GFB2 tmp = GCDFreeBasis2(ans.Bfactor, B);
    if (!IsInvertible(tmp.Afactor)) ans.GFBCommonFactors.push_back(tmp.Afactor);
    /// concat -- SLUG SLUG SLUG, do this better!
    for (int i=0; i < len(tmp.GFBCommonFactors); ++i)
      ans.GFBCommonFactors.push_back(tmp.GFBCommonFactors[i]);
    ans.Bfactor = tmp.Bfactor;
    return ans;
  }

  RingElem RefineGCDFreeBasis(std::vector<RingElem>& basis, RingElem N)
  {
    const ring& R = owner(N);
    if (basis.empty()) { basis.push_back(gcd(N,zero(R))); return one(R); }
    RingElem RemainingFactor = one(R);
    vector<RingElem> NewBasis;
    const int n = len(basis);
    for (int i=0; i < n; ++i)
    {
      if (IsInvertible(N)) { NewBasis.push_back(basis[i]); continue; }
      GFB2 tmp = GCDFreeBasis2(basis[i], N);
      N = tmp.Bfactor;
      if (!IsInvertible(tmp.Afactor)) NewBasis.push_back(tmp.Afactor);
      // concat  SLUG SLUG SLUG
      for (int j=0; j < len(tmp.GFBCommonFactors); ++j)
        NewBasis.push_back(tmp.GFBCommonFactors[j]);
    }
    swap(basis, NewBasis);
    return N;
  }
  
} // end of namespace CoCoA


// RCS header/log in the next few lines
// $Header: /Volumes/Home_1/cocoa/cvs-repository/CoCoALib-0.99/src/AlgebraicCore/GCDFreeBasis.C,v 1.1 2017/02/01 10:36:49 abbott Exp $
// $Log: GCDFreeBasis.C,v $
// Revision 1.1  2017/02/01 10:36:49  abbott
// Summary: IMpl of GCDFreeBasis (transl from CoCoA-5)
//
//
