// Copyright (c) 2017  John Abbott
// This file is part of the CoCoALib suite of examples.
// You are free to use any part of this example in your own programs.

#include "CoCoA/library.H"

using namespace std;

//----------------------------------------------------------------------
const string ShortDescription =
  "This program illustrates use of the RootBound function for      \n"
  "obtaining an upper bound on the abs value of the complex roots. \n";

const string LongDescription =
  "This program illustrates use of the RootBound function for      \n"
  "obtaining an upper bound on the abs value of the complex roots. \n"
  "The function has an optional second arg.  The second arg says   \n"
  "how many Graeffe iterations to perform; these lead to a better  \n"
  "bound, but can be slow for large polynomials (as the example    \n"
  "makes clear).                                                   \n";
//----------------------------------------------------------------------

namespace CoCoA
{

  void program()
  {
    GlobalManager CoCoAFoundations;

    cout << ShortDescription << endl;
    cout << boolalpha; // so that bools print out as true/false

    ring P = NewPolyRing(RingQQ(), symbols("x,y"));
    RingElem f= RingElem(P, "x^2-2");
    BigRat B = RootBound(f);

    //-------------------------------------------------------
    // Now we try a larger polynomial g = product((x+k*k*k), k=1,...,200)
    // For such a big poly "RootBound" does just a few Graeffe steps
    // but we can use the second arg to force it to make more Graeffe steps
    // (though this will take extra time).
    RingElem g = one(P);
    const RingElem& x = indet(P,0);
    for (int k=0; k <= 250; ++k)
      g *= (x+k*k*k);

    cout << "Let g = product((x+k*k*k), k=1,...,200)" << endl
         << "We know the true root bound for g is 15625000." << endl << endl;

    double start = CpuTime();
    BigRat Bg = RootBound(g);
    const double t = CpuTime() - start;
    cout << "RootBound uses a heuristic to decide how many Graffe iterations to use." << endl
         << "RootBound(g) = " << Bg << "   time to compute: " << t << endl;    

    cout << endl << "We can specify how many Graeffe iterations to perform as a second arg." << endl
         << "We obtain a trade-off between speed and tightness:" << endl;
    for (int NumGraeffe = 0; NumGraeffe < 6; ++NumGraeffe)
    {
      start = CpuTime();
      Bg = RootBound(g,NumGraeffe);
      const double t1 = CpuTime() - start;
      cout << "RootBound(g," << NumGraeffe << ") = " << Bg << "  time to compute: " << t1 << endl;
    }

  }

} // end of namespace CoCoA

//----------------------------------------------------------------------
// Use main() to handle any uncaught exceptions and warn the user about them.
int main()
{
  try
  {
    CoCoA::program();
    return 0;
  }
  // catch (const CoCoA::InterruptReceived& intr)
  // {
  //   cerr << endl
  //        << "------------------------------" << endl
  //        << ">>>  CoCoALib interrupted  <<<" << endl
  //        << "------------------------------" << endl
  //        << "-->>  " << intr << "  <<--" << endl;
  //   return 2;
  // }
  catch (const CoCoA::ErrorInfo& err)
  {
    cerr << "***ERROR***  UNCAUGHT CoCoA error";
    ANNOUNCE(cerr, err);
  }
  catch (const std::exception& exc)
  {
    cerr << "***ERROR***  UNCAUGHT std::exception: " << exc.what() << endl;
  }
  catch(...)
  {
    cerr << "***ERROR***  UNCAUGHT UNKNOWN EXCEPTION" << endl;
  }

  CoCoA::BuildInfo::PrintAll(cerr);
  return 1;
}

//----------------------------------------------------------------------
// RCS header/log in the next few lines
// $Header: /Volumes/Home_1/cocoa/cvs-repository/CoCoALib-0.99/examples/ex-RootBound1.C,v 1.2 2018/01/17 10:24:15 abbott Exp $
// $Log: ex-RootBound1.C,v $
// Revision 1.2  2018/01/17 10:24:15  abbott
// Summary: Changed so that it works even when SmallExponent_t is unsigned char
//
// Revision 1.1  2017/09/14 15:54:54  abbott
// Summary: Added RootBound
//
// Revision 1.12  2017/07/08 19:07:02  abbott
// Summary: Removed comment out (dodgy) code for reporting unhandled interrupts
//
// Revision 1.11  2016/11/18 18:05:15  abbott
// Summary: Added commented out code to catch InterruptReceived
//
// Revision 1.10  2015/06/29 14:23:19  abbott
// Summary: added missing CoCoA:: prefix
// Author: JAA
//
// Revision 1.9  2015/06/29 13:25:54  bigatti
// -- code in namespace CoCoA
//
// Revision 1.8  2015/06/25 14:19:02  abbott
// Summary: Added call to CoCoA::BuildInfo::Printall
// Author: JAA
//
// Revision 1.7  2013/05/28 07:07:04  bigatti
// -- added "cout << boolalpha": useful for testing
//
// Revision 1.6  2012/11/30 14:04:55  abbott
// Increased visibility of comment saying "put your code here".
//
// Revision 1.5  2010/12/17 16:07:54  abbott
// Ensured that all i/o in examples is on standard C++ streams
// (rather than GlobalInput(), etc).
//
// Revision 1.4  2008/10/07 12:12:54  abbott
// Removed useless commented out #include.
//
// Revision 1.3  2007/05/31 16:06:16  bigatti
// -- removed previous unwanted checked-in version
//
// Revision 1.1.1.1  2007/03/09 15:16:11  abbott
// Imported files
//
// Revision 1.9  2007/03/07 11:51:40  bigatti
// -- improved test alignment
//
// Revision 1.8  2007/03/03 14:15:45  bigatti
// -- "foundations" renamed into "GlobalManager"
//
// Revision 1.7  2007/03/02 17:46:40  bigatti
// -- unique RingZ and RingQ
// -- requires foundations.H ;  foundations blah;  (thik of a better name)
//
// Revision 1.6  2007/03/02 10:47:53  cocoa
// First stage of RingZ modifications -- tests do not compile currently, Anna will fix this.
//
// Revision 1.5  2007/03/01 13:52:59  bigatti
// -- minor: fixed typo
//
// Revision 1.4  2007/02/28 15:15:56  bigatti
// -- minor: removed quotes in description
//
// Revision 1.3  2007/02/12 16:27:43  bigatti
// -- added strings ShortDescription and LongDescription for indexing
//
// Revision 1.2  2007/02/10 18:44:03  cocoa
// Added "const" twice to each test and example.
// Eliminated dependency on io.H in several files.
// Improved BuildInfo, and added an example about how to use it.
// Some other minor cleaning.
//
// Revision 1.1.1.1  2006/05/30 11:39:36  cocoa
// Imported files
//
// Revision 1.1  2006/03/12 21:28:34  cocoa
// Major check in after many changes
//
