// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include <awali/dyn/loading/locations.hh>
#include <awali/dyn/core/context-description.hh>


#include<dirent.h>
#include<fstream>
#include<map>
#include<vector>
#include<stdexcept>
#include<iostream>

namespace awali { namespace cora {



//Options
std::string output_format="default";
std::string input_format="default";
std::string letter_type="char";
std::string alphabet="ab";
std::string alphabetb="ab";
std::string semiring="B";
bool verbose=true;
std::string algo="default";
std::string name="tmp";
bool allow_eps_transition = false;

const char* args[10];

enum coms_t {
  /*TODO :
   * - chain
   */
  //utilities
  HELP, LIST, DISPLAY, INFO,
  //make_automaton
  CAT, EMPTY_AUT, STRIP,
  NUM_STATE, NUM_TRANS, NUM_INIT, NUM_FIN, SUPPORT, CHARACTERISTIC, ALLOW_EPS,
  //make_ratexp
  CAT_EXP, DISPLAY_EXP, AUT_TO_EXP, EXPAND, SNF, STAR_HEIGHT, VALID_EXP, CST_TERM,
  //accessible
  NUM_ACC, NUM_COACC, ACCESSIBLE, COACC, TRIM, IS_TRIM, IS_USELESS, IS_ACC, IS_COACC, IS_EMPTY,
  //factor
  PREFIX, SUFFIX, FACTOR,
  //derivation
  EXP_TO_AUT, DERIVATION,
  //factories
  DE_BRUIJN, DIVKBASEB, DOUBLERING, LADYBIRD, WITNESS, CERNY,
  //determinize
  DETERMINIZE, COMPLEMENT, COMPLETE, REDUCE, L_REDUCE, R_REDUCE, IS_DET, IS_COMPLETE, IS_AMB, ARE_EQVT,
  //edit
  EDIT,
  //eval
  EVAL, ENUMERATE, SHORTEST,
  //minimize
  MINQUOTIENT, MINIMAL, MINIMALE, MINCOQUOTIENT,
  //product
  PRODUCT,LEFT_MULT, RIGHT_MULT, SHUFFLE,INFILTRATION,UNION,ARE_ISOMORPHIC,POWER,
  //proper
  PROPER, IS_PROPER, IS_VALID,
  //random
  RAND_DFA,
  //transpose
  TRANSPOSE,
  //standard
  IS_STANDARD, CONCATENATION, SUM, STAR, STANDARD,
  //transducer
  EMPTY_TDC, TDC_DOMAIN, IMAGE, INVERSE, COMPOSE, PARTIAL_ID, EVAL_T, EVAL_TW, IS_FUNC, //TDC_AUT,
  //
  SYNCHRONIZE, IS_SYNCHRONIZABLE, REALTIME,
  //options
  INPUT_FMT, OUTPUT_FMT, ALPHABET,ALPHABET_B, QUIET, LABELS, WEIGHTS, VERBOSE, METHOD, HISTORY, NAME
};

    enum arg_kind_t {  ALPH, AUT, BOOL, CMD, EXP, FMT, INT, MTD, NONE, LBL, SMR, TDC, WEIGHT, WORD, NM };

struct argument_t {
  arg_kind_t kind;
  std::string commentary = "";
  argument_t(arg_kind_t k) { kind=k; }
  argument_t(arg_kind_t k, std::string str) { kind=k; commentary=str; }
};

std::ostream& operator<<(std::ostream& os, const arg_kind_t& kind)
{
  switch (kind) {
    case ALPH: os << "alphabet";  break;
    case AUT: os << "aut"; break;
    case BOOL: os << "bool"; break;
    case CMD: os << "cmd";  break;
    case EXP: os << "exp"; break;
    case FMT: os << "format"; break;
    case INT: os << "int"; break;
    case MTD: os << "method"; break;
    case NONE: os << "none";  break;
    case SMR: os << "semiring";  break;
    case LBL: os << "letters";  break;
    case TDC: os << "tdc"; break;
    case WEIGHT: os << "weight"; break;
    case WORD: os << "word"; break;
    case NM: os << "name"; break;
  };
  return os;
}

std::ostream& operator<<(std::ostream& os, const argument_t& arg)
{
  os << '<' << arg.kind;
  if (arg.commentary.length() != 0)
    os << ':' << arg.commentary;
  os << '>';
  return os;
}

std::ostream& operator<<(std::ostream& os, const std::vector<argument_t>& args)
{
  for (argument_t arg: args)
    os << arg << " ";
  return os;
}

struct command {
  std::string desc;
  std::string longdesc;
  std::vector<argument_t> args;
  unsigned argc;
  coms_t cst;
};

std::map<std::string,command> commands;
std::map<std::string,command> options;
std::map<std::string,command> commands_base;
std::map<std::string,command> commands_aut_tdc;
std::map<std::string,command> commands_aut;
std::map<std::string,command> commands_nfa;
std::map<std::string,command> commands_tdc;
std::map<std::string,command> commands_exp;
std::map<std::string,command> commands_fact;
std::map<std::string,command> commands_std;

void init() {
  //utilities
  commands_base.emplace(std::make_pair("help", command{"print this help or a command help : help <cmd>",
    "Print the list of commands or details on a command.",
    {{CMD}},1
    ,HELP}));
  commands_base.emplace(std::make_pair("list", command{"list stored automata",
    "List automata stored in share/automata directory.",
    {},0
    ,LIST}));
  commands_aut_tdc.emplace(std::make_pair("display", command{"display an automaton using Graphviz",
    "Display an automaton using Graphviz.\n"
    "Use option -Opdf (default) to display a pdf export from Graphviz,\n"
    "Use option -Odot to display in a Graphviz window.\n"
    "Use option -N to give a name to the output and obtain a fresh window.",
    {{AUT}},1,
    DISPLAY}));
  commands_aut_tdc.emplace(std::make_pair("info", command{"basic characteristics of an automaton",
    "Print the number of states, transitions, initial states and final states.",
    {{AUT}},1,
    INFO}));
  //make_automaton
  commands_aut_tdc.emplace(std::make_pair("cat", command{"load and output an automaton",
    "Load and output an automaton.\n"
    "Use with -I[input-format] -O[output-format] options to convert an automaton:\n"
    "available formats are json (default), dot (output only), pdf (output only), svg (output only), grail (Boolean only) and fado (Boolean only).",
                                          {{AUT}},1,
                                          CAT}));
  commands_aut_tdc.emplace(std::make_pair("allow-eps", command{"allow epsilon transition",
    "Return a copy of input automaton in which spontaneous transitions are allowed.",
                                          {{AUT}},1,
                                          ALLOW_EPS}));
  commands_aut.emplace(std::make_pair("create", command{"create and edit an empty automaton",
    "Create and edit an empty automaton.\n"
    "Use option -W to choose the weight : B (default), Z, Q, R, F2, Z-min-plus, R-max-prod.\n"
    "Use option -L to chose whether letters are char or int.\n"
    "Use option -A followed by a sequence of letters to chose the alphabet.\n"
    "If option -O is set, instead of editing, the empty automaton is output.",
    {},0,EMPTY_AUT}));
  commands_aut_tdc.emplace(std::make_pair("strip", command{"strip the history",
    "Remove every history information.",
    {{AUT}},1,STRIP}));
  commands_aut_tdc.emplace(std::make_pair("num-states", command{"number of states",
    "Print the number of states.",
    {{AUT}},1,NUM_STATE}));
  commands_aut_tdc.emplace(std::make_pair("num-initial-states", command{"number of initial states",
    "Print the number of initial states.",
    {{AUT}},1,NUM_INIT}));
  commands_aut_tdc.emplace(std::make_pair("num-final-states", command{"number of final states",
    "Print the number of final states.",
    {{AUT}},1,NUM_FIN}));
  commands_aut_tdc.emplace(std::make_pair("num-transitions", command{"number of transitions",
    "Print the number of transitions.",
    {{AUT}},1,NUM_TRANS}));
  commands_aut.emplace(std::make_pair("support", command{"support of a weighted automaton",
    "Build the NFA with the same transition function as the input",
    {{AUT}},1,SUPPORT}));
  commands_nfa.emplace(std::make_pair("characteristic", command{"characteristic automaton of a NFA",
    "Build the weighted with the same transition function as the input\n"
    "The weight of each transition is the one of the semiring\n"
    "The semiring can be specified with the option -W",
    {{AUT}},1,CHARACTERISTIC}));
  //make_ratexp
  commands_exp.emplace(std::make_pair("catE", command{"load and output a rational expression",
    "Load and output a rational expression.\n"
    "Use with -I or -O option for conversion:\n"
    "Available formats : json or text.",
    {{EXP}},1,CAT_EXP}));
  commands_exp.emplace(std::make_pair("displayE", command{"display a rational expression with graphviz",
    "Load and display a rational expression.\n"
    "Use -N to give a name to the output and obtain a fresh window.",
    {{EXP}},1,DISPLAY_EXP}));
  commands_aut_tdc.emplace(std::make_pair("aut-to-exp", command{"convert an automaton into a rational expression",
    "Convert an automaton into a rational expression.",
    {{AUT}},1,AUT_TO_EXP}));
  commands_exp.emplace(std::make_pair("expand", command{"expand a rational expression",
    "Expand a rational expression.",
    {{EXP}},1,EXPAND}));
  commands_exp.emplace(std::make_pair("star-normal-form", command{"star normal form of a rational expression",
    "Star normal form of a rational expression.",
    {{EXP}},1,SNF}));
  commands_exp.emplace(std::make_pair("is-validE", command{"test whether a rational expression is valid",
    "test whether a rational expression is valid.",
    {{EXP}},1,VALID_EXP}));
  commands_exp.emplace(std::make_pair("constant-term", command{"compute the constant term of the expression",
    "compute the constant term of a valid rational expression,\n"
	  "i.e. the weight of the empty word in the described series.", 
    {{EXP}},1,CST_TERM}));
  commands_exp.emplace(std::make_pair("minimal-automaton-E", command{"minimal automaton of a language",
    "Compute the minimal automaton of a language.\n"
    "The input is a Boolean rational expression.\n"
    "Use -M to set the algorithm: 'moore' (default), 'hopcroft' or 'brzozowski'." ,
                                      {{AUT}},1,MINIMALE}));

  //accessible
  commands_aut_tdc.emplace(std::make_pair("num-acc-states", command{"number of accessible states",
    "Print the number of accessible states.",
    {{AUT}},1, NUM_ACC}));
  commands_aut_tdc.emplace(std::make_pair("num-coacc-states", command{"number of coaccessible states",
    "Print the number of coaccessible states.",
    {{AUT}},1, NUM_COACC}));
  commands_aut_tdc.emplace(std::make_pair("accessible", command{"accessible part of the automaton",
    "Compute the accessible subautomaton.",
    {{AUT}},1, ACCESSIBLE}));
  commands_aut_tdc.emplace(std::make_pair("coaccessible", command{"coaccessible part of the automaton",
    "Compute the coaccessible subautomaton.",
    {{AUT}},1, COACC}));
  commands_aut_tdc.emplace(std::make_pair("trim", command{"trim part of the automaton",
    "Compute the trim subautomaton.",
    {{AUT}},1, TRIM}));
  commands_aut_tdc.emplace(std::make_pair("is-trim", command{"test whether the automaton is trim",
    "Test whether the automaton is trim.\n"
    "Exit with 0 if true.\n"
    "Use -Q to not print the result.",
    {{AUT}},1, IS_TRIM}));
  commands_aut_tdc.emplace(std::make_pair("is-accessible", command{"test whether the automaton is accessible",
    "Test whether the automaton is accessible.\n"
    "Exit with 0 if true.\n"
    "Use -Q to not print the result.",
    {{AUT}},1, IS_ACC}));
  commands_aut_tdc.emplace(std::make_pair("is-coaccessible", command{"test whether the automaton is coaccessible",
    "Test whether the automaton is coaccessible.\n"
    "Exit with 0 if true.\n"
    "Use -Q to not print the result.",
    {{AUT}},1, IS_COACC}));
  commands_aut_tdc.emplace(std::make_pair("is-empty", command{"test whether the automaton is empty",
    "Test whether the automaton is empty.\n"
    "Exit with 0 if true.\n"
    "Use -Q to not print the result.",
    {{AUT}},1, IS_EMPTY}));
  commands_aut_tdc.emplace(std::make_pair("is-useless", command{"test whether the automaton has no useful state",
    "Test whether the trim part of the automaton is empty.\n"
    "Exit with 0 if true.\n"
    "Use -Q to not print the result.",
    {{AUT}},1, IS_USELESS}));
  //factor
  commands_aut_tdc.emplace(std::make_pair("prefix", command{"automaton accepting the language of prefixes",
    "Make every state of the automaton final.",
    {{AUT}},1, PREFIX}));
  commands_aut_tdc.emplace(std::make_pair("suffix", command{"automaton accepting the language of suffixes",
    "Make every state of the automaton initial.",
    {{AUT}},1, SUFFIX}));
  commands_aut_tdc.emplace(std::make_pair("factor", command{"automaton accepting the language of factors",
    "Make every state of the automaton both initial and final.",
    {{AUT}},1, FACTOR}));
  //derivation
  commands_exp.emplace(std::make_pair("exp-to-aut", command{"build an automaton from a rational expression",
    "Build an automaton from a rational expression.\n"
    "Use -M to choose the method:\n"
    "- derivation : derived term method (default);\n"
    "- breaking : broken derived term method;\n"
    "- standard : incremental with standard automata;\n"
    "- thompson : Thompson automaton.\n"
    "- zpc : a Thompson-like automaton; safe with weights.",
    {{EXP}},1,EXP_TO_AUT}));
  commands_exp.emplace(std::make_pair("derivation", command{"derivation of a rational expression",
    "Derivation of a rational expression with respect to a word.\n"
    "Use -M to chose the method:\n"
    "- derivation : usual derivation (default);\n"
    "- breaking : breaking derivation.\n",
    {{EXP}, {WORD}},2,DERIVATION}));
  //determinize
  commands_nfa.emplace(std::make_pair("determinize", command{"determinize an NFA",
    "Determinize an NFA. The result may be not complete.",
    {{AUT}},1,DETERMINIZE}));
  commands_nfa.emplace(std::make_pair("complement", command{"complement a complete DFA",
    "Complement a complete DFA",
    {{AUT}},1,COMPLEMENT}));
  commands_nfa.emplace(std::make_pair("complete", command{"complete a NFA          ",
    "Complete a DFA.",
    {{AUT}},1,COMPLETE}));
  commands_aut.emplace(std::make_pair("reduce", command{"reduce a WFA",
    "Compute a reduced automaton equivalent to the given weighted automaton.\n"
    "The weights must belong to Z or to a field.",
    {{AUT}},1,REDUCE}));
  commands_aut.emplace(std::make_pair("left-reduce", command{"left reduce a WFA",
    "Compute a controllable automaton equivalent to the given weighted automaton.\n"
    "The weights must belong to Z or to a field.",
    {{AUT}},1,L_REDUCE}));
  commands_aut.emplace(std::make_pair("right-reduce", command{"right reduce a WFA",
    "Compute an observable automaton equivalent to the given weighted automaton.\n"
    "The weights must belong to Z or to a field.",
    {{AUT}},1,R_REDUCE}));
  commands_aut.emplace(std::make_pair("is-deterministic", command{"test whether an automaton is deterministic",
    "Test whether an automaton is deterministic.\n"
    "Exit with 0 if true.\n"
    "Use -V to print the result.",
    {{AUT}},1,IS_DET}));
  commands_nfa.emplace(std::make_pair("is-complete", command{"test whether a DFA is complete",
    "Test whether a DFA is complete\n"
    "Exit with 0 if true.\n"
    "Use -V to print the result.",
    {{AUT}},1,IS_COMPLETE}));
  commands_aut.emplace(std::make_pair("is-ambiguous", command{"test whether an automaton is ambiguous",
    "Test whether an automaton is ambiguous.\n"
    "Exit with 0 if true.\n"
    "Use -V to print the result.",
    {{AUT}},1,IS_AMB}));
  commands_aut.emplace(std::make_pair("are-equivalent", command{"test whether two automata are equivalent",
    "Test whether two automata are equivalent.\n"
    "Both automata must have the same type : NFA, Z- or F-FA, where F is a field.\n"
    "Exit with 0 if true.\n"
    "Use -V to print the result.",
    {{AUT,"1"}, {AUT,"2"}},2,ARE_EQVT}));
  //edit
  commands_aut_tdc.emplace(std::make_pair("edit", command{"edit an automaton",
    "Enter an interactive mode to edit the automaton.",
    {{AUT}},1,EDIT}));
  //eval
  commands_aut.emplace(std::make_pair("eval",
                                      command{"compute the behaviour of a word",
                                              "Compute the weigth of a word in an automaton.",
                                              {{AUT}, {WORD}},
                                              2,
                                              EVAL}));
  commands_aut.emplace(std::make_pair("enumerate", command{"enumerate the first words of the language",
    "Enumerate the words w.r.t. radix ordering accepted by the automaton,\n"
    "with their weight, up to length n",
    {{AUT},{INT,"n"}},2,ENUMERATE}));
  commands_aut.emplace(std::make_pair("shortest", command{"enumerate the shortest words of the language",
    "Enumerate the n shortest words (radix order) accepted by the automaton,\n"
    "with their weight.",
    {{AUT},{INT,"n"}},2,SHORTEST}));
  //factories
  commands_fact.emplace(std::make_pair("de-Bruijn",
            command{"build an automaton whose determinisation is the deBruijn graph",
    "Build the automaton with n+1 states that recognizes words with 'a' n-th letter from end.\n"
    "The determinisation of this automaton is the de Bruijn graph with 2^n states.\n"
    "The 2-alphabet can be modified with -A option."
    ,
    {{INT, "n"}},1,DE_BRUIJN}));
  commands_fact.emplace(std::make_pair("divkbaseb",
            command{"build an automaton that recognizes multiple of k written in base b",
    "Build the automaton with k states that recognizes words representing multiple of k written in base b.\n"
    "The k-letter alphabet can be modified with -A option."
    ,
    {{INT,"k"},{INT, "b"}},2,DIVKBASEB}));
  commands_fact.emplace(std::make_pair("double-ring",
            command{"build an automaton that recognizes words \n"
    "                       with the same number of 'a's as 'b's modulo n",
    "Build an automaton with n states that recognizes words with the same number of 'a' as 'b' modulo n.\n"
    "The 2-letter alphabet can be modified with -A option."
    ,
    {{INT, "n"}},1,DOUBLERING}));
  commands_fact.emplace(std::make_pair("ladybird", command{"build an automaton whose determinisation is exponential",
    "Build an automaton with n states whose determinization (complete) has 2^n states.\n"
    "When n=7, the automaton can be drawn like a ladybird.\n"
    "The 3-letter alphabet can be modified with -A option."
    ,
    {{INT, "n"}},1,LADYBIRD}));
  commands_fact.emplace(std::make_pair("cerny", command{"build an automaton with long synchronizing word",
    "Build an automaton with long synchronizing word\n"
    "The 2-letter alphabet can be modified with -A option."
    ,
    {{INT, "n"}},1,CERNY}));
  commands_fact.emplace(std::make_pair("witness", command{"build Brzozowski's universal witness",
    "Build Brzozowski's universal witness with n states.\n"
    "This automaton is a candidate for worst case of many algorithms.\n"
    "The 3-letter alphabet can be modified with -A option."
    ,
    {{INT, "n"}},1,WITNESS}));
  //minquotient
  commands_nfa.emplace(std::make_pair("minimal-automaton", command{"minimal automaton of a language",
    "Compute the minimal automaton of a language.\n"
    "The input must be a NFA.\n"
    "Use -M to set the algorithm: 'moore' (default), 'hopcroft' or 'brzozowski'." ,
                                      {{AUT}},1,MINIMAL}));
  commands_aut.emplace(std::make_pair("min-quotient", command{"minimal quotient of an automaton",
    "Compute the minimal quotient of an automaton.\n"
    "If the automaton is deterministic this computes the minimal automaton, up to "
   "the fact that the result may be not complete.\n"
    "Use -M to set the algorithm: 'hopcroft' or 'moore' (default)",
                                      {{AUT}},1,MINQUOTIENT}));
  commands_aut.emplace(std::make_pair("min-coquotient", command{"minimal coquotient of an automaton",
    "Compute the minimal coquotient of an automaton.\n"
    "Use -M to set the algorithm: 'hopcroft' or 'moore' (default)",
                                      {{AUT}},1,MINCOQUOTIENT}));
  //product
  commands_aut.emplace(std::make_pair("product", command{"product of two automata",
    "Compute the product of two automata.\n"
    "Both automata may have different types.",
    {{AUT,"1"},{AUT,"2"}},2,PRODUCT}));
  commands_aut.emplace(std::make_pair("power", command{"power of an automaton",
    "Compute the n-th power of an automaton.",
    {{AUT},{INT,"n"}},2,POWER}));
  commands_aut.emplace(std::make_pair("shuffle", command{"shuffle of two automata",
    "Compute the shuffle of two automata.\n"
    "Both automata may have different types.",
    {{AUT,"1"},{AUT,"2"}},2,SHUFFLE}));
  commands_aut.emplace(std::make_pair("infiltration", command{"infiltration of two automata",
    "Compute the infiltration of two automata.\n"
    "Both automata may have different types.",
    {{AUT,"1"},{AUT,"2"}},2,INFILTRATION}));
  commands_aut.emplace(std::make_pair("union", command{"union of two automata",
    "Compute the union of two automata.\n"
    "Both automata may have different types.\n",
    {{AUT,"1"},{AUT,"2"}},2,UNION}));
  commands_std.emplace(std::make_pair("left-mult", command{"external product of a weight and an  automaton",
    "Build an automaton that recognizes the left product of a weight  by the series recognized by the input.\n"
    "If the automaton is not standard, every initial weight is multiplied by the scalar;\n"
	  "otherwise, the weight of every transition leaving the initial state is.\n",
      {{WEIGHT},{AUT}},2,LEFT_MULT}));
  commands_std.emplace(std::make_pair("right-mult", command{"external product of an automaton and a weight",
    "Build an automaton that recognizes the right product of a weight by the series recognized by the input.\n"
	  "Every final weight is multiplied by the scalar.\n",
    {{AUT},{WEIGHT}},2,RIGHT_MULT}));
  commands_aut.emplace(std::make_pair("are-isomorphic", command{"test whether the two automata are isomorphic",
    "Test whether the two automata are isomorphic.\n"
    "Both automata may have different types."
    "Exit with 0 if true.\n"
    "Use -V to print the result.",
    {{AUT,"1"},{AUT,"2"}},2,ARE_ISOMORPHIC}));
  //proper
  commands_aut_tdc.emplace(std::make_pair("proper", command{"epsilon removal",
    "Compute an equivalent automaton without epsilon transitions.\n"
    "Use -Mforward or -Mbackward (default) to chose the direction of removal",
                                          {{AUT}},1,PROPER}));
  commands_aut_tdc.emplace(std::make_pair("is-proper", command{"test whether the automaton has some epsilon transitions",
    "Test whether the automaton has some epsilon transitions.\n"
    "Exit with 0 if true.\n"
    "Use -V to print the result.",
    {{AUT}},1,IS_PROPER}));
  commands_aut_tdc.emplace(std::make_pair("is-valid", command{"test whether the automaton  is valid",
    "Test whether the automaton is valid.\n"
    "A WFA with epsilon transitions may have an infinite number of path labeled by a given word;\n"
    "If the set of the weights of these paths is not summable, the automaton is not valid.\n"
    "Exit with 0 if true.\n"
    "Use -V to print the result.",
    {{AUT}},1,IS_VALID}));
  //random
  commands_fact.emplace(std::make_pair("random-dfa", command{"generate a random DFA",
    "Generate a random DFA.\n"
    "A DFA with n states is generated.\n"
    "Use option -A followed by a sequence of letters to chose the alphabet.",
    {{INT,"n"}},1,RAND_DFA}));
  //standard
  commands_std.emplace(std::make_pair("is-standard", command{"test whether the automaton is standard",
    "Test whether the automaton is standard.\n"
    "An automaton if standard if it has a single initial state with no incoming transition.\n"
    "Exit with 0 if true.\n"
    "Use -V to print the result.",
    {{AUT}},1,STANDARD}));
  commands_std.emplace(std::make_pair("standard", command{"make an automaton standard",
    "Make an automaton standard; add one state.\n"
    "An automaton if standard if it has a single initial state with no incoming transition.\n",
    {{AUT}},1,STANDARD}));
  commands_std.emplace(std::make_pair("concatenation", command{"concatenation of standard automata",
    "Build a standard automaton that recognizes the concatenation of languages (or series) recognized by the inputs,\n"
    "which are standard automata.\n"
    "An automaton if standard if it has a single initial state with no incoming transition.\n",
    {{AUT,"1"},{AUT,"2"}},2,CONCATENATION}));
  commands_std.emplace(std::make_pair("sum", command{"sum of standard automata",
    "Build a standard automaton that recognizes the union (or sum) of languages (or series) recognized by the inputs,\n"
    "which are standard automata.\n"
    "An automaton if standard if it has a single initial state with no incoming transition.\n",
    {{AUT,"1"},{AUT,"2"}},2,SUM}));
  commands_std.emplace(std::make_pair("star", command{"star of a standard automaton",
    "Build a standard automaton that recognizes the star of the language (or series) recognized by the input,\n"
    "which is a standard automaton.\n"
    "An automaton if standard if it has a single initial state with no incoming transition.\n",
    {{AUT}},1,STAR}));
  //transducer
  commands_tdc.emplace(std::make_pair("create-tdc", command{"create and edit an empty transducer",
    "Create and edit an empty transducer."
    "Use option -W to choose the weight : B (default), Z, Q, R, F2, Z-min-plus, R-max-prod.\n"
    "Use option -A followed by a sequence of letters to chose the input alphabet.\n"
    "Use option -B followed by a sequence of letters to chose the output alphabet.\n"
    "If option -O is set, instead of editing, the empty automaton is output.",
    {},0,EMPTY_TDC}));
  commands_tdc.emplace(std::make_pair("domain", command{"domain of a transducer",
    "Build the automaton which is the projection of the transducer w.r.t the input.",
    {TDC},1,TDC_DOMAIN}));
  commands_tdc.emplace(std::make_pair("image", command{"image of a transducer",
    "Build the automaton which is the projection of the transducer w.r.t the output.",
    {TDC},1,IMAGE}));
  commands_tdc.emplace(std::make_pair("inverse", command{"inverse of a transducer",
    "Swap the input and the output of a transducer.",
    {TDC},1,INVERSE}));
  commands_tdc.emplace(std::make_pair("compose", command{"compose two transducers",
    "Build the transducer composition of the both inputs.",
    {{TDC,"1"},{TDC,"2"}},2,COMPOSE}));
  commands_aut.emplace(std::make_pair("partial-identity", command{"partial-identity from automaton",
    "Build the transducer which realizes the partial identity on the language defined by the automaton.",
    {{AUT}},1,PARTIAL_ID}));
  commands_tdc.emplace(std::make_pair("is-functional", command{"test whether a transducer is functional",
    "Test whether a transducer is functional.\n"
    "Exit with 0 if true.\n"
    "Use -V to print the result.",
    {TDC},
    1,IS_FUNC}));
  commands_tdc.emplace(std::make_pair("eval-tdc", command{"image of an automaton by a transducer",
    "Apply the transducer to the automaton",
    {{AUT},{TDC}},2,EVAL_T}));
  commands_tdc.emplace(std::make_pair("eval-word", command{"image of a word by a transducer",
    "Apply the transducer to the word",
    {{TDC},{WORD}},2,EVAL_TW}));
  /*
  commands_tdc.emplace(std::make_pair("tdc-to-aut", command{"convert a transducer to a weighted automaton",
    "Convert a transducer to a weighted automaton.\n"
    "The labels of the automaton are the projection of the first tape of the transducer;\n"
    "the weights are series whose coefficients are the weights of the transducers,\n"
    "over the projection of the n-1 last tapes of the transducer.",
    {TDC},1,TDC_AUT}));
  */

  commands_tdc.emplace(std::make_pair("is-synchronizable", command{"test whether a transducer is synchronizable",
    "Test whether a subnormalized transducer can be synchronized.\n"
    "Exit with 0 if true.\n"
    "Use -V to print the result.",
    {TDC},
    1,IS_SYNCHRONIZABLE}));
  commands_tdc.emplace(std::make_pair("synchronize", command{"build a synchronized transducer",
    "Build a synchronized transducer from a subnormalized transducer.\n"
    "The synchronization is applied until a contradiction occurs.",
    {TDC},
    1,SYNCHRONIZE}));
  commands_tdc.emplace(std::make_pair("realtime", command{"build a realtime transducer",
    "Build a transducer where the input of each transition is a letter and the output is a word from a subnormalized transducer.\n"
    "This algorithm must only be applied to transducers where each word has a finte number of images.",
    {TDC},
    1,REALTIME}));
  //transpose
  commands_aut.emplace(std::make_pair("transpose", command{"transpose an automaton",
    "Build the transposed automaton.",
    {{AUT}},1,TRANSPOSE}));

  commands.insert(commands_base.begin(),commands_base.end());
  commands.insert(commands_aut_tdc.begin(),commands_aut_tdc.end());
  commands.insert(commands_aut.begin(),commands_aut.end());
  commands.insert(commands_nfa.begin(),commands_nfa.end());
  commands.insert(commands_tdc.begin(),commands_tdc.end());
  commands.insert(commands_exp.begin(),commands_exp.end());
  commands.insert(commands_fact.begin(),commands_fact.end());
  commands.insert(commands_std.begin(),commands_std.end());

  //options
  options.emplace(std::make_pair("A",
        command{"alphabet",
    "set the alphabet; comes after -L option if the latter is present.\n"
    "                      If letters are char, alphabet is given as a word,\n"
    "                          e.g. with -Axy if the alphabet is {'x','y'}.\n"
    "                      If letters are int, the letters are the nonnegative integers\n"
    "                          smaller than the given bound, e.g. with -A3 the alphabet is {0,1,2}.",
    {{ALPH}},1,ALPHABET}));
  options.emplace(std::make_pair("B", command{"alphabet2",
    "for transducers, set the output alphabet",
    {{ALPH}},1,ALPHABET_B}));
  options.emplace(std::make_pair("I", command{"input",
    "set input format: automata : json (default), fado, or grail;\n"
    "                    ratexp : text (default) or json",
    {{FMT}},1,INPUT_FMT}));
  options.emplace(std::make_pair("H", command{"history",
    "print history in dot format or display",
    {},0,HISTORY}));
  options.emplace(std::make_pair("M", command{"method",
    "set method for some commands: exp-to-aut, minimize, minimal, proper, ...",
    {{MTD}},1,METHOD}));
  options.emplace(std::make_pair("N", command{"name",
    "set name of the figure for display",
    {{NM}},1,NAME}));
  options.emplace(std::make_pair("O",
        command{"output",
    "set output format: for automata : json (default), dot, fado, or grail;\n"
    "                                      for automata display : pdf (default), dot \n"
    "                                      for ratexp : text (default) or json",
    {{FMT}},1,OUTPUT_FMT}));
  options.emplace(std::make_pair("Q", command{"quiet",
    "do not print result of test commands",
    {},0,QUIET}));
  options.emplace(std::make_pair("V", command{"verbose",
    "print result of test commands (default)",
    {},0,VERBOSE}));
  options.emplace(std::make_pair("W",
        command{"weights",
    std::string("set the semiring of weights;\n")
    +std::string("                   choice among ")
    +dyn::all_weightset_public_static_names_as_string()
    //B, Z, Q, R, C, Z-min-plus, Z-max-plus, R-max-prod, F2, Z/<n>Z."
    ,
    {{SMR}},1,
    WEIGHTS}));
  options.emplace(std::make_pair("L", command{"labels",
    "set the type of labels : char (default) or int;\n"
	  "                     for char, the default alphabet is {'a','b'}, for int, it is {0,1};\n"
          "                     it can be modified with option -A;\n"
          "                     if both -L and -A are used, -A should be set after -L on the command line",
    {{LBL}},1,
    LABELS}));
}

}}//end of ns awali::cora
