// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_ALPHABETS_CHAR_HH
# define AWALI_ALPHABETS_CHAR_HH

# include <cassert>
# include <string>
# include <iostream>

#include <awali/sttc/misc/escape.hh>

namespace awali { namespace sttc {

  class char_letters
  {
  public:
    using letter_t = char;
    using word_t = std::string;

    static std::string sname()
    {
      return "char_letter";
    }

    virtual std::string vname(bool = true) const
    {
      return sname();
    }

    word_t
    to_word(const letter_t l) const
    {
      return {l};
    }

    const word_t&
    to_word(const word_t& l) const
    {
      return l;
    }

    word_t
    concat(const letter_t l, const letter_t r) const
    {
      return {l, r};
    }

    word_t
    concat(const word_t& l, const letter_t r) const
    {
      return l + r;
    }

    word_t
    concat(const letter_t l, const word_t& r) const
    {
      return l + r;
    }

    word_t
    concat(const word_t& l, const word_t& r) const
    {
      return l + r;
    }

    /// Add the special character first and last.
    word_t delimit(const word_t& w) const
    {
      return concat(concat(special_letter(), w), special_letter());
    }

    /// Remove first and last characters, that must be "special".
    word_t undelimit(const word_t& w) const
    {
      size_t s = w.size();
      assert(2 <= s);
      assert(w[0] == special_letter());
      assert(w[s-1] == special_letter());
      return w.substr(1, s-2);
    }

    static word_t
    empty_word()
    {
      return {};
    }

    static bool
    is_empty_word(const word_t& w)
    {
      return w.empty();
    }

    word_t
    transpose(const word_t& w) const
    {
      // C++11 lacks std::rbegin/rend...
      return {w.rbegin(), w.rend()};
    }

    letter_t
    transpose(letter_t l) const
    {
      return l;
    }

    bool
    equals(const letter_t& l1, const letter_t& l2) const
    {
      return l1 == l2;
    }

    bool
    equals(const word_t& w1, const word_t& w2) const
    {
      return w1 == w2;
    }

    bool
    is_letter(const letter_t&) const
    {
      return true;
    }

    bool
    is_letter(const word_t& w) const
    {
      return w.size() == 1;
    }

    /// The reserved letter used to forge the "one" label (the unit,
    /// the identity).
    static constexpr letter_t one_letter() { return 0; }

  private:
    /// The reserved letter used to forge the labels for initial and
    /// final transitions.
    ///
    /// Use the public special() interface.
    static constexpr letter_t special_letter() { return 127; }

  public:
    std::ostream&
    print(const letter_t& l, std::ostream& o) const
    {
      if (l != one_letter() && l != special_letter())
        o << l;
      return o;
    }

    std::ostream&
    print(const word_t& w, std::ostream& o) const
    {
      return o << format(w);
    }

    letter_t
    parse_one_letter(const std::string& s, size_t& p) const {
      --p;
      return s[p];
    }


    letter_t
    conv_one_letter(std::istream& i) const
    {
      return i.get();
     }

    std::string
    format(const letter_t l) const
    {
      if (l == one_letter() || l == special_letter())
        return {};
      else
        return str_escape(l);
    }

    std::string
    format(const word_t& w) const
    {
      size_t s = w.size();

      std::string res;
      if (s == 0
          || (s == 1 && w[0] == one_letter()))
        res = "\\e";

      // If the string starts or ends with the special letter, skip
      // it.  If the resulting string is empty, format it this way.
      // (We DON'T want to format it as "\\e".)
      else if (w[0] == special_letter())
        res = (s == 1) ? "" : str_escape(w.substr(1));
      else if (s > 1 && w[s - 1] == special_letter())
        res = str_escape(w.substr(0, s - 1));
      else
        res = str_escape(w);
      return res;
    }

    /// Special character, used to label transitions from pre() and to
    /// post().
    template <typename T = letter_t>
    static T special();
  };

  template <>
  inline
  char_letters::letter_t
  char_letters::special<char_letters::letter_t>()
  {
    return special_letter();
  }

  template <>
  inline
  char_letters::word_t
  char_letters::special<char_letters::word_t>()
  {
    return {special_letter()};
  }

}}//end of ns awali::stc

#endif // !AWALI_ALPHABETS_CHAR_HH
