// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_ALGOS_ZPC_HH
# define AWALI_ALGOS_ZPC_HH

#include <awali/sttc/ctx/fwd.hh>
#include <awali/sttc/ctx/traits.hh>
#include <awali/sttc/core/mutable-automaton.hh>
#include <awali/sttc/core/rat/visitor.hh>
#include <awali/sttc/misc/raise.hh>

namespace awali { namespace sttc {

  namespace rat
  {
    /// \tparam Aut      relative the generated automaton
    /// \tparam Context  relative to the RatExp.
    template <typename Aut,
	      typename Ratexp,
              typename Context = context_t_of<Aut>
	      >
    class zpc_visitor
      : public Context::const_visitor
    {
    public:
      using automaton_t = Aut;
      using context_t = Context;
      using weightset_t = weightset_t_of<context_t>;
      using weight_t = weight_t_of<context_t>;
      using history_t = std::shared_ptr<string_history>;

      using super_type = typename Context::const_visitor;

      static_assert(context_t::has_one(), "requires nullable labels");

      constexpr static const char* me() { return "zpc"; }

      zpc_visitor(const context_t& ctx)
        : res_(make_shared_ptr<automaton_t>(ctx)), history_(std::make_shared<string_history>())
      {}

      automaton_t
      operator()(const Ratexp& v)
      {
	res_->set_history(history_);
        v->accept(*this);
        res_->set_initial(initial_);
        res_->set_final(final_);
	res_->set_final(initial_,cst_);
        return std::move(res_);
      }

      AWALI_RAT_VISIT(zero,)
      {
        initial_ = res_->add_state();
        final_ = res_->add_state();
	cst_ = ws_.zero();
	history_->add_state(initial_,"z[");
	history_->add_state(final_,"]z");
      }

      AWALI_RAT_VISIT(one,)
      {
        initial_ = res_->add_state();
        final_ = res_->add_state();
	cst_ = ws_.one();
	history_->add_state(initial_,"e[");
	history_->add_state(final_,"]e");
      }

      AWALI_RAT_VISIT(atom, e)
      {
        initial_ = res_->add_state();
        final_ = res_->add_state();
        res_->new_transition(initial_, final_, e.value());
	cst_ = ws_.zero();
	history_->add_state(initial_,"l[");
	history_->add_state(final_,"]l");
      }

      AWALI_RAT_VISIT(sum, e)
      {
        state_t initial = res_->add_state();
        state_t final = res_->add_state();
	weight_t w=ws_.zero();
        for (auto c: e)
          {
            c->accept(*this);
            res_->new_transition(initial, initial_, epsilon_);
            res_->new_transition(final_, final, epsilon_);
	    w=ws_.add(w,cst_);
          }
        initial_ = initial;
        final_ = final;
	cst_ = w;
	history_->add_state(initial_,"+[");
	history_->add_state(final_,"]+");
      }

      AWALI_RAT_UNSUPPORTED(complement)
      AWALI_RAT_UNSUPPORTED(conjunction)
      AWALI_RAT_UNSUPPORTED(ldiv)
      AWALI_RAT_UNSUPPORTED(shuffle)
      AWALI_RAT_UNSUPPORTED(transposition)

      AWALI_RAT_VISIT(prod, e)
      {
        state_t initial = res_->add_state();
        e.head()->accept(*this);
	weight_t w = cst_;
	res_->new_transition(initial, initial_, epsilon_);
	state_t final, current = final_;
	for(auto c: e.tail()) {
            c->accept(*this);
	    final = res_->add_state();
	    history_->add_state(final,"].[");
            res_->new_transition(current, initial_, epsilon_);
            res_->new_transition(final_, final, epsilon_);
            res_->new_transition(initial, initial_, epsilon_, w);
            res_->new_transition(current, final, epsilon_, cst_);
	    w = ws_.mul(w, cst_);
	    current = final;
	}
        initial_ = initial;
	final_ = final;
	cst_ = w;
	history_->add_state(initial_,".[");
	history_->add_state(final_,"].");
      }

      AWALI_RAT_VISIT(star, e)
      {
        e.sub()->accept(*this);
        state_t initial = res_->add_state();
        state_t final = res_->add_state();
	weight_t w = ws_.star(cst_);
        res_->new_transition(initial, initial_, epsilon_,w);
        res_->new_transition(final_,  final,    epsilon_,w);
        res_->new_transition(final_,  initial_, epsilon_,w);
        initial_ = initial;
        final_ = final;
	cst_ = w;
	history_->add_state(initial_,"*[");
	history_->add_state(final_,"]*");
      }

      AWALI_RAT_VISIT(lweight, e)
      {
        e.sub()->accept(*this);

        const weight_t& w = e.weight();
        for (auto t: res_->out(initial_))
          res_->set_weight(t, ws_.mul(w, res_->weight_of(t)));
	cst_ = ws_.mul(w, cst_);
      }

      AWALI_RAT_VISIT(rweight, e)
      {
        e.sub()->accept(*this);

        const weight_t& w = e.weight();
        for (auto t: res_->in(final_))
          res_->set_weight(t, ws_.mul(res_->weight_of(t), w));
	cst_ = ws_.mul(cst_, w);
      }

    private:
      automaton_t res_;
      const weightset_t& ws_ = *res_->weightset();
      using label_t = label_t_of<automaton_t>;
      const label_t epsilon_ = res_->labelset()->one();
      state_t initial_ = automaton_t::element_type::null_state();
      state_t final_ = automaton_t::element_type::null_state();
      weight_t cst_;
      history_t history_;
    };

  } // rat::

  /// \tparam Aut      relative to the generated automaton.
  /// \tparam Context  relative to the RatExp.
  template <typename Aut,
	    typename Ratexp,
            typename Context = context_t_of<Aut>>
  Aut
  zpc(const Context& ctx, const Ratexp& e)
  {
    rat::zpc_visitor<Aut, Ratexp, Context> zpc{ctx};
    return zpc(e);
  }

}}//end of ns awali::stc

#endif // !AWALI_ALGOS_ZPC_HH
