// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef DYN_CONTEXT_DESCRIPTION_HH
#define DYN_CONTEXT_DESCRIPTION_HH

#include <iostream>
#include <sstream>
#include <stdexcept>
#include <vector>
#include <string>
#include <memory>
#include <algorithm>
#include <awali/sttc/misc/json.hh>

//Definitions and factories for
// -labelset_description
// -weightset_description
// -context_description
#include <awali/dyn/core/context-description/description-types.hh>

//Weightset descriptions are handled by object corresponding to
//each semiring
#include <awali/dyn/core/context-description/weightsets/abstract-weightset.hh>
//The simples case (B, Q, Z-max-plus, etc...) are basic_weightset
#include <awali/dyn/core/context-description/weightsets/basic-weightset.hh>
//The other cases may have specific type
#include <awali/dyn/core/context-description/weightsets/cyclic-weightset.hh>

namespace awali {
  namespace dyn {
    //To be register in abstract_weighset, each semiring requires
    //the instanciation of a static object

    const std::vector<abstract_weightset*>& instances() {
          static basic_weightset _b_description("B","b");
	  static basic_weightset _z_description("Z","z");
	  static basic_weightset _q_description("Q","q");
	  static basic_weightset _r_description("R","r");
	  static basic_weightset _c_description("C","c");
	  static basic_weightset _f2_description("F2","f2");
	  static basic_weightset _zmin_description("Z-min-plus","zmin");
	  static basic_weightset _zmax_description("Z-max-plus","zmax");
	  static basic_weightset _rmaxp_description("R-max-prod","pmax");
	  static cyclic_weightset _cyclic_description;
	  static std::vector<abstract_weightset*> v{
	    &_b_description,
	      &_z_description,
	      &_q_description,
	      &_r_description,
	      &_c_description,
	      &_f2_description,
	      &_zmin_description,
	      &_zmax_description,
	      &_rmaxp_description,
	      &_cyclic_description};
	  return v;

    }

    labelset_description oneset() {
      labelset_description ls = make_labelset_description();
      ls->type_=CTypes::ONESET;
      return ls;
    }

    labelset_description letterset(std::string s) {
      labelset_description ls = make_labelset_description();
      ls->type_=CTypes::LETTERSET;
      for(auto c : s)
	ls->alphabet.emplace_back(std::string(1,c));
      return ls;
    }

    labelset_description intletterset(int a, int b) {
      labelset_description ls = make_labelset_description();
      ls->type_=CTypes::INTLETTERSET;
      for(int i=a; i<=b; ++i) {
	std::ostringstream o;
	o<<i;
	ls->alphabet.emplace_back(o.str());
      }
      return ls;
    }

    labelset_description intletterset(int n) {
      return intletterset(0, n-1);
    }

    labelset_description wordset(std::string s) {
      labelset_description ls = make_labelset_description();
      ls->type_=CTypes::WORDSET;
      for(auto c : s)
	ls->alphabet.emplace_back(std::string(1,c));
      return ls;
    }

    labelset_description nullableset(labelset_description ls1) {
      labelset_description ls = make_labelset_description();
      ls->type_=CTypes::NULLABLE;
      ls->children_.emplace_back(ls1);
      return ls;
    }

    labelset_description ltupleset(std::vector<labelset_description> lss) {
      labelset_description ls = make_labelset_description();
      ls->type_=CTypes::TUPLE;
      for(auto ls1 : lss)
	ls->children_.emplace_back(ls1);
      return ls;
    }

    weightset_description weightset(const std::string &k) {
      weightset_description ws;
      for(auto wgt : instances()) {
	ws=wgt->fromstring(k);
	if(ws.use_count() >0)
	  break;
      }
      if(ws.use_count()==0)
	throw std::invalid_argument("Unknown semiring: "+k);
      return ws;
    }

    weightset_description wtupleset(std::vector<weightset_description> wss) {
      weightset_description ws = make_weightset_description();
      ws->type_=WTypes::TUPLE;
      for(auto ws1 : wss)
	ws->children_.emplace_back(ws1);
      return ws;
    }

    weightset_description ratweight(context_description cd) {
      weightset_description ws = make_weightset_description();
      ws->type_ = WTypes::RATEXP;
      ws->ct_ = cd;
      return ws;
    }

    context_description c_desc(labelset_description ls, weightset_description ws) {
      context_description cd = make_context_description();
      cd->ls_=ls;
      cd->ws_ =ws;
      return cd;
    }

    //************************


    context_description parse_context(std::istream& i);

    int parse_characteristic(std::istream& i) {
      std::string key = sttc::parsestring(i);
      if (key == "Characteristic") {
	sttc::check(i, ':');
	return std::stoi(sttc::parsestring(i));
      } else
	throw std::runtime_error("json: expected \"Characteristic\", got \""+key+"\"");
    }

    weightset_description parse_weightset(std::istream& i) {
      weightset_description ws=make_weightset_description();
      std::string aut = sttc::get_first_attr(i);
      if(aut == "Semiring") {
    	for(auto wgt : instances()) {
	  std::streampos position = i.tellg();
	  ws=wgt->parse_weightset(i);
	  if(ws.use_count() >0)
	    break;
          i.seekg(position);
	}
	if(ws.use_count()==0)
	  throw std::runtime_error("json: Semiring");
	sttc::check(i,'}');
	return ws;
      }
      else if(aut=="Rational Expression") {
	ws->type_ = WTypes::RATEXP;
	ws->ct_ = parse_context(i);
	sttc::check(i,'}');
	return ws;
      }
      else if(aut == "Tuple") {
	ws->type_ = WTypes::TUPLE;
	sttc::check(i, '[');
	char c;
	std::vector<weightset_description> vecls;
	do {
	  ws->children_.emplace_back(parse_weightset(i));
	  i >> c;
	} while(c==',');
	if(c!=']')
	  throw std::runtime_error("json: Elements ]");
	return ws;
      }
      else
	throw std::runtime_error("json: Weights");
    }

    labelset_description parse_labelset(std::istream& i) {
      labelset_description ls= make_labelset_description();
      std::string aut = sttc::get_first_attr(i);
      if(aut != "Label Type") {
	if(aut == "Tuple") {
	  ls->type_ = CTypes::TUPLE;
	  sttc::check(i, '[');
	  char c;
	  std::vector<labelset_description> vecls;
	  do {
	    ls->children_.emplace_back(parse_labelset(i));
	    i >> c;
	  } while(c==',');
	  if(c!=']')
	    throw std::runtime_error("json: Elements ]");
	  if(sttc::peek(i)!='}') {
	    std::string al = sttc::parsestring(i);
	    if(al=="Epsilon transitions") {
	      sttc::check(i,':');
	      if(sttc::parsecst(i)=='t') {
		labelset_description ls2= make_labelset_description();
		ls2->type_ = CTypes::NULLABLE;
		ls2->children_.emplace_back(ls);
		return ls2;
	      }
	    }
	  }
	}
	return ls;
      }
      std::string ltype = sttc::parsestring(i);
      if(ltype == "No Label") {
	ls->type_ = CTypes::ONESET;
	sttc::check(i, '}');
	return ls;
      }
      if(ltype == "Words") {
	ls->type_ = CTypes::WORDSET;
	std::string al = sttc::parsestring(i);
	if(al!="Alphabet")
	  throw std::runtime_error("json: Alphabet");
	sttc::check(i, ':');
	sttc::check(i, '[');
	char c;
	do {
	  al = sttc::parsestring(i);
	  ls->alphabet.emplace_back(al);
	  i >> c;
	} while(c==',');
	if(c!=']')
	  throw std::runtime_error("json: Alphabet ]");
	sttc::check(i, '}');
	return ls;
      }
      if(ltype == "Letters" || ltype == "Int Letters") {
	bool eps=false;
	if(ltype == "Letters")
	  ls->type_ = CTypes::LETTERSET;
	else
	  ls->type_ = CTypes::INTLETTERSET;
	std::string al = sttc::parsestring(i);
	if(al=="Epsilon transitions") {
	  sttc::check(i,':');
	  if(sttc::parsecst(i)=='t')
	    eps=true;
	  al = sttc::parsestring(i);
	}
	if(al!="Alphabet")
	  throw std::runtime_error("json: Alphabet");
	sttc::check(i, ':');
	sttc::check(i, '[');
	char c;
	do {
	  al = sttc::parsestring(i);
	  ls->alphabet.emplace_back(al);
	  i >> c;
	} while(c==',');
	if(c!=']')
	  throw std::runtime_error("json: Alphabet ]");
	sttc::check(i, '}');
	if(eps) {
	  labelset_description ls2= make_labelset_description();
	  ls2->type_ = CTypes::NULLABLE;
	  ls2->children_.emplace_back(ls);
	  return ls2;
	}
	return ls;
      }
      throw std::runtime_error("json: Labels");
    }

    labelset_description parse_labels(std::istream& i) {
      std::string aut = sttc::get_first_attr(i);
      if(aut != "Labels")
	throw std::runtime_error("json: Labels");
      labelset_description ls=parse_labelset(i);
      sttc::check(i, '}');
      return ls;
    }

    weightset_description parse_weights(std::istream& i) {
      std::string aut = sttc::get_first_attr(i);
      if(aut != "Weights")
	throw std::runtime_error("json: Weights");
      weightset_description ws=parse_weightset(i);
      sttc::check(i, '}');
      return ws;
    }

    context_description parse_context(std::istream& i) {
      std::string aut = sttc::get_first_attr(i);
      if(aut != "Context")
	throw std::runtime_error("json: Context");
      context_description ct= make_context_description();
      sttc::check(i, '[');
      ct->ls_=parse_labels(i);
      sttc::check(i, ',');
      ct->ws_=parse_weights(i);
      sttc::check(i, ']');
      sttc::check(i, '}');
      return ct;
    }



    //******************

    std::string tostring(labelset_description ls, bool dynamic){
      std::string res;
      switch(ls->type_){
      case CTypes::ONESET: return "lao";
      case CTypes::LETTERSET:
	res="lal_char";
	if(dynamic) {
	  res+="(";
	  for(auto l: ls-> alphabet)
	    res+=l;
	  res+=")";
	}
	return res;
      case CTypes::INTLETTERSET:
	res="lal_int";
	if(dynamic) {
	  res+="(";
	  for(auto l: ls-> alphabet)
	    res+=l;
	  res+=")";
	}
	return res;
      case CTypes::WORDSET:
	res="law_char";
	if(dynamic) {
	  res+="(";
	  for(auto l: ls-> alphabet)
	    res+=l;
	  res+=")";
	}
	return res;
      case CTypes::NULLABLE: return "lan<"+tostring(ls->children_.front(), dynamic)+">";
      case CTypes::TUPLE:
	{
	  res="lat";
	  char s='<';
	  for(auto l:ls->children_) {
	    res+= s+tostring(l, dynamic);
	    s=',';
	  }
	  return res+">";
	}
      default: break;
      }
      return "";
    }

    std::string tostring(context_description ct, bool dynamic);

    std::string tostring(weightset_description ws, bool dynamic){
      std::string r="tuple";
      std::string sep="<";
      if(ws->type_ >= 0)
	return instances()[ws->type_]->tostring(ws, dynamic);
      switch(ws->type_){
      case WTypes::RATEXP: return "ratexpset<"+tostring(ws->ct_,dynamic)+">";
      case WTypes::SERIES: return "series<"+tostring(ws->ct_, dynamic)+">";
      case WTypes::TUPLE:
	for(auto wsc : ws->children_) {
	  r+=sep+tostring(wsc, dynamic);
	  sep=",";
	}
	return r+">";
      default: break;
      }
      return "";
    }

    std::string tostring(context_description ct, bool dynamic) {
      return tostring(ct->ls_, dynamic)+"_"+tostring(ct->ws_, dynamic);
    }

    std::vector<std::string> all_weightset_public_static_names() {
      std::vector<std::string> result;
      for (auto ws : instances())
        result.push_back(std::string(ws->static_public_name()));
      return result;
    }
    std::string all_weightset_public_static_names_as_string() {
      std::stringstream ss;
      ss << "[";
      bool b = false;
      for (auto ws : instances()) {
        if (b) 
          ss << ", ";
        else
          b= true;
        ss << ws->static_public_name();
      }
      ss << "]";
      return ss.str();
    }
  
  }//end of ns awali::dyn
} //end of ns awali

#endif
