// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef COMMON_QFRACTION_HH
#define COMMON_QFRACTION_HH

#include <awali/utils/arith.hh>
#include <awali/utils/hash.hh>
#include <complex>

namespace awali {

  struct q_fraction_t
  {
    int num;
    unsigned int den;

    q_fraction_t(int n = 0, unsigned d = 1)
    : num(n)
    , den(d)
    {}

    // Puts the fraction in normal form.
    q_fraction_t& reduce()
    {
      int gc = utils::gcd(abs(num), den);
      num /= gc;
      den /= gc;
      return *this;
    }

    bool operator==(const q_fraction_t& w) const {
      return num * w.den == w.num * den;
    }

    bool operator<(const q_fraction_t& w) const {
      return this->num * w.den < w.num * this->den;
    }

  };

  std::ostream& operator<<(std::ostream& o, const q_fraction_t& v){
    o << v.num << '/' << v.den;
    return o;
  }

}

namespace std {
  //hash_function

  bool operator< (const std::complex<double>& lhs, const std::complex<double>& rhs) {
    if(std::norm(lhs) == std::norm(rhs))
      return std::arg(lhs) < std::arg(rhs);
    return std::norm(lhs) < std::norm(rhs);
  }

  std::ostream& operator<<(std::ostream& o, const std::complex<double>& v){
    if(v.imag()==0)
      o << v.real();
    else {
      if(v.real()==0)
	o << v.imag() << 'i';
      else
	o << v.real() << '+' << v.imag() << 'i';
    }
    return o;
  }

  template<>
  struct hash<awali::q_fraction_t> {
    size_t operator()(const awali::q_fraction_t& f) const
    {
	  size_t res = 0;
	  std::hash_combine(res, awali::utils::hash_value(f.num));
	  std::hash_combine(res, awali::utils::hash_value(f.den));
	  return res;
    }
  };

  template<typename T>
  struct hash<std::complex<T>> {
    size_t operator()(const std::complex<T>& c) const
    {
	  size_t res = 0;
	  std::hash_combine(res, awali::utils::hash_value(c.real()));
	  std::hash_combine(res, awali::utils::hash_value(c.imag()));
	  return res;
    }
  };

}//end of ns awali

#endif
