// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_ALGOS_SUM_HH
# define AWALI_ALGOS_SUM_HH

# include <map>

#include <awali/sttc/algos/product.hh> // join_automata
#include <awali/sttc/algos/standard.hh> // is_standard
#include <awali/sttc/ctx/traits.hh>
#include <awali/sttc/misc/raise.hh> // require

namespace awali { namespace sttc {


  /*----------------------------.
  | sum(automaton, automaton).  |
  `----------------------------*/

  /// Merge transitions of \a b into those of \a res.
  ///
  /// \pre The context of \a res must include that of \a b.
  /// \pre res and b must be standard.
  template <typename A, typename B>
  A&
  sum_here(A& res, const B& b)
  {
    require(is_standard(res), __func__, ": lhs must be standard");
    require(is_standard(b), __func__, ": rhs must be standard");

    // State in B -> state in Res.
    std::map<state_t, state_t> m;
    state_t initial = res->dst_of(*(res->initial_transitions().begin()));
    for (auto s: b->states())
      m.emplace(s, b->is_initial(s) ? initial : res->add_state());
    m.emplace(b->pre(), res->pre());
    m.emplace(b->post(), res->post());

    // Add b.
    for (auto t: b->all_transitions())
      // Do not add initial transitions, the unique initial state is
      // already declared as such, and its weight must remain 1.
      if (b->src_of(t) != b->pre())
        {
          if (b->dst_of(t) == b->post())
            res->add_transition(m[b->src_of(t)], m[b->dst_of(t)],
                               b->label_of(t), b->weight_of(t));
          else
            res->new_transition(m[b->src_of(t)], m[b->dst_of(t)],
                               b->label_of(t), b->weight_of(t));
        }
    return res;
  }


  template <typename A, typename B>
  inline
  auto
  sum(const A& lhs, const B& rhs)
    -> decltype(join_automata(lhs, rhs))
  {
    auto res = join_automata(lhs, rhs);
    // A standard automaton has a single initial state.
    res->set_initial(res->add_state());
    sum_here(res, lhs);
    sum_here(res, rhs);
    return res;
  }

  /*------------------------------.
  | sum(polynomial, polynomial).  |
  `------------------------------*/

  /// Sums of values.
  template <typename ValueSet>
  inline
  typename ValueSet::value_t
  sum(const ValueSet& vs,
      const typename ValueSet::value_t& lhs,
      const typename ValueSet::value_t& rhs)
  {
    return vs.add(lhs, rhs);
  }

}}//end of ns awali::stc

#endif // !AWALI_ALGOS_SUM_HH
