// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_ALGOS_JS_PARSER_HH
# define AWALI_ALGOS_JS_PARSER_HH

#include <awali/sttc/misc/raise.hh>
#include <awali/sttc/misc/json.hh>
#include <awali/sttc/misc/add-epsilon-trans.hh>
#include <awali/sttc/core/rat/ratexpset.hh>
#include <awali/sttc/core/rat/ratexp.hh>
#include <awali/sttc/algos/print_exp.hh>
#include <awali/sttc/history/string_history.hh>
# include <stdexcept>
# include <stack>
# include <iostream>
# include <sstream>

namespace awali { namespace sttc {


  namespace internal {

    template<typename RatExpSet>
    struct js_exp_parser {
      using ratexpset_t = RatExpSet;
      using context_t = context_t_of<ratexpset_t>;
      using labelset_t = labelset_t_of<context_t>;
      using ratexp_t = typename ratexpset_t::value_t;
      using weightset_t = weightset_t_of<ratexpset_t>;

      js_exp_parser(const ratexpset_t& rs, std::istream& i)
        : rs_(rs), i_(i)
      {}

      ratexp_t parseNode(const std::string& key) {
	ratexp_t e;
	char c;
	if(key == "Sum") {
	  bool first=true;
	  check(i_,'[');
	  do {
	    ratexp_t f= parseNode();
	    if(first) {
	      e=f;
	      first=false;
	    }
	    else {
	      e=rs_.add(e,f);
	    }
	    i_ >> c;
	  } while (c==',');
	  if(c!=']')
	    throw std::runtime_error("Json parser: ratexp");
	}
	else if(key == "Prod") {
	  bool first=true;
	  check(i_,'[');
	  do {
	    ratexp_t f= parseNode();
	    if(first) {
	      e=f;
	      first=false;
	    }
	    else
	      e=rs_.mul(e,f);
	    i_ >> c;
	  } while (c==',');
	  if(c!=']')
	    throw std::runtime_error("Json parser: ratexp");
	}
	else if(key == "Star") {
	  e= rs_.star(parseNode());
	}
	else if(key == "Label") {
	  e= rs_.atom(ls_.js_parse(i_));
	}
	else if(key == "LWeight") {
	  auto w = ws_.js_parse(i_);
	  std::string key2 = parsestring(i_);
	  check(i_,':');
	  e=rs_.lmul(w,parseNode(key2));
	}
	else if(key == "RWeight") {
	  auto w = ws_.js_parse(i_);
	  std::string key2 = parsestring(i_);
	  check(i_,':');
	  e=rs_.rmul(parseNode(key2),w);
	}
	else if(key == "One") {
	  char c=parsecst(i_);
	  if(c!='n')
	    throw std::runtime_error("Json parser: ratexp");
	  e = rs_.one();
	}
	else if(key == "Zero") {
	  char c=parsecst(i_);
	  if(c!='n')
	    throw std::runtime_error("Json parser: ratexp");
	  e = rs_.zero();
	}
	else
	  throw std::runtime_error("Json parser: ratexp");
	return e;
      }

      ratexp_t parseNode() {
	std::string key = get_first_attr(i_);
	ratexp_t e=parseNode(key);
	check(i_,'}');
	return e;
      }

    private:
      const ratexpset_t& rs_;
      std::istream& i_;
      weightset_t ws_ = *rs_.weightset();
      labelset_t ls_ = *rs_.labelset();

    };
  }

  template <typename RatExpSet>
  inline
  typename RatExpSet::ratexp_t
  js_parse_exp_content(const RatExpSet& rs,
		       std::istream& i)
  {
    internal::js_exp_parser<RatExpSet> parser{rs,i};
    return parser.parseNode();
  }

  template <typename Context>
  mutable_automaton<Context>
  js_parse_aut_content(const Context& context, std::istream& i) {
    // using aut_t = mutable_automaton<Context>;
    auto ws = context.weightset();
    auto ls = context.labelset();
    mutable_automaton<Context> aut = make_mutable_automaton(context);
    char c;
    std::unordered_map<unsigned,state_t> states;
    unsigned s;
    std::string key = get_first_attr(i);
    if(key != "Content")
      throw std::runtime_error("json: Content");
    check(i, '[');
    key = get_first_attr(i);
    if(key != "States")
      throw std::runtime_error("json: States");
    check(i, '[');
    do {
      if(peek(i)==']') { //no state
	i >> c;
	break;
      }
      key =get_first_attr(i);
      if(key != "Id")
	throw std::runtime_error("json: State Id");
      s=parseint(i);
      states[s]=aut->add_state();
      key=parsestring(i);
      if(key != "Name")
	throw std::runtime_error("json: Name");
      check(i, ':');
      key=parsestring(i);
      aut->set_state_name(states[s],key);
      if(peek(i)!='}') {
	std::string tmp=parsestring(i);
	check(i, ':');
	if(tmp == "History") {
	  key=parsestring(i);
	  auto history=aut->history();
	  if(history->get_nature() == history_kind_t::NO_HISTORY) {
	    history=std::make_shared<string_history>();
	    aut->set_history(history);
	  }
	  auto& hs = dynamic_cast<sttc::string_history&>(*history);
	  hs.add_state(states[s], key);
	}
	else
	  parseignore(i);
      }
      check(i, '}');
      i >> c;
    } while(c==',');
    if(c!=']')
      throw std::runtime_error("json: States ]");
    check(i, '}');
    if(peek(i)==']') {
      check(i, ']');
      check(i, '}');
      return aut;
    }
    check(i, ',');
    key = get_first_attr(i);
    if(key == "Initial States") {
      check(i, '[');
      do {
	if(peek(i)!='{') {
	  s=parseint(i);
	  aut->set_initial(states[s]);
	}
	else {
	  key =get_first_attr(i);
	  if(key != "Id")
	    throw std::runtime_error("json: IState Id");
	  s=parseint(i);
	  if(peek(i)=='"') {
	    key=parsestring(i);
	    typename Context::labelset_t::value_t l;
	    bool has_label=false;
	    if(key == "Label") {
	      has_label=true;
	      check(i, ':');
	      l=ls->js_parse(i);
	      if(peek(i)!='"') {
		aut->new_transition(states[s], aut->post(), l);
		check(i, '}');
		i >> c;
		continue;
	      }
	    }
	    if(key != "Weight")
	      throw std::runtime_error("json: FWeight");
	    check(i, ':');
	    auto w=ws->js_parse(i);
	    if(has_label)
	      aut->new_transition(states[s], aut->post(), l, w);
	    else
	      aut->set_initial(states[s], w);
	  }
	  else
	    aut->set_initial(states[s]);
	  check(i, '}');
	}
	i >> c;
      } while(c==',');
      if(c!=']')
	throw std::runtime_error("json: IStates ]");
      check(i, '}');
      if(peek(i)==']') {
	check(i, ']');
	check(i, '}');
	check(i, ']');
	check(i, '}');
	return aut;
      }
      check(i,',');
      key=get_first_attr(i);
    }
    if(key == "Final States") {
      check(i, '[');
      do {
	if(peek(i)!='{') {
	  s=parseint(i);
	  aut->set_final(states[s]);
	}
	else {
	  key =get_first_attr(i);
	  if(key != "Id")
	    throw std::runtime_error("json: FState Id");
	  s=parseint(i);
	  if(peek(i)=='"') {
	    key=parsestring(i);
	    typename Context::labelset_t::value_t l;
	    bool has_label=false;
	    if(key == "Label") {
	      has_label=true;
	      check(i, ':');
	      l=ls->js_parse(i);
	      if(peek(i)!='"') {
		aut->new_transition(states[s], aut->post(), l);
		check(i, '}');
		i >> c;
		continue;
	      }
	    }
	    if(key != "Weight")
	      throw std::runtime_error("json: FWeight");
	    check(i, ':');
	    auto w=ws->js_parse(i);
	    if(has_label)
	      aut->new_transition(states[s], aut->post(), l, w);
	    else
	      aut->set_final(states[s], w);
	  }
	  else
	    aut->set_final(states[s]);
	  check(i, '}');
	}
	i >> c;
      } while(c==',');
      if(c!=']')
	throw std::runtime_error("json: FStates ]");
      check(i, '}');
      if(peek(i)==']') {
	check(i, ']');
	check(i, '}');
	check(i, ']');
	check(i, '}');
	return aut;
      }
      check(i,',');
      key=get_first_attr(i);
    }
    if(key == "Transitions") {
      unsigned t;
      check(i, '[');
      do {
	if(peek(i)==']') { //no transition
	  i >> c;
	  break;
	}
	key =get_first_attr(i);
	if(key == "Id") {
	  parseint(i);
	  key=parsestring(i);
	  check(i, ':');
	}
	if(key != "Src")
	  throw std::runtime_error("json: Src");
	s=parseint(i);
	key=parsestring(i);
	check(i, ':');
	if(key != "Dst")
	  throw std::runtime_error("json: Dst");
	t=parseint(i);
	if(peek(i)!='"') {
	  new_epsilon_trans(aut, states[s], states[t]);
	  check(i, '}');
	  i >> c;
	  continue;
	}
	key=parsestring(i);
	check(i, ':');
	typename Context::labelset_t::value_t l;
	bool has_label=false;
	if(key == "Label") {
	  l = ls->js_parse(i);
	  has_label=true;
	  if(peek(i)!='"') {
	    aut->new_transition(states[s], states[t],l);
	    check(i, '}');
	    i >> c;
	    continue;
	  }
	  key=parsestring(i);
	  check(i, ':');
	}
	if(key != "Weight")
	  throw std::runtime_error("json: Weight");
	auto w  = ws->js_parse(i);
	if(has_label)
	  aut->new_transition(states[s], states[t],l,w);
	else
	  new_epsilon_trans(aut, states[s], states[t], w);
	check(i, '}');
	i >> c;
      } while(c==',');
      if(c!=']')
	throw std::runtime_error("json: Trans ]");
      check(i, '}');// end object Transition
      check(i, ']');// end array Content
      check(i, '}');// end objet Content
    }
    return aut;
  }


}}//end of ns awali::stc

#endif // !AWALI_ALGOS_JS_PARSER_HH
