// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef AWALI_ALGOS_ARE_EQUIVALENT_HH
# define AWALI_ALGOS_ARE_EQUIVALENT_HH

#include <awali/common/priority.hh>

#include <awali/sttc/algos/accessible.hh> // is_useless
#include <awali/sttc/algos/complement.hh>
#include <awali/sttc/algos/complete.hh>
#include <awali/sttc/algos/determinize.hh>
#include <awali/sttc/algos/left-mult.hh>
#include <awali/sttc/algos/product.hh>
#include <awali/sttc/algos/reduce.hh>
#include <awali/sttc/algos/union.hh>
#include <awali/sttc/algos/lal-lan-conversion.hh>

namespace awali { namespace sttc {


  /*---------------------------------------.
  | are_equivalent(automaton, automaton).  |
  `---------------------------------------*/

  /// Check equivalence between Boolean automata on a free labelset.
  template <typename Aut1, typename Aut2, typename P>
  auto
  are_equivalent_(const Aut1& a1, const Aut2& a2, priority::FIVE<P>)
    -> typename std::enable_if<(labelset_t_of<Aut1>::is_free()
                                && std::is_same<weightset_t_of<Aut1>, b>::value
                                && labelset_t_of<Aut2>::is_free()
                                && std::is_same<weightset_t_of<Aut2>, b>::value),
                               bool>::type
  {
    return (   is_useless(difference(a1, a2))
            && is_useless(difference(a2, a1)));
  }


  /// Check equivalence between Boolean automata on fields, or Z.
  template <typename Aut1, typename Aut2, typename T>
  auto
  are_equivalent_(const Aut1& a1, const Aut2& a2, priority::FOUR<T>)
    -> decltype((a1->weightset()->sub( a1->weightset()->zero(),
                                       a1->weightset()->one()   ),
                 true))
  {
    const auto& ws2 = *a2->weightset();
    // d = a1 U -a2.
    auto d = union_a(a1,
                     left_mult(a2, ws2.sub(ws2.zero(), ws2.one())));
    return is_empty(reduce(d));
  }


  template <typename Aut1, typename Aut2, typename P>
  bool
  are_equivalent_(const Aut1& a1, const Aut2& a2, priority::ONE<P>)
  {
    throw std::runtime_error("are_equivalent is only supported for weigh-sets with subtraction and for boolean automata over a free label-sets.");
  }

  /*-----------------------------------.
  | difference(automaton, automaton).  |
  `-----------------------------------*/

  /// An automaton that computes weights of \a lhs, but not by \a rhs.
  template <typename Lhs, typename Rhs>
  typename Lhs::element_type::automaton_nocv_t
  difference(const Lhs& lhs, const Rhs& rhs)
  {
    // Meet complement()'s requirements.
    auto r = rhs;
    if (!is_deterministic(r))
      r = complete(determinize(r,false));
    else if (!is_complete(r))
      r = complete(r);
    return product(lhs, complement(r));
  }

  template <typename Aut1, typename Aut2>
  bool are_equivalent(const Aut1& a1, const Aut2& a2)
  {
    return are_equivalent_(to_lal(a1), to_lal(a2),priority::value);
  }

}}//end of ns awali::stc

#endif // !AWALI_ALGOS_ARE_EQUIVALENT_HH
