// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef DYN_MODULES_TRANSDUCER_CC
#define DYN_MODULES_TRANSDUCER_CC

#include <awali/dyn/modules/automaton.hh>
#include <awali/dyn/modules/proper.hh>

namespace awali {
  namespace dyn {

    automaton_t make_transducer(std::vector<std::string> alphabets, std::string semiring) {
      std::vector<labelset_description> v;
      for(auto al : alphabets)
	v.emplace_back(nullableset(letterset(al)));
      return make_automaton_from_context(ltupleset(v),weightset(semiring));
    }

    automaton_t make_transducer(std::vector<std::string> alphabets) {
      return make_transducer(alphabets, "B");
    }

    unsigned num_tapes(automaton_t tdc) {
      return loading::call<unsigned, &num_tapes>("num_tapes", "transducer", tdc);
    }

    transition_t set_tdc_transition(automaton_t tdc, state_t src, state_t dst, const std::vector<std::string>& labels) {
      return loading::call4<transition_t, state_t, state_t, const std::vector<std::string>&, &set_tdc_transition>("set_tdc_transition", "transducer", tdc, src, dst, labels);
    }

    transition_t get_tdc_transition(automaton_t tdc, state_t src, state_t dst, const std::vector<std::string>& labels) {
      return loading::call4<transition_t, state_t, state_t, const std::vector<std::string>&, &get_tdc_transition>("get_tdc_transition", "transducer", tdc, src, dst, labels);
    }

    bool has_tdc_transition(automaton_t tdc, state_t src, state_t dst, const std::vector<std::string>& labels) {
      return loading::call4<bool, state_t, state_t, const std::vector<std::string>&, &has_tdc_transition>("has_tdc_transition", "transducer", tdc, src, dst, labels);
    }

    weight_t add_tdc_transition(automaton_t tdc, state_t src, state_t dst, const std::vector<std::string>& labels) {
      return loading::call4<weight_t, state_t, state_t, const std::vector<std::string>&, &add_tdc_transition>("add_tdc_transition", "transducer", tdc, src, dst, labels);
    }

    void del_tdc_transition(automaton_t tdc, state_t src, state_t dst, const std::vector<std::string>& labels) {
      loading::call4<void, state_t, state_t, const std::vector<std::string>&, &del_tdc_transition>("del_tdc_transition", "transducer", tdc, src, dst, labels);
    }

    transition_t set_tdc_transition(automaton_t tdc, state_t src, state_t dst, const std::vector<std::string>& labels, weight_t w) {
      std::string stat_ctx = tdc->get_context()->sname();
      typedef transition_t (*bridge_t)(automaton_t,  state_t , state_t , const std::vector<std::string>& , weight_t);
      static std::unordered_map<std::string, bridge_t> bridges;
      auto it = bridges.find(stat_ctx);
      if(it == bridges.end()) {
	auto bridge = (bridge_t) loading::get_handler("set_tdc_wtransition", "transducer", stat_ctx);
	bridges.emplace(stat_ctx, bridge);
	return bridge(tdc, src, dst, labels, w);
      }
      else {
	return it->second(tdc, src, dst, labels, w);
      }
    }

    weight_t add_tdc_transition(automaton_t tdc, state_t src, state_t dst, const std::vector<std::string>& labels, weight_t w) {
      std::string stat_ctx = tdc->get_context()->sname();
      typedef weight_t (*bridge_t)(automaton_t,  state_t , state_t , const std::vector<std::string>& , weight_t);
      static std::unordered_map<std::string, bridge_t> bridges;
      auto it = bridges.find(stat_ctx);
      if(it == bridges.end()) {
	auto bridge = (bridge_t) loading::get_handler("add_tdc_wtransition", "transducer", stat_ctx);
	bridges.emplace(stat_ctx, bridge);
	return bridge(tdc, src, dst, labels, w);
      }
      else {
	return it->second(tdc, src, dst, labels, w);
      }
    }

    std::vector<std::string> get_tdc_label(automaton_t tdc, transition_t tr) {
      return loading::call2<std::vector<std::string>, transition_t, &get_tdc_label>("get_tdc_label", "transducer", tdc, tr);
    }

    std::vector<state_t> tdc_successors(automaton_t tdc, state_t s, const std::vector<std::string>& label) {
      return loading::call3<std::vector<state_t>, state_t, const std::vector<std::string>&, &tdc_successors>("tdc_successors", "transducer", tdc, s, label);
    }

    std::vector<state_t> tdc_predecessors(automaton_t tdc, state_t s, const std::vector<std::string>& label) {
      return loading::call3<std::vector<state_t>, state_t, const std::vector<std::string>&, &tdc_predecessors>("tdc_predecessors", "transducer", tdc, s, label);
    }

    std::vector<transition_t> tdc_in(automaton_t tdc, state_t s, const std::vector<std::string>& label) {
      return loading::call3<std::vector<transition_t>, state_t, const std::vector<std::string>&, &tdc_in>("tdc_in", "transducer", tdc, s, label);
    }

    std::vector<transition_t> tdc_out(automaton_t tdc, state_t s, const std::vector<std::string>& label) {
      return loading::call3<std::vector<transition_t>, state_t, const std::vector<std::string>&, &tdc_out>("tdc_out", "transducer", tdc, s, label);
    }

    std::vector<char> input_alphabet(automaton_t tdc) {
      return loading::call<std::vector<char>, &input_alphabet>("input_alphabet", "transducer", tdc);
    }

    std::vector<char> output_alphabet(automaton_t tdc) {
      return loading::call<std::vector<char>, &output_alphabet>("output_alphabet", "transducer", tdc);
    }

    void set_final_output(automaton_t tdc, state_t src, const std::string& output) {
      loading::call3<void, state_t, const std::string&, &set_final_output>("set_final_output", "transducer", tdc, src, output);
    }

    std::string get_final_output(automaton_t tdc, state_t src) {
      return loading::call2<std::string, state_t, &get_final_output>("get_final_output", "transducer", tdc, src);
    }


    std::vector<std::vector<char>> alphabets(automaton_t tdc) {
      return loading::call<std::vector<std::vector<char>>, &alphabets>("alphabets", "transducer", tdc);
    }

    bool has_label(automaton_t tdc, unsigned i, std::string l) {
      auto v=alphabets(tdc);
      auto al=v[i];
      for(auto k: al)
	if(k==l[0])
	  return true;
      return false;
    }

    bool has_input_label(automaton_t tdc, std::string l) {
      return has_label(tdc, 0, l);
    }

    bool has_output_label(automaton_t tdc, std::string l) {
      return has_label(tdc, 1, l);
    }

    automaton_t domain(automaton_t tdc) {
      return loading::call<automaton_t, &domain>("domain", "transducer", tdc);
    }

    automaton_t image(automaton_t tdc) {
      return loading::call<automaton_t, &image>("image", "transducer", tdc);
    }

    automaton_t images(automaton_t tdc) {
      return loading::call<automaton_t, &image>("images", "transducer", tdc);
    }

    automaton_t projection(automaton_t tdc, unsigned i) {
      if(i==0)
	return domain(tdc);
      else if(i==1)
	return image(tdc);
      else return projection(images(tdc),i-1);
    }

    automaton_t inverse(automaton_t tdc) {
      return loading::call<automaton_t, &inverse>("inverse", "transducer", tdc);
    }

    automaton_t compose(automaton_t tdc1, automaton_t tdc2) {
      return loading::call_mixte<automaton_t, &compose>("compose", "compose", tdc1, tdc2);
    }

    bool is_functional(automaton_t tdc) {
      return loading::call<bool, &is_functional>("is_functional", "transducer", tdc);
    }

    automaton_t lift_tdc(automaton_t tdc) {
      return loading::call<automaton_t , &lift_tdc>("lift_tdc", "transducer", tdc);
    }

    bool is_synchronizable(automaton_t tdc) {
      return loading::call<bool, &is_synchronizable>("is_synchronizable", "transducer", tdc);
    }

    automaton_t synchronize(automaton_t tdc) {
      return loading::call<automaton_t, &synchronize>("synchronize", "transducer", tdc);
    }

    automaton_t realtime(automaton_t tdc) {
      return loading::call<automaton_t, &realtime>("realtime", "transducer", tdc);
    }

  }
}//end of ns awali::dyn
#endif
