// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include <awali/dyn/modules/determinize.hh>

namespace awali {
  namespace dyn {

    automaton_t min_quotient_det(automaton_t aut, param_t dir) {
      return loading::call2<automaton_t, param_t, &min_quotient_det>("min_quotient_det", "quotient", aut, dir);
    }

    automaton_t min_quotient(automaton_t aut, param_t dir) {
      try {
	if(is_deterministic(aut))
	  return min_quotient_det(aut, dir);
      }
      catch(...) {}
      return loading::call2<automaton_t, param_t, &min_quotient>("min_quotient", "quotient", aut, dir);
    }

    bool is_quotient(automaton_t aut1, automaton_t aut2) {
      return loading::call2<bool, automaton_t, &is_quotient>("is_quotient", "quotient", aut1, aut2);
    }

    static void normalize_equiv(automaton_t aut, std::vector<std::vector<state_t>>& equiv) {
      std::unordered_set<state_t> m;
      unsigned count =0;
      for(auto v: equiv) {
	count += v.size();
	for(auto s: v) {
	  if(!aut->has_state(s))
	    throw std::runtime_error("Quotient: Wrong state in partition");
	  if(m.find(s)!=m.end())
	    throw std::runtime_error("Quotient: Duplicate state in partition");
	  m.emplace(s);
	}
      }
      for(auto s: aut->all_states())
	if(m.find(s)==m.end()) {
	  std::vector<state_t> vs{s};
	  equiv.emplace_back(vs);
	}
    }

    automaton_t merge(automaton_t aut, std::vector<std::vector<state_t>>& equiv) {
      normalize_equiv(aut, equiv);
      return loading::call2<automaton_t, std::vector<std::vector<state_t>>&, &merge>("merge", "quotient", aut, equiv);
    }

    static bool is_congruence_exact(automaton_t aut, std::vector<std::vector<state_t>>& equiv) {
      return loading::call2<bool, std::vector<std::vector<state_t>>&, &is_congruence_exact>("is_congruence", "quotient", aut, equiv);
    }

    automaton_t quotient(automaton_t aut, std::vector<std::vector<state_t>>& equiv) {

      normalize_equiv(aut, equiv);
      if(is_congruence_exact(aut,equiv))
	return merge(aut, equiv);
      throw std::runtime_error("Quotient: The equivalence is not a congruence");
    }

    bool is_congruence(automaton_t aut, std::vector<std::vector<state_t>>& equiv) {
      normalize_equiv(aut, equiv);
      return is_congruence_exact(aut,equiv);
    }

  }

}//end of ns awali::dyn
