// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include<cstdlib>
#include<iostream>
#include <vector>
#include <map>

#include <dirent.h>
#include <sys/stat.h>
#include <errno.h>
#include <awali/dyn/config.hh>

namespace awali { namespace dyn {
  namespace loading {
    std::vector<std::string> get_lib_directory() {
      std::string local(std::getenv("HOME"));
      local.append("/.awali/otf");

      #define STR_VALUE(arg)      #arg
      #define STR(name) STR_VALUE(name)
      #ifdef AWALI_INST_OTF_LIB_PATH
      std::string global =std::string(STR(AWALI_INST_OTF_LIB_PATH));
      #undef STR
      #undef STR_VALUE
      #else
      throw std::runtime_error("Installed Static-libraries directory not set.");
      #endif
      #undef STR
      #undef STR_VALUE

      return {local,global};
    }

    std::vector<std::string> get_dynlib_directory() {
      #define STR_VALUE(arg)      #arg
      #define STR(name) STR_VALUE(name)

      #ifdef AWALI_DYNLIB_PATH
      std::string local = std::string(STR(AWALI_DYNLIB_PATH));
      #else
      throw std::runtime_error("Dynamic-library directory not set.");
      #endif

      #ifdef AWALI_INST_DYNLIB_PATH
      std::string global = std::string(STR(AWALI_INST_DYNLIB_PATH));
      #else
      throw std::runtime_error("Installed Dynamic-library directory not set.");
      #endif

      return {local,global};
    }

    std::vector<std::string> get_share_directory() {
      #define STR_VALUE(arg)      #arg
      #define STR(name) STR_VALUE(name)
      #ifdef AWALI_SHARE_PATH
      std::string local(STR(AWALI_SHARE_PATH));
      #else
      throw std::runtime_error("Share directory not set.");
      #endif
      #ifdef AWALI_INST_SHARE_PATH
      std::string global(STR(AWALI_INST_SHARE_PATH));
      #else
      throw std::runtime_error("Installed share directory not set.");
      #endif

      #undef STR
      #undef STR_VALUE

      return {local, global};
    }

    std::vector<std::string> get_include_directory() {
      #define STR_VALUE(arg)      #arg
      #define STR(name) STR_VALUE(name)

      #ifdef AWALI_INCLUDE_PATH
      std::string local(STR(AWALI_INCLUDE_PATH));
      #else
      throw std::runtime_error("Include directory not set.");;
      #endif

      #ifdef AWALI_INCLUDE_PATH
      std::string global(STR(AWALI_INST_INCLUDE_PATH));
      #else
      throw std::runtime_error("Include directory not set.");;
      #endif

      #undef STR
      #undef STR_VALUE

      return {local, global};
    }

//     std::string get_source_directory() {
//       #ifdef AWALI_SOURCE_PATH
//       #define STR_VALUE(arg)      #arg
//       #define STR(name) STR_VALUE(name)
//       return std::string(STR(AWALI_SOURCE_PATH));
//       #undef STR
//       #undef STR_VALUE
//       #else
//       throw std::runtime_error("Source directory not set.");
//       #endif
//     }

//     std::string get_awali_directory() {
//       #ifdef AWALI_PATH
//       #define STR_VALUE(arg)      #arg
//       #define STR(name) STR_VALUE(name)
//       return std::string(STR(AWALI_PATH));
//       #undef STR
//       #undef STR_VALUE
//       #else
//       const char* dir=std::getenv("AWALI_PATH");
//       if(dir==NULL) {
//         std::cerr << "Environment variable AWALI_PATH is not set" << std::endl;
//         std::cerr << "Please define : export AWALI_PATH=[...]/awali" << std::endl;
//         exit(0);
//       }
//       return std::string(dir);
//       #endif
//     }

    int make_awali_dir() {
      struct stat st;
      std::string dir1= get_lib_directory()[0];
      std::string dir2= get_compilation_directory("")[0];
      std::string surdir= dir1.substr(0,dir1.length()-4); //removes '/otf'
      for(std::string dir : {surdir,dir1,dir2}) {
        if(stat(dir.data(), &st)==-1)
        {
          if(errno == ENOENT) {
            if(mkdir(dir.data(), S_IRWXU)!=0) {
              std::cerr << "Unable to create directory :"
                        << dir << std::endl;
              return -1;
            }

          }
          else {
            std::cerr << "Something wrong with this directory name:"
                      << dir << std::endl;
            return -1;
          }
        }
        else if (!S_ISDIR(st.st_mode)) {
          std::cerr << "The following file exists but is not a directory:"
                    << dir << std::endl;
          return -1;
        }
      }
      return 0;
    }

    std::vector<std::string> get_compilation_directory(std::string lname)
    {
      std::string local(std::getenv("HOME"));
      local.append("/.awali/tmp/");
      local.append(lname);
      return {local,"/tmp/"+lname};
    }

    std::vector<std::string> get_dyn_config_hh_directory() {
      #ifdef AWALI_DYN_CONFIG_HH_PATH
      #define STR_VALUE(arg)      #arg
      #define STR(name) STR_VALUE(name)
        return {std::string(STR(AWALI_DYN_CONFIG_HH_PATH))};
      #undef STR
      #undef STR_VALUE
      #else
        throw std::runtime_error("Directory of \"config.hh\" not set.");
      #endif

    }

    std::vector<file_loc_t> example_automata() {
      std::map<std::string, file_loc_t> result;
      std::vector<std::string> dirs = get_share_directory();
      for (std::string dir : dirs) {
        dir+="/awali/automata";
        DIR* d = opendir(dir.c_str());
        struct dirent * entry;
        while( (d != nullptr) && ((entry=readdir(d))!=nullptr) ) {
          std::string file (entry->d_name);
          size_t u = file.find_last_of('.');
          if ( (file.substr(u+1)=="json")) {
            std::string file_no_ext = file.substr(0,u);

            std::ifstream fic(dir+'/'+file);
            std::string desc = sttc::get_first_attr(fic);
            if(desc=="Description") {
              desc = sttc::parsestring(fic);
            } else {
              desc = "";
            }

            fic.close();
            if (result.find(file_no_ext) == result.end())
              result[file_no_ext] = file_loc_t{dir,file_no_ext,file.substr(u+1),desc};
          }

        }
      }
      std::vector<file_loc_t> real_result;
      real_result.reserve(result.size());
      for (const auto &s : result) {
        real_result.push_back(s.second);
//         std::cerr << s.second.dir << " " << s.second.name << " . " << s.second.ext << std::endl;
      }
      return real_result;
    }

  }

}}//end of ns awali::dyn
