// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

/** @file transducer.hh
 *  This files contains the definition of the transducer type for the dynamic library
 *
 *  It allows direct access to some services which are also accessible through functions
 */
#ifndef DYN_TRANSDUCER_HH
#define DYN_TRANSDUCER_HH

#include <awali/dyn/modules/transducer.hh>
#include <awali/dyn/modules/proper.hh>

namespace awali {
  namespace dyn {

    struct transducer_t { // A transducer is an automaton
    private:
      automaton_t aut_;
    public:
      transducer_t(const automaton_t& a) : aut_(a) {}

      //For compatibility with the behaviour of automata
      automaton_t operator ->() {
	return aut_;
      }

      operator automaton_t() {
	return aut_;
      }

      unsigned num_tapes() {
	return dyn::num_tapes(*this);
      }

      transition_t get_transition(state_t src, state_t dst, const std::vector<std::string>& labels) {
	return dyn::get_tdc_transition(*this, src, dst, labels);
      }

      bool has_transition(state_t src, state_t dst, const std::vector<std::string>& labels) {
	return dyn::has_tdc_transition(*this, src, dst, labels);
      }

      void set_transition(state_t src, state_t dst, const std::vector<std::string>& labels) {
	dyn::set_tdc_transition(*this, src, dst, labels);
      }

      void set_transition(state_t src, state_t dst, const std::vector<std::string>& labels, weight_t w) {
	dyn::set_tdc_transition(*this, src, dst, labels, w);
      }

      weight_t add_transition(state_t src, state_t dst, const std::vector<std::string>& labels) {
	return dyn::add_tdc_transition(*this, src, dst, labels);
      }

      weight_t add_transition(automaton_t tdc, state_t src, state_t dst, const std::vector<std::string>& labels, weight_t w) {
	return dyn::add_tdc_transition(*this, src, dst, labels, w);
      }

      void del_transition(state_t src, state_t dst, const std::vector<std::string>& labels) {
	return dyn::del_tdc_transition(*this, src, dst, labels);
      }

      std::vector<state_t> successors(state_t s, const std::vector<std::string>& label) {
	return tdc_successors(*this, s, label);
      }

      std::vector<state_t> predecessors(state_t s, const std::vector<std::string>& label) {
	return tdc_predecessors(*this, s, label);
      }

      std::vector<transition_t> in(state_t s, const std::vector<std::string>& label) {
	return tdc_in(*this, s, label);
      }

      std::vector<transition_t> out(state_t s, const std::vector<std::string>& label) {
	return tdc_out(*this, s, label);
      }

      std::vector<std::string> label_of(transition_t tr) {
	return dyn::get_tdc_label(*this, tr);
      }

      std::vector<std::vector<char>> alphabets() {
	return dyn::alphabets(*this);
      }

    };

  }
}//end of ns awali::dyn
#endif
