// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef DYN_ABSTRACT_AUTOMATON_HH
#define DYN_ABSTRACT_AUTOMATON_HH

#include<vector>
#include<iostream>
#include<string>
#include <awali/dyn/core/abstract-context.hh>
#include <awali/common/qfraction.hh>
#include<memory>

namespace awali { namespace dyn {

  using state_t = unsigned;
  using transition_t = unsigned;
  using label_t = any_t;
  using weight_t = any_t;

  struct abstract_automaton_t {
    //  Statistics
    virtual size_t num_states() const = 0;
    virtual size_t num_initials() const = 0;
    virtual size_t num_finals() const = 0;
    virtual size_t num_transitions() const = 0;

    //  States (reading)
    virtual bool has_state(state_t s) const = 0;
    virtual bool is_initial(state_t s) const = 0;
    virtual bool is_final(state_t s) const = 0;
    virtual weight_t get_initial_weight(state_t s) const = 0;
    virtual weight_t get_final_weight(state_t s) const = 0;

    //  States (writing)
    virtual state_t add_state() = 0;
    virtual state_t add_state(std::string name) = 0;
    virtual void del_state(state_t s) = 0;
    virtual void set_initial(state_t s, weight_t w) = 0;
    virtual void set_initial(state_t s) = 0;
    //return the resulting weight
    virtual weight_t add_initial(state_t s, weight_t w) = 0;
    virtual void unset_initial(state_t s) = 0;
    virtual void set_final(state_t s, weight_t w) = 0;
    virtual void set_final(state_t s) = 0;
    //return the resulting weight
    virtual weight_t add_final(state_t s, weight_t w) = 0;
    virtual void unset_final(state_t s) = 0;

    //  Transitions (reading)
    virtual transition_t get_transition(state_t src, state_t dst, label_t label) const = 0;
    virtual bool has_transition(state_t src, state_t dst, label_t label) const = 0;
    virtual bool has_transition(transition_t t) const = 0;
    virtual state_t src_of(transition_t t) const = 0;
    virtual state_t dst_of(transition_t t) const = 0;
    virtual label_t label_of(transition_t t) const = 0;
    virtual weight_t weight_of(transition_t t) const = 0;
    virtual bool is_eps_transition(transition_t t) const = 0;

    //  Transitions (writing)
    virtual transition_t set_transition(state_t src, state_t dst, label_t label, weight_t weight) = 0;
    virtual transition_t set_transition(state_t src, state_t dst, label_t label) = 0;
    virtual transition_t set_eps_transition(state_t src, state_t dst) = 0;
    virtual transition_t set_eps_transition(state_t src, state_t dst, weight_t weight) = 0;
    virtual weight_t add_transition(state_t src, state_t dst, label_t label, weight_t weight) = 0;
    virtual weight_t add_transition(state_t src, state_t dst, label_t label) = 0;
    virtual weight_t add_eps_transition(state_t src, state_t dst) = 0;
    virtual weight_t add_eps_transition(state_t src, state_t dst, weight_t weight) = 0;
    virtual weight_t set_weight(transition_t tr, weight_t weight) = 0;
    virtual weight_t add_weight(transition_t tr, weight_t weight) = 0;
    virtual weight_t lmul_weight(transition_t tr, weight_t weight) = 0;
    virtual weight_t rmul_weight(transition_t tr, weight_t weight) = 0;
    virtual void del_transition(transition_t tr) = 0;
    virtual transition_t del_transition(state_t src, state_t dst, label_t label) = 0;
    virtual void del_transition(state_t src, state_t dst) = 0;
    virtual void del_eps_transition(state_t src, state_t dst) = 0;

    //  Exploration
    virtual std::vector<state_t> states() const = 0;
    virtual std::vector<transition_t> transitions() const  = 0;
    virtual std::vector<state_t> initial_states() const = 0;
    virtual std::vector<state_t> final_states() const = 0;
    virtual std::vector<transition_t> final_transitions() const = 0;
    virtual std::vector<transition_t> initial_transitions() const = 0;
    virtual std::vector<state_t> successors(state_t s) const = 0;
    virtual std::vector<state_t> successors(state_t s, any_t label) const = 0;
    virtual std::vector<state_t> predecessors(state_t s) const = 0;
    virtual std::vector<state_t> predecessors(state_t s, any_t label) const = 0;
    virtual std::vector<transition_t> out(state_t s) const = 0;
    virtual std::vector<transition_t> out(state_t s, any_t label) const = 0;
    virtual std::vector<transition_t> in(state_t s) const = 0;
    virtual std::vector<transition_t> in(state_t s, any_t label) const = 0;
    virtual std::vector<transition_t> outin(state_t src, state_t dst) const = 0;

    // Hidden states
    virtual state_t pre() const = 0 ;
    virtual state_t post() const = 0 ;
    virtual std::vector<state_t> all_states() const = 0; //including pre() and post()
    virtual std::vector<transition_t> all_transitions() const  = 0; // including initial and final
    virtual std::vector<transition_t> all_out(state_t s) const = 0; // including final
    virtual std::vector<transition_t> all_in(state_t s) const = 0; // including initial

    //  Misc
    virtual std::vector<label_t> alphabet() const = 0;
    virtual bool is_transducer() const =0;
    virtual bool allow_eps_transition() const = 0;

    virtual context_t get_context() const = 0;

    virtual void strip_history() = 0;
    virtual std::string get_state_name(state_t s) const =0;
    virtual void set_state_name(state_t s, std::string name) =0;
    virtual state_t get_state_by_name(std::string name) const =0 ;


    virtual std::ostream& json(std::ostream&) const = 0;

    //for handling by languages where identifier 'in' is not allowed
    // incoming is the same as in, and outgoing the same as out
    virtual std::vector<transition_t> incoming(state_t s) const = 0;
    virtual std::vector<transition_t> incoming(state_t s, any_t label) const = 0;
    virtual std::vector<transition_t> outgoing(state_t s) const = 0;
    virtual std::vector<transition_t> outgoing(state_t s, any_t label) const = 0;
    virtual ~abstract_automaton_t() {}
  };

    using automaton_t=std::shared_ptr<abstract_automaton_t>;
}
}//end of ns awali::dyn

#endif
