// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef DYN_ANY_HH
#define DYN_ANY_HH

#include <awali/dyn/core/internal/untyped_value.hh>
#include <awali/dyn/core/internal/value.hh>
#include<string>



namespace awali {
  namespace dyn {

    template<typename T>
    bool any_typeof(const any_t& a);

    template<typename T>
    T any_cast(const any_t& a);

    std::ostream& operator<<(std::ostream& o, const dyn::any_t& a);

    struct any_t {
      template<typename T>
      friend T any_cast(const any_t& a);

      template<typename T>
      friend bool any_typeof(const any_t& a);
      friend std::ostream& operator<<(std::ostream& o, const dyn::any_t& a);

    private:
      internal::untyped_value* val;

      std::ostream& output (std::ostream& o) const {
	return val->output(o);
      }

    public:
      template<typename T>
      any_t(const T& t) : val(new internal::Value<T>(t)) {}

      any_t(const char *s) : val(new internal::Value<std::string>(s)) {}


      any_t(const any_t& a) : val(a.val->clone()) {}

      any_t& operator= (const any_t& t) {
	if(&t==this) return *this;
	delete val;
	val=t.val->clone();
	return *this;
      }

      bool operator< (const any_t& a) const {
	return val->less(*(a.val));
      }


      bool operator==(const any_t& a) const {
	return val->equal(*(a.val));
      }

      bool operator!=(const any_t& a) const {
	return !val->equal(*(a.val));
      }

      bool operator>=(const any_t& a) const {
	return !val->less(*(a.val));
      }

      bool operator<=(const any_t& a) const {
	return val->equal(*(a.val)) || val->less(*(a.val));
      }

      bool operator>(const any_t& a) const {
	return !val->equal(*(a.val)) && !val->less(*(a.val));
      }

      template<typename T>
      operator T() const {
	return any_cast<T>(*this);
      }

      std::ostream& real_type_name(std::ostream &o) const {
	return val->real_type_name(o);
      }

      ~any_t() {
	delete val;
      }
    };

    std::ostream&
    operator<< (std::ostream& o, const any_t& a){
      return a.output(o);
    }


    template<typename T>
    bool any_typeof(const any_t& a){
      try{
	const internal::Value<T>& test=dynamic_cast<const internal::Value<T>&>(*(a.val));
	return true;
      }
      catch(std::bad_cast& e){
	return false;
      }
    }

    template<typename T>
    T any_cast(const any_t& a){
      try {
	const internal::Value<T>& test=dynamic_cast<const internal::Value<T>&>(*(a.val));
	return test.val;
      }
      catch(...) {
	std::cerr << "Extract " << typeid(T).name() << " from any enclosing " << typeid(*(a.val)).name() << std::endl;
	throw;
      }
    }

  }
}//end awali::dyn

#endif //DYN_ANY_HH
