// This file is part of Awali.
// Copyright 2016-2019 Sylvain Lombardy, Victor Marsault, Jacques Sakarovitch
//
// Awali is a free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include <awali/sttc/algos/determinize.hh>
#include <awali/sttc/algos/min_quotient.hh>
#include <awali/sttc/algos/proper.hh>
#include <awali/sttc/algos/complete.hh>
#include <awali/sttc/weightset/b.hh>
#include <awali/sttc/labelset/traits.hh>
#include <awali/dyn/bridge-sttc/explicit-automaton.cc>
#include<stdexcept>

#include<set-types.hh>

namespace awali {

  template<typename Context,
	   typename LS=typename Context::labelset_t,
	   typename WS=typename Context::weightset_t>
  struct dispatch { //default
    static dyn::automaton_t minimal_automaton(dyn::automaton_t aut, param_t algo) {
      throw std::runtime_error("minimal_automaton only supported for Boolean");
    }
  };

  template<typename Context, typename T>
  struct dispatch<Context,sttc::letterset<T>, sttc::b> {
    static dyn::automaton_t minimal_automaton(dyn::automaton_t aut, param_t algo) {
      auto a=dyn::get_stc_automaton<context_t>(aut);
      switch(algo) {
      case BRZOZOWSKI:
	a=sttc::determinize(sttc::transpose_view(a));
	a=determinize(sttc::transpose_view(a));
	break;
      case MOORE:
	if(!sttc::is_deterministic(a))
	  a=determinize(a);
	a=sttc::min_quotient(a,MOORE);
	break;
      case HOPCROFT:
	if(!sttc::is_deterministic(a))
	  a=determinize(a);
	a=sttc::min_quotient(a,HOPCROFT);
	break;
      default:
	throw std::runtime_error("Supported algorithms for minimisation are BRZOZOWSKI, MOORE and HOPCROFT");
      }
      sttc::complete_here(a);
      return dyn::make_automaton(a);
    }
  };

  template<typename Context, typename T>
  struct dispatch<Context, sttc::nullableset<sttc::letterset<T>>, sttc::b> {
    static dyn::automaton_t minimal_automaton(dyn::automaton_t aut, param_t algo) {
      auto res=proper(dyn::get_stc_automaton<context_t>(aut));
      auto a = sttc::make_mutable_automaton<sttc::not_nullable_of<context_t>>(sttc::get_not_nullable_context(res->context()));
      sttc::copy_into(res,a);
      switch(algo) {
      case BRZOZOWSKI:
	a=sttc::determinize(sttc::transpose_view(a));
	a=determinize(sttc::transpose_view(a));
	break;
      case MOORE:
	if(!sttc::is_deterministic(a))
	  a=determinize(a);
	a=sttc::min_quotient(a,MOORE);
	break;
      case HOPCROFT:
	if(!sttc::is_deterministic(a))
	  a=determinize(a);
	a=sttc::min_quotient(a,HOPCROFT);
	break;
      default:
	throw std::runtime_error("Supported algorithms for minimisation are BRZOZOWSKI, MOORE and HOPCROFT");
      }
      sttc::complete_here(a);
      return dyn::make_automaton(a);
    }
  };


  extern "C" dyn::automaton_t minimal_automaton(dyn::automaton_t aut, param_t algo) {
    return dispatch<context_t>::minimal_automaton(aut, algo);
  }
}
