#ifndef ATLAS_MMTESTTIME_H
   #define ATLAS_MMTESTTIME_H

#include "atlas_mmparse.h"
#include "atlas_gentesttime.h"

/* procedure 1 */
char *MMGetGenName(char pre, int kb, ATL_mmnode_t *mp)
{
   char *rout;
   rout = malloc(32);
   assert(rout);
   /* ATL_<pre>amm<kb>_<vlen>[k,m]_<mu>x<nu>x<ku>.c */
   sprintf(rout, "ATL_%camm%d_%d%c_%dx%dx%d.c", pre, kb, mp->vlen,
           FLAG_IS_SET(mp->flag,MMF_KVEC)?'k':'m', mp->mu, mp->nu, mp->ku);
   return(rout);
}
/* procedure 2 */
char *MMGetGenString
(
   char pre,                    /* precision/type prefix */
   ATL_mmnode_t *mp             /* mmkern ptr */
)
{
   char *frm=
      "make gen_amm pre=%c rt=%s vec=%s vlen=%d mu=%d nu=%d ku=%d bcast=%d";
   char *vec="mdim";
   char *gs;
   int len;
   len = strlen(frm) + strlen(mp->rout) + 4;
   gs = malloc(len);
   assert(gs);
   if (mp->vlen < 2)
      vec = "no";
   else if (FLAG_IS_SET(mp->flag, MMF_KVEC))
   {
      vec = "kdim";
      assert(mp->ku % mp->vlen == 0);
   }
   else
      assert(mp->mu % mp->vlen == 0);
   sprintf(gs, frm, pre, mp->rout, vec, mp->vlen, mp->mu, mp->nu, mp->ku,
           !FLAG_IS_SET(mp->flag, MMF_NOBCAST));
   return(gs);
}

/* procedure 3 */
void MMFillInGenStrings(char pre, ATL_mmnode_t *mmb)
{
   ATL_mmnode_t *mp;

   if (pre == 'z')
      pre = 'd';
   else if (pre == 'c')
      pre = 's';

   for (mp=mmb; mp; mp = mp->next)
   {
      if (mp->ID == 0 && !mp->genstr)
         mp->genstr = MMGetGenString(pre, mp);
   }
}

/* procedure 4 */
ATL_mmnode_t *MMGetNodeGEN(char pre, int flag, int kb, int mu, int nu, int ku,
                           int vlen, int KVEC, char *rt)
{
   ATL_mmnode_t *mp;

   mp = GetMMNode();
   mp->mu = mu;
   mp->nu = nu;
   mp->ku = ku;
   mp->vlen = vlen;
   if (flag&1)
      mp->flag |= (1<<MMF_NOBCAST);
   if (!kb)
      mp->flag |= (1<<MMF_KRUNTIME);
   else
      mp->kbB = kb;
   mp->vlen = vlen;
   if (KVEC && vlen > 1)
      mp->flag |= (1<<MMF_KVEC);
   else if (vlen < 2)
      KVEC=1;
   if (!rt)
      rt = MMGetGenName(pre, kb, mp);
   mp->rout = rt;
   mp->genstr = MMGetGenString(pre, mp);
   return(mp);
}

/* procedure 5 */
int MMDoGenString(int verb, char *genstr)
{
   int err=0;
   if (!genstr)
      err=1;
   else if (verb < 3) /* want to redirect output */
   {
      char *ln;
      ln = NewMergedString(genstr, " > /dev/null 2>&1");
      err = system(ln);
      free(ln);
   }
   else
      err = system(genstr);
   if (err)
      fprintf(stderr, "UNABLE TO GENERATE WITH COMMAND: '%s'\n",
              genstr ? genstr : "NULL");
   return(err);
}

/* procedure 6 */
char *MMGetTestString
(
   char pre,                    /* precision/type prefix */
   int mb, int nb, int kb,      /* dimensions to test */
   int beta,                    /* beta case to test */
   ATL_mmnode_t *umm            /* mmkern ptr */
)
{
   int i, k, lnlen, Dd, Ud, Vd;
   int len0, len1, len2, len3, len4, len5;
   char *ln;
/*
 * Horrific code so we don't have to rely on snprintf, which some ancient
 * compiler may lack
 */
   Ud = NumDecDigits(umm->mu);
   i = NumDecDigits(umm->nu);
   Ud = Mmax(Ud,i);
   i = NumDecDigits(umm->ku);
   Ud = Mmax(Ud,i);
   Dd = NumDecDigits(mb);
   i = NumDecDigits(mb);
   Dd = Mmax(Dd,i);
   i = NumDecDigits(kb);
   Dd = Mmax(Dd,i);
   Vd = NumDecDigits(umm->vlen);
   len0 = 64+strlen(umm->rout) + 3*Dd; /* guess for first print */
   len1 = 8 + Vd;  /* kmaj */
   len2 = 22;      /* beta */
   if (umm->comp)
      len3 = 24 + strlen(umm->comp) + strlen(umm->cflags);
   else
      len3 = 0;
   len4 = 40 + 3*(Dd+Ud);  /* dim/unroll print */
   len5 = 20; /* redirect */
   lnlen = len0 + len1 + len2 + len3 + len4 + len5;
   ln = malloc(lnlen);
   assert(ln);

   if (umm->ID > 0)
      i = sprintf(ln, "make %cammmtst mmrout=AMMCASES/%s mu=%d nu=%d ku=%d ",
                  pre, umm->rout, umm->mu, umm->nu, umm->ku);
   else
      i = sprintf(ln, "make %cammmtst mmrout=%s mu=%d nu=%d ku=%d ",
                  pre, umm->rout, umm->mu, umm->nu, umm->ku);
   assert(i < len0);
   if (FLAG_IS_SET(umm->flag, MMF_KVEC))
   {
      k = sprintf(ln+i, "kmaj=%d ", umm->vlen);
      i += k;
      assert(k < len1);
   }
   if (beta != 1)
   {
      if (beta == -1)
         k = sprintf(ln+i, "beta=-1 betan=\"N1\" ");
      else if (beta == 0)
         k = sprintf(ln+i, "beta=%d ", beta);
      else
         assert(0);
      assert(k < len2);
      i += k;
   }
   if (umm->comp)
   {
      char ch = (pre == 'c' || pre == 's') ? 'S' : 'D';
      k = sprintf(ln+i, "%cMC=\"%s\" %cMCFLAGS=\"%s\" ",
                  ch, umm->comp, ch, umm->cflags);
      assert(k < len3);
      i += k;
   }
   k = sprintf(ln+i, "pre=%c M=%d N=%d K=%d mb=%d nb=%d kb=%d ",
               pre, mb, nb, kb, 0, 0,
               FLAG_IS_SET(umm->flag, MMF_KRUNTIME) ? 0 : kb);
   assert(k < len4);
   i += k;
   k = sprintf(ln+i, " > /dev/null 2>&1\n");
   assert(k < len5);
   i += k;
   assert(i < lnlen);
   return(ln);
}

/* procedure 7 */
int MMKernelFailsTest
(
   char pre,                    /* precision/type prefix */
   int mb, int nb, int kb,      /* dimensions to test */
   int beta,                    /* beta case to test */
   ATL_mmnode_t *umm            /* mmkern ptr */
)
/*
 * RETURNS: 0 on success, non-zero on failure
 */
{
   char *ln;
   int i;
   char ch;

/*
 * If the file is generated, call generator to create it
 */
   if (umm->ID == 0 && !umm->genstr)
      umm->genstr = MMGetGenString(pre, umm);
   if (umm->genstr)
      assert(!MMDoGenString(0, umm->genstr));
   ln = MMGetTestString(pre, mb, nb, kb, beta, umm);
   i = system(ln);
   if (i)
   {
      fprintf(stderr, "%d of %s: FAILED COMMAND : %s\n",__LINE__,__FILE__,ln);
      if (umm->genstr)
         fprintf(stderr, "   genstr was = '%s'\n", umm->genstr);
   }
   free(ln);
   return(i);
}

/* procedure 8 */
static ATL_mmnode_t *DelBadMMKernels(char pre, int verb, ATL_mmnode_t *bp)
/*
 * Deletes all kernels that can't pass basic usage test
 * RETURNS: modifed bp queue wt failing kernels removed
 */
{
   ATL_mmnode_t *p, *prev;
   int die;

   if (verb > 0)
       printf("\nBEGIN BASIC MATMUL KERNEL TESTS:\n");

   prev = p = bp;
   while (p)
   {
      if (MMKernelFailsTest(pre, p->mbB, p->nbB, p->kbB, 0, p) ||
          MMKernelFailsTest(pre, p->mbB, p->nbB, p->kbB, 1, p) ||
          MMKernelFailsTest(pre, p->mbB, p->nbB, p->kbB, -1, p))
      {
         if (verb > 0)
            printf("   NUKING bad kernel %s(%d)\n", p->rout, p->ID);
         if (p == bp)
            bp = p = KillMMNode(p);
         else
            prev->next = p = KillMMNode(p);
      }
      else
      {
         if (verb > 0)
            printf("   Kernel %s(%d) passes basic tests\n", p->rout, p->ID);
         prev = p;
         p = p->next;
      }
   }
   printf("DONE BASIC KERNEL TESTS.\n\n");
   return(bp);
}


/* procedure 9 */
double TimeMMKernel
(
   int verb,                    /* 0: no output, 1 min output, 2: full output */
   int flag,                   /* 1: ignore any prior output file */
   ATL_mmnode_t *mmp,           /* ptr to mmkern struct */
   char pre,                    /* type/prec prefix: z,c,d,s */
   int mb, int nb, int kb,      /* dimensions to time */
   int beta,                    /* beta to time */
   int mflop,                   /* >0: force mflop MFLOPs in each time interv */
   int cflush                   /* >=0: size of cache flush, else ignored */
)
/*
 * flag - take actions the following actions if bit location is set:
 *    0 : ignore any prior output file on output
 *    1 : time in serial rather than parallel
 */
{
   char fnam[128], ln[4096];  /* security from 1991 */
   const char *LO = FLAG_IS_SET(mmp->flag, MMF_AOUTER) ? "IJK": "JIK";
   const int ku=mmp->ku;
   const int KB = (!FLAG_IS_SET(mmp->flag,MMF_KVEC)) ? kb : ((kb+ku-1)/ku)*ku;
   const int FORCETIME = flag&1, SERIAL=flag&2;
   int DOTIME=1;
   int MV=3;  /* bit pattern on move CBA (C=4, B=2, A=1) */
   char *be;
   int i, j;
   char ch;
   double *dp;
   MV = ((mmp->flag) >> MMF_MVA)&7;
/*
 * If it's a emit_mm generated file w/o the genstring, create the genstring
 * assuming it is a mmK
 */
   if (mmp->ID == 0 && !mmp->genstr)
      mmp->genstr = MMGetGenString(pre, mmp);
/*
 * If the file is generated, call generator to create it
 */
   if (mmp->genstr)
      assert(!MMDoGenString(verb, mmp->genstr));
   else if (verb > 2)
      printf("NO genstr\n");
   #ifndef ATL_SERIAL_INSTALL
      #define RESACT 2   /* want average frm ReadResultsFile for parallel */
/*
 *    Figure out the name of the output file
 */
      if (FORCETIME)
         strcpy(fnam, "res/tmpout.ktim");
/*
 *    PREammID_MBxNBxKB_MUxNUxKU_FLAG_v[M,K]VLENbBETA_CFLUSH
 */
      else
      {
         if (mmp->vlen < 2)
            ch = 'S';
         else
            ch = (FLAG_IS_SET(mmp->flag,MMF_KVEC)) ? 'K' : 'M';

         sprintf(fnam, "res/%cammm%d_%dx%dx%d_%dx%dx%d_%d_v%c%db%d_%d.ktim",
                 pre, mmp->ID, mb, nb, KB, mmp->mu, mmp->nu, mmp->ku, mmp->flag,
                 ch, mmp->vlen, beta, cflush);
      }
/*
 *    If we actually need to do timing, must also construct timer call
 */
      if (!FORCETIME)
         DOTIME = !FileExists(fnam);
      if (DOTIME)
      {
         i = sprintf(ln, "make x%cammtime_pt mb=%d nb=%d kb=%d",
                     pre, mb, nb, KB);
         if (SERIAL)
            i += sprintf(ln+i, " NPROC=1");
         if (mmp->genstr)
            i += sprintf(ln+i, " mmrout=%s", mmp->rout);
         else
            i += sprintf(ln+i, " mmrout=AMMCASES/%s", mmp->rout);
         i += sprintf(ln+i, " mu=%d nu=%d ku=%d", mmp->mu, mmp->nu, mmp->ku);
         i += sprintf(ln+i, " mvA=%d mvB=%d mvC=%d", ((mmp->flag >> MMF_MVA)&1),
                      ((mmp->flag >> MMF_MVB)&1), ((mmp->flag >> MMF_MVC)&1));
         i += sprintf(ln+i, " kmoves=\"");
         if (FLAG_IS_SET(mmp->flag, MMF_MVA))
            i += sprintf(ln+i, " -DATL_MOVEA");
         if (FLAG_IS_SET(mmp->flag, MMF_MVB))
            i += sprintf(ln+i, " -DATL_MOVEB");
         if (FLAG_IS_SET(mmp->flag, MMF_MVC))
            i += sprintf(ln+i, " -DATL_MOVEC");
         i += sprintf(ln+i, "\"");
         if (beta == 1 || beta == 0)
            i += sprintf(ln+i, " beta=%d", beta);
         else
            i += sprintf(ln+i, " beta=-1 betan=\"N1\"");
         ch = (pre == 'c' || pre == 's') ? 'S' : 'D';
         if (mmp->comp)
            i += sprintf(ln+i, " %cMC=\"%s\"", ch, mmp->comp);
         if (mmp->cflags)
            i += sprintf(ln+i, " %cMCFLAGS=\"%s\"", ch, mmp->cflags);
         i += sprintf(ln+i, " outF=\"-f %s\"", fnam);
      }
   #else
      #define RESACT 0   /* ReadResultsFile uses ACTION=0 */

      if (beta == 0)
         be = "b0";
      else if (beta == 1)
         be = "b1";
      else if (beta == -1)
         be = "bn1";
      else
         be = "bX";
      if (FORCETIME)
         sprintf(fnam, "res/tmpout");
      else /*dmmMNK%d_JMMBxNBxKB_muxnuk#xku_rtK_LDTOP_vlen_pf_a1_bX_MV_flushKB*/
         sprintf(fnam,
         "res/%cmm%s%d_%dx%dd%dx%d_%dx%dx%d_rtK%d__v%d_%d_%d_a1_%s_%d_%d.ktim",
                 pre, FLAG_IS_SET(mmp->flag, MMF_AOUTER) ? "MNK" : "NMK",
                 mmp->ID, mb, nb, FLAG_IS_SET(mmp->flag,MMF_KVEC), kb,
                 mmp->mu, mmp->nu, mmp->ku,
                 FLAG_IS_SET(mmp->flag, MMF_KRUNTIME),
                 FLAG_IS_SET(mmp->flag, MMF_LDCTOP),
                 mmp->vlen, mmp->pref, be, MV, cflush);

      if (!FORCETIME)
         DOTIME = !FileExists(fnam);
      if (DOTIME)
      {
         if (pre == 'c' || pre == 'z')
            i = sprintf(ln, "make cmmucase mmrout=%s csC=2 ", mmp->rout);
         else
         {
            if (mmp->ID > 0)
               i = sprintf(ln, "make mmucaseN mmrout=AMMCASES/%s ", mmp->rout);
            else
               i = sprintf(ln, "make mmucaseN mmrout=%s ", mmp->rout);
            if (FLAG_IS_SET(mmp->flag,MMF_KVEC))
               i += sprintf(ln+i, "kmaj=%d ", mmp->vlen);
         }
         if (mmp->cflags)
         {
            ch = (pre == 'c' || pre == 's') ? 'S' : 'D';
            i += sprintf(ln+i, "%cMCFLAGS=\"%s\" ", ch, mmp->cflags);
         }
         if (mmp->comp)
         {
            ch = (pre == 'c' || pre == 's') ? 'S' : 'D';
            i += sprintf(ln+i, "%cMC=\"%s\" ", ch, mmp->comp);
         }
         if (mmp->moves)
            i += sprintf(ln+i, " moves=\"%s\" ", mmp->moves);
         if (mmp->exflags)
            i += sprintf(ln+i, " %s ", mmp->exflags);

         if (!cflush)
            i += sprintf(ln+i, "moves=\"\" ");

         i += sprintf(ln+i, "casnam=%s ", fnam);
         i += sprintf(ln+i, "pre=%c M=%d N=%d K=%d mb=%d nb=%d kb=%d ",
                      pre, mb, nb, KB, 0, 0,
                      FLAG_IS_SET(mmp->flag, MMF_KRUNTIME) ? 0:KB);
         i += sprintf(ln+i, "mu=%d nu=%d ku=%d beta=%d",
                      mmp->mu, mmp->nu, mmp->ku, beta);
      }
   }
   #endif
   if (FORCETIME || !FileExists(fnam))
   {
      if (verb < 3)
         i += sprintf(ln+i, " > /dev/null 2>&1\n");
      else
         i += sprintf(ln+i, "\n");
      if (verb > 1)
         fprintf(stdout, "SYSTEM: %s", ln);
      if (system(ln))
      {
         fprintf(stderr, "ERROR IN COMMAND: %s", ln);
         fprintf(stderr, "   PROPOSED FILENAME: %s\n", fnam);
         if (mmp->genstr)
            fprintf(stderr, "   GENSTR='%s'\n", mmp->genstr);
         sprintf(ln, "rm -f %s\n", fnam);
         assert(!system(ln));
         exit(-1);
      }
   }
   dp = ReadResultsFile(RESACT, 0, fnam);
   if (!dp)
   {
      fprintf(stderr, "\nEmpty file '%s'!\n", fnam);
      fprintf(stderr, "From command: '%s'\n", ln);
      fprintf(stderr, "DOTIME=%d, genstr='%s'\n", DOTIME,
              (mmp->genstr) ? mmp->genstr : "");
      exit(-1);
   }
   if (mmp->genstr && DOTIME)
   {
      sprintf(ln, "rm %s\n", mmp->rout);
      i = system(ln);  /* return value unused, just to shut gcc up */
   }
   if (kb != KB)
   {
      double mf;
      mf = *((double*)ReadResultsFile(RESACT, 0, fnam));
      mf = (mf / KB)*kb;
      return(mf);
   }

   return(*((double*)ReadResultsFile(RESACT, 0, fnam)));
   #undef RESACT
}
/* procedure 10 */
int MMKernCanHandleCase(ATL_mmnode_t *mp, int mb, int nb, int kb)
/*
 * RETURNS: 0 if mp cannot handle GEMM of size mbxnbxkb
 *          1 if it can handle w/o extra computation
 *          2 if handling it requires extra comptutation (KVEC only: on kb)
 */
{
   if (mb%mp->mu == 0 && nb%mp->nu == 0)
   {
      const int KRUN=FLAG_IS_SET(mp->flag, MMF_KRUNTIME), ku=mp->ku;
      if (mp->kbB == kb)
         return(1);
      if (KRUN && kb%ku == 0)
         return(1);
      if (FLAG_IS_SET(mp->flag, MMF_KVEC))
      {
         const int vl = mp->vlen, kbB=mp->kbB;
         if (KRUN && kb%ku < vl)
            return(2);
         if (kbB > kb && (kbB-kb < vl))
            return(2);
         if (kbB < kb && (kb-kbB < vl))
            return(2);
      }
   }
   return(0);
}

/* procedure 11 */
ATL_mmnode_t *MMTimeKernWithAlt
(
   int verb,                    /* 0: no output, 1 min output, 2: full output */
   int imf,                     /* which mflop entry to store res in */
   int flag,                    /* 1: ignore any prior output file */
   ATL_mmnode_t *mp0,           /* ptr to mmkern struct */
   ATL_mmnode_t *mp1,           /* ptr to mmkern struct */
   char pre,                    /* type/prec prefix: z,c,d,s */
   int mb, int nb, int kb,      /* dimensions to time */
   int beta,                    /* beta to time */
   int mflop,                   /* >0: force mflop MFLOPs in each time interv */
   int cflush                   /* >=0: size of cache flush, else ignored */
)
{
   int i;
   double mf0=0.0, mf1=0.0;
   i = MMKernCanHandleCase(mp0, mb, nb, kb);
   if (i)
   {
      mf0 = TimeMMKernel(verb, flag, mp0, pre, mb, nb, kb, beta, mflop, cflush);
      if (i == 2)
      {
         int k = mp0->vlen;
         k = ((kb+k-1)/k)*k;
         mf0 *= kb;
         mf0 /= k;
      }
      mp0->mflop[imf] = mf0;
   }
/*
 * If we couldn't use first kernel, or if first kernel needed extra flops
 */
   if (!i || i == 2)
   {
      int j;
      j = MMKernCanHandleCase(mp1, mb, nb, kb);
      if (j)
      {
         mf1 = TimeMMKernel(verb, flag, mp1, pre, mb, nb, kb,beta,mflop,cflush);
         if (j == 2)
         {
            int k = mp1->vlen;
            k = ((kb+k-1)/k)*k;
            mf1 *= kb;
            mf1 /= k;
         }
         mp1->mflop[imf] = mf1;
      }
      else if (!i)
         return(NULL);
   }
   if (mf1 > mf0)
      return(mp1);
   return(mp0);
}

/* procedure 12 */
ATL_mmnode_t *MMTimeAllKernsWithAlt
(
   int verb,                    /* 0: no output, 1 min output, 2: full output */
   int imf,
   int flag,                   /* 1: ignore any prior output file */
   ATL_mmnode_t *mmb,           /* ptr to mmkern struct */
   ATL_mmnode_t *mmA,           /* ptr to mmkern struct */
   char pre,                    /* type/prec prefix: z,c,d,s */
   int mb, int nb, int kb,      /* dimensions to time */
   int beta,                    /* beta to time */
   int mflop,                   /* >0: force mflop MFLOPs in each time interv */
   int cflush                   /* >=0: size of cache flush, else ignored */
)
/*
 * For given problem size, times all kernels in mmb.  If any entry in mmb
 * cannot handle this exact problem (eg., unrolling mismatch), or requires
 * extra flops to handle the problem, the corresponding
 * entry in the alternate list mmA will be tried.
 * RETURNS: ptr to fastest timed kernel, or NULL if none worked.
 * NOTE: neither list is changed by this function (mflop is overwritten!),
 *       but ptr into original listis returned.
 */
{
   ATL_mmnode_t *mpB=NULL, *mp0, *mp1;
   double mfB=0.0;
   for (mp0=mmb, mp1=mmA; mp0; mp0 = mp0->next, mp1 = mp1->next)
   {
      ATL_mmnode_t *mp;
      mp = MMTimeKernWithAlt(verb, imf, flag, mp0, mp1, pre, mb, nb, kb,
                             beta, mflop, cflush);
      if (mp)
      {
         double mf;
         mf = mp->mflop[imf];
         if (mf > mfB)
         {
            mfB = mf;
            mpB = mp;
         }
      }
   }
   return(mpB);
}

/* procedure 13 */
int TimeNegMMKernels            /* RET: 0 if no retiming required */
(
   int imf,                     /* index of mflop array to check/set */
   int verb,                    /* 0: no output, 1 min output, 2: full output */
   int FORCETIME,               /* 1: ignore any prior output file */
   ATL_mmnode_t *mmb,           /* ptr to mmkern struct queue */
   char pre,                    /* type/prec prefix: z,c,d,s */
   int beta,                    /* beta to time */
   int mflop,                   /* >0: force mflop MFLOPs in each time interv */
   int cflush                   /* >0: size of cache flush, else ignored */
)
{
   ATL_mmnode_t *mp;
   int RETIME=0;
   for (mp=mmb; mp; mp = mp->next)
   {
      if (mp->mflop[imf] <= 0.0)
      {
         RETIME++;
         mp->mflop[imf] = TimeMMKernel(verb, FORCETIME ? 1:0, mp, pre,
                                       mp->mbB, mp->nbB, mp->kbB,
                                       beta, mflop, cflush);
      }
   }
   return(RETIME);
}


/* procedure 14 */
static ATL_mmnode_t *TimeMMFileWithPath
(
   char pre,
   char *path,
   char *file,
   int imf,                     /* index of mflop array to check/set */
   int verb,                    /* 0: no output, 1 min output, 2: full output */
   int FORCETIME,               /* 1: ignore any prior output file */
   int beta,                    /* beta to time */
   int mflop,                   /* >0: force mflop MFLOPs in each time interv */
   int cflush                   /* >0: size of cache flush, else ignored */
)
{
   ATL_mmnode_t *mmb;
   mmb = ReadMMFileWithPath(pre, path, file);
   MMFillInGenStrings(pre, mmb);
   if (TimeNegMMKernels(imf, verb, FORCETIME, mmb, pre, beta, mflop, cflush))
      WriteMMFileWithPath(pre, path, file, mmb);
   return(mmb);
}

/* procedure 15 */
void TimeAllMMKernels
(
   int itime,                   /* index of mflop array to set */
   int verb,                    /* 0: no output, 1 min output, 2: full output */
   int FORCETIME,               /* 1: ignore any prior output file */
   ATL_mmnode_t *mmb,           /* ptr to mmkern struct queue */
   char pre,                    /* type/prec prefix: z,c,d,s */
   int beta,                    /* beta to time */
   int mflop,                   /* >0: force mflop MFLOPs in each time interv */
   int cflush                   /* >0: size of cache flush, else ignored */
)
{
   ATL_mmnode_t *mmp;
   for (mmp=mmb; mmp; mmp = mmp->next)
      mmp->mflop[itime] = TimeMMKernel(verb, FORCETIME?1:0, mmp, pre,
                                       mmp->mbB, mmp->nbB, mmp->kbB,
                                       beta, mflop, cflush);
}

double TimeTSKernel
(
   int verb,                    /* 0: no output, 1 min output, 2: full output */
   int FORCETIME,               /* 1: ignore any prior output file */
   char pre,                    /* type/prec prefix: z,c,d,s */
   int mb,                      /* triangle is mbxmb (kb=mb) */
   int nb,                      /* NRHS */
   int mflop                    /* >0: force mflop MFLOPs in each time interv */
)
{
   int i, DOTIME=1;
   char ln[2048], resf[256];

   if (FORCETIME)
      strcpy(resf, "res/tmpout.ktim");
   else
   {
      sprintf(resf, "res/%ctrsm%dx%d_F%d.ktim", pre, mb, nb, mflop);
      DOTIME = !FileExists(resf);
   }

   if (DOTIME)
   {
      i = sprintf(ln, "make %ctrsmKtime mb=%d nb=%d outF=\" -f %s \"",
                  pre, mb, nb, resf);
      if (mflop)
         i += sprintf(ln+i, " FMF=%d", mflop);

      sprintf(ln+i, "\n");
      if (verb > 1)
         printf("SYSTEM: %s", ln);
      if (system(ln))
      {
         fprintf(stderr, "ERROR IN COMMAND: %s", ln);
         fprintf(stderr, "   PROPOSED FILENAME: %s\n", resf);
         sprintf(ln, "rm -f %s\n", resf);
         assert(!system(ln));
         exit(-1);
      }
   }
   return(*((double*)ReadResultsFile(0, 0, resf)));
}

#endif  /* end guard around atlas_mmtesttime.h */
