/************************************************************

  This example shows how to iterate over group members using
  H5Literate.

Note: This example includes older cases from previous versions
  of HDF5 for historical reference and to illustrate how to
  migrate older code to newer functions. However, readers are
  encouraged to avoid using deprecated functions and earlier
  schemas from those versions.

 ************************************************************/

#include "hdf5.h"
#include <stdio.h>

#define FILENAME "h5ex_g_iterate.h5"

/*
 * Operator function to be called by H5Literate.
 */
herr_t op_func(hid_t loc_id, const char *name, const H5L_info_t *info, void *operator_data);

int
main(void)
{
    hid_t  file; /* Handle */
    herr_t status;

    /*
     * Open file.
     */
    file = H5Fopen(FILENAME, H5F_ACC_RDONLY, H5P_DEFAULT);

    /*
     * Begin iteration.
     */
    printf("Objects in root group:\n");
    status = H5Literate(file, H5_INDEX_NAME, H5_ITER_NATIVE, NULL, op_func, NULL);

    /*
     * Close and release resources.
     */
    status = H5Fclose(file);

    return 0;
}

/************************************************************

  Operator function.  Prints the name and type of the object
  being examined.

 ************************************************************/
herr_t
op_func(hid_t loc_id, const char *name, const H5L_info_t *info, void *operator_data)
{
    herr_t     status;
    H5O_info_t infobuf;

    /*
     * Get type of the object and display its name and type.
     * The name of the object is passed to this function by
     * the Library.
     */
#if H5_VERSION_GE(1, 12, 0) && !defined(H5_USE_110_API) && !defined(H5_USE_18_API) && !defined(H5_USE_16_API)
    status = H5Oget_info_by_name(loc_id, name, &infobuf, H5O_INFO_ALL, H5P_DEFAULT);
#else
    status = H5Oget_info_by_name(loc_id, name, &infobuf, H5P_DEFAULT);
#endif
    switch (infobuf.type) {
        case H5O_TYPE_GROUP:
            printf("  Group: %s\n", name);
            break;
        case H5O_TYPE_DATASET:
            printf("  Dataset: %s\n", name);
            break;
        case H5O_TYPE_NAMED_DATATYPE:
            printf("  Datatype: %s\n", name);
            break;
        default:
            printf("  Unknown: %s\n", name);
    }

    return 0;
}
