/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.spark.sql.execution.streaming

import org.apache.spark.{SparkException, SparkRuntimeException}

/**
 * Object for grouping error messages from streaming query exceptions
 */
object StreamingErrors {
  def cannotLoadCheckpointFileManagerClass(path: String, className: String, err: Throwable):
  Throwable = {
    new SparkException(
      errorClass = "CANNOT_LOAD_CHECKPOINT_FILE_MANAGER.ERROR_LOADING_CLASS",
      messageParameters = Map("path" -> path, "className" -> className, "msg" -> err.toString),
      cause = err
    )
  }

  def cannotLoadCheckpointFileManager(path: String, err: Throwable):
  Throwable = {
    new SparkException(
      errorClass = "CANNOT_LOAD_CHECKPOINT_FILE_MANAGER.UNCATEGORIZED",
      messageParameters = Map("path" -> path),
      cause = err
    )
  }

  def statefulOperatorMissingStateDirectory(
      opsInCurBatch: Map[Long, String]): Throwable = {
    def formatPairString(pair: (Long, String)): String =
      s"(OperatorId: ${pair._1} -> OperatorName: ${pair._2})"

    new SparkRuntimeException(
      errorClass = "STREAMING_STATEFUL_OPERATOR_MISSING_STATE_DIRECTORY",
      messageParameters = Map(
        "OpsInCurBatchSeq" -> opsInCurBatch.map(formatPairString).mkString(", ")
      )
    )
  }
}
