package org.apache.jetspeed.portlet;

import java.util.Locale;
import java.util.Map;
import java.util.Enumeration;

import java.io.IOException;
import java.io.InputStream;

/**
 ** The <CODE>PortletRequest</CODE> encapsulates the request sent by the
 ** client to the portlet.
 **
 ** @see   PortletResponse
 **
 ** @author   <A HREF="mailto:shesmer@apache.org">Stephan Hesmer</A>
 **/

public interface PortletRequest extends javax.servlet.http.HttpServletRequest
{
    /**
     ** Returns the mode that the portlet is running in.
     **
     ** @return   the portlet mode
     **/

    public Portlet.Mode getMode ();

    /**
     ** Returns the mode that the portlet was running at last, or
     ** <CODE>null</CODE> if no previous mode exists.
     **
     ** @return   the previous portlet mode
     **/

    public Portlet.Mode getPreviousMode ();

    /**
     ** Defines which portlet mode is shown next. Once the mode
     ** is changed it cannot be changed back to
     ** <CODE>Portlet.ModeModifier.REQUESTED</CODE>.
     **
     ** <P>
     ** This function may only be used during event processing,
     ** in any other case the call has no effect.
     **
     ** @param   modeModifier
     **          the mode modifier
     ** @exception   AccessDeniedException
     **              if the portlet tries to access this function
     **              outside of the action event processing
     **/

    public void setModeModifier(Portlet.ModeModifier modeModifier) throws AccessDeniedException;

    /**
     ** Returns the HTTP method of this request. The most commonly
     ** used request methods are GET and POST.
     **
     ** @return   the method
     **/

    public String getMethod ();

    /**
     ** Retrieves the body of the request as binary data using
     ** an <CODE>InputStream</CODE>.
     **
     ** @return   an input stream containing the body of the request
     **
     ** @exception   IllegalStateException
     **              if the portlet tries to access this function
     **              outside of the action event processing
     ** @exception   IOException
     **              if an input or output exception occurred
     **/

    public javax.servlet.ServletInputStream getInputStream ()
        throws IOException;

    /**
     ** Returns the locale of the preferred language. The preference
     ** is based on the user's choice of language(s) and/or the
     ** client's Accept-Language header.
     **
     ** <P>
     ** If more than one language is preferred, the locale returned by
     ** this method is the one with the highest preference.
     **
     ** @return   the locale of the preferred language
     **/

    public Locale getLocale ();

    /**
     *
     * Returns an <code>Enumeration</code> of <code>Locale</code> objects
     * indicating, in decreasing order starting with the preferred locale, the
     * locales that are acceptable to the client based on the Accept-Language
     * header.
     * If the client request doesn't provide an Accept-Language header,
     * this method returns an <code>Enumeration</code> containing one 
     * <code>Locale</code>, the default locale for the server.
     *
     *
     * @return		an <code>Enumeration</code> of preferred 
     *                  <code>Locale</code> objects for the client
     *
     */

    public Enumeration getLocales();

    /**
     ** Returns an object representing the client device that the
     ** user connects to the portal with.
     **
     ** @return   the client
     **/

    public Client getClient ();

    /**
     ** Returns the value of the parameter with the given name,
     ** or <CODE>null</CODE> if no such parameter exists.
     **
     ** <P>
     ** You should only use this method when you are sure the
     ** parameter has only one value. If not, use
     ** <CODE>getParameterValues(String)</CODE>
     **
     ** @param   name
     **          the parameter name
     **
     ** @return   the parameter value
     **
     ** @see   #getParameterMap()
     ** @see   #getParameterValues(String)
     **/

    public String getParameter (String name);

    /**
     ** Returns a map of the parameters of this request.
     **
     ** @return   a map of parameters
     **
     ** @see   #getParameter(String)
     ** @see   #getParameterValues(String)
     **/

    public Map getParameterMap ();

    /**
     ** Returns an enumeration of all parameter names. If this
     ** request
     **
     ** @return   the enumeration
     **/

    public Enumeration getParameterNames ();

    /**
     ** Returns the values of all parameters with the
     ** given name.
     **
     ** <P>
     ** A request can carry more than one parameter with a
     ** certain name. This method returns these parameters
     ** in the order of appearance.
     **
     ** @param   name
     **          the parameter name
     **
     ** @return   the array of parameter values
     **
     ** @see   #getParameterMap()
     ** @see   #getParameter(String)
     **/

    public String [] getParameterValues (String name);

    /**
     ** Associates an attribute with the given name and value
     ** with this request. If a portlet needs to communicate
     ** information to embedded servlets or JSP, this methods
     ** can used carry the information along.
     **
     ** <P>
     ** The portlet provider should take care that the
     ** the namespace of attribute names is not unnecessarily
     ** polluted. It is recommended to prefix all attributes
     ** the package and class name of the portlet that
     ** makes use of this method.
     **
     ** @param   name
     **          the attribute name
     ** @param   value
     **          the attribute value
     **/

    public void setAttribute (String name, Object value);

    /**
     ** Removes the attribute with the given name.
     **
     ** @param   name
     **          the name of attribute to be removed
     **/

    public void removeAttribute (String name);

   /**
     ** Returns the value of the attribute with the given name,
     ** or <CODE>null</CODE> if no attribute with the given name exists.
     **
     ** @param   name
     **          the attribute name
     **
     ** @return   the attribute value
     **/

    public Object getAttribute (String name);

    /**
     ** Returns an enumeration of names of all attributes available to
     ** this request. This method returns an empty enumeration if the
     ** request has no attributes available to it.
     **
     ** @return   an enumeration of attribute names
     **/

    public Enumeration getAttributeNames ();

    /**
     ** Returns whether this request was made using a secure channel, such
     ** as HTTPS.
     **
     ** @return   <CODE>true</CODE> if channel is secure, <BR>
     **           <CODE>false</CODE> otherwise
     **/

    public boolean isSecure ();

    /**
     ** Returns the window that the portlet is running in.
     **
     ** @return   the portlet window
     **/

    public PortletWindow getWindow ();

    /**
     ** Returns the data of the concrete portlet instance
     ** <P>
     ** If the portlet is run in CONFIGURE mode, the
     ** portlet data is not accessible and this method will
     ** return <CODE>null</CODE>.
     ** </P>
     **
     ** @return   the portlet data
     **/

    public PortletData getData ();

    /**
     ** Returns the current session or, if there is no current session,
     ** it creates one and returns it.
     **
     ** @return   the portlet session
     **/

    public PortletSession getPortletSession ();

    /**
     ** Returns the current session or, if there is no current session
     ** and the given flag is <CODE>true</CODE>, it creates one and returns
     ** it.
     **
     ** <P>
     ** If the given flag is <CODE>false</CODE> and there is no current
     ** portlet session, this method returns <CODE>null</CODE>.
     **
     ** @param   create
     **          <CODE>true</CODE> to create a news session, <BR>
     **          <CODE>false</CODE> to return <CODE>null</CODE> of there
     **          is no current session
     **
     ** @return   the portlet session
     **/

    public PortletSession getPortletSession (boolean create);

    /**
     *
     * Returns an array containing all of the <code>Cookie</code>
     * objects the client sent with this request.
     * This method returns <code>null</code> if no cookies were sent.
     *
     * @return		an array of all the <code>Cookies</code>
     *			included with this request, or <code>null</code>
     *			if the request has no cookies
     *
     *
     */

    public javax.servlet.http.Cookie[] getCookies();

    /**
     * Returns the user object. The user object contains useful
     * information about the user and his or her preferences.
     * 
     * <P>
     * If the user has not logged in or does not grant access to
     * the portlet, this method returns <CODE>null</CODE>.
     * 
     * @return the user object
     */
    public User getUser ();

    /**
     * Returns the <CODE>PortletSettings</CODE> object of the concrete portlet.
     * 
     * @return the PortletSettings object, or NULL if no PortletSettings object is available.
     */
    public PortletSettings getPortletSettings();

    /**
     *
     * Returns the value of the specified request header
     * as a <code>long</code> value that represents a 
     * <code>Date</code> object. Use this method with
     * headers that contain dates, such as
     * <code>If-Modified-Since</code>. 
     *
     * <p>The date is returned as
     * the number of milliseconds since January 1, 1970 GMT.
     * The header name is case insensitive.
     *
     * <p>If the request did not have a header of the
     * specified name, this method returns -1. If the header
     * can't be converted to a date, the method throws
     * an <code>IllegalArgumentException</code>.
     *
     * @param name		a <code>String</code> specifying the
     *				name of the header
     *
     * @return			a <code>long</code> value
     *				representing the date specified
     *				in the header expressed as
     *				the number of milliseconds
     *				since January 1, 1970 GMT,
     *				or -1 if the named header
     *				was not included with the
     *				reqest
     *
     * @exception	IllegalArgumentException	If the header value
     *							can't be converted
     *							to a date
     *
     */

    public long getDateHeader(String name);
    
    
    

    /**
     *
     * Returns the value of the specified request header
     * as a <code>String</code>. If the request did not include a header
     * of the specified name, this method returns <code>null</code>.
     * The header name is case insensitive. You can use
     * this method with any request header.
     *
     * @param name		a <code>String</code> specifying the
     *				header name
     *
     * @return			a <code>String</code> containing the
     *				value of the requested
     *				header, or <code>null</code>
     *				if the request does not
     *				have a header of that name
     *
     */			

    public String getHeader(String name); 




    /**
     *
     * Returns all the values of the specified request header
     * as an <code>Enumeration</code> of <code>String</code> objects.
     *
     * <p>Some headers, such as <code>Accept-Language</code> can be sent
     * by clients as several headers each with a different value rather than
     * sending the header as a comma separated list.
     *
     * <p>If the request did not include any headers
     * of the specified name, this method returns an empty
     * <code>Enumeration</code>.
     * The header name is case insensitive. You can use
     * this method with any request header.
     *
     * @param name		a <code>String</code> specifying the
     *				header name
     *
     * @return			a <code>Enumeration</code> containing the
     *				values of the requested
     *				header, or <code>null</code>
     *				if the request does not
     *				have any headers of that name
     *
     */			

    public Enumeration getHeaders(String name); 

    /**
     *
     * Returns an enumeration of all the header names
     * this request contains. If the request has no
     * headers, this method returns an empty enumeration.
     *
     * <p>Some portlet containers do not allow do not allow
     * portlets to access headers using this method, in
     * which case this method returns <code>null</code>
     *
     * @return			an enumeration of all the
     *				header names sent with this
     *				request; if the request has
     *				no headers, an empty enumeration;
     *				if the portlet container does not
     *				allow portlets to use this method,
     *				<code>null</code>
     *
     */

    public Enumeration getHeaderNames();
    
    
    

    /**
     *
     * Returns the value of the specified request header
     * as an <code>int</code>. If the request does not have a header
     * of the specified name, this method returns -1. If the
     * header cannot be converted to an integer, this method
     * throws a <code>NumberFormatException</code>.
     *
     * <p>The header name is case insensitive.
     *
     * @param name		a <code>String</code> specifying the name
     *				of a request header
     *
     * @return			an integer expressing the value 
     * 				of the request header or -1
     *				if the request doesn't have a
     *				header of this name
     *
     * @exception	NumberFormatException		If the header value
     *							can't be converted
     *							to an <code>int</code>
     */

    public int getIntHeader(String name);

    /**
     * Invalidates the cache entry of the portlet associated with this request, so
     * that the next portlet's response is not served from the cache, but is the
     * actual output of the portlet.
     * <P>
     * This method may always be called when a request is at hand.
     */
    public void invalidateCache ();

}
