/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.servlet;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpOptions;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpTrace;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyService;
import org.apache.portals.applications.webcontent2.proxy.impl.ServletRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Generic reverse proxy servlet invoking {@link ReverseProxyService} injected.
 */
public class GenericReverseProxyServlet extends HttpServlet
{

    private static final long serialVersionUID = 1L;

    /**
     * Default allowed HTTP methods.
     */
    private static final Set<String> DEFAULT_ALLOWED_HTTP_METHODS = 
        new HashSet<String>(Arrays.asList(HttpGet.METHOD_NAME, 
                                          HttpHead.METHOD_NAME, 
                                          HttpPost.METHOD_NAME, 
                                          HttpPut.METHOD_NAME, 
                                          HttpDelete.METHOD_NAME,
                                          HttpOptions.METHOD_NAME,
                                          HttpTrace.METHOD_NAME));

    private static Logger log = LoggerFactory.getLogger(GenericReverseProxyServlet.class);

    /**
     * Internal {@link ReverseProxyService} to invoke while processing reverse proxy requests.
     */
    private ReverseProxyService proxyService;

    /**
     * Zero-constructor default constructor.
     */
    public GenericReverseProxyServlet()
    {
    }

    /**
     * Returns the internal {@link ReverseProxyService} instance.
     * @return
     */
    public ReverseProxyService getProxyService()
    {
        return proxyService;
    }

    /**
     * Sets the internal {@link ReverseProxyService} instance.
     * @param proxyService
     */
    public void setProxyService(ReverseProxyService proxyService)
    {
        this.proxyService = proxyService;
    }

    /**
     * Invokes the internal {@link ReverseProxyService} on allowed HTTP requests.
     */
    @Override
    protected void service(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
    {
        if (!isAllowedHttpMethod(request))
        {
            super.service(request, response);
        }
        else
        {
            try
            {
                ServletRequestContext requestContext = new ServletRequestContext(request, response);
                ProxyContext proxyContext = new ProxyContext(requestContext);
                ProxyContext.setCurrentProxyContext(proxyContext);
                getProxyService().invoke(proxyContext);
            }
            catch (ReverseProxyException e)
            {
                if (e.getStatusCode() > 0)
                {
                    if (log.isDebugEnabled())
                    {
                        log.error("Response proxy processing exception occurred.", e);
                    }
                    else
                    {
                        log.error("Response proxy processing exception occurred. " + e);
                    }

                    response.sendError(e.getStatusCode(), e.getLocalizedMessage());
                }
                else
                {
                    throw new ServletException(e);
                }
            }
            catch (IOException e)
            {
                throw e;
            }
            catch (Exception e)
            {
                throw new ServletException(e);
            }
            finally
            {
                ProxyContext.removeCurrentProxyContext();
            }
        }
    }

    /**
     * Returns true if the HTTP method used by <code>request</code> is allowed by this.
     * Otherwise returns false.
     * @param request
     * @return
     */
    protected boolean isAllowedHttpMethod(HttpServletRequest request)
    {
        return DEFAULT_ALLOWED_HTTP_METHODS.contains(request.getMethod());
    }
}
