/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.servlet;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;

import org.apache.commons.chain.Command;
import org.apache.commons.lang.StringUtils;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.conn.ssl.X509HostnameVerifier;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.portals.applications.webcontent2.proxy.HttpClientContextBuilder;
import org.apache.portals.applications.webcontent2.proxy.ProxyMappingRegistry;
import org.apache.portals.applications.webcontent2.proxy.builder.ProxyServices;
import org.apache.portals.applications.webcontent2.proxy.impl.DefaultProxyMappingRegistry;
import org.apache.portals.applications.webcontent2.proxy.impl.DefaultReverseProxyService;
import org.apache.portals.applications.webcontent2.proxy.impl.ProxyProcessingChain;
import org.apache.portals.applications.webcontent2.proxy.util.ProxyCommandUtils;
import org.apache.portals.applications.webcontent2.proxy.util.YamlConfigUtils;

/**
 * Simple reverse proxy servlet implementation as an example and a base class to be overriden.
 */
public class SimpleReverseProxyServlet extends GenericReverseProxyServlet
{

    private static final long serialVersionUID = 1L;

    /**
     * Reverse proxy mapping initialization parameter name.
     */
    public static final String MAPPINGS_PARAM_NAME = "mappings";

    /**
     * Internal reverse proxy mapping registry.
     */
    private ProxyMappingRegistry proxyMappingRegistry;

    /**
     * Internal {@link HttpClientBuilder} instance.
     */
    private HttpClientBuilder httpClientBuilder;

    /**
     * Internal {@link HttpClientContextBuilder} instance.
     */
    private HttpClientContextBuilder httpClientContextBuilder;

    /**
     * SSL Socket Connection Factory.
     */
    private LayeredConnectionSocketFactory sslSocketFactory;

    /**
     * SSL Hostname Verifier.
     */
    private X509HostnameVerifier hostnameVerifier = SSLConnectionSocketFactory.BROWSER_COMPATIBLE_HOSTNAME_VERIFIER;

    /**
     * Internal root reverse proxy processing chain instance.
     */
    private ProxyProcessingChain proxyServiceCommand;

    /**
     * Zero-argument default constructor.
     */
    public SimpleReverseProxyServlet()
    {
        super();
    }

    /**
     * Creates the root reverse proxy processing chain instance,
     * and initializes all the {@link Command}s inside the chain.
     */
    @Override
    public void init(ServletConfig servletConfig) throws ServletException
    {
        super.init(servletConfig);
        initHostnameVerifier(servletConfig);

        proxyServiceCommand = createProxyServiceCommand();
        ProxyCommandUtils.initializeAllCommands(proxyServiceCommand);

        DefaultReverseProxyService proxyService = new DefaultReverseProxyService(proxyServiceCommand);
        setProxyService(proxyService);
    }

    /**
     * Initializes SSL hostnameVerifier (type of {@link X509HostnameVerifier})
     * from the init parameter, "ssl-hostname-verifier".
     * <p>
     * The init parameter can be any of "ALLOW_ALL_HOSTNAME_VERIFIER", "BROWSER_COMPATIBLE_HOSTNAME_VERIFIER"
     * or "STRICT_HOSTNAME_VERIFIER", case-insensitively. The default value is "BROWSER_COMPATIBLE_HOSTNAME_VERIFIER".
     * </p>
     * @param servletConfig
     */
    protected void initHostnameVerifier(ServletConfig servletConfig) {
        String param = StringUtils.trim(servletConfig.getInitParameter("ssl-hostname-verifier"));

        if (StringUtils.isNotEmpty(param))
        {
            if (StringUtils.equalsIgnoreCase("ALLOW_ALL_HOSTNAME_VERIFIER", param))
            {
                setHostnameVerifier(SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
            }
            else if (StringUtils.equalsIgnoreCase("BROWSER_COMPATIBLE_HOSTNAME_VERIFIER", param))
            {
                setHostnameVerifier(SSLConnectionSocketFactory.BROWSER_COMPATIBLE_HOSTNAME_VERIFIER);
            }
            else if (StringUtils.equalsIgnoreCase("STRICT_HOSTNAME_VERIFIER", param))
            {
                setHostnameVerifier(SSLConnectionSocketFactory.STRICT_HOSTNAME_VERIFIER);
            }
        }
    }

    /**
     * Returns the internal {@link ProxyMappingRegistry} instance.
     * @return
     */
    public ProxyMappingRegistry getProxyMappingRegistry()
    {
        if (proxyMappingRegistry == null)
        {
            proxyMappingRegistry = new DefaultProxyMappingRegistry();
            final String param = getServletConfig().getInitParameter(MAPPINGS_PARAM_NAME);
            proxyMappingRegistry.addAllProxyMappings(YamlConfigUtils.loadProxyMappings(param, getServletContext()));
            return proxyMappingRegistry;
        }

        return proxyMappingRegistry;
    }

    /**
     * Sets the internal {@link ProxyMappingRegistry} instance.
     * @param proxyMappingRegistry
     */
    public void setProxyMappingRegistry(ProxyMappingRegistry proxyMappingRegistry)
    {
        this.proxyMappingRegistry = proxyMappingRegistry;
    }

    /**
     * Returns the internal {@link HttpClientBuilder} instance.
     * @return
     */
    public HttpClientBuilder getHttpClientBuilder()
    {
        if (httpClientBuilder == null)
        {
            httpClientBuilder = createHttpClientBuilder();
        }

        return httpClientBuilder;
    }

    /**
     * Sets the internal {@link HttpClientBuilder} instance.
     * @param httpClientBuilder
     */
    public void setHttpClientBuilder(HttpClientBuilder httpClientBuilder)
    {
        this.httpClientBuilder = httpClientBuilder;
    }

    /**
     * Returns the internal {@link HttpClientContextBuilder} instance.
     * @return
     */
    public HttpClientContextBuilder getHttpClientContextBuilder()
    {
        return httpClientContextBuilder;
    }

    /**
     * Sets the internal {@link HttpClientContextBuilder} instance.
     * @param httpClientContextBuilder
     */
    public void setHttpClientContextBuilder(HttpClientContextBuilder httpClientContextBuilder)
    {
        this.httpClientContextBuilder = httpClientContextBuilder;
    }

    /**
     * Returns the internal SSL Socket Connection Factory ({@link LayeredConnectionSocketFactory}).
     * If not set, it invokes {@link #createSslSocketFactory()} to create a default one.
     * @return
     */
    public LayeredConnectionSocketFactory getSslSocketFactory()
    {
        if (sslSocketFactory == null)
        {
            sslSocketFactory = createSslSocketFactory();
        }

        return sslSocketFactory;
    }

    /**
     * Sets the internal SSL Socket Connection Factory ({@link LayeredConnectionSocketFactory}).
     * @param sslSocketFactory
     */
    public void setSslSocketFactory(LayeredConnectionSocketFactory sslSocketFactory)
    {
        this.sslSocketFactory = sslSocketFactory;
    }

    /**
     * Returns the internal SSL Hostname Verifier.
     * @return
     */
    public X509HostnameVerifier getHostnameVerifier()
    {
        return hostnameVerifier;
    }

    /**
     * Sets the internal SSL Hostname Verifier.
     * @param hostnameVerifier
     */
    public void setHostnameVerifier(X509HostnameVerifier hostnameVerifier)
    {
        this.hostnameVerifier = hostnameVerifier;
    }

    /**
     * Disposes all the {@link Command}s in the root reverse proxy processing chain.
     */
    @Override
    public void destroy()
    {
        ProxyCommandUtils.destroyAllCommands(proxyServiceCommand);
        super.destroy();
    }

    /**
     * {@link #init(ServletConfig)} method invokes this method to
     * create the root reverse proxy processing chain ({@link ProxyProcessingChain}).
     * @return
     */
    protected ProxyProcessingChain createProxyServiceCommand() {
        return ProxyServices.createDefault().build(getProxyMappingRegistry(), getHttpClientBuilder(), getHttpClientContextBuilder());
    }

    /**
     * Creates {@link HttpClientBuilder} instance.
     * This method is invoked by {@link #getHttpClientBuilder()}
     * if an {@link HttpClientBuilder} instance was not set before.
     * @return
     */
    protected HttpClientBuilder createHttpClientBuilder() {
        HttpClientBuilder builder = HttpClients.custom().disableRedirectHandling().useSystemProperties();

        LayeredConnectionSocketFactory sslSocketFactory = getSslSocketFactory();

        if (sslSocketFactory != null)
        {
            builder.setSSLSocketFactory(sslSocketFactory);
        }

        return builder;
    }

    /**
     * Creates the default internal SSL Socket Connection Factory ({@link LayeredConnectionSocketFactory}).
     * @return
     */
    protected LayeredConnectionSocketFactory createSslSocketFactory()
    {
        return new SSLConnectionSocketFactory(SSLContexts.createDefault(), getHostnameVerifier());
    }
}
