/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewriter;

/**
 * Abstract base implementation of {@link ProxyMapping}
 * to provide the basic management features for attributes and content rewriters.
 */
public abstract class AbstractProxyMapping implements ProxyMapping
{

    /**
     * Attribute key-value map.
     */
    private Map<String, Object> attributes;

    /**
     * {@link ContentRewriter} map per content mime type.
     */
    private Map<String, ContentRewriter> contentRewriters;

    /**
     * {@inheritDoc}
     */
    public Object getAttribute(String name)
    {
        if (attributes != null)
        {
            return attributes.get(name);
        }

        return null;
    }

    /**
     * {@inheritDoc}
     */
    public void setAttribute(String name, Object value)
    {
        if (attributes == null)
        {
            attributes = new HashMap<String, Object>();
        }

        attributes.put(name, value);
    }

    /**
     * {@inheritDoc}
     */
    public void removeAttribute(String name)
    {
        if (attributes != null)
        {
            attributes.remove(name);
        }
    }

    /**
     * {@inheritDoc}
     */
    public Map<String, Object> getAttributesMap()
    {
        if (attributes == null)
        {
            return Collections.emptyMap();
        }
        else
        {
            return Collections.unmodifiableMap(attributes);
        }
    }

    /**
     * Stores all the given <code>attributes</code> map in mapping.
     */
    public void setAttributesMap(String name, Map<String, Object> attributes)
    {
        this.attributes = new HashMap<String, Object>();

        if (attributes != null)
        {
            this.attributes.putAll(attributes);
        }
    }

    /**
     * Returns an unmodifiable map of the internal content rewriters.
     * If nothing exists, then it returns an empty map.
     * @return
     */
    public Map<String, ContentRewriter> getContentRewriters()
    {
        if (contentRewriters == null)
        {
            return Collections.emptyMap();
        }

        return Collections.unmodifiableMap(contentRewriters);
    }

    /**
     * Sets the internal content rewriters map by copying <code>contentRewriters</code>.
     * @param contentRewriters
     */
    public void setContentRewriters(Map<String, ContentRewriter> contentRewriters)
    {
        this.contentRewriters = new HashMap<String, ContentRewriter>();

        if (contentRewriters != null)
        {
            this.contentRewriters.putAll(contentRewriters);
        }
    }

    /**
     * {@inheritDoc}
     */
    public ContentRewriter getContentRewriter(String mimeType)
    {
        if (mimeType != null)
        {
            return getContentRewriters().get(mimeType);
        }

        return null;
    }

}
