/* Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.portlet.faces.tckharness;

import com.thoughtworks.selenium.SeleneseTestCase;
import com.thoughtworks.selenium.Selenium;
import com.thoughtworks.selenium.DefaultSelenium;
import com.thoughtworks.selenium.SeleniumException;

import java.text.MessageFormat;

import java.util.HashMap;
import java.util.Properties;
import java.util.Enumeration;
import java.util.Collection;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import java.util.regex.Matcher;
import java.util.logging.Level;
import java.util.logging.Logger;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.FileNotFoundException;

import org.junit.BeforeClass;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized.Parameters;

import static junit.framework.Assert.assertTrue;
import static junit.framework.Assert.fail;

@RunWith(TckParameterized.class)
public class TckTestCase
{
  private static String MAX_WAIT = "10000";

  protected static Selenium sSelenium;

  private String mPageName, mTestName;

  private static final String TEST_RESULT_NAME_XPATH = 
    "//span[@id=\"{0}-test-name\"]";
  private static final String TEST_RESULT_STATUS_XPATH = 
    "//span[@id=\"{0}-result-status\"]";
  private static final String TEST_RESULT_DETAIL_XPATH = 
    "//span[@id=\"{0}-result-detail\"]";

  private static final String[] ACTION_XPATHS =
  { "//input[@type=\"submit\" and @value=\"Run Test\"]", 
    "//a[text()=\"Run Test\"]" };

  private static final String TEST_PASSED_XPATH = 
    "//span[@id=\"{0}-result-status\" and text()=\"SUCCESS\"]";
  private static final String TEST_FAILED_XPATH = 
    "//span[@id=\"{0}-result-status\" and text()=\"FAILED\"]";

  private static final String TCK_HOME = "BRIDGE_301_TCK_HOME";
  private static final String PROPS_FILE = "bridge.tck.properties";
  private static final String LOGIN_FILE = "bridge.tck.login.properties";
  private static final String LOGIN_FILE_KEY = "bridge.tck.login.file";

  private static final String CLIENT_BASE_URL_KEY = 
    "bridge.tck.test.base-url";
  private static final String TEST_FILE_KEY = "bridge.tck.test.file";
  private static final String TEST_FILTER_KEY = "bridge.tck.test.filter";
  private static final String TEST_EXCLUSIONS_FILE_KEY = 
    "bridge.tck.test.exclusions.file";
  private static final String SELENIUM_BROWSER_KEY = "bridge.tck.browser";
  private static final String SELENIUM_HOST_KEY = 
    "bridge.tck.selenium.host";
  private static final String SELENIUM_PORT_KEY = 
    "bridge.tck.selenium.port";

  private static final String DEFAULT_SELENIUM_BROWSER = "*firefox";
  private static final String DEFAULT_SELENIUM_HOST = "localhost";
  private static final String DEFAULT_SELENIUM_PORT = "4444";

  private static String sClientBaseUrl;
  private static String sTestFile;
  private static String sTestExclusionsFile;
  private static String sTestFilter;
  private static String sSeleniumBrowser; // = DEFAULT_SELENIUM_BROWSER;
  private static String sSeleniumHost; // = DEFAULT_SELENIUM_HOST;
  private static int sSeleniumPort; // = DEFAULT_SELENIUM_PORT;

  // Name value pairs for fields in login page
  private static List<String[]> sLoginFields;

  // Name of button in login page
  private static String sLoginButton;

  public static void log(String msg)
  {
    Logger.getLogger("global").log(Level.INFO, msg);
  }

  public static void errLog(String msg)
  {
    Logger.getLogger("global").log(Level.SEVERE, msg);
  }

  private static String getProperty(Properties fileProperties, String key, 
                                    String defaultValue, boolean required)
  {
    boolean checkFirst = (defaultValue != null);
    if (System.getProperty(key) != null)
    {
      return System.getProperty(key);
    }
    if (fileProperties != null && (String) fileProperties.get(key) != null)
    {
      return (String) fileProperties.get(key);
    }
    if (defaultValue != null)
    {
      return defaultValue;
    }
    else if (required)
    {
      errLog("Property " + key + " has not been set.");
      System.exit(1);
    }
    return defaultValue;
  }

  static {
    log("static block");
    String tckHomeDir = System.getenv(TCK_HOME);
    if (tckHomeDir == null)
    {
      // Default value for the TCK home.
      tckHomeDir = 
          System.getProperty("user.home") + File.separator + "BridgeTckHome";
    }

    File propsFile = new File(tckHomeDir, PROPS_FILE);
    Properties fileProps = null;
    try
    {
      log("Home dir: " + tckHomeDir);
      FileInputStream fis = null;

      fis = new FileInputStream(propsFile);
      fileProps = new Properties();
      fileProps.load(fis);
    }
    catch (FileNotFoundException fnfe)
    {
      // Not necessarily an error.
      log("No TCK properties file, " + propsFile + " does not exist");
      propsFile = null;
    }
    catch (IOException ioe)
    {
      errLog("Error loading TCK properties file, " + propsFile.getName());
      System.exit(1);
    }

    sClientBaseUrl = 
        getProperty(fileProps, CLIENT_BASE_URL_KEY, null, true);
    sTestFilter = getProperty(fileProps, TEST_FILTER_KEY, null, false);
    sTestFile = getProperty(fileProps, TEST_FILE_KEY, null, true);
    sTestExclusionsFile = 
        getProperty(fileProps, TEST_EXCLUSIONS_FILE_KEY, null, true);

    sSeleniumHost = 
        getProperty(fileProps, SELENIUM_HOST_KEY, DEFAULT_SELENIUM_HOST, 
                    false);
    sSeleniumPort = 
        Integer.parseInt(getProperty(fileProps, SELENIUM_PORT_KEY, 
                                     DEFAULT_SELENIUM_PORT, false));

    sSeleniumBrowser = 
        getProperty(fileProps, SELENIUM_BROWSER_KEY, DEFAULT_SELENIUM_BROWSER, 
                    false);

    // Login properties file
    try
    {
      FileInputStream fis = null;
      if (System.getProperty(LOGIN_FILE_KEY) != null)
      {
        fis = new FileInputStream(System.getProperty(LOGIN_FILE_KEY));
      }
      else
      {
        fis = new FileInputStream(new File(tckHomeDir, LOGIN_FILE));
      }

      Properties props = new Properties();
      props.load(fis);

      Enumeration propsNameList = props.propertyNames();
      while (propsNameList.hasMoreElements())
      {
        String loginProp = (String) propsNameList.nextElement();
        if (props.get(loginProp).equals(""))
        {
          // A button login property is defined as name only
          sLoginButton = loginProp;
        }
        else
        {
          if (sLoginFields == null)
          {
            sLoginFields = new ArrayList<String[]>();
          }
          sLoginFields.add(new String[]
              { loginProp, (String) props.get(loginProp) });
        }
      }
    }
    catch (FileNotFoundException fnfe)
    {
      // No login properties file.  Not an error condition.
    }
    catch (IOException ioe)
    {
      errLog("Error loading login properties file, " + 
             propsFile.getName());
      System.exit(1);
    }

    log("test file: " + sTestFile);
    log("test filter: " + sTestFilter);
    log("End of static block");
  }

  public TckTestCase(String pageName, String testName)
  {
    mPageName = pageName;
    mTestName = testName;
  }

  @BeforeClass
  public static void setUpSession()
    throws Exception
  {
    log("setUpSession");
    try
    {
      log("SeleniumHost: " + sSeleniumHost);
      log("SeleniumPort: " + sSeleniumPort);
      log("SeleniumBrowser: " + sSeleniumBrowser);
      log("ClientBaseUrl: " + sClientBaseUrl);
      sSelenium = 
          new DefaultSelenium(sSeleniumHost, sSeleniumPort, sSeleniumBrowser, 
                              sClientBaseUrl);
      sSelenium.start();
      sSelenium.setTimeout(MAX_WAIT);
    }
    catch (Exception exception)
    {
      exception.printStackTrace();
      throw exception;
    }
  }

  @AfterClass
  public static void runAfterClass()
    throws Exception
  {
    if (sSelenium != null)
    {
      sSelenium.stop();
      sSelenium.shutDownSeleniumServer();
    }
  }

  @Before
  public void runBeforeEachTest()
    throws Exception
  {
    sSelenium.open(mPageName);
    sSelenium.waitForPageToLoad(MAX_WAIT);
  }

  @Test
  public void testPage()
  {
    login();

    performActions();

    String failMsg = null;
    String testName = null;
    String testStatus = null;
    String testDetails = null;
    String bodyText = sSelenium.getBodyText();
    try
    {
      testName = 
          sSelenium.getText(MessageFormat.format(TEST_RESULT_NAME_XPATH, 
                                                 new Object[]
              { mTestName }));
      testStatus = 
          sSelenium.getText(MessageFormat.format(TEST_RESULT_STATUS_XPATH, 
                                                 new Object[]
              { mTestName }));
      testDetails = 
          sSelenium.getText(MessageFormat.format(TEST_RESULT_DETAIL_XPATH, 
                                                 new Object[]
              { mTestName }));
    }
    catch (SeleniumException se)
    {
      // If any of the tags are missing from the portlet content the full screen mark-up
      // will be returned.
      failMsg = bodyText;
    }

    if (failMsg == null)
    {
      if ("FAILED".equals(testStatus))
      {
        // The portlet content is valid but the test has failed so display details
        failMsg = 
            new StringBuilder().append(testStatus).append("\n").append(testDetails).toString();
      }
    }

    assertTrue(failMsg, "SUCCESS".equals(testStatus));
  }

  public void performActions()
  {
    boolean actionsLeftToPerform = true;

    outer:
    while (actionsLeftToPerform)
    {
      for (String action: ACTION_XPATHS)
      {
        if (sSelenium.isElementPresent(action))
        {
          sSelenium.click(action);
          sSelenium.waitForPageToLoad(MAX_WAIT);
          continue outer;
        }
      }
      actionsLeftToPerform = false;
    }
  }

  public void login()
  {
    if (sLoginFields != null)
    {
      for (String[] field: sLoginFields)
      {
        if (!sSelenium.isElementPresent("//input[@name='" + field[0] + 
                                        "']"))
        {
          // Not a login page
          return;
        }
      }
      for (String[] field: sLoginFields)
      {
        sSelenium.type(field[0], field[1]);
      }
      sSelenium.click(sLoginButton);
      sSelenium.waitForPageToLoad(MAX_WAIT);
    }
  }

  public String getName()
  {
    return mTestName;
  }

  @Parameters
  public static Collection testData()
  {
    log("testData()");
    List testList = new ArrayList<String[]>(200);
    FileInputStream exIs = null;
    FileInputStream tis = null;
    Pattern filterPattern = null;

    if ((sTestFilter != null) && (sTestFilter.length() > 0))
    {
      filterPattern = Pattern.compile(sTestFilter);
    }

    try
    {
      exIs = new FileInputStream(sTestExclusionsFile);
    }
    catch (Exception e)
    {
      errLog("Unable to acccess test exclusions file, " + 
             sTestExclusionsFile + " Exception thrown: " + e.getMessage());
      System.exit(1);
    }
    Properties exProps = new Properties();
    try
    {
      exProps.loadFromXML(exIs);
    }
    catch (Exception e)
    {
      errLog("Unable to parse test exclusions file, " + 
             sTestExclusionsFile + " Exception thrown: " + e.getMessage());
      System.exit(1);
    }

    try
    {
      tis = new FileInputStream(sTestFile);
    }
    catch (Exception e)
    {
      errLog("Unable to acccess test description file, " + sTestFile + 
             " Exception thrown: " + e.getMessage());
      System.exit(1);
    }


    Properties testProps = new Properties();
    try
    {
      testProps.loadFromXML(tis);
    }
    catch (Exception e)
    {
      errLog("Unable to parse test description file, " + sTestFile + 
             " Exception thrown: " + e.getMessage());
      System.exit(1);
    }

    Enumeration tests = testProps.propertyNames();
    while (tests.hasMoreElements())
    {
      String page = (String) tests.nextElement();
      String testName = testProps.getProperty(page);
      boolean filterMatch = true;
      if (filterPattern != null)
      {
        Matcher m = filterPattern.matcher(testName);
        filterMatch = m.matches();
      }

      if (filterMatch)
      {
        if (exProps.getProperty(testName) == null)
        {
          testList.add(new String[]
              { page, testName });
        }
      }
    }

    return testList;
  }

  public static void main(String[] args)
  {
    // junit.textui.TestRunner.run(suite());
    new org.junit.runner.JUnitCore().runClasses(org.apache.myfaces.portlet.faces.tckharness.TckTestCase.class);
  }
}
