/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.rest.client;

import static org.apache.juneau.TestUtils.*;
import static org.apache.juneau.commons.utils.CollectionUtils.*;
import static org.apache.juneau.http.HttpParts.*;
import static org.apache.juneau.httppart.HttpPartSchema.*;

import java.io.*;

import org.apache.http.*;
import org.apache.juneau.*;
import org.apache.juneau.collections.*;
import org.apache.juneau.http.part.*;
import org.apache.juneau.httppart.*;
import org.apache.juneau.rest.annotation.*;
import org.apache.juneau.rest.mock.*;
import org.apache.juneau.rest.servlet.*;
import org.apache.juneau.serializer.*;
import org.apache.juneau.testutils.pojos.*;
import org.apache.juneau.uon.*;
import org.apache.juneau.utest.utils.*;
import org.junit.jupiter.api.*;

class RestClient_FormData_Test extends TestBase {

	@Rest
	public static class A extends BasicRestObject {
		@RestPost
		public Reader formData(org.apache.juneau.rest.RestRequest req) {
			return reader(req.getFormParams().asQueryString());
		}
	}

	//------------------------------------------------------------------------------------------------------------------
	// Method tests
	//------------------------------------------------------------------------------------------------------------------

	@Test void a01_formData_String_Object() throws Exception {
		client().formData("foo","bar").formData(part("foo",new StringBuilder("baz"),null)).build().post("/formData").run().assertContent("foo=bar&foo=baz");
		client().build().post("/formData").formData("foo","bar").formData("foo",new StringBuilder("baz")).run().assertContent("foo=bar&foo=baz");
		client().build().post("/formData").formData(null,"bar").run().assertContent("");
		client().build().post("/formData").formData("foo",null).run().assertContent("");
		client().build().post("/formData").formData(null,(String)null).run().assertContent("");
	}

	@Test void a03_formData_NameValuePair() throws Exception {
		client().formData(part("foo","bar")).build().post("/formData").formData(part("foo","baz")).run().assertContent("foo=bar&foo=baz");
	}

	@Test void a04_formDatas_Objects() throws Exception {
		client().formData(part("foo","bar")).build().post("/formData").run().assertContent("foo=bar");
		client().formData(parts("foo","bar","foo","baz").getAll()).build().post("/formData").run().assertContent("foo=bar&foo=baz");
		client().formData(part("foo","bar"),part("foo","baz")).build().post("/formData").run().assertContent("foo=bar&foo=baz");

		client().build().post("/formData").formData(part("foo","bar")).run().assertContent("foo=bar");
		client().build().post("/formData").formData(part("foo","bar"),part("foo","baz")).run().assertContent("foo=bar&foo=baz");

		client().build().post("/formData").formDataBean(ABean.get()).run().assertContent("a=1&b=foo");

		client().formData(part("foo","bar"),null).build().post("/formData").run().assertContent("foo=bar");
		client().formData(part("foo",null)).build().post("/formData").run().assertContent("");
		client().formData(part(null,null)).build().post("/formData").run().assertContent("");

		client().build().post("/formData").formData(part("foo",null)).run().assertContent("");
		client().build().post("/formData").formData(part(null,"foo")).run().assertContent("");
		client().build().post("/formData").formData(part(null,null)).run().assertContent("");

		client().formData(part("foo","bar",null)).build().post("/formData").run().assertContent("foo=bar");
		client().formData(part("foo",null,null)).build().post("/formData").run().assertContent("");
		client().formData(part("foo",null,null).skipIfEmpty().schema(HttpPartSchema.create().default_("bar").build())).build().post("/formData").run().assertContent("foo=bar");
	}

	@Test void a06_formData_String_Object_Schema() throws Exception {
		var l = l("bar","baz");
		var l2 = l("qux","quux");
		client().formData(part("foo",l,T_ARRAY_PIPES)).build().post("/formData").formData(part("foo",l2,T_ARRAY_PIPES)).run().assertContent().asString().asUrlDecode().is("foo=bar|baz&foo=qux|quux");
	}

	@Test void a07_formData_String_Object_Schema_Serializer() throws Exception {
		var l = l("bar","baz");
		client().formData(part("foo",l,T_ARRAY_PIPES).serializer(UonSerializer.DEFAULT)).build().post("/formData").run().assertContent().asString().asUrlDecode().is("foo=@(bar,baz)");
	}

	@Test void a09_formData_String_Supplier() throws Exception {
		var s = MutableSupplier.of(null);

		var x1 = client().formData(part("foo",s,null)).build();
		s.set(JsonList.of("foo","bar"));
		x1.post("/formData").run().assertContent().asString().asUrlDecode().is("foo=foo,bar");
		s.set(JsonList.of("bar","baz"));
		x1.post("/formData").run().assertContent().asString().asUrlDecode().is("foo=bar,baz");

		var x2 = client().build();
		s.set(JsonList.of("foo","bar"));
		x2.post("/formData").formData("foo",s).run().assertContent().asString().asUrlDecode().is("foo=foo,bar");
		s.set(JsonList.of("bar","baz"));
		x2.post("/formData").formData("foo",s).run().assertContent().asString().asUrlDecode().is("foo=bar,baz");
	}

	@Test void a10_formData_String_Supplier_Schema_Serializer() throws Exception {
		var s = MutableSupplier.of(JsonList.of("foo","bar"));
		var x = client().formData(part("foo",s,T_ARRAY_PIPES).serializer(FakeWriterSerializer.X)).build();
		x.post("/formData").run().assertContent().asString().asUrlDecode().is("foo=xfoo|barx");
		s.set(JsonList.of("bar","baz"));
		x.post("/formData").run().assertContent().asString().asUrlDecode().is("foo=xbar|bazx");
	}

	@Test void a11_formData_String_Supplier_Schema() throws Exception {
		var l1 = l("foo","bar");
		var l2 = l("bar","baz");
		var s = MutableSupplier.of(null);

		var x1 = client().formData(part("foo",s,T_ARRAY_PIPES)).build();
		s.set(l1);
		x1.post("/formData").run().assertContent().asString().asUrlDecode().is("foo=foo|bar");
		s.set(l2);
		x1.post("/formData").run().assertContent().asString().asUrlDecode().is("foo=bar|baz");

		var x2 = client().build();
		s.set(l1);
		x2.post("/formData").formData(part("foo",s,T_ARRAY_PIPES)).run().assertContent().asString().asUrlDecode().is("foo=foo|bar");
		s.set(l2);
		x2.post("/formData").formData(part("foo",s,T_ARRAY_PIPES)).run().assertContent().asString().asUrlDecode().is("foo=bar|baz");
	}

	public static class A12 implements HttpPartSerializer {
		@Override
		public HttpPartSerializerSession getPartSession() {
			return (type, schema, value) -> {
				throw new SerializeException("bad");
			};
		}
	}

	@Test void a12_badSerialization() {
		assertThrowsWithMessage(Exception.class, "bad", ()->client().formData(part("Foo","bar",null).serializer(new A12())).build().post("/").run());
	}

	//------------------------------------------------------------------------------------------------------------------
	// Helper methods.
	//------------------------------------------------------------------------------------------------------------------

	private static NameValuePair part(String name, Object val) {
		return basicPart(name, val);
	}

	private static SerializedPart part(String name, Object val, HttpPartSchema schema) {
		return serializedPart(name, val).schema(schema);
	}

	private static PartList parts(String...pairs) {
		return partList(pairs);
	}

	private static RestClient.Builder client() {
		return MockRestClient.create(A.class).json5();
	}
}