/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.client.deployment.application;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.flink.annotation.Internal;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.JobID;
import org.apache.flink.api.common.time.Time;
import org.apache.flink.client.ClientUtils;
import org.apache.flink.client.cli.ClientOptions;
import org.apache.flink.client.deployment.application.ApplicationExecutionException;
import org.apache.flink.client.deployment.application.JobStatusPollingUtils;
import org.apache.flink.client.deployment.application.UnsuccessfulExecutionException;
import org.apache.flink.client.deployment.application.executors.EmbeddedExecutorServiceLoader;
import org.apache.flink.client.program.PackagedProgram;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.DeploymentOptions;
import org.apache.flink.configuration.PipelineOptionsInternal;
import org.apache.flink.runtime.client.DuplicateJobSubmissionException;
import org.apache.flink.runtime.clusterframework.ApplicationStatus;
import org.apache.flink.runtime.dispatcher.DispatcherBootstrap;
import org.apache.flink.runtime.dispatcher.DispatcherGateway;
import org.apache.flink.runtime.jobmanager.HighAvailabilityMode;
import org.apache.flink.runtime.jobmaster.JobResult;
import org.apache.flink.runtime.messages.Acknowledge;
import org.apache.flink.runtime.messages.FlinkJobNotFoundException;
import org.apache.flink.runtime.rpc.FatalErrorHandler;
import org.apache.flink.util.ExceptionUtils;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.concurrent.FutureUtils;
import org.apache.flink.util.concurrent.ScheduledExecutor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class ApplicationDispatcherBootstrap
implements DispatcherBootstrap {
    public static final JobID ZERO_JOB_ID = new JobID(0L, 0L);
    @VisibleForTesting
    static final String FAILED_JOB_NAME = "(application driver)";
    private static final Logger LOG = LoggerFactory.getLogger(ApplicationDispatcherBootstrap.class);
    private final PackagedProgram application;
    private final Collection<JobID> recoveredJobIds;
    private final Configuration configuration;
    private final FatalErrorHandler errorHandler;
    private final CompletableFuture<Void> applicationCompletionFuture;
    private final CompletableFuture<Acknowledge> bootstrapCompletionFuture;
    private ScheduledFuture<?> applicationExecutionTask;

    private static boolean isCanceledOrFailed(ApplicationStatus applicationStatus) {
        return applicationStatus == ApplicationStatus.CANCELED || applicationStatus == ApplicationStatus.FAILED;
    }

    public ApplicationDispatcherBootstrap(PackagedProgram application, Collection<JobID> recoveredJobIds, Configuration configuration, DispatcherGateway dispatcherGateway, ScheduledExecutor scheduledExecutor, FatalErrorHandler errorHandler) {
        this.configuration = (Configuration)Preconditions.checkNotNull((Object)configuration);
        this.recoveredJobIds = (Collection)Preconditions.checkNotNull(recoveredJobIds);
        this.application = (PackagedProgram)Preconditions.checkNotNull((Object)application);
        this.errorHandler = (FatalErrorHandler)Preconditions.checkNotNull((Object)errorHandler);
        this.applicationCompletionFuture = this.fixJobIdAndRunApplicationAsync(dispatcherGateway, scheduledExecutor);
        this.bootstrapCompletionFuture = this.finishBootstrapTasks(dispatcherGateway);
    }

    public void stop() {
        if (this.applicationExecutionTask != null) {
            this.applicationExecutionTask.cancel(true);
        }
        if (this.applicationCompletionFuture != null) {
            this.applicationCompletionFuture.cancel(true);
        }
    }

    @VisibleForTesting
    ScheduledFuture<?> getApplicationExecutionFuture() {
        return this.applicationExecutionTask;
    }

    @VisibleForTesting
    CompletableFuture<Void> getApplicationCompletionFuture() {
        return this.applicationCompletionFuture;
    }

    @VisibleForTesting
    CompletableFuture<Acknowledge> getBootstrapCompletionFuture() {
        return this.bootstrapCompletionFuture;
    }

    private CompletableFuture<Acknowledge> finishBootstrapTasks(DispatcherGateway dispatcherGateway) {
        CompletionStage shutdownFuture = ((CompletableFuture)this.applicationCompletionFuture.handle((ignored, t) -> {
            if (t == null) {
                LOG.info("Application completed SUCCESSFULLY");
                return this.finish(dispatcherGateway, ApplicationStatus.SUCCEEDED);
            }
            Optional<ApplicationStatus> maybeApplicationStatus = this.extractApplicationStatus((Throwable)t);
            if (maybeApplicationStatus.isPresent() && ApplicationDispatcherBootstrap.isCanceledOrFailed(maybeApplicationStatus.get())) {
                ApplicationStatus applicationStatus = maybeApplicationStatus.get();
                LOG.info("Application {}: ", (Object)applicationStatus, t);
                return this.finish(dispatcherGateway, applicationStatus);
            }
            if (t instanceof CancellationException) {
                LOG.warn("Application has been cancelled because the {} is being stopped.", (Object)ApplicationDispatcherBootstrap.class.getSimpleName());
                return CompletableFuture.completedFuture(Acknowledge.get());
            }
            LOG.warn("Application failed unexpectedly: ", t);
            return FutureUtils.completedExceptionally((Throwable)t);
        })).thenCompose(Function.identity());
        FutureUtils.handleUncaughtException((CompletableFuture)shutdownFuture, (t, e) -> this.errorHandler.onFatalError(e));
        return shutdownFuture;
    }

    private CompletableFuture<Acknowledge> finish(DispatcherGateway dispatcherGateway, ApplicationStatus applicationStatus) {
        boolean shouldShutDownOnFinish = this.configuration.getBoolean(DeploymentOptions.SHUTDOWN_ON_APPLICATION_FINISH);
        return shouldShutDownOnFinish ? dispatcherGateway.shutDownCluster(applicationStatus) : CompletableFuture.completedFuture(Acknowledge.get());
    }

    private Optional<ApplicationStatus> extractApplicationStatus(Throwable t) {
        Optional maybeException = ExceptionUtils.findThrowable((Throwable)t, UnsuccessfulExecutionException.class);
        return maybeException.map(UnsuccessfulExecutionException::getStatus);
    }

    private CompletableFuture<Void> fixJobIdAndRunApplicationAsync(DispatcherGateway dispatcherGateway, ScheduledExecutor scheduledExecutor) {
        Optional configuredJobId = this.configuration.getOptional(PipelineOptionsInternal.PIPELINE_FIXED_JOB_ID);
        boolean submitFailedJobOnApplicationError = this.configuration.getBoolean(DeploymentOptions.SUBMIT_FAILED_JOB_ON_APPLICATION_ERROR);
        if (!HighAvailabilityMode.isHighAvailabilityModeActivated((Configuration)this.configuration) && !configuredJobId.isPresent()) {
            return this.runApplicationAsync(dispatcherGateway, scheduledExecutor, false, submitFailedJobOnApplicationError);
        }
        if (!configuredJobId.isPresent()) {
            this.configuration.set(PipelineOptionsInternal.PIPELINE_FIXED_JOB_ID, (Object)ZERO_JOB_ID.toHexString());
        }
        return this.runApplicationAsync(dispatcherGateway, scheduledExecutor, true, submitFailedJobOnApplicationError);
    }

    private CompletableFuture<Void> runApplicationAsync(DispatcherGateway dispatcherGateway, ScheduledExecutor scheduledExecutor, boolean enforceSingleJobExecution, boolean submitFailedJobOnApplicationError) {
        CompletableFuture applicationExecutionFuture = new CompletableFuture();
        Set tolerateMissingResult = Collections.synchronizedSet(new HashSet());
        this.applicationExecutionTask = scheduledExecutor.schedule(() -> this.runApplicationEntryPoint(applicationExecutionFuture, tolerateMissingResult, dispatcherGateway, scheduledExecutor, enforceSingleJobExecution, submitFailedJobOnApplicationError), 0L, TimeUnit.MILLISECONDS);
        return applicationExecutionFuture.thenCompose(jobIds -> this.getApplicationResult(dispatcherGateway, (Collection<JobID>)jobIds, tolerateMissingResult, scheduledExecutor));
    }

    private void runApplicationEntryPoint(CompletableFuture<List<JobID>> jobIdsFuture, Set<JobID> tolerateMissingResult, DispatcherGateway dispatcherGateway, ScheduledExecutor scheduledExecutor, boolean enforceSingleJobExecution, boolean submitFailedJobOnApplicationError) {
        if (submitFailedJobOnApplicationError && !enforceSingleJobExecution) {
            jobIdsFuture.completeExceptionally((Throwable)((Object)new ApplicationExecutionException(String.format("Submission of failed job in case of an application error ('%s') is not supported in non-HA setups.", DeploymentOptions.SUBMIT_FAILED_JOB_ON_APPLICATION_ERROR.key()))));
            return;
        }
        ArrayList<JobID> applicationJobIds = new ArrayList<JobID>(this.recoveredJobIds);
        try {
            EmbeddedExecutorServiceLoader executorServiceLoader = new EmbeddedExecutorServiceLoader(applicationJobIds, dispatcherGateway, scheduledExecutor);
            ClientUtils.executeProgram(executorServiceLoader, this.configuration, this.application, enforceSingleJobExecution, true);
            if (applicationJobIds.isEmpty()) {
                jobIdsFuture.completeExceptionally((Throwable)((Object)new ApplicationExecutionException("The application contains no execute() calls.")));
            } else {
                jobIdsFuture.complete(applicationJobIds);
            }
        }
        catch (Throwable t) {
            Optional maybeDuplicate = ExceptionUtils.findThrowable((Throwable)t, DuplicateJobSubmissionException.class);
            if (enforceSingleJobExecution && maybeDuplicate.isPresent() && ((DuplicateJobSubmissionException)maybeDuplicate.get()).isGloballyTerminated()) {
                JobID jobId = ((DuplicateJobSubmissionException)maybeDuplicate.get()).getJobID();
                tolerateMissingResult.add(jobId);
                jobIdsFuture.complete(Collections.singletonList(jobId));
            }
            if (submitFailedJobOnApplicationError && applicationJobIds.isEmpty()) {
                JobID failedJobId = JobID.fromHexString((String)((String)this.configuration.get(PipelineOptionsInternal.PIPELINE_FIXED_JOB_ID)));
                dispatcherGateway.submitFailedJob(failedJobId, FAILED_JOB_NAME, t).thenAccept(ignored -> jobIdsFuture.complete(Collections.singletonList(failedJobId)));
            }
            jobIdsFuture.completeExceptionally((Throwable)((Object)new ApplicationExecutionException("Could not execute application.", t)));
        }
    }

    private CompletableFuture<Void> getApplicationResult(DispatcherGateway dispatcherGateway, Collection<JobID> applicationJobIds, Set<JobID> tolerateMissingResult, ScheduledExecutor executor) {
        List jobResultFutures = applicationJobIds.stream().map(jobId -> this.unwrapJobResultException(this.getJobResult(dispatcherGateway, (JobID)jobId, executor, tolerateMissingResult.contains(jobId)))).collect(Collectors.toList());
        return FutureUtils.waitForAll(jobResultFutures);
    }

    private CompletableFuture<JobResult> getJobResult(DispatcherGateway dispatcherGateway, JobID jobId, ScheduledExecutor scheduledExecutor, boolean tolerateMissingResult) {
        Time timeout = Time.milliseconds((long)((Duration)this.configuration.get(ClientOptions.CLIENT_TIMEOUT)).toMillis());
        Time retryPeriod = Time.milliseconds((long)((Duration)this.configuration.get(ClientOptions.CLIENT_RETRY_PERIOD)).toMillis());
        CompletableFuture<JobResult> jobResultFuture = JobStatusPollingUtils.getJobResult(dispatcherGateway, jobId, scheduledExecutor, timeout, retryPeriod);
        if (tolerateMissingResult) {
            return FutureUtils.handleException(jobResultFuture, FlinkJobNotFoundException.class, exception -> new JobResult.Builder().jobId(jobId).applicationStatus(ApplicationStatus.UNKNOWN).netRuntime(Long.MAX_VALUE).build());
        }
        return jobResultFuture;
    }

    private CompletableFuture<JobResult> unwrapJobResultException(CompletableFuture<JobResult> jobResult) {
        return jobResult.thenApply(result -> {
            if (result.isSuccess()) {
                return result;
            }
            throw new CompletionException((Throwable)((Object)UnsuccessfulExecutionException.fromJobResult(result, this.application.getUserCodeClassLoader())));
        });
    }
}

