/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.partition.hybrid.tiered.file;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.buffer.BufferHeader;
import org.apache.flink.runtime.io.network.buffer.BufferRecycler;
import org.apache.flink.runtime.io.network.buffer.CompositeBuffer;
import org.apache.flink.runtime.io.network.buffer.NetworkBuffer;
import org.apache.flink.runtime.io.network.buffer.ReadOnlySlicedNetworkBuffer;
import org.apache.flink.runtime.io.network.partition.BufferReaderWriterUtil;
import org.apache.flink.runtime.io.network.partition.hybrid.tiered.common.TieredStoragePartitionId;
import org.apache.flink.runtime.io.network.partition.hybrid.tiered.common.TieredStorageSubpartitionId;
import org.apache.flink.runtime.io.network.partition.hybrid.tiered.file.PartitionFileReader;
import org.apache.flink.runtime.io.network.partition.hybrid.tiered.file.ProducerMergedPartitionFileIndex;
import org.apache.flink.util.ExceptionUtils;
import org.apache.flink.util.IOUtils;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProducerMergedPartitionFileReader
implements PartitionFileReader {
    private static final Logger LOG = LoggerFactory.getLogger(ProducerMergedPartitionFileReader.class);
    private final ByteBuffer reusedHeaderBuffer = BufferReaderWriterUtil.allocatedHeaderBuffer();
    private final Path dataFilePath;
    private final ProducerMergedPartitionFileIndex dataIndex;
    private volatile FileChannel fileChannel;

    @VisibleForTesting
    ProducerMergedPartitionFileReader(Path dataFilePath, ProducerMergedPartitionFileIndex dataIndex) {
        this.dataFilePath = dataFilePath;
        this.dataIndex = dataIndex;
    }

    @Override
    public PartitionFileReader.ReadBufferResult readBuffer(TieredStoragePartitionId partitionId, TieredStorageSubpartitionId subpartitionId, int segmentId, int bufferIndex, MemorySegment memorySegment, BufferRecycler recycler, @Nullable PartitionFileReader.ReadProgress readProgress, @Nullable CompositeBuffer partialBuffer) throws IOException {
        this.lazyInitializeFileChannel();
        Tuple2<Long, Long> startAndEndOffset = this.getReadStartAndEndOffset(subpartitionId, bufferIndex, readProgress, partialBuffer);
        if (startAndEndOffset == null) {
            return null;
        }
        long readStartOffset = (Long)startAndEndOffset.f0;
        long readEndOffset = (Long)startAndEndOffset.f1;
        int numBytesToRead = Math.min(memorySegment.size(), (int)(readEndOffset - readStartOffset));
        if (numBytesToRead == 0) {
            return null;
        }
        LinkedList<Buffer> readBuffers = new LinkedList<Buffer>();
        ByteBuffer byteBuffer = memorySegment.wrap(0, numBytesToRead);
        this.fileChannel.position(readStartOffset);
        this.readFileDataToBuffer(memorySegment, recycler, byteBuffer);
        Tuple2<Integer, Integer> partial = this.sliceBuffer(byteBuffer, memorySegment, partialBuffer, recycler, readBuffers);
        return ProducerMergedPartitionFileReader.getReadBufferResult(readBuffers, readStartOffset, readEndOffset, numBytesToRead, (Integer)partial.f0, (Integer)partial.f1);
    }

    @Override
    public long getPriority(TieredStoragePartitionId partitionId, TieredStorageSubpartitionId subpartitionId, int segmentId, int bufferIndex, @Nullable PartitionFileReader.ReadProgress readProgress) {
        this.lazyInitializeFileChannel();
        ProducerMergedReadProgress progress = ProducerMergedPartitionFileReader.convertToCurrentReadProgress(readProgress);
        if (progress != null && progress.getCurrentBufferOffset() != progress.getEndOfRegionOffset()) {
            return progress.getCurrentBufferOffset();
        }
        return this.dataIndex.getRegion(subpartitionId, bufferIndex).map(ProducerMergedPartitionFileIndex.FixedSizeRegion::getRegionStartOffset).orElse(Long.MAX_VALUE);
    }

    @Override
    public void release() {
        if (this.fileChannel != null) {
            try {
                this.fileChannel.close();
            }
            catch (IOException e) {
                ExceptionUtils.rethrow((Throwable)e, (String)"Failed to close file channel.");
            }
        }
        IOUtils.deleteFileQuietly((Path)this.dataFilePath);
    }

    private void lazyInitializeFileChannel() {
        if (this.fileChannel == null) {
            try {
                this.fileChannel = FileChannel.open(this.dataFilePath, StandardOpenOption.READ);
            }
            catch (IOException e) {
                ExceptionUtils.rethrow((Throwable)e, (String)"Failed to open file channel.");
            }
        }
    }

    private Tuple2<Integer, Integer> sliceBuffer(ByteBuffer byteBuffer, MemorySegment memorySegment, @Nullable CompositeBuffer partialBuffer, BufferRecycler bufferRecycler, List<Buffer> readBuffers) {
        Preconditions.checkState((this.reusedHeaderBuffer.position() == 0 ? 1 : 0) != 0);
        Preconditions.checkState((partialBuffer == null || partialBuffer.missingLength() > 0 ? 1 : 0) != 0);
        NetworkBuffer buffer = new NetworkBuffer(memorySegment, bufferRecycler);
        buffer.setSize(byteBuffer.remaining());
        try {
            BufferHeader header;
            int numPartialBytes;
            int numSlicedBytes = 0;
            if (partialBuffer != null) {
                buffer.retainBuffer();
                int position = byteBuffer.position() + partialBuffer.missingLength();
                numPartialBytes = partialBuffer.missingLength();
                partialBuffer.addPartialBuffer(buffer.readOnlySlice(byteBuffer.position(), numPartialBytes));
                numSlicedBytes += numPartialBytes;
                byteBuffer.position(position);
                readBuffers.add(partialBuffer);
            }
            partialBuffer = null;
            while (byteBuffer.hasRemaining() && (header = this.parseBufferHeader(byteBuffer)) != null) {
                numSlicedBytes += 8;
                if (header.getLength() <= byteBuffer.remaining()) {
                    buffer.retainBuffer();
                    ReadOnlySlicedNetworkBuffer slicedBuffer = buffer.readOnlySlice(byteBuffer.position(), header.getLength());
                    slicedBuffer.setDataType(header.getDataType());
                    slicedBuffer.setCompressed(header.isCompressed());
                    byteBuffer.position(byteBuffer.position() + header.getLength());
                    numSlicedBytes += header.getLength();
                    readBuffers.add(slicedBuffer);
                    continue;
                }
                buffer.retainBuffer();
                numPartialBytes = byteBuffer.remaining();
                numSlicedBytes += numPartialBytes;
                partialBuffer = new CompositeBuffer(header);
                partialBuffer.addPartialBuffer(buffer.readOnlySlice(byteBuffer.position(), numPartialBytes));
                readBuffers.add(partialBuffer);
                break;
            }
            Tuple2 tuple2 = Tuple2.of((Object)numSlicedBytes, (Object)ProducerMergedPartitionFileReader.getPartialBufferReadBytes(partialBuffer));
            return tuple2;
        }
        catch (Throwable throwable) {
            LOG.error("Failed to slice the read buffer {}.", (Object)byteBuffer, (Object)throwable);
            throw throwable;
        }
        finally {
            buffer.recycleBuffer();
        }
    }

    @Nullable
    private Tuple2<Long, Long> getReadStartAndEndOffset(TieredStorageSubpartitionId subpartitionId, int bufferIndex, @Nullable PartitionFileReader.ReadProgress currentReadProgress, @Nullable CompositeBuffer partialBuffer) {
        long readEndOffset;
        long readStartOffset;
        ProducerMergedReadProgress readProgress = ProducerMergedPartitionFileReader.convertToCurrentReadProgress(currentReadProgress);
        if (readProgress == null || readProgress.getCurrentBufferOffset() == readProgress.getEndOfRegionOffset()) {
            Optional<ProducerMergedPartitionFileIndex.FixedSizeRegion> regionOpt = this.dataIndex.getRegion(subpartitionId, bufferIndex);
            if (!regionOpt.isPresent()) {
                return null;
            }
            readStartOffset = regionOpt.get().getRegionStartOffset();
            readEndOffset = regionOpt.get().getRegionEndOffset();
        } else {
            readStartOffset = readProgress.getCurrentBufferOffset() + (long)ProducerMergedPartitionFileReader.getPartialBufferReadBytes(partialBuffer);
            readEndOffset = readProgress.getEndOfRegionOffset();
        }
        Preconditions.checkState((readStartOffset <= readEndOffset ? 1 : 0) != 0);
        return Tuple2.of((Object)readStartOffset, (Object)readEndOffset);
    }

    private static PartitionFileReader.ReadBufferResult getReadBufferResult(List<Buffer> readBuffers, long readStartOffset, long readEndOffset, int numBytesToRead, int numBytesRealRead, int numBytesReadPartialBuffer) {
        boolean shouldContinueRead = readStartOffset + (long)numBytesRealRead < readEndOffset;
        ProducerMergedReadProgress readProgress = new ProducerMergedReadProgress(readStartOffset + (long)numBytesRealRead - (long)numBytesReadPartialBuffer, readEndOffset);
        Preconditions.checkState((numBytesRealRead <= numBytesToRead && numBytesToRead - numBytesRealRead < 8 ? 1 : 0) != 0);
        return new PartitionFileReader.ReadBufferResult(readBuffers, shouldContinueRead, readProgress);
    }

    private void readFileDataToBuffer(MemorySegment memorySegment, BufferRecycler recycler, ByteBuffer byteBuffer) throws IOException {
        try {
            BufferReaderWriterUtil.readByteBufferFully(this.fileChannel, byteBuffer);
            byteBuffer.flip();
        }
        catch (Throwable throwable) {
            recycler.recycle(memorySegment);
            throw throwable;
        }
    }

    private static int getPartialBufferReadBytes(@Nullable CompositeBuffer partialBuffer) {
        return partialBuffer == null ? 0 : partialBuffer.readableBytes() + 8;
    }

    private static ProducerMergedReadProgress convertToCurrentReadProgress(@Nullable PartitionFileReader.ReadProgress readProgress) {
        if (readProgress == null) {
            return null;
        }
        Preconditions.checkState((boolean)(readProgress instanceof ProducerMergedReadProgress));
        return (ProducerMergedReadProgress)readProgress;
    }

    private BufferHeader parseBufferHeader(ByteBuffer buffer) {
        Preconditions.checkArgument((this.reusedHeaderBuffer.position() == 0 ? 1 : 0) != 0);
        BufferHeader header = null;
        try {
            if (buffer.remaining() >= 8) {
                header = BufferReaderWriterUtil.parseBufferHeader(buffer);
            }
        }
        catch (Throwable throwable) {
            this.reusedHeaderBuffer.clear();
            LOG.error("Failed to parse buffer header.", throwable);
            throw throwable;
        }
        this.reusedHeaderBuffer.clear();
        return header;
    }

    public static class ProducerMergedReadProgress
    implements PartitionFileReader.ReadProgress {
        private final long currentBufferOffset;
        private final long endOfRegionOffset;

        public ProducerMergedReadProgress(long currentBufferOffset, long endOfRegionOffset) {
            this.currentBufferOffset = currentBufferOffset;
            this.endOfRegionOffset = endOfRegionOffset;
        }

        public long getCurrentBufferOffset() {
            return this.currentBufferOffset;
        }

        public long getEndOfRegionOffset() {
            return this.endOfRegionOffset;
        }
    }
}

