/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.spark.source;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import org.apache.iceberg.FileScanTask;
import org.apache.iceberg.MicroBatches;
import org.apache.iceberg.ScanTaskGroup;
import org.apache.iceberg.Schema;
import org.apache.iceberg.SchemaParser;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.Table;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.io.InputFile;
import org.apache.iceberg.io.OutputFile;
import org.apache.iceberg.io.PositionOutputStream;
import org.apache.iceberg.io.SeekableInputStream;
import org.apache.iceberg.relocated.com.google.common.base.Joiner;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Iterables;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.spark.SparkReadConf;
import org.apache.iceberg.spark.source.SerializableTableWithSize;
import org.apache.iceberg.spark.source.SparkInputPartition;
import org.apache.iceberg.spark.source.SparkScan;
import org.apache.iceberg.spark.source.StreamingOffset;
import org.apache.iceberg.util.PropertyUtil;
import org.apache.iceberg.util.SnapshotUtil;
import org.apache.iceberg.util.TableScanUtil;
import org.apache.iceberg.util.Tasks;
import org.apache.iceberg.util.ThreadPools;
import org.apache.spark.api.java.JavaSparkContext;
import org.apache.spark.broadcast.Broadcast;
import org.apache.spark.sql.connector.read.InputPartition;
import org.apache.spark.sql.connector.read.PartitionReaderFactory;
import org.apache.spark.sql.connector.read.streaming.MicroBatchStream;
import org.apache.spark.sql.connector.read.streaming.Offset;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SparkMicroBatchStream
implements MicroBatchStream {
    private static final Joiner SLASH = Joiner.on((String)"/");
    private static final Logger LOG = LoggerFactory.getLogger(SparkMicroBatchStream.class);
    private final Table table;
    private final boolean caseSensitive;
    private final String expectedSchema;
    private final Broadcast<Table> tableBroadcast;
    private final long splitSize;
    private final int splitLookback;
    private final long splitOpenFileCost;
    private final boolean localityPreferred;
    private final StreamingOffset initialOffset;
    private final boolean skipDelete;
    private final boolean skipOverwrite;
    private final long fromTimestamp;

    SparkMicroBatchStream(JavaSparkContext sparkContext, Table table, SparkReadConf readConf, Schema expectedSchema, String checkpointLocation) {
        this.table = table;
        this.caseSensitive = readConf.caseSensitive();
        this.expectedSchema = SchemaParser.toJson((Schema)expectedSchema);
        this.localityPreferred = readConf.localityEnabled();
        this.tableBroadcast = sparkContext.broadcast((Object)SerializableTableWithSize.copyOf(table));
        this.splitSize = readConf.splitSize();
        this.splitLookback = readConf.splitLookback();
        this.splitOpenFileCost = readConf.splitOpenFileCost();
        this.fromTimestamp = readConf.streamFromTimestamp();
        InitialOffsetStore initialOffsetStore = new InitialOffsetStore(table, checkpointLocation, this.fromTimestamp);
        this.initialOffset = initialOffsetStore.initialOffset();
        this.skipDelete = readConf.streamingSkipDeleteSnapshots();
        this.skipOverwrite = readConf.streamingSkipOverwriteSnapshots();
    }

    public Offset latestOffset() {
        this.table.refresh();
        if (this.table.currentSnapshot() == null) {
            return StreamingOffset.START_OFFSET;
        }
        if (this.table.currentSnapshot().timestampMillis() < this.fromTimestamp) {
            return StreamingOffset.START_OFFSET;
        }
        Snapshot latestSnapshot = this.table.currentSnapshot();
        long addedFilesCount = PropertyUtil.propertyAsLong((Map)latestSnapshot.summary(), (String)"added-data-files", (long)-1L);
        addedFilesCount = addedFilesCount == -1L ? (long)Iterables.size((Iterable)latestSnapshot.addedDataFiles(this.table.io())) : addedFilesCount;
        return new StreamingOffset(latestSnapshot.snapshotId(), addedFilesCount, false);
    }

    public InputPartition[] planInputPartitions(Offset start, Offset end) {
        Preconditions.checkArgument((boolean)(end instanceof StreamingOffset), (String)"Invalid end offset: %s is not a StreamingOffset", (Object)end);
        Preconditions.checkArgument((boolean)(start instanceof StreamingOffset), (String)"Invalid start offset: %s is not a StreamingOffset", (Object)start);
        if (end.equals((Object)StreamingOffset.START_OFFSET)) {
            return new InputPartition[0];
        }
        StreamingOffset endOffset = (StreamingOffset)end;
        StreamingOffset startOffset = (StreamingOffset)start;
        List<FileScanTask> fileScanTasks = this.planFiles(startOffset, endOffset);
        CloseableIterable splitTasks = TableScanUtil.splitFiles((CloseableIterable)CloseableIterable.withNoopClose(fileScanTasks), (long)this.splitSize);
        ArrayList combinedScanTasks = Lists.newArrayList((Iterable)TableScanUtil.planTasks((CloseableIterable)splitTasks, (long)this.splitSize, (int)this.splitLookback, (long)this.splitOpenFileCost));
        InputPartition[] partitions = new InputPartition[combinedScanTasks.size()];
        Tasks.range((int)partitions.length).stopOnFailure().executeWith(this.localityPreferred ? ThreadPools.getWorkerPool() : null).run(index -> {
            partitions[index.intValue()] = new SparkInputPartition((ScanTaskGroup)combinedScanTasks.get((int)index), this.tableBroadcast, this.expectedSchema, this.caseSensitive, this.localityPreferred);
        });
        return partitions;
    }

    public PartitionReaderFactory createReaderFactory() {
        return new SparkScan.ReaderFactory(0);
    }

    public Offset initialOffset() {
        return this.initialOffset;
    }

    public Offset deserializeOffset(String json) {
        return StreamingOffset.fromJson(json);
    }

    public void commit(Offset end) {
    }

    public void stop() {
    }

    private List<FileScanTask> planFiles(StreamingOffset startOffset, StreamingOffset endOffset) {
        ArrayList fileScanTasks = Lists.newArrayList();
        StreamingOffset batchStartOffset = StreamingOffset.START_OFFSET.equals((Object)startOffset) ? SparkMicroBatchStream.determineStartingOffset(this.table, this.fromTimestamp) : startOffset;
        StreamingOffset currentOffset = null;
        do {
            if (currentOffset == null) {
                currentOffset = batchStartOffset;
            } else {
                Snapshot snapshotAfter = SnapshotUtil.snapshotAfter((Table)this.table, (long)currentOffset.snapshotId());
                currentOffset = new StreamingOffset(snapshotAfter.snapshotId(), 0L, false);
            }
            Snapshot snapshot = this.table.snapshot(currentOffset.snapshotId());
            if (snapshot == null) {
                throw new IllegalStateException(String.format("Cannot load current offset at snapshot %d, the snapshot was expired or removed", currentOffset.snapshotId()));
            }
            if (!this.shouldProcess(this.table.snapshot(currentOffset.snapshotId()))) {
                LOG.debug("Skipping snapshot: {} of table {}", (Object)currentOffset.snapshotId(), (Object)this.table.name());
                continue;
            }
            MicroBatches.MicroBatch latestMicroBatch = MicroBatches.from((Snapshot)this.table.snapshot(currentOffset.snapshotId()), (FileIO)this.table.io()).caseSensitive(this.caseSensitive).specsById(this.table.specs()).generate(currentOffset.position(), Long.MAX_VALUE, currentOffset.shouldScanAllFiles());
            fileScanTasks.addAll(latestMicroBatch.tasks());
        } while (currentOffset.snapshotId() != endOffset.snapshotId());
        return fileScanTasks;
    }

    private boolean shouldProcess(Snapshot snapshot) {
        String op;
        switch (op = snapshot.operation()) {
            case "append": {
                return true;
            }
            case "replace": {
                return false;
            }
            case "delete": {
                Preconditions.checkState((boolean)this.skipDelete, (String)"Cannot process delete snapshot: %s, to ignore deletes, set %s=true", (long)snapshot.snapshotId(), (Object)"streaming-skip-delete-snapshots");
                return false;
            }
            case "overwrite": {
                Preconditions.checkState((boolean)this.skipOverwrite, (String)"Cannot process overwrite snapshot: %s, to ignore overwrites, set %s=true", (long)snapshot.snapshotId(), (Object)"streaming-skip-overwrite-snapshots");
                return false;
            }
        }
        throw new IllegalStateException(String.format("Cannot process unknown snapshot operation: %s (snapshot id %s)", op.toLowerCase(Locale.ROOT), snapshot.snapshotId()));
    }

    private static StreamingOffset determineStartingOffset(Table table, Long fromTimestamp) {
        if (table.currentSnapshot() == null) {
            return StreamingOffset.START_OFFSET;
        }
        if (fromTimestamp == null) {
            return new StreamingOffset(SnapshotUtil.oldestAncestor((Table)table).snapshotId(), 0L, false);
        }
        if (table.currentSnapshot().timestampMillis() < fromTimestamp) {
            return StreamingOffset.START_OFFSET;
        }
        try {
            Snapshot snapshot = SnapshotUtil.oldestAncestorAfter((Table)table, (long)fromTimestamp);
            if (snapshot != null) {
                return new StreamingOffset(snapshot.snapshotId(), 0L, false);
            }
            return StreamingOffset.START_OFFSET;
        }
        catch (IllegalStateException e) {
            return new StreamingOffset(SnapshotUtil.oldestAncestor((Table)table).snapshotId(), 0L, false);
        }
    }

    private static class InitialOffsetStore {
        private final Table table;
        private final FileIO io;
        private final String initialOffsetLocation;
        private final Long fromTimestamp;

        InitialOffsetStore(Table table, String checkpointLocation, Long fromTimestamp) {
            this.table = table;
            this.io = table.io();
            this.initialOffsetLocation = SLASH.join((Object)checkpointLocation, (Object)"offsets/0", new Object[0]);
            this.fromTimestamp = fromTimestamp;
        }

        public StreamingOffset initialOffset() {
            InputFile inputFile = this.io.newInputFile(this.initialOffsetLocation);
            if (inputFile.exists()) {
                return this.readOffset(inputFile);
            }
            this.table.refresh();
            StreamingOffset offset = SparkMicroBatchStream.determineStartingOffset(this.table, this.fromTimestamp);
            OutputFile outputFile = this.io.newOutputFile(this.initialOffsetLocation);
            this.writeOffset(offset, outputFile);
            return offset;
        }

        private void writeOffset(StreamingOffset offset, OutputFile file) {
            try (PositionOutputStream outputStream = file.create();){
                BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)outputStream, StandardCharsets.UTF_8));
                writer.write(offset.json());
                writer.flush();
            }
            catch (IOException ioException) {
                throw new UncheckedIOException(String.format("Failed writing offset to: %s", this.initialOffsetLocation), ioException);
            }
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        private StreamingOffset readOffset(InputFile file) {
            try (SeekableInputStream in = file.newStream();){
                StreamingOffset streamingOffset = StreamingOffset.fromJson((InputStream)in);
                return streamingOffset;
            }
            catch (IOException ioException) {
                throw new UncheckedIOException(String.format("Failed reading offset from: %s", this.initialOffsetLocation), ioException);
            }
        }
    }
}

