/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hudi.common.util.collection;

import org.apache.hudi.common.model.HoodieRecord;
import org.apache.hudi.common.model.HoodieRecordPayload;
import org.apache.hudi.common.serialization.DefaultSerializer;
import org.apache.hudi.common.testutils.HoodieCommonTestHarness;
import org.apache.hudi.common.testutils.SchemaTestUtil;
import org.apache.hudi.common.testutils.SpillableMapTestUtils;

import org.apache.avro.generic.GenericRecord;
import org.apache.avro.generic.IndexedRecord;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import static org.junit.jupiter.api.Assertions.assertEquals;

/**
 * Tests RocksDB based map {@link RocksDbDiskMap}.
 */
public class TestRocksDbBasedMap extends HoodieCommonTestHarness {

  @BeforeEach
  public void setUp() {
    initPath();
  }

  @Test
  public void testSimple() throws IOException, URISyntaxException {
    try (RocksDbDiskMap records = new RocksDbDiskMap(basePath, new DefaultSerializer<>())) {
      SchemaTestUtil testUtil = new SchemaTestUtil();
      List<IndexedRecord> iRecords = testUtil.generateHoodieTestRecords(0, 100);
      ((GenericRecord) iRecords.get(0)).get(HoodieRecord.COMMIT_TIME_METADATA_FIELD).toString();
      List<String> recordKeys = SpillableMapTestUtils.upsertRecords(iRecords, records);

      // make sure records have spilled to disk
      Iterator<HoodieRecord<? extends HoodieRecordPayload>> itr = records.iterator();
      List<HoodieRecord> oRecords = new ArrayList<>();
      while (itr.hasNext()) {
        HoodieRecord<? extends HoodieRecordPayload> rec = itr.next();
        oRecords.add(rec);
        assert recordKeys.contains(rec.getRecordKey());
      }
      assertEquals(recordKeys.size(), oRecords.size());
    }
  }
}
