/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.partition.hybrid;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Deque;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.buffer.BufferRecycler;
import org.apache.flink.runtime.io.network.partition.BufferReaderWriterUtil;
import org.apache.flink.runtime.io.network.partition.ResultSubpartition;
import org.apache.flink.runtime.io.network.partition.hybrid.HsFileDataIndex;
import org.apache.flink.runtime.io.network.partition.hybrid.HsSubpartitionFileReader;
import org.apache.flink.runtime.io.network.partition.hybrid.HsSubpartitionViewInternalOperations;
import org.apache.flink.util.ExceptionUtils;
import org.apache.flink.util.Preconditions;

public class HsSubpartitionFileReaderImpl
implements HsSubpartitionFileReader {
    private final ByteBuffer headerBuf;
    private final int subpartitionId;
    private final FileChannel dataFileChannel;
    private final HsSubpartitionViewInternalOperations operations;
    private final CachedRegionManager cachedRegionManager;
    private final BufferIndexManager bufferIndexManager;
    private final Deque<BufferIndexOrError> loadedBuffers = new LinkedBlockingDeque<BufferIndexOrError>();
    private final Consumer<HsSubpartitionFileReader> fileReaderReleaser;
    private volatile boolean isFailed;

    public HsSubpartitionFileReaderImpl(int subpartitionId, FileChannel dataFileChannel, HsSubpartitionViewInternalOperations operations, HsFileDataIndex dataIndex, int maxBufferReadAhead, Consumer<HsSubpartitionFileReader> fileReaderReleaser, ByteBuffer headerBuf) {
        this.subpartitionId = subpartitionId;
        this.dataFileChannel = dataFileChannel;
        this.operations = operations;
        this.headerBuf = headerBuf;
        this.bufferIndexManager = new BufferIndexManager(maxBufferReadAhead);
        this.cachedRegionManager = new CachedRegionManager(subpartitionId, dataIndex);
        this.fileReaderReleaser = fileReaderReleaser;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        HsSubpartitionFileReaderImpl that = (HsSubpartitionFileReaderImpl)o;
        return this.subpartitionId == that.subpartitionId;
    }

    public int hashCode() {
        return Objects.hash(this.subpartitionId);
    }

    @Override
    public synchronized void readBuffers(Queue<MemorySegment> buffers, BufferRecycler recycler) throws IOException {
        int indexToLoad;
        if (this.isFailed) {
            throw new IOException("subpartition reader has already failed.");
        }
        int firstBufferToLoad = this.bufferIndexManager.getNextToLoad();
        if (firstBufferToLoad < 0) {
            return;
        }
        int numRemainingBuffer = this.cachedRegionManager.getRemainingBuffersInRegion(firstBufferToLoad);
        if (numRemainingBuffer == 0) {
            return;
        }
        this.moveFileOffsetToBuffer(firstBufferToLoad);
        int numLoaded = 0;
        while (!buffers.isEmpty() && (indexToLoad = this.bufferIndexManager.getNextToLoad()) >= 0 && numRemainingBuffer-- > 0) {
            Buffer buffer;
            MemorySegment segment = buffers.poll();
            try {
                buffer = BufferReaderWriterUtil.readFromByteChannel(this.dataFileChannel, this.headerBuf, segment, recycler);
                if (buffer == null) {
                    buffers.add(segment);
                    break;
                }
            }
            catch (Throwable throwable) {
                buffers.add(segment);
                throw throwable;
            }
            this.loadedBuffers.add(BufferIndexOrError.newBuffer(buffer, indexToLoad));
            this.bufferIndexManager.updateLastLoaded(indexToLoad);
            this.cachedRegionManager.advance(buffer.readableBytes() + 8);
            ++numLoaded;
        }
        if (this.loadedBuffers.size() <= numLoaded) {
            this.operations.notifyDataAvailable();
        }
    }

    @Override
    public synchronized void fail(Throwable failureCause) {
        BufferIndexOrError bufferIndexOrError;
        if (this.isFailed) {
            return;
        }
        this.isFailed = true;
        while ((bufferIndexOrError = this.loadedBuffers.pollLast()) != null) {
            if (!bufferIndexOrError.getBuffer().isPresent()) continue;
            ((Buffer)Preconditions.checkNotNull((Object)bufferIndexOrError.buffer)).recycleBuffer();
        }
        this.loadedBuffers.add(BufferIndexOrError.newError(failureCause));
        this.operations.notifyDataAvailable();
    }

    @Override
    public void prepareForScheduling() {
        int consumingOffset = this.operations.getConsumingOffset(true);
        this.bufferIndexManager.updateLastConsumed(consumingOffset);
        this.cachedRegionManager.updateConsumingOffset(consumingOffset);
    }

    @Override
    public int compareTo(HsSubpartitionFileReader that) {
        Preconditions.checkArgument((boolean)(that instanceof HsSubpartitionFileReaderImpl));
        return Long.compare(this.getNextOffsetToLoad(), ((HsSubpartitionFileReaderImpl)that).getNextOffsetToLoad());
    }

    public Deque<BufferIndexOrError> getLoadedBuffers() {
        return this.loadedBuffers;
    }

    @Override
    public Optional<ResultSubpartition.BufferAndBacklog> consumeBuffer(int nextBufferToConsume) throws Throwable {
        if (!this.checkAndGetFirstBufferIndexOrError(nextBufferToConsume).isPresent()) {
            return Optional.empty();
        }
        BufferIndexOrError current = (BufferIndexOrError)Preconditions.checkNotNull((Object)this.loadedBuffers.poll());
        BufferIndexOrError next = this.loadedBuffers.peek();
        Buffer.DataType nextDataType = next == null ? Buffer.DataType.NONE : next.getDataType();
        int backlog = this.loadedBuffers.size();
        int bufferIndex = current.getIndex();
        Buffer buffer = current.getBuffer().orElseThrow(() -> new NullPointerException("Get a non-throwable and non-buffer bufferIndexOrError, which is not allowed"));
        return Optional.of(ResultSubpartition.BufferAndBacklog.fromBufferAndLookahead(buffer, nextDataType, backlog, bufferIndex));
    }

    @Override
    public Buffer.DataType peekNextToConsumeDataType(int nextBufferToConsume) {
        Buffer.DataType dataType = Buffer.DataType.NONE;
        try {
            dataType = this.checkAndGetFirstBufferIndexOrError(nextBufferToConsume).map(BufferIndexOrError::getDataType).orElse(Buffer.DataType.NONE);
        }
        catch (Throwable throwable) {
            ExceptionUtils.rethrow((Throwable)throwable);
        }
        return dataType;
    }

    @Override
    public void releaseDataView() {
        this.fileReaderReleaser.accept(this);
    }

    @Override
    public int getBacklog() {
        return this.loadedBuffers.size();
    }

    private Optional<BufferIndexOrError> checkAndGetFirstBufferIndexOrError(int expectedBufferIndex) throws Throwable {
        if (this.loadedBuffers.isEmpty()) {
            return Optional.empty();
        }
        BufferIndexOrError peek = this.loadedBuffers.peek();
        if (peek.getThrowable().isPresent()) {
            throw peek.getThrowable().get();
        }
        if (peek.getIndex() != expectedBufferIndex) {
            return Optional.empty();
        }
        return Optional.of(peek);
    }

    private void moveFileOffsetToBuffer(int bufferIndex) throws IOException {
        Tuple2 indexAndOffset = this.cachedRegionManager.getNumSkipAndFileOffset(bufferIndex);
        this.dataFileChannel.position((Long)indexAndOffset.f1);
        for (int i = 0; i < (Integer)indexAndOffset.f0; ++i) {
            BufferReaderWriterUtil.positionToNextBuffer(this.dataFileChannel, this.headerBuf);
        }
        this.cachedRegionManager.skipAll(this.dataFileChannel.position());
    }

    private long getNextOffsetToLoad() {
        int bufferIndex = this.bufferIndexManager.getNextToLoad();
        if (bufferIndex < 0) {
            return Long.MAX_VALUE;
        }
        return this.cachedRegionManager.getFileOffset(bufferIndex);
    }

    public static class Factory
    implements HsSubpartitionFileReader.Factory {
        public static final Factory INSTANCE = new Factory();

        private Factory() {
        }

        @Override
        public HsSubpartitionFileReader createFileReader(int subpartitionId, FileChannel dataFileChannel, HsSubpartitionViewInternalOperations operation, HsFileDataIndex dataIndex, int maxBuffersReadAhead, Consumer<HsSubpartitionFileReader> fileReaderReleaser, ByteBuffer headerBuffer) {
            return new HsSubpartitionFileReaderImpl(subpartitionId, dataFileChannel, operation, dataIndex, maxBuffersReadAhead, fileReaderReleaser, headerBuffer);
        }
    }

    private static class CachedRegionManager {
        private final int subpartitionId;
        private final HsFileDataIndex dataIndex;
        private int consumingOffset = -1;
        private int currentBufferIndex;
        private int numSkip;
        private int numReadable;
        private long offset;

        private CachedRegionManager(int subpartitionId, HsFileDataIndex dataIndex) {
            this.subpartitionId = subpartitionId;
            this.dataIndex = dataIndex;
        }

        public void updateConsumingOffset(int consumingOffset) {
            this.consumingOffset = consumingOffset;
        }

        private long getFileOffset(int bufferIndex) {
            this.updateCachedRegionIfNeeded(bufferIndex);
            return this.currentBufferIndex == -1 ? Long.MAX_VALUE : this.offset;
        }

        private int getRemainingBuffersInRegion(int bufferIndex) {
            this.updateCachedRegionIfNeeded(bufferIndex);
            return this.numReadable;
        }

        private void skipAll(long newOffset) {
            this.offset = newOffset;
            this.numSkip = 0;
        }

        private Tuple2<Integer, Long> getNumSkipAndFileOffset(int bufferIndex) {
            this.updateCachedRegionIfNeeded(bufferIndex);
            Preconditions.checkState((this.numSkip >= 0 ? 1 : 0) != 0, (Object)"num skip must be greater than or equal to 0");
            Preconditions.checkState((this.currentBufferIndex <= bufferIndex ? 1 : 0) != 0);
            return new Tuple2((Object)this.numSkip, (Object)this.offset);
        }

        private void advance(long bufferSize) {
            if (this.isInCachedRegion(this.currentBufferIndex + 1)) {
                ++this.currentBufferIndex;
                --this.numReadable;
                this.offset += bufferSize;
            }
        }

        private void updateCachedRegionIfNeeded(int bufferIndex) {
            if (this.isInCachedRegion(bufferIndex)) {
                int numAdvance = bufferIndex - this.currentBufferIndex;
                this.numSkip += numAdvance;
                this.numReadable -= numAdvance;
                this.currentBufferIndex = bufferIndex;
                return;
            }
            Optional<HsFileDataIndex.ReadableRegion> lookupResultOpt = this.dataIndex.getReadableRegion(this.subpartitionId, bufferIndex, this.consumingOffset);
            if (!lookupResultOpt.isPresent()) {
                this.currentBufferIndex = -1;
                this.numReadable = 0;
                this.numSkip = 0;
                this.offset = -1L;
            } else {
                HsFileDataIndex.ReadableRegion cachedRegion = lookupResultOpt.get();
                this.currentBufferIndex = bufferIndex;
                this.numSkip = cachedRegion.numSkip;
                this.numReadable = cachedRegion.numReadable;
                this.offset = cachedRegion.offset;
            }
        }

        private boolean isInCachedRegion(int bufferIndex) {
            return bufferIndex < this.currentBufferIndex + this.numReadable && bufferIndex >= this.currentBufferIndex;
        }
    }

    static class BufferIndexManager {
        private final int maxBuffersReadAhead;
        private int lastLoaded = -1;
        private int lastConsumed = -1;

        BufferIndexManager(int maxBuffersReadAhead) {
            this.maxBuffersReadAhead = maxBuffersReadAhead;
        }

        private void updateLastLoaded(int lastLoaded) {
            Preconditions.checkState((this.lastLoaded <= lastLoaded ? 1 : 0) != 0);
            this.lastLoaded = lastLoaded;
        }

        private void updateLastConsumed(int lastConsumed) {
            this.lastConsumed = lastConsumed;
        }

        private int getNextToLoad() {
            int maxToLoad;
            int nextToLoad = Math.max(this.lastLoaded, this.lastConsumed) + 1;
            return nextToLoad <= (maxToLoad = this.lastConsumed + this.maxBuffersReadAhead) ? nextToLoad : -1;
        }
    }

    public static class BufferIndexOrError {
        @Nullable
        private final Buffer buffer;
        private final int index;
        @Nullable
        private final Throwable throwable;

        private BufferIndexOrError(@Nullable Buffer buffer, int index, @Nullable Throwable throwable) {
            this.buffer = buffer;
            this.index = index;
            this.throwable = throwable;
        }

        public Buffer.DataType getDataType() {
            return this.buffer == null ? Buffer.DataType.NONE : this.buffer.getDataType();
        }

        private static BufferIndexOrError newError(Throwable throwable) {
            return new BufferIndexOrError(null, -1, (Throwable)Preconditions.checkNotNull((Object)throwable));
        }

        private static BufferIndexOrError newBuffer(Buffer buffer, int index) {
            return new BufferIndexOrError((Buffer)Preconditions.checkNotNull((Object)buffer), index, null);
        }

        public Optional<Buffer> getBuffer() {
            return Optional.ofNullable(this.buffer);
        }

        public Optional<Throwable> getThrowable() {
            return Optional.ofNullable(this.throwable);
        }

        public int getIndex() {
            Preconditions.checkNotNull((Object)this.buffer, (String)"Is error, cannot get index.");
            return this.index;
        }
    }
}

