/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.catalog;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.apache.flink.annotation.Internal;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.ExecutionConfig;
import org.apache.flink.configuration.ReadableConfig;
import org.apache.flink.table.api.CatalogNotExistException;
import org.apache.flink.table.api.TableException;
import org.apache.flink.table.api.ValidationException;
import org.apache.flink.table.catalog.Catalog;
import org.apache.flink.table.catalog.CatalogBaseTable;
import org.apache.flink.table.catalog.CatalogChange;
import org.apache.flink.table.catalog.CatalogDatabase;
import org.apache.flink.table.catalog.CatalogDescriptor;
import org.apache.flink.table.catalog.CatalogMaterializedTable;
import org.apache.flink.table.catalog.CatalogModel;
import org.apache.flink.table.catalog.CatalogPartition;
import org.apache.flink.table.catalog.CatalogPartitionSpec;
import org.apache.flink.table.catalog.CatalogRegistry;
import org.apache.flink.table.catalog.CatalogStore;
import org.apache.flink.table.catalog.CatalogStoreHolder;
import org.apache.flink.table.catalog.CatalogTable;
import org.apache.flink.table.catalog.CatalogView;
import org.apache.flink.table.catalog.Column;
import org.apache.flink.table.catalog.ContextResolvedModel;
import org.apache.flink.table.catalog.ContextResolvedTable;
import org.apache.flink.table.catalog.DataTypeFactory;
import org.apache.flink.table.catalog.DataTypeFactoryImpl;
import org.apache.flink.table.catalog.DefaultSchemaResolver;
import org.apache.flink.table.catalog.ObjectIdentifier;
import org.apache.flink.table.catalog.ObjectPath;
import org.apache.flink.table.catalog.QueryOperationCatalogView;
import org.apache.flink.table.catalog.ResolvedCatalogBaseTable;
import org.apache.flink.table.catalog.ResolvedCatalogMaterializedTable;
import org.apache.flink.table.catalog.ResolvedCatalogModel;
import org.apache.flink.table.catalog.ResolvedCatalogTable;
import org.apache.flink.table.catalog.ResolvedCatalogView;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.catalog.SchemaResolver;
import org.apache.flink.table.catalog.TableChange;
import org.apache.flink.table.catalog.TableDistribution;
import org.apache.flink.table.catalog.TemporaryOperationListener;
import org.apache.flink.table.catalog.UnresolvedIdentifier;
import org.apache.flink.table.catalog.exceptions.CatalogException;
import org.apache.flink.table.catalog.exceptions.DatabaseAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotEmptyException;
import org.apache.flink.table.catalog.exceptions.DatabaseNotExistException;
import org.apache.flink.table.catalog.exceptions.ModelNotExistException;
import org.apache.flink.table.catalog.exceptions.PartitionNotExistException;
import org.apache.flink.table.catalog.exceptions.TableAlreadyExistException;
import org.apache.flink.table.catalog.exceptions.TableNotExistException;
import org.apache.flink.table.catalog.listener.AlterDatabaseEvent;
import org.apache.flink.table.catalog.listener.AlterModelEvent;
import org.apache.flink.table.catalog.listener.AlterTableEvent;
import org.apache.flink.table.catalog.listener.CatalogContext;
import org.apache.flink.table.catalog.listener.CatalogModificationListener;
import org.apache.flink.table.catalog.listener.CreateDatabaseEvent;
import org.apache.flink.table.catalog.listener.CreateModelEvent;
import org.apache.flink.table.catalog.listener.CreateTableEvent;
import org.apache.flink.table.catalog.listener.DropDatabaseEvent;
import org.apache.flink.table.catalog.listener.DropModelEvent;
import org.apache.flink.table.catalog.listener.DropTableEvent;
import org.apache.flink.table.delegation.Parser;
import org.apache.flink.table.expressions.resolver.ExpressionResolver;
import org.apache.flink.table.factories.FactoryUtil;
import org.apache.flink.table.operations.Operation;
import org.apache.flink.table.operations.QueryOperation;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public final class CatalogManager
implements CatalogRegistry,
AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(CatalogManager.class);
    private final Map<String, Catalog> catalogs;
    private final Map<ObjectIdentifier, CatalogBaseTable> temporaryTables;
    private final Map<ObjectIdentifier, CatalogModel> temporaryModels;
    @Nullable
    private String currentCatalogName;
    @Nullable
    private String currentDatabaseName;
    private DefaultSchemaResolver schemaResolver;
    private Parser parser;
    private final String builtInCatalogName;
    private final DataTypeFactory typeFactory;
    private final List<CatalogModificationListener> catalogModificationListeners;
    private final CatalogStoreHolder catalogStoreHolder;

    private CatalogManager(String defaultCatalogName, Catalog defaultCatalog, DataTypeFactory typeFactory, List<CatalogModificationListener> catalogModificationListeners, CatalogStoreHolder catalogStoreHolder) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)defaultCatalogName) ? 1 : 0) != 0, (Object)"Default catalog name cannot be null or empty");
        Preconditions.checkNotNull((Object)defaultCatalog, (String)"Default catalog cannot be null");
        this.catalogs = new LinkedHashMap<String, Catalog>();
        this.catalogs.put(defaultCatalogName, defaultCatalog);
        this.currentCatalogName = defaultCatalogName;
        this.currentDatabaseName = defaultCatalog.getDefaultDatabase();
        this.temporaryTables = new HashMap<ObjectIdentifier, CatalogBaseTable>();
        this.temporaryModels = new HashMap<ObjectIdentifier, CatalogModel>();
        this.builtInCatalogName = defaultCatalogName;
        this.typeFactory = typeFactory;
        this.catalogModificationListeners = catalogModificationListeners;
        this.catalogStoreHolder = catalogStoreHolder;
    }

    @VisibleForTesting
    public List<CatalogModificationListener> getCatalogModificationListeners() {
        return this.catalogModificationListeners;
    }

    public Optional<CatalogDescriptor> getCatalogDescriptor(String catalogName) {
        return this.catalogStoreHolder.catalogStore().getCatalog(catalogName);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    @Override
    public void close() throws CatalogException {
        ArrayList<Throwable> errors = new ArrayList<Throwable>();
        for (Map.Entry<String, Catalog> entry : this.catalogs.entrySet()) {
            String catalogName = entry.getKey();
            Catalog catalog = entry.getValue();
            try {
                catalog.close();
            }
            catch (Throwable e) {
                LOG.error(String.format("Failed to close catalog %s: %s", catalogName, e.getMessage()), e);
                errors.add(e);
            }
        }
        try {
            this.catalogStoreHolder.close();
        }
        catch (Throwable e) {
            errors.add(e);
            LOG.error(String.format("Failed to close catalog store holder: %s", e.getMessage()), e);
        }
        if (!errors.isEmpty()) {
            CatalogException exception = new CatalogException("Failed to close catalog manager");
            for (Throwable e : errors) {
                exception.addSuppressed(e);
            }
            throw exception;
        }
    }

    public void initSchemaResolver(boolean isStreamingMode, ExpressionResolver.ExpressionResolverBuilder expressionResolverBuilder, Parser parser) {
        this.schemaResolver = new DefaultSchemaResolver(isStreamingMode, this.typeFactory, expressionResolverBuilder);
        this.parser = parser;
    }

    public SchemaResolver getSchemaResolver() {
        return this.schemaResolver;
    }

    public DataTypeFactory getDataTypeFactory() {
        return this.typeFactory;
    }

    public void createCatalog(String catalogName, CatalogDescriptor catalogDescriptor, boolean ignoreIfExists) throws CatalogException {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be null or empty.");
        Preconditions.checkNotNull((Object)catalogDescriptor, (String)"Catalog descriptor cannot be null");
        boolean catalogExistsInStore = this.catalogStoreHolder.catalogStore().contains(catalogName);
        boolean catalogExistsInMemory = this.catalogs.containsKey(catalogName);
        if (catalogExistsInStore || catalogExistsInMemory) {
            if (!ignoreIfExists) {
                throw new CatalogException(String.format("Catalog %s already exists.", catalogName));
            }
        } else {
            Catalog catalog = this.initCatalog(catalogName, catalogDescriptor);
            catalog.open();
            this.catalogs.put(catalogName, catalog);
            this.catalogStoreHolder.catalogStore().storeCatalog(catalogName, catalogDescriptor);
        }
    }

    public void createCatalog(String catalogName, CatalogDescriptor catalogDescriptor) {
        this.createCatalog(catalogName, catalogDescriptor, false);
    }

    public void alterCatalog(String catalogName, CatalogChange catalogChange) throws CatalogException {
        Catalog newCatalog;
        CatalogDescriptor newDescriptor;
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be null or empty.");
        Preconditions.checkNotNull((Object)catalogChange, (String)"Catalog change cannot be null.");
        CatalogStore catalogStore = this.catalogStoreHolder.catalogStore();
        Optional<CatalogDescriptor> oldDescriptorOpt = this.getCatalogDescriptor(catalogName);
        if (catalogStore.contains(catalogName) && oldDescriptorOpt.isPresent()) {
            newDescriptor = catalogChange.applyChange(oldDescriptorOpt.get());
            newCatalog = this.initCatalog(catalogName, newDescriptor);
            catalogStore.removeCatalog(catalogName, false);
            if (this.catalogs.containsKey(catalogName)) {
                this.catalogs.get(catalogName).close();
            }
        } else {
            throw new CatalogException(String.format("Catalog %s does not exist in the catalog store.", catalogName));
        }
        newCatalog.open();
        this.catalogs.put(catalogName, newCatalog);
        this.catalogStoreHolder.catalogStore().storeCatalog(catalogName, newDescriptor);
    }

    private Catalog initCatalog(String catalogName, CatalogDescriptor catalogDescriptor) {
        return FactoryUtil.createCatalog(catalogName, catalogDescriptor.getConfiguration().toMap(), this.catalogStoreHolder.config(), this.catalogStoreHolder.classLoader());
    }

    @Deprecated
    public void registerCatalog(String catalogName, Catalog catalog) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be null or empty.");
        Preconditions.checkNotNull((Object)catalog, (String)"Catalog cannot be null");
        if (this.catalogs.containsKey(catalogName)) {
            throw new CatalogException(String.format("Catalog %s already exists.", catalogName));
        }
        catalog.open();
        this.catalogs.put(catalogName, catalog);
    }

    public void unregisterCatalog(String catalogName, boolean ignoreIfNotExists) {
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be null or empty.");
        if (this.catalogs.containsKey(catalogName) || this.catalogStoreHolder.catalogStore().contains(catalogName)) {
            if (catalogName.equals(this.currentCatalogName)) {
                throw new CatalogException("Cannot drop a catalog which is currently in use.");
            }
            if (this.catalogs.containsKey(catalogName)) {
                Catalog catalog = this.catalogs.remove(catalogName);
                catalog.close();
            }
            if (this.catalogStoreHolder.catalogStore().contains(catalogName)) {
                this.catalogStoreHolder.catalogStore().removeCatalog(catalogName, ignoreIfNotExists);
            }
        } else if (!ignoreIfNotExists) {
            throw new CatalogException(String.format("Catalog %s does not exist.", catalogName));
        }
    }

    public Optional<Catalog> getCatalog(String catalogName) {
        if (this.catalogs.containsKey(catalogName)) {
            return Optional.of(this.catalogs.get(catalogName));
        }
        Optional<CatalogDescriptor> optionalDescriptor = this.getCatalogDescriptor(catalogName);
        return optionalDescriptor.map(descriptor -> {
            Catalog catalog = this.initCatalog(catalogName, (CatalogDescriptor)descriptor);
            catalog.open();
            this.catalogs.put(catalogName, catalog);
            return catalog;
        });
    }

    public Catalog getCatalogOrThrowException(String catalogName) {
        return this.getCatalog(catalogName).orElseThrow(() -> new ValidationException(String.format("Catalog %s does not exist", catalogName)));
    }

    @Override
    public Catalog getCatalogOrError(String catalogName) throws CatalogNotExistException {
        return this.getCatalog(catalogName).orElseThrow(() -> new CatalogNotExistException(catalogName));
    }

    @Override
    @Nullable
    public String getCurrentCatalog() {
        return this.currentCatalogName;
    }

    public void setCurrentCatalog(@Nullable String catalogName) throws CatalogNotExistException {
        if (catalogName == null) {
            this.currentCatalogName = null;
            this.currentDatabaseName = null;
            return;
        }
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)catalogName) ? 1 : 0) != 0, (Object)"Catalog name cannot be empty.");
        Catalog potentialCurrentCatalog = this.getCatalog(catalogName).orElseThrow(() -> new CatalogException(String.format("A catalog with name [%s] does not exist.", catalogName)));
        if (!catalogName.equals(this.currentCatalogName)) {
            this.currentCatalogName = catalogName;
            this.currentDatabaseName = potentialCurrentCatalog.getDefaultDatabase();
            LOG.info("Set the current default catalog as [{}] and the current default database as [{}].", (Object)this.currentCatalogName, (Object)this.currentDatabaseName);
        }
    }

    @Override
    @Nullable
    public String getCurrentDatabase() {
        return this.currentDatabaseName;
    }

    public void setCurrentDatabase(@Nullable String databaseName) {
        if (databaseName == null) {
            this.currentDatabaseName = null;
            return;
        }
        Preconditions.checkArgument((!StringUtils.isNullOrWhitespaceOnly((String)databaseName) ? 1 : 0) != 0, (Object)"The database name cannot be empty.");
        if (this.currentCatalogName == null) {
            throw new CatalogException("Current catalog has not been set.");
        }
        if (!this.getCatalogOrThrowException(this.currentCatalogName).databaseExists(databaseName)) {
            throw new CatalogException(String.format("A database with name [%s] does not exist in the catalog: [%s].", databaseName, this.currentCatalogName));
        }
        if (!databaseName.equals(this.currentDatabaseName)) {
            this.currentDatabaseName = databaseName;
            LOG.info("Set the current default database as [{}] in the current default catalog [{}].", (Object)this.currentDatabaseName, (Object)this.currentCatalogName);
        }
    }

    public String getBuiltInCatalogName() {
        return this.builtInCatalogName;
    }

    public String getBuiltInDatabaseName() {
        return this.getCatalogOrThrowException(this.getBuiltInCatalogName()).getDefaultDatabase();
    }

    public Optional<ContextResolvedTable> getTable(ObjectIdentifier objectIdentifier) {
        CatalogBaseTable temporaryTable = this.temporaryTables.get(objectIdentifier);
        if (temporaryTable != null) {
            ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(temporaryTable);
            return Optional.of(ContextResolvedTable.temporary(objectIdentifier, resolvedTable));
        }
        return this.getPermanentTable(objectIdentifier, null);
    }

    public Optional<ContextResolvedTable> getTable(ObjectIdentifier objectIdentifier, long timestamp) {
        CatalogBaseTable temporaryTable = this.temporaryTables.get(objectIdentifier);
        if (temporaryTable != null) {
            ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(temporaryTable);
            return Optional.of(ContextResolvedTable.temporary(objectIdentifier, resolvedTable));
        }
        return this.getPermanentTable(objectIdentifier, timestamp);
    }

    @Override
    public Optional<ResolvedCatalogBaseTable<?>> getCatalogBaseTable(ObjectIdentifier objectIdentifier) {
        ContextResolvedTable resolvedTable = this.getTable(objectIdentifier).orElse(null);
        return resolvedTable == null ? Optional.empty() : Optional.of(resolvedTable.getResolvedTable());
    }

    @Override
    public boolean isTemporaryTable(ObjectIdentifier objectIdentifier) {
        return this.temporaryTables.containsKey(objectIdentifier);
    }

    public ContextResolvedTable getTableOrError(ObjectIdentifier objectIdentifier) {
        return this.getTable(objectIdentifier).orElseThrow(() -> new TableException(String.format("Cannot find table '%s' in any of the catalogs %s, nor as a temporary table.", objectIdentifier, this.listCatalogs())));
    }

    @Override
    public Optional<CatalogPartition> getPartition(ObjectIdentifier tableIdentifier, CatalogPartitionSpec partitionSpec) {
        Optional<Catalog> catalogOptional = this.getCatalog(tableIdentifier.getCatalogName());
        if (catalogOptional.isPresent()) {
            try {
                return Optional.of(catalogOptional.get().getPartition(tableIdentifier.toObjectPath(), partitionSpec));
            }
            catch (PartitionNotExistException partitionNotExistException) {
                // empty catch block
            }
        }
        return Optional.empty();
    }

    private Optional<ContextResolvedTable> getPermanentTable(ObjectIdentifier objectIdentifier, @Nullable Long timestamp) {
        Optional<Catalog> catalogOptional = this.getCatalog(objectIdentifier.getCatalogName());
        ObjectPath objectPath = objectIdentifier.toObjectPath();
        if (catalogOptional.isPresent()) {
            Catalog currentCatalog = catalogOptional.get();
            try {
                CatalogBaseTable table;
                if (timestamp != null) {
                    table = currentCatalog.getTable(objectPath, timestamp);
                    if (table.getTableKind() == CatalogBaseTable.TableKind.VIEW) {
                        throw new TableException(String.format("%s is a view, but time travel is not supported for view.", objectIdentifier.asSummaryString()));
                    }
                } else {
                    table = currentCatalog.getTable(objectPath);
                }
                ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(table);
                return Optional.of(ContextResolvedTable.permanent(objectIdentifier, currentCatalog, resolvedTable));
            }
            catch (TableNotExistException tableNotExistException) {
                // empty catch block
            }
        }
        return Optional.empty();
    }

    private Optional<CatalogBaseTable> getUnresolvedTable(ObjectIdentifier objectIdentifier) {
        Optional<Catalog> currentCatalog = this.getCatalog(objectIdentifier.getCatalogName());
        ObjectPath objectPath = objectIdentifier.toObjectPath();
        if (currentCatalog.isPresent()) {
            try {
                CatalogBaseTable table = currentCatalog.get().getTable(objectPath);
                return Optional.of(table);
            }
            catch (TableNotExistException tableNotExistException) {
                // empty catch block
            }
        }
        return Optional.empty();
    }

    public Set<String> listCatalogs() {
        return Collections.unmodifiableSet(Stream.concat(this.catalogs.keySet().stream(), this.catalogStoreHolder.catalogStore().listCatalogs().stream()).collect(Collectors.toSet()));
    }

    public Set<String> listTables() {
        return this.listTables(this.getCurrentCatalog(), this.getCurrentDatabase());
    }

    public Set<String> listTables(String catalogName, String databaseName) {
        Catalog catalog = this.getCatalogOrThrowException(catalogName);
        if (catalog == null) {
            throw new ValidationException(String.format("Catalog %s does not exist", catalogName));
        }
        try {
            return Stream.concat(catalog.listTables(databaseName).stream(), this.listTemporaryTablesInternal(catalogName, databaseName).map(e -> ((ObjectIdentifier)e.getKey()).getObjectName())).collect(Collectors.toSet());
        }
        catch (DatabaseNotExistException e2) {
            throw new ValidationException(String.format("Database %s does not exist", databaseName), e2);
        }
    }

    public Set<String> listTemporaryTables() {
        return this.listTemporaryTablesInternal(this.getCurrentCatalog(), this.getCurrentDatabase()).map(e -> ((ObjectIdentifier)e.getKey()).getObjectName()).collect(Collectors.toSet());
    }

    public Set<String> listTemporaryViews() {
        return this.listTemporaryViewsInternal(this.getCurrentCatalog(), this.getCurrentDatabase()).map(e -> ((ObjectIdentifier)e.getKey()).getObjectName()).collect(Collectors.toSet());
    }

    private Stream<Map.Entry<ObjectIdentifier, CatalogBaseTable>> listTemporaryTablesInternal(String catalogName, String databaseName) {
        return this.temporaryTables.entrySet().stream().filter(e -> {
            ObjectIdentifier identifier = (ObjectIdentifier)e.getKey();
            return identifier.getCatalogName().equals(catalogName) && identifier.getDatabaseName().equals(databaseName);
        });
    }

    public Set<String> listViews() {
        return this.listViews(this.getCurrentCatalog(), this.getCurrentDatabase());
    }

    public Set<String> listViews(String catalogName, String databaseName) {
        Catalog catalog = this.getCatalogOrThrowException(catalogName);
        if (catalog == null) {
            throw new ValidationException(String.format("Catalog %s does not exist", catalogName));
        }
        try {
            return Stream.concat(catalog.listViews(databaseName).stream(), this.listTemporaryViewsInternal(catalogName, databaseName).map(e -> ((ObjectIdentifier)e.getKey()).getObjectName())).collect(Collectors.toSet());
        }
        catch (DatabaseNotExistException e2) {
            throw new ValidationException(String.format("Database %s does not exist", databaseName), e2);
        }
    }

    private Stream<Map.Entry<ObjectIdentifier, CatalogBaseTable>> listTemporaryViewsInternal(String catalogName, String databaseName) {
        return this.listTemporaryTablesInternal(catalogName, databaseName).filter(e -> e.getValue() instanceof CatalogView);
    }

    public Set<String> listSchemas() {
        return Stream.concat(this.catalogs.keySet().stream(), this.temporaryTables.keySet().stream().map(ObjectIdentifier::getCatalogName)).collect(Collectors.toSet());
    }

    public Set<String> listSchemas(String catalogName) {
        return Stream.concat(this.getCatalog(catalogName).map(Catalog::listDatabases).orElse(Collections.emptyList()).stream(), this.temporaryTables.keySet().stream().filter(i -> i.getCatalogName().equals(catalogName)).map(ObjectIdentifier::getDatabaseName)).collect(Collectors.toSet());
    }

    public boolean schemaExists(String catalogName) {
        return this.getCatalog(catalogName).isPresent() || this.temporaryTables.keySet().stream().anyMatch(i -> i.getCatalogName().equals(catalogName));
    }

    public boolean schemaExists(String catalogName, String databaseName) {
        return this.temporaryDatabaseExists(catalogName, databaseName) || this.permanentDatabaseExists(catalogName, databaseName);
    }

    private boolean temporaryDatabaseExists(String catalogName, String databaseName) {
        return this.temporaryTables.keySet().stream().anyMatch(i -> i.getCatalogName().equals(catalogName) && i.getDatabaseName().equals(databaseName));
    }

    private boolean permanentDatabaseExists(String catalogName, String databaseName) {
        return this.getCatalog(catalogName).map(c -> c.databaseExists(databaseName)).orElse(false);
    }

    @Override
    public ObjectIdentifier qualifyIdentifier(UnresolvedIdentifier identifier) {
        return ObjectIdentifier.of(identifier.getCatalogName().orElseGet(() -> this.qualifyCatalog(this.getCurrentCatalog())), identifier.getDatabaseName().orElseGet(() -> this.qualifyDatabase(this.getCurrentDatabase())), identifier.getObjectName());
    }

    public String qualifyCatalog(@Nullable String catalogName) {
        if (!StringUtils.isNullOrWhitespaceOnly((String)catalogName)) {
            return catalogName;
        }
        String currentCatalogName = this.getCurrentCatalog();
        if (StringUtils.isNullOrWhitespaceOnly((String)currentCatalogName)) {
            throw new ValidationException("A current catalog has not been set. Please use a fully qualified identifier (such as 'my_catalog.my_database.my_table') or set a current catalog using 'USE CATALOG my_catalog'.");
        }
        return currentCatalogName;
    }

    public String qualifyDatabase(@Nullable String databaseName) {
        if (!StringUtils.isNullOrWhitespaceOnly((String)databaseName)) {
            return databaseName;
        }
        String currentDatabaseName = this.getCurrentDatabase();
        if (StringUtils.isNullOrWhitespaceOnly((String)currentDatabaseName)) {
            throw new ValidationException("A current database has not been set. Please use a fully qualified identifier (such as 'my_database.my_table' or 'my_catalog.my_database.my_table') or set a current database using 'USE my_database'.");
        }
        return currentDatabaseName;
    }

    public boolean createTable(CatalogBaseTable table, ObjectIdentifier objectIdentifier, boolean ignoreIfExists) {
        boolean result;
        if (ignoreIfExists) {
            Optional<CatalogBaseTable> resultOpt = this.getUnresolvedTable(objectIdentifier);
            result = resultOpt.isEmpty();
        } else {
            result = true;
        }
        this.execute((catalog, path) -> {
            ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(table);
            catalog.createTable(path, resolvedTable, ignoreIfExists);
            if (resolvedTable instanceof CatalogTable || resolvedTable instanceof CatalogMaterializedTable) {
                this.catalogModificationListeners.forEach(listener -> listener.onEvent(CreateTableEvent.createEvent(CatalogContext.createContext(objectIdentifier.getCatalogName(), catalog), objectIdentifier, resolvedTable, ignoreIfExists, false)));
            }
        }, objectIdentifier, false, "CreateTable");
        return result;
    }

    public void createTemporaryTable(CatalogBaseTable table, ObjectIdentifier objectIdentifier, boolean ignoreIfExists) {
        Optional<TemporaryOperationListener> listener = this.getTemporaryOperationListener(objectIdentifier);
        this.temporaryTables.compute(objectIdentifier, (k, v) -> {
            if (v != null) {
                if (!ignoreIfExists) {
                    throw new ValidationException(String.format("Temporary table '%s' already exists", objectIdentifier));
                }
                return v;
            }
            ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(table);
            Catalog catalog = this.getCatalog(objectIdentifier.getCatalogName()).orElse(null);
            if (listener.isPresent()) {
                return ((TemporaryOperationListener)listener.get()).onCreateTemporaryTable(objectIdentifier.toObjectPath(), resolvedTable);
            }
            if (resolvedTable instanceof CatalogTable) {
                this.catalogModificationListeners.forEach(l -> l.onEvent(CreateTableEvent.createEvent(CatalogContext.createContext(objectIdentifier.getCatalogName(), catalog), objectIdentifier, resolvedTable, ignoreIfExists, true)));
            }
            return resolvedTable;
        });
    }

    public void dropTemporaryTable(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.dropTemporaryTableInternal(objectIdentifier, table -> table instanceof CatalogTable, ignoreIfNotExists, true);
    }

    public void dropTemporaryView(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.dropTemporaryTableInternal(objectIdentifier, table -> table instanceof CatalogView, ignoreIfNotExists, false);
    }

    private void dropTemporaryTableInternal(ObjectIdentifier objectIdentifier, Predicate<CatalogBaseTable> filter, boolean ignoreIfNotExists, boolean isDropTable) {
        CatalogBaseTable catalogBaseTable = this.temporaryTables.get(objectIdentifier);
        if (filter.test(catalogBaseTable)) {
            this.getTemporaryOperationListener(objectIdentifier).ifPresent(l -> l.onDropTemporaryTable(objectIdentifier.toObjectPath()));
            Catalog catalog = this.getCatalog(objectIdentifier.getCatalogName()).orElse(null);
            ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(catalogBaseTable);
            this.temporaryTables.remove(objectIdentifier);
            if (isDropTable) {
                this.catalogModificationListeners.forEach(listener -> listener.onEvent(DropTableEvent.createEvent(CatalogContext.createContext(objectIdentifier.getCatalogName(), catalog), objectIdentifier, resolvedTable, ignoreIfNotExists, true)));
            }
        } else if (!ignoreIfNotExists) {
            throw new ValidationException(String.format("Temporary table or view with identifier '%s' does not exist.", objectIdentifier.asSummaryString()));
        }
    }

    protected Optional<TemporaryOperationListener> getTemporaryOperationListener(ObjectIdentifier identifier) {
        return this.getCatalog(identifier.getCatalogName()).map(c -> c instanceof TemporaryOperationListener ? (TemporaryOperationListener)((Object)c) : null);
    }

    public void alterTable(CatalogBaseTable table, ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.execute((catalog, path) -> {
            ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(table);
            catalog.alterTable(path, resolvedTable, ignoreIfNotExists);
            if (resolvedTable instanceof CatalogTable || resolvedTable instanceof CatalogMaterializedTable) {
                this.catalogModificationListeners.forEach(listener -> listener.onEvent(AlterTableEvent.createEvent(CatalogContext.createContext(objectIdentifier.getCatalogName(), catalog), objectIdentifier, resolvedTable, ignoreIfNotExists)));
            }
        }, objectIdentifier, ignoreIfNotExists, "AlterTable");
    }

    public void alterTable(CatalogBaseTable table, List<TableChange> changes, ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.execute((catalog, path) -> {
            ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable(table);
            catalog.alterTable(path, resolvedTable, changes, ignoreIfNotExists);
            if (resolvedTable instanceof CatalogTable) {
                this.catalogModificationListeners.forEach(listener -> listener.onEvent(AlterTableEvent.createEvent(CatalogContext.createContext(objectIdentifier.getCatalogName(), catalog), objectIdentifier, resolvedTable, ignoreIfNotExists)));
            }
        }, objectIdentifier, ignoreIfNotExists, "AlterTable");
    }

    public boolean dropTable(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        return this.dropTableInternal(objectIdentifier, ignoreIfNotExists, CatalogBaseTable.TableKind.TABLE);
    }

    public boolean dropMaterializedTable(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        return this.dropTableInternal(objectIdentifier, ignoreIfNotExists, CatalogBaseTable.TableKind.MATERIALIZED_TABLE);
    }

    public boolean dropView(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        return this.dropTableInternal(objectIdentifier, ignoreIfNotExists, CatalogBaseTable.TableKind.VIEW);
    }

    private boolean dropTableInternal(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists, CatalogBaseTable.TableKind kind) {
        String kindStr;
        Predicate<CatalogBaseTable> filter;
        switch (kind) {
            case VIEW: {
                filter = table -> table instanceof CatalogView;
                kindStr = "View";
                break;
            }
            case TABLE: {
                filter = table -> table instanceof CatalogTable;
                kindStr = "Table";
                break;
            }
            case MATERIALIZED_TABLE: {
                filter = table -> table instanceof CatalogMaterializedTable;
                kindStr = "Materialized Table";
                break;
            }
            default: {
                throw new ValidationException("Not supported table kind: " + kind);
            }
        }
        if (filter.test(this.temporaryTables.get(objectIdentifier))) {
            String lowerKindStr = kindStr.toLowerCase(Locale.ROOT);
            throw new ValidationException(String.format("Temporary %s with identifier '%s' exists. Drop it first before removing the permanent %s.", lowerKindStr, objectIdentifier, lowerKindStr));
        }
        Optional<CatalogBaseTable> resultOpt = this.getUnresolvedTable(objectIdentifier);
        if (resultOpt.isPresent() && filter.test(resultOpt.get())) {
            this.execute((catalog, path) -> {
                ResolvedCatalogBaseTable<?> resolvedTable = this.resolveCatalogBaseTable((CatalogBaseTable)resultOpt.get());
                catalog.dropTable(path, ignoreIfNotExists);
                if (kind != CatalogBaseTable.TableKind.VIEW) {
                    this.catalogModificationListeners.forEach(listener -> listener.onEvent(DropTableEvent.createEvent(CatalogContext.createContext(objectIdentifier.getCatalogName(), catalog), objectIdentifier, resolvedTable, ignoreIfNotExists, false)));
                }
            }, objectIdentifier, ignoreIfNotExists, "DropTable");
            return true;
        }
        if (!ignoreIfNotExists) {
            throw new ValidationException(String.format("%s with identifier '%s' does not exist.", kindStr, objectIdentifier.asSummaryString()));
        }
        return false;
    }

    public Optional<ContextResolvedModel> getModel(ObjectIdentifier objectIdentifier) {
        CatalogModel temporaryModel = this.temporaryModels.get(objectIdentifier);
        if (temporaryModel != null) {
            ResolvedCatalogModel resolvedModel = this.resolveCatalogModel(temporaryModel);
            return Optional.of(ContextResolvedModel.temporary(objectIdentifier, resolvedModel));
        }
        Optional<Catalog> catalogOptional = this.getCatalog(objectIdentifier.getCatalogName());
        ObjectPath objectPath = objectIdentifier.toObjectPath();
        if (catalogOptional.isPresent()) {
            Catalog currentCatalog = catalogOptional.get();
            try {
                CatalogModel model = currentCatalog.getModel(objectPath);
                if (model != null) {
                    ResolvedCatalogModel resolvedModel = this.resolveCatalogModel(model);
                    return Optional.of(ContextResolvedModel.permanent(objectIdentifier, currentCatalog, resolvedModel));
                }
            }
            catch (ModelNotExistException modelNotExistException) {
            }
            catch (UnsupportedOperationException unsupportedOperationException) {
                // empty catch block
            }
        }
        return Optional.empty();
    }

    public ContextResolvedModel getModelOrError(ObjectIdentifier objectIdentifier) {
        return this.getModel(objectIdentifier).orElseThrow(() -> new TableException(String.format("Cannot find model '%s' in any of the catalogs %s.", objectIdentifier, this.listCatalogs())));
    }

    public boolean isTemporaryModel(ObjectIdentifier objectIdentifier) {
        return this.temporaryModels.containsKey(objectIdentifier);
    }

    public Set<String> listModels() {
        return this.listModels(this.getCurrentCatalog(), this.getCurrentDatabase());
    }

    public Set<String> listModels(String catalogName, String databaseName) {
        Catalog catalog = this.getCatalogOrThrowException(catalogName);
        if (catalog == null) {
            throw new ValidationException(String.format("Catalog %s does not exist", catalogName));
        }
        try {
            return new HashSet<String>(catalog.listModels(databaseName));
        }
        catch (DatabaseNotExistException e) {
            throw new ValidationException(String.format("Database %s does not exist", databaseName), e);
        }
    }

    public void createModel(CatalogModel model, ObjectIdentifier objectIdentifier, boolean ignoreIfExists) {
        this.execute((catalog, path) -> {
            ResolvedCatalogModel resolvedModel = this.resolveCatalogModel(model);
            catalog.createModel(path, resolvedModel, ignoreIfExists);
            this.catalogModificationListeners.forEach(listener -> listener.onEvent(CreateModelEvent.createEvent(CatalogContext.createContext(objectIdentifier.getCatalogName(), catalog), objectIdentifier, resolvedModel, ignoreIfExists, false)));
        }, objectIdentifier, false, "CreateModel");
    }

    public void createTemporaryModel(CatalogModel model, ObjectIdentifier objectIdentifier, boolean ignoreIfExists) {
        Optional<TemporaryOperationListener> listener = this.getTemporaryOperationListener(objectIdentifier);
        this.temporaryModels.compute(objectIdentifier, (k, v) -> {
            if (v != null) {
                if (!ignoreIfExists) {
                    throw new ValidationException(String.format("Temporary model '%s' already exists", objectIdentifier));
                }
                return v;
            }
            ResolvedCatalogModel resolvedModel = this.resolveCatalogModel(model);
            Catalog catalog = this.getCatalog(objectIdentifier.getCatalogName()).orElse(null);
            if (listener.isPresent()) {
                return ((TemporaryOperationListener)listener.get()).onCreateTemporaryModel(objectIdentifier.toObjectPath(), resolvedModel);
            }
            this.catalogModificationListeners.forEach(l -> l.onEvent(CreateModelEvent.createEvent(CatalogContext.createContext(objectIdentifier.getCatalogName(), catalog), objectIdentifier, resolvedModel, ignoreIfExists, true)));
            return resolvedModel;
        });
    }

    public void alterModel(CatalogModel modelChange, ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.execute((catalog, path) -> {
            ResolvedCatalogModel resolvedModel = this.resolveCatalogModel(modelChange);
            catalog.alterModel(path, resolvedModel, ignoreIfNotExists);
            this.catalogModificationListeners.forEach(listener -> listener.onEvent(AlterModelEvent.createEvent(CatalogContext.createContext(objectIdentifier.getCatalogName(), catalog), objectIdentifier, resolvedModel, ignoreIfNotExists)));
        }, objectIdentifier, ignoreIfNotExists, "AlterModel");
    }

    public void dropModel(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        this.execute((catalog, path) -> {
            Optional<ContextResolvedModel> resultOpt = this.getModel(objectIdentifier);
            if (resultOpt.isPresent()) {
                ResolvedCatalogModel resolvedModel = resultOpt.get().getResolvedModel();
                catalog.dropModel(path, ignoreIfNotExists);
                this.catalogModificationListeners.forEach(listener -> listener.onEvent(DropModelEvent.createEvent(CatalogContext.createContext(objectIdentifier.getCatalogName(), catalog), objectIdentifier, resolvedModel, ignoreIfNotExists, false)));
            } else if (!ignoreIfNotExists) {
                throw new ModelNotExistException(objectIdentifier.getCatalogName(), objectIdentifier.toObjectPath());
            }
        }, objectIdentifier, ignoreIfNotExists, "DropModel");
    }

    public void dropTemporaryModel(ObjectIdentifier objectIdentifier, boolean ignoreIfNotExists) {
        CatalogModel model = this.temporaryModels.get(objectIdentifier);
        if (model != null) {
            this.getTemporaryOperationListener(objectIdentifier).ifPresent(l -> l.onDropTemporaryModel(objectIdentifier.toObjectPath()));
            Catalog catalog = this.getCatalog(objectIdentifier.getCatalogName()).orElse(null);
            ResolvedCatalogModel resolvedModel = this.resolveCatalogModel(model);
            this.temporaryModels.remove(objectIdentifier);
            this.catalogModificationListeners.forEach(listener -> listener.onEvent(DropModelEvent.createEvent(CatalogContext.createContext(objectIdentifier.getCatalogName(), catalog), objectIdentifier, resolvedModel, ignoreIfNotExists, true)));
        } else if (!ignoreIfNotExists) {
            throw new ValidationException(String.format("Temporary model with identifier '%s' does not exist.", objectIdentifier.asSummaryString()));
        }
    }

    public ResolvedCatalogModel resolveCatalogModel(CatalogModel model) {
        Preconditions.checkNotNull((Object)this.schemaResolver, (String)"Schema resolver is not initialized.");
        if (model instanceof ResolvedCatalogModel) {
            return (ResolvedCatalogModel)model;
        }
        ResolvedSchema resolvedInputSchema = model.getInputSchema().resolve(this.schemaResolver);
        ResolvedSchema resolvedOutputSchema = model.getOutputSchema().resolve(this.schemaResolver);
        return ResolvedCatalogModel.of(model, resolvedInputSchema, resolvedOutputSchema);
    }

    private void execute(ModifyCatalog command, ObjectIdentifier objectIdentifier, boolean ignoreNoCatalog, String commandName) {
        Optional<Catalog> catalog = this.getCatalog(objectIdentifier.getCatalogName());
        if (catalog.isPresent()) {
            try {
                command.execute(catalog.get(), objectIdentifier.toObjectPath());
            }
            catch (DatabaseNotExistException | TableAlreadyExistException | TableNotExistException e) {
                throw new ValidationException(this.getErrorMessage(objectIdentifier, commandName), e);
            }
            catch (Exception e) {
                throw new TableException(this.getErrorMessage(objectIdentifier, commandName), e);
            }
        } else if (!ignoreNoCatalog) {
            throw new ValidationException(String.format("Catalog %s does not exist.", objectIdentifier.getCatalogName()));
        }
    }

    private String getErrorMessage(ObjectIdentifier objectIdentifier, String commandName) {
        return String.format("Could not execute %s in path %s", commandName, objectIdentifier);
    }

    public ResolvedCatalogBaseTable<?> resolveCatalogBaseTable(CatalogBaseTable baseTable) {
        Preconditions.checkNotNull((Object)this.schemaResolver, (String)"Schema resolver is not initialized.");
        if (baseTable instanceof CatalogTable) {
            return this.resolveCatalogTable((CatalogTable)baseTable);
        }
        if (baseTable instanceof CatalogMaterializedTable) {
            return this.resolveCatalogMaterializedTable((CatalogMaterializedTable)baseTable);
        }
        if (baseTable instanceof CatalogView) {
            return this.resolveCatalogView((CatalogView)baseTable);
        }
        throw new IllegalArgumentException("Unknown kind of catalog base table: " + baseTable.getClass());
    }

    public ResolvedCatalogTable resolveCatalogTable(CatalogTable table) {
        Preconditions.checkNotNull((Object)this.schemaResolver, (String)"Schema resolver is not initialized.");
        if (table instanceof ResolvedCatalogTable) {
            return (ResolvedCatalogTable)table;
        }
        ResolvedSchema resolvedSchema = table.getUnresolvedSchema().resolve(this.schemaResolver);
        List physicalColumns = resolvedSchema.getColumns().stream().filter(Column::isPhysical).map(Column::getName).collect(Collectors.toList());
        Consumer<TableDistribution> distributionValidation = distribution -> {
            distribution.getBucketKeys().forEach(bucketKey -> {
                if (!physicalColumns.contains(bucketKey)) {
                    throw new ValidationException(String.format("Invalid bucket key '%s'. A bucket key for a distribution must reference a physical column in the schema. Available columns are: %s", bucketKey, physicalColumns));
                }
            });
            distribution.getBucketCount().ifPresent(c -> {
                if (c <= 0) {
                    throw new ValidationException(String.format("Invalid bucket count '%s'. The number of buckets for a distributed table must be at least 1.", c));
                }
            });
        };
        table.getDistribution().ifPresent(distributionValidation);
        table.getPartitionKeys().forEach(partitionKey -> {
            if (!physicalColumns.contains(partitionKey)) {
                throw new ValidationException(String.format("Invalid partition key '%s'. A partition key must reference a physical column in the schema. Available columns are: %s", partitionKey, physicalColumns));
            }
        });
        return new ResolvedCatalogTable(table, resolvedSchema);
    }

    public ResolvedCatalogMaterializedTable resolveCatalogMaterializedTable(CatalogMaterializedTable table) {
        Preconditions.checkNotNull((Object)this.schemaResolver, (String)"Schema resolver is not initialized.");
        if (table instanceof ResolvedCatalogMaterializedTable) {
            return (ResolvedCatalogMaterializedTable)table;
        }
        ResolvedSchema resolvedSchema = table.getUnresolvedSchema().resolve(this.schemaResolver);
        List physicalColumns = resolvedSchema.getColumns().stream().filter(Column::isPhysical).map(Column::getName).collect(Collectors.toList());
        table.getPartitionKeys().forEach(partitionKey -> {
            if (!physicalColumns.contains(partitionKey)) {
                throw new ValidationException(String.format("Invalid partition key '%s'. A partition key must reference a physical column in the schema. Available columns are: %s", partitionKey, physicalColumns));
            }
        });
        return new ResolvedCatalogMaterializedTable(table, resolvedSchema);
    }

    public ResolvedCatalogView resolveCatalogView(CatalogView view) {
        List<Operation> parse;
        Preconditions.checkNotNull((Object)this.schemaResolver, (String)"Schema resolver is not initialized.");
        if (view instanceof ResolvedCatalogView) {
            return (ResolvedCatalogView)view;
        }
        if (view instanceof QueryOperationCatalogView) {
            QueryOperation queryOperation = ((QueryOperationCatalogView)view).getQueryOperation();
            return new ResolvedCatalogView(view, queryOperation.getResolvedSchema());
        }
        ResolvedSchema resolvedSchema = view.getUnresolvedSchema().resolve(this.schemaResolver);
        try {
            parse = this.parser.parse(view.getExpandedQuery());
        }
        catch (Throwable e) {
            return new ResolvedCatalogView(view, resolvedSchema);
        }
        if (parse.size() != 1 || !(parse.get(0) instanceof QueryOperation)) {
            return new ResolvedCatalogView(view, resolvedSchema);
        }
        QueryOperation operation = (QueryOperation)parse.get(0);
        ResolvedSchema querySchema = operation.getResolvedSchema();
        if (querySchema.getColumns().size() != resolvedSchema.getColumns().size()) {
            return new ResolvedCatalogView(view, resolvedSchema);
        }
        ResolvedSchema renamedQuerySchema = new ResolvedSchema(IntStream.range(0, resolvedSchema.getColumnCount()).mapToObj(i -> querySchema.getColumn(i).get().rename(resolvedSchema.getColumnNames().get(i))).collect(Collectors.toList()), resolvedSchema.getWatermarkSpecs(), resolvedSchema.getPrimaryKey().orElse(null));
        return new ResolvedCatalogView(new QueryOperationCatalogView(operation, view), renamedQuerySchema);
    }

    public void createDatabase(String catalogName, String databaseName, CatalogDatabase database, boolean ignoreIfExists) throws DatabaseAlreadyExistException, CatalogException {
        Catalog catalog = this.getCatalogOrThrowException(catalogName);
        catalog.createDatabase(databaseName, database, ignoreIfExists);
        this.catalogModificationListeners.forEach(listener -> listener.onEvent(CreateDatabaseEvent.createEvent(CatalogContext.createContext(catalogName, catalog), databaseName, database, ignoreIfExists)));
    }

    public void dropDatabase(String catalogName, String databaseName, boolean ignoreIfNotExists, boolean cascade) throws DatabaseNotExistException, DatabaseNotEmptyException, CatalogException {
        if (Objects.equals(this.currentCatalogName, catalogName) && Objects.equals(this.currentDatabaseName, databaseName)) {
            throw new ValidationException("Cannot drop a database which is currently in use.");
        }
        Catalog catalog = this.getCatalogOrError(catalogName);
        catalog.dropDatabase(databaseName, ignoreIfNotExists, cascade);
        this.catalogModificationListeners.forEach(listener -> listener.onEvent(DropDatabaseEvent.createEvent(CatalogContext.createContext(catalogName, catalog), databaseName, ignoreIfNotExists, cascade)));
    }

    public void alterDatabase(String catalogName, String databaseName, CatalogDatabase newDatabase, boolean ignoreIfNotExists) throws DatabaseNotExistException, CatalogException {
        Catalog catalog = this.getCatalogOrError(catalogName);
        catalog.alterDatabase(databaseName, newDatabase, ignoreIfNotExists);
        this.catalogModificationListeners.forEach(listener -> listener.onEvent(AlterDatabaseEvent.createEvent(CatalogContext.createContext(catalogName, catalog), databaseName, newDatabase, ignoreIfNotExists)));
    }

    private static interface ModifyCatalog {
        public void execute(Catalog var1, ObjectPath var2) throws Exception;
    }

    @Internal
    public static final class Builder {
        @Nullable
        private ClassLoader classLoader;
        @Nullable
        private ReadableConfig config;
        @Nullable
        private String defaultCatalogName;
        @Nullable
        private Catalog defaultCatalog;
        @Nullable
        private ExecutionConfig executionConfig;
        @Nullable
        private DataTypeFactory dataTypeFactory;
        private List<CatalogModificationListener> catalogModificationListeners = Collections.emptyList();
        private CatalogStoreHolder catalogStoreHolder;

        public Builder classLoader(ClassLoader classLoader) {
            this.classLoader = classLoader;
            return this;
        }

        public Builder config(ReadableConfig config) {
            this.config = config;
            return this;
        }

        public Builder defaultCatalog(String defaultCatalogName, Catalog defaultCatalog) {
            this.defaultCatalogName = defaultCatalogName;
            this.defaultCatalog = defaultCatalog;
            return this;
        }

        public Builder executionConfig(ExecutionConfig executionConfig) {
            this.executionConfig = executionConfig;
            return this;
        }

        public Builder dataTypeFactory(DataTypeFactory dataTypeFactory) {
            this.dataTypeFactory = dataTypeFactory;
            return this;
        }

        public Builder catalogModificationListeners(List<CatalogModificationListener> catalogModificationListeners) {
            this.catalogModificationListeners = catalogModificationListeners;
            return this;
        }

        public Builder catalogStoreHolder(CatalogStoreHolder catalogStoreHolder) {
            this.catalogStoreHolder = catalogStoreHolder;
            return this;
        }

        public CatalogManager build() {
            Preconditions.checkNotNull((Object)this.classLoader, (String)"Class loader cannot be null");
            Preconditions.checkNotNull((Object)this.config, (String)"Config cannot be null");
            Preconditions.checkNotNull((Object)this.catalogStoreHolder, (String)"CatalogStoreHolder cannot be null");
            return new CatalogManager(this.defaultCatalogName, this.defaultCatalog, this.dataTypeFactory != null ? this.dataTypeFactory : new DataTypeFactoryImpl(this.classLoader, this.config, this.executionConfig == null ? null : this.executionConfig.getSerializerConfig()), this.catalogModificationListeners, this.catalogStoreHolder);
        }
    }
}

