/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.dolphinscheduler.e2e.cases;

import static org.assertj.core.api.Assertions.assertThat;

import org.apache.dolphinscheduler.e2e.core.DolphinScheduler;
import org.apache.dolphinscheduler.e2e.pages.LoginPage;
import org.apache.dolphinscheduler.e2e.pages.project.ProjectPage;

import java.util.UUID;

import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junitpioneer.jupiter.DisableIfTestFails;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.testcontainers.shaded.org.awaitility.Awaitility;

@DolphinScheduler(composeFiles = "docker/basic/docker-compose.yaml")
@DisableIfTestFails
class ProjectE2ETest {

    private static final String project = "test-project-" + UUID.randomUUID();

    private static final String workerGroup = "default";

    private static RemoteWebDriver browser;

    @BeforeAll
    public static void setup() {
        new LoginPage(browser)
                .login("admin", "dolphinscheduler123")
                .goToNav(ProjectPage.class);
    }

    @Test
    @Order(1)
    void testCreateProject() {
        final ProjectPage page = new ProjectPage(browser);
        page.create(project);

        Awaitility.await().untilAsserted(() -> {
            browser.navigate().refresh();
            assertThat(
                    page.projectList()).anyMatch(
                            it -> it.getText().contains(project));
        });
    }

    @Test
    @Order(5)
    void testAssignWorkerGroup() {
        final ProjectPage page = new ProjectPage(browser);
        page.assignWorkerGroup(project, workerGroup);
        page.verifyAssignedWorkerGroup(project, workerGroup);
    }

    @Test
    @Order(30)
    void testDeleteProject() {
        final ProjectPage page = new ProjectPage(browser);
        page.delete(project);

        Awaitility.await().untilAsserted(() -> {
            browser.navigate().refresh();
            assertThat(
                    page.projectList()).noneMatch(
                            it -> it.getText().contains(project));
        });
    }
}
