/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.dolphinscheduler.meter.metrics;

import org.apache.dolphinscheduler.common.utils.OSUtils;

import lombok.extern.slf4j.Slf4j;
import io.micrometer.core.instrument.MeterRegistry;

@Slf4j
public class DefaultMetricsProvider implements MetricsProvider {

    private final MeterRegistry meterRegistry;

    public DefaultMetricsProvider(MeterRegistry meterRegistry) {
        this.meterRegistry = meterRegistry;
    }

    private SystemMetrics systemMetrics;

    private long lastRefreshTime = 0;

    private double lastSystemCpuUsage = 0.0d;
    private double lastProcessCpuUsage = 0.0d;

    private static final long SYSTEM_METRICS_REFRESH_INTERVAL = 1_000L;

    @Override
    public SystemMetrics getSystemMetrics() {
        if (System.currentTimeMillis() - lastRefreshTime < SYSTEM_METRICS_REFRESH_INTERVAL) {
            return systemMetrics;
        }

        double systemCpuUsage = meterRegistry.get("system.cpu.usage").gauge().value();
        if (Double.compare(systemCpuUsage, Double.NaN) == 0) {
            systemCpuUsage = lastSystemCpuUsage;
        } else {
            lastSystemCpuUsage = systemCpuUsage;
        }
        double processCpuUsage = meterRegistry.get("process.cpu.usage").gauge().value();
        if (Double.compare(processCpuUsage, Double.NaN) == 0) {
            processCpuUsage = lastProcessCpuUsage;
        } else {
            lastProcessCpuUsage = processCpuUsage;
        }

        double jvmMemoryUsed = meterRegistry.get("jvm.memory.used").meter().measure().iterator().next().getValue();
        double jvmMemoryMax = meterRegistry.get("jvm.memory.max").meter().measure().iterator().next().getValue();

        long totalSystemMemory = OSUtils.getTotalSystemMemory();
        long systemMemoryAvailable = OSUtils.getSystemAvailableMemoryUsed();

        double diskToTalBytes = meterRegistry.get("disk.total").gauge().value();
        double diskFreeBytes = meterRegistry.get("disk.free").gauge().value();

        systemMetrics = SystemMetrics.builder()
                .systemCpuUsagePercentage(systemCpuUsage)
                .jvmCpuUsagePercentage(processCpuUsage)
                .jvmMemoryUsed(jvmMemoryUsed)
                .jvmMemoryMax(jvmMemoryMax)
                .jvmMemoryUsedPercentage(jvmMemoryUsed / jvmMemoryMax)
                .systemMemoryUsed(totalSystemMemory - systemMemoryAvailable)
                .systemMemoryMax(totalSystemMemory)
                .systemMemoryUsedPercentage((double) (totalSystemMemory - systemMemoryAvailable) / totalSystemMemory)
                .diskUsed(diskToTalBytes - diskFreeBytes)
                .diskTotal(diskToTalBytes)
                .diskUsedPercentage((diskToTalBytes - diskFreeBytes) / diskToTalBytes)
                .build();
        lastRefreshTime = System.currentTimeMillis();
        return systemMetrics;
    }

}
