/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 1997, 1998, 1999
 *	Sleepycat Software.  All rights reserved.
 */

#include "db_config.h"

#ifndef lint
static const char revid[] = "$Id: cxx_app.cpp,v 1.1.1.9.2.2 2000/02/08 00:43:41 noriko Exp $";
#endif /* not lint */

#include "db_cxx.h"
#include "cxx_int.h"

extern "C" {          // needed for __db_errcall
#include "db_int.h"
#include "common_ext.h"
}

#include <errno.h>
#include <stdio.h>              // needed for set_error_stream
#include <string.h>


// The reason for a static variable is that some structures
// (like Dbts) have no connection to any Db or DbEnv, so when
// errors occur in their methods, we must have some reasonable
// way to determine whether to throw or return errors.
//
// This variable is taken from flags whenever a DbEnv is constructed.
// Normally there is only one DbEnv per program, and even if not,
// there is typically a single policy of throwing or returning.
//
static int last_known_error_policy = ON_ERROR_UNKNOWN;

////////////////////////////////////////////////////////////////////////
//                                                                    //
//                            DbEnv                                   //
//                                                                    //
////////////////////////////////////////////////////////////////////////

// A truism for the DbEnv object is that there is always a valid
// DB_ENV handle for as long as the DbEnv object is live.  That means
// that when the underlying handle is invalidated, during
// DbEnv::close() and DbEnv::remove(), a new one is obtained immediately.
// This strategy is necessary to correctly allow reopen, and have
// a valid handle available for other set calls, e.g.:
//
//    DbEnv dbenv(0);
//    dbenv.open(...);
//    dbenv.close(...);
//    dbenv.set_errpfx(...);
//    dbenv.open(...);
//
// An odd side effect is that even if the user does a final close,
// we'll still have a DB_ENV handle that we must close in the destructor.
// Getting and Freeing of the handle correspond to internal calls of
// DbEnv::initialize() and DbEnv::cleanup().  Note that this strategy
// is also used with Db:: and DB handles.

ostream *DbEnv::error_stream_ = 0;

DbEnv::DbEnv(u_int32_t flags)
:	imp_(NULL)
,	construct_error_(0)
,	construct_flags_(flags)
,	tx_recover_callback_(0)
,	paniccall_callback_(0)
{
	int err;

	if ((err = initialize(NULL)) != 0)
		DB_ERROR("DbEnv::DbEnv", err, error_policy());
}

DbEnv::DbEnv(DB_ENV *env, u_int32_t flags)
:	imp_(NULL)
,	construct_error_(0)
,	construct_flags_(flags)
,	tx_recover_callback_(0)
,	paniccall_callback_(0)
{
	int err;

	if ((err = initialize(env)) != 0)
		DB_ERROR("DbEnv::DbEnv", err, error_policy());
}

// Note: we don't check for error codes in this destructor,
// since we can't return an error and throwing an exception
// in a destructor is unexpected.  If the user is curious
// about errors, they will call close() directly.
//
DbEnv::~DbEnv()
{
	DB_ENV *env = unwrap(this);

	cleanup();
	if (env != NULL)
		(void)env->close(env, 0);
}

// used internally (and also by Db) before closing and in destructor.
void DbEnv::cleanup()
{
	DB_ENV *env = unwrap(this);

	if (env != NULL) {
		// extra safety
		env->cj_internal = 0;
		imp_ = 0;
	}
	construct_error_ = 0;
}

int DbEnv::close(u_int32_t flags)
{
	DB_ENV *env = unwrap(this);
	int err, init_err;

	// after a close (no matter if success or failure),
	// the underlying DB_ENV object must not be accessed,
	// so we clean up in advance.
	//
	cleanup();

	// It's safe to throw an error after the close,
	// since our error mechanism does not peer into
	// the DB* structures.
	//
	if ((err = env->close(env, flags)) != 0) {
		DB_ERROR("DbEnv::close", err, error_policy());
	}
	if ((init_err = initialize(NULL)) != 0) {
		if (err == 0) {
			err = init_err;
			DB_ERROR("DbEnv::close", err, error_policy());
		}
	}

	return err;
}

void DbEnv::err(int error, const char *format, ...)
{
	va_list args;
	DB_ENV *env = unwrap(this);

	va_start(args, format);
	__db_real_err(env, error, 1, 1, format, args);
	va_end(args);
}

void DbEnv::errx(const char *format, ...)
{
	va_list args;
	DB_ENV *env = unwrap(this);

	va_start(args, format);
	__db_real_err(env, 0, 0, 1, format, args);
	va_end(args);
}

// used internally during constructor or after a close,
// to associate an existing DB_ENV with this DbEnv,
// or create a new one.  If there is an error,
// construct_error_ is set; this is examined during open.
//
int DbEnv::initialize(DB_ENV *env)
{
	int err;

	last_known_error_policy = error_policy();

	if (env == 0) {
		// Create a new DB_ENV environment.
		if ((err = ::db_env_create(&env,
			construct_flags_ & ~DB_CXX_NO_EXCEPTIONS)) != 0) {
			construct_error_ = err;
			return err;
		}
	}
	imp_ = wrap(env);
	env->cj_internal = this;    // for DB_ENV* to DbEnv* conversion
	return 0;
}

// Return a tristate value corresponding to whether we should
// throw exceptions on errors:
//   ON_ERROR_RETURN
//   ON_ERROR_THROW
//   ON_ERROR_UNKNOWN
//
int DbEnv::error_policy()
{
	if ((construct_flags_ & DB_CXX_NO_EXCEPTIONS) != 0) {
		return ON_ERROR_RETURN;
	}
	else {
		return ON_ERROR_THROW;
	}
}

// If an error occurred during the constructor, report it now.
// Otherwise, call the underlying DB->open method.
//
int DbEnv::open(const char *db_home, char * const *db_config,
		u_int32_t flags, int mode)
{
	DB_ENV *env = unwrap(this);
	int err;

	if ((err = construct_error_) != 0)
		DB_ERROR("Db::open", err, error_policy());
	else if ((err = env->open(env, db_home, db_config, flags, mode)) != 0)
		DB_ERROR("DbEnv::open", err, error_policy());

	return err;
}

int DbEnv::remove(const char *db_home, char *const *db_config,
		  u_int32_t flags)
{
	DB_ENV *env;
	int ret;

	env = unwrap(this);
	cleanup();
	if ((ret = env->remove(env, db_home, db_config, flags)) != 0)
		DB_ERROR("DbEnv::remove", ret, error_policy());

	return ret;
}

// Report an error associated with the DbEnv.
// error_policy is one of:
//   ON_ERROR_THROW     throw an error
//   ON_ERROR_RETURN    do nothing here, the caller will return an error
//   ON_ERROR_UNKNOWN   defer the policy to policy saved in DbEnv::DbEnv
//
void DbEnv::runtime_error(const char *caller, int error, int error_policy)
{
	if (error_policy == ON_ERROR_UNKNOWN)
		error_policy = last_known_error_policy;
	if (error_policy == ON_ERROR_THROW)
		throw DbException(caller, error);
}

// static method
char *DbEnv::strerror(int error)
{
	return db_strerror(error);
}

// Note: This actually behaves a bit like a static function,
// since DB_ENV.db_errcall has no information about which
// db_env triggered the call.  A user that has multiple DB_ENVs
// will simply not be able to have different streams for each one.
//
void DbEnv::set_error_stream(ostream *stream)
{
	DB_ENV *dbenv = unwrap(this);

	error_stream_ = stream;
	dbenv->set_errcall(dbenv, (stream == 0) ? 0 : stream_error_function);
}

void DbEnv::stream_error_function(const char *prefix, char *message)
{
	if (error_stream_) {
		if (prefix) {
			(*error_stream_) << prefix << ": ";
		}
		if (message) {
			(*error_stream_) << message;
		}
		(*error_stream_) << "\n";
	}
}

// static method
char *DbEnv::version(int *major, int *minor, int *patch)
{
	return db_version(major, minor, patch);
}

// This is a variant of the DB_WO_ACCESS macro to define a simple set_
// method calling the underlying C method, but unlike a simple
// set method, it may return an error or raise an exception.
// Note this macro expects that input _argspec is an argument
// list element (e.g. "char *arg") defined in terms of "arg".
//
#define DB_DBENV_ACCESS(_name, _argspec)                       \
                                                               \
int DbEnv::set_##_name(_argspec)                               \
{                                                              \
	int ret;                                               \
	DB_ENV *dbenv = unwrap(this);                          \
                                                               \
	if ((ret = (*(dbenv->set_##_name))(dbenv, arg)) != 0) {\
		DB_ERROR("DbEnv::set_" # _name, ret, error_policy()); \
	}                                                      \
	return ret;                                            \
}

#define DB_DBENV_ACCESS_NORET(_name, _argspec)                 \
                                                               \
void DbEnv::set_##_name(_argspec)                              \
{                                                              \
	DB_ENV *dbenv = unwrap(this);                          \
                                                               \
	(*(dbenv->set_##_name))(dbenv, arg);                   \
	return;                                                \
}

DB_DBENV_ACCESS_NORET(errcall, void (*arg)(const char *, char *))
DB_DBENV_ACCESS_NORET(errfile, FILE *arg)
DB_DBENV_ACCESS_NORET(errpfx, const char *arg)
DB_DBENV_ACCESS(func_close, int (*arg)(int))
DB_DBENV_ACCESS(func_dirfree, void (*arg)(char **, int))
DB_DBENV_ACCESS(func_dirlist, int (*arg)(const char *, char ***, int *))
DB_DBENV_ACCESS(func_exists, int (*arg)(const char *, int *))
DB_DBENV_ACCESS(func_free, void (*arg)(void *))
DB_DBENV_ACCESS(func_fsync, int (*arg)(int))
DB_DBENV_ACCESS(func_ioinfo, int (*arg)(const char *,
		int, u_int32_t *, u_int32_t *, u_int32_t *))
DB_DBENV_ACCESS(func_malloc, void *(*arg)(size_t))
DB_DBENV_ACCESS(func_map, int (*arg)(char *, size_t, int, int, void **))
DB_DBENV_ACCESS(func_open, int (*arg)(const char *, int, ...))
DB_DBENV_ACCESS(func_read, ssize_t (*arg)(int, void *, size_t))
DB_DBENV_ACCESS(func_realloc, void *(*arg)(void *, size_t))
DB_DBENV_ACCESS(func_seek,
		int (*arg)(int, size_t, db_pgno_t, u_int32_t, int, int))
DB_DBENV_ACCESS(func_sleep, int (*arg)(u_long, u_long))
DB_DBENV_ACCESS(func_unlink, int (*arg)(const char *))
DB_DBENV_ACCESS(func_unmap, int (*arg)(void *, size_t))
DB_DBENV_ACCESS(func_write, ssize_t (*arg)(int, const void *, size_t))
DB_DBENV_ACCESS(func_yield, int (*arg)(void))
DB_DBENV_ACCESS(lg_bsize, u_int32_t arg)
DB_DBENV_ACCESS(lg_max, u_int32_t arg)
DB_DBENV_ACCESS(lk_detect, u_int32_t arg)
DB_DBENV_ACCESS(lk_max, u_int32_t arg)
DB_DBENV_ACCESS(mp_mmapsize, size_t arg)
DB_DBENV_ACCESS(mutexlocks, int arg)
DB_DBENV_ACCESS(pageyield, int arg)
DB_DBENV_ACCESS(region_init, int arg)
DB_DBENV_ACCESS(tas_spins, u_int32_t arg)
DB_DBENV_ACCESS(tx_max, u_int32_t arg)

// Here are the set methods that don't fit the above mold.
//

int DbEnv::set_cachesize(u_int32_t gbytes, u_int32_t bytes, int ncache)
{
	int ret;
	DB_ENV *dbenv = unwrap(this);

	if ((ret = (*(dbenv->set_cachesize))(dbenv, gbytes, bytes, ncache)) != 0) {
		DB_ERROR("DbEnv::set_cachesize", ret, error_policy());
	}
	return ret;
}

int DbEnv::set_lk_conflicts(u_int8_t *lk_conflicts, int lk_max)
{
	int ret;
	DB_ENV *dbenv = unwrap(this);

	if ((ret = (*(dbenv->set_lk_conflicts))
	     (dbenv, lk_conflicts, lk_max)) != 0) {
		DB_ERROR("DbEnv::set_lk_conflicts", ret, error_policy());
	}
	return ret;
}

int DbEnv::set_verbose(u_int32_t which, int onoff)
{
	int ret;
	DB_ENV *dbenv = unwrap(this);

	if ((ret = (*(dbenv->set_verbose))(dbenv, which, onoff)) != 0) {
		DB_ERROR("DbEnv::set_verbose", ret, error_policy());
	}
	return ret;
}

int DbEnv::tx_recover_intercept(DB_ENV *env, DBT *dbt,
				DB_LSN *lsn, db_recops op, void *info)
{
	if (env == 0) {
		DB_ERROR("DbEnv::tx_recover_callback", EINVAL, ON_ERROR_UNKNOWN);
		return EINVAL;
	}
	DbEnv *cxxenv = (DbEnv *)env->cj_internal;
	if (cxxenv == 0) {
		DB_ERROR("DbEnv::tx_recover_callback", EINVAL, ON_ERROR_UNKNOWN);
		return EINVAL;
	}
	if (cxxenv->tx_recover_callback_ == 0) {
		DB_ERROR("DbEnv::tx_recover_callback", EINVAL, cxxenv->error_policy());
		return EINVAL;
	}
	Dbt *cxxdbt = (Dbt *)dbt;
	DbLsn *cxxlsn = (DbLsn *)lsn;
	return (*cxxenv->tx_recover_callback_)(cxxenv, cxxdbt, cxxlsn, op, info);
}

int DbEnv::set_tx_recover
    (int (*arg)(DbEnv *, Dbt *, DbLsn *, db_recops, void *))
{
	int ret;
	DB_ENV *dbenv = unwrap(this);

	tx_recover_callback_ = arg;
	if ((ret =
	    (*(dbenv->set_tx_recover))(dbenv, tx_recover_intercept)) != 0) {
		DB_ERROR("DbEnv::set_tx_recover", ret, error_policy());
	}
	return ret;
}

void DbEnv::paniccall_intercept(DB_ENV *env, int errval)
{
	if (env == 0) {
		DB_ERROR("DbEnv::paniccall_callback", EINVAL, ON_ERROR_UNKNOWN);
	}
	DbEnv *cxxenv = (DbEnv *)env->cj_internal;
	if (cxxenv == 0) {
		DB_ERROR("DbEnv::paniccall_callback", EINVAL, ON_ERROR_UNKNOWN);
	}
	if (cxxenv->paniccall_callback_ == 0) {
		DB_ERROR("DbEnv::paniccall_callback", EINVAL, cxxenv->error_policy());
	}
	(*cxxenv->paniccall_callback_)(cxxenv, errval);
}

void DbEnv::set_paniccall(void (*arg)(DbEnv *, int))
{
	DB_ENV *dbenv = unwrap(this);

	paniccall_callback_ = arg;

	(*(dbenv->set_paniccall))(dbenv, paniccall_intercept);
}

int DbEnv::recovery_init_intercept(DB_ENV *env)
{
	if (env == 0) {
		DB_ERROR("DbEnv::recovery_init_callback", EINVAL,
			 ON_ERROR_UNKNOWN);
	}
	DbEnv *cxxenv = (DbEnv *)env->cj_internal;
	if (cxxenv == 0) {
		DB_ERROR("DbEnv::recovery_init_callback", EINVAL,
			 ON_ERROR_UNKNOWN);
	}
	if (cxxenv->recovery_init_callback_ == 0) {
		DB_ERROR("DbEnv::recovery_init_callback", EINVAL,
			 cxxenv->error_policy());
	}
	return (*cxxenv->recovery_init_callback_)(cxxenv);
}

void DbEnv::set_recovery_init(int (*arg)(DbEnv *))
{
	DB_ENV *dbenv = unwrap(this);

	recovery_init_callback_ = arg;

	(*(dbenv->set_recovery_init))(dbenv, recovery_init_intercept);
}

void DbEnv::feedback_intercept(DB_ENV *env, int opcode, int pct)
{
	if (env == 0) {
		DB_ERROR("DbEnv::feedback_callback", EINVAL, ON_ERROR_UNKNOWN);
		return;
	}
	DbEnv *cxxenv = (DbEnv *)env->cj_internal;
	if (cxxenv == 0) {
		DB_ERROR("DbEnv::feedback_callback", EINVAL, ON_ERROR_UNKNOWN);
		return;
	}
	if (cxxenv->feedback_callback_ == 0) {
		DB_ERROR("DbEnv::feedback_callback", EINVAL,
			 cxxenv->error_policy());
		return;
	}
	(*cxxenv->feedback_callback_)(cxxenv, opcode, pct);
}

void DbEnv::set_feedback(void (*arg)(DbEnv *, int, int))
{
	DB_ENV *dbenv = unwrap(this);

	feedback_callback_ = arg;

	(*(dbenv->set_feedback))(dbenv, feedback_intercept);
}
