package Acme::Pony::Pink;

use 5.010;
use autodie;
use strict;
no warnings;
use utf8;

BEGIN {
	$Acme::Pony::Pink::AUTHORITY = 'cpan:TOBYINK';
	$Acme::Pony::Pink::VERSION   = '0.003';
}

use Carp qw<croak>;
use MIME::Base64 qw<decode_base64>;
use List::MoreUtils qw<firstidx>;

use Config;
use constant _path_to_perl => $Config{perlpath};

sub new
{
	shift;
}

sub _image
{
	state $image = do {
		local $/ = <DATA>;
		decode_base64($/);
	};
	return $image;
}

sub _find_shebang
{
	my ($class, $lines) = @_;
	
	my $idx = firstidx { /^\#\!.*\bperl\b/ } @$lines;
	return if $idx < 0; # no shebang
	
	# look for code before shebang... that's not a real shebang
	for (0 .. $idx)
	{
		next if $lines->[$_] =~ m{^\s*#};
		next if $lines->[$_] =~ m{^\s*$};
		return;
	}
	
	return $lines->[$idx];
}

sub _filehandle
{
	my ($class, $args, $mode) = @_;
	return unless @$args;
	$mode //= '<';
	
	my $arg = shift @$args;
	my $fh;
	
	if (ref $arg eq 'SCALAR')
		{ open $fh, $mode, \$arg }
	elsif (!ref $arg)
		{ open $fh, $mode, $arg }
	else
		{ $fh = $arg }
	
	return $fh;
}

sub ponify
{
	my $class = shift;
	local $/ = "\n";
	
	my $fh_in  = $class->_filehandle(\@_, '<') or croak "must supply input handle";
	my $fh_out = $class->_filehandle(\@_, '>') || \*STDOUT;
		
	my @lines = <$fh_in>;
	unshift @lines, sprintf("#!%s\n", $class->_path_to_perl)
		unless $class->_find_shebang(\@lines);
	
	print {$fh_out} $class->_image;
	print {$fh_out} qq(\n\n);
	print {$fh_out} @lines;
	return;
}

=head1 NAME

Acme::Pony::Pink - make your script look like a pink pony

=head1 SYNOPSIS

 my $converter = Acme::Pony::Pink->new;
 $converter->ponify('myscript.pl' => 'myscript.gif');

myscript.gif is now a small picture of a pink pony, and you can run the
original script using:

 perl -x myscript.gif

=head1 DESCRIPTION

This module turns a Perl script into a picture of a pink pony.

The picture is executable via C<< perl -x >>.

=head2 Unconstrutive Constructor

=over

=item C<< new >>

Returns the invocant.

=back

=head2 Method

=over

=item C<< ponify($input, $output) >>

The input is a Perl script; the output is a GIF file. (No, the "F" in "GIF"
does not stand for "file".)

Input and output may be filehandles, file names, or scalar references if you
just want to play with strings. The input source is required; the default
output destination is STDOUT.

=back

=head1 BUGS

Please report any bugs to someone who cares.

=head1 SEE ALSO

L<Acme::Pony>,
L<Unicornify::URL>.

=head1 AUTHOR

Toby Inkster E<lt>tobyink@cpan.orgE<gt>.

=head1 COPYRIGHT AND LICENCE

This software is copyright (c) 2012 by Toby Inkster.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 DISCLAIMER OF WARRANTIES

THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.

=cut

__PACKAGE__
__DATA__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