/* This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/****************************************************************************
 * This module is mostly all new
 * by Rob Nation
 * A little of it is borrowed from ctwm.
 * Copyright 1993 Robert Nation. No restrictions are placed on this code,
 * as long as the copyright notice is preserved
 ****************************************************************************/
/***********************************************************************
 *
 * fvwm icon code
 *
 ***********************************************************************/

#include "config.h"

#include <stdio.h>
#include <signal.h>

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#include <X11/Intrinsic.h>
#ifdef XPM
#include <X11/xpm.h>
#endif /* XPM */

#include "fvwm.h"
#include "icons.h"
#include "misc.h"
#include "parse.h"
#include "screen.h"
#include "bindings.h"
#include "module.h"
#include "focus.h"

#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif /* SHAPE */

#include <libs/Picture.h>


static void GetBitmapFile(FvwmWindow *tmp_win);
static void GetXPMFile(FvwmWindow *tmp_win);



/****************************************************************************
 *
 * Creates an icon window as needed
 *
 ****************************************************************************/
void CreateIconWindow(FvwmWindow *tmp_win, int def_x, int def_y)
{
  int final_x, final_y;
  /* mask for create windows */
  unsigned long valuemask;
  /* attributes for create windows */
  XSetWindowAttributes attributes;

  SET_ICON_OURS(tmp_win, 1);
  SET_PIXMAP_OURS(tmp_win, 0);
  SET_ICON_SHAPED(tmp_win, 0);
  tmp_win->icon_pixmap_w = None;
  tmp_win->iconPixmap = None;
  tmp_win->iconDepth = 0;

  if(IS_ICON_SUPPRESSED(tmp_win))
    return;

  /* First, see if it was specified in the .fvwmrc */
  tmp_win->icon_p_height = 0;
  tmp_win->icon_p_width = 0;

  /* First, check for a monochrome bitmap */
  if(tmp_win->icon_bitmap_file != NULL)
    GetBitmapFile(tmp_win);

#ifdef XPM
  /* Next, check for a color pixmap */
  if((tmp_win->icon_bitmap_file != NULL)&&
     (tmp_win->icon_p_height == 0)&&(tmp_win->icon_p_width == 0))
    GetXPMFile(tmp_win);
#endif /* XPM */

  /* Next, See if the app supplies its own icon window */
  if((tmp_win->icon_p_height == 0)&&(tmp_win->icon_p_width == 0)&&
     (tmp_win->wmhints) && (tmp_win->wmhints->flags & IconWindowHint))
    GetIconWindow(tmp_win);

  /* Finally, try to get icon bitmap from the application */
  if((tmp_win->icon_p_height == 0)&&(tmp_win->icon_p_width == 0)&&
	  (tmp_win->wmhints)&&(tmp_win->wmhints->flags & IconPixmapHint))
    GetIconBitmap(tmp_win);

  /* figure out the icon window size */
  if (!(HAS_NO_ICON_TITLE(tmp_win))||(tmp_win->icon_p_height == 0))
    {
      tmp_win->icon_t_width = XTextWidth(Scr.IconFont.font,
					 tmp_win->icon_name,
                                         strlen(tmp_win->icon_name));
      tmp_win->icon_w_height = ICON_HEIGHT;
    }
  else
    {
      tmp_win->icon_t_width = 0;
      tmp_win->icon_w_height = 0;
    }
  if((IS_ICON_OURS(tmp_win))&&(tmp_win->icon_p_height >0))
    {
      tmp_win->icon_p_width += 4;
      tmp_win->icon_p_height +=4;
    }

  if(tmp_win->icon_p_width == 0)
    tmp_win->icon_p_width = tmp_win->icon_t_width+6;
  tmp_win->icon_w_width = tmp_win->icon_p_width;

  final_x = def_x;
  final_y = def_y;
  if(final_x <0)
    final_x = 0;
  if(final_y <0)
    final_y = 0;

  if(final_x + tmp_win->icon_w_width >=Scr.MyDisplayWidth)
    final_x = Scr.MyDisplayWidth - tmp_win->icon_w_width-1;
  if(final_y + tmp_win->icon_w_height >=Scr.MyDisplayHeight)
    final_y = Scr.MyDisplayHeight - tmp_win->icon_w_height-1;

  tmp_win->icon_x_loc = final_x;
  tmp_win->icon_xl_loc = final_x;
  tmp_win->icon_y_loc = final_y;

  /* clip to fit on screen */
  valuemask = CWColormap | CWBorderPixel
              | CWBackPixel | CWCursor | CWEventMask;
  attributes.colormap = Scr.cmap;
  attributes.background_pixel = Scr.StdColors.back;
  attributes.cursor = Scr.FvwmCursors[DEFAULT];
  attributes.border_pixel = 0;
  attributes.event_mask = (ButtonPressMask | ButtonReleaseMask
			   | VisibilityChangeMask | ExposureMask | KeyPressMask
			   | EnterWindowMask | LeaveWindowMask
			   | FocusChangeMask );
  if (!(HAS_NO_ICON_TITLE(tmp_win)) || (tmp_win->icon_p_height == 0))
    tmp_win->icon_w = XCreateWindow(dpy, Scr.Root, final_x,
				    final_y + tmp_win->icon_p_height,
				    tmp_win->icon_w_width,
				    tmp_win->icon_w_height, 0, Scr.depth,
				    InputOutput, Scr.viz, valuemask,
				    &attributes);

  if((IS_ICON_OURS(tmp_win)) && (tmp_win->icon_p_width > 0)
     && (tmp_win->icon_p_height > 0)) {
      /* fixme: if a client pixmap is supplied use it's depth, visual and colormap */
      tmp_win->icon_pixmap_w = XCreateWindow(dpy, Scr.Root, final_x, final_y,
					     tmp_win->icon_p_width,
					     tmp_win->icon_p_height, 0,
					     Scr.depth, InputOutput, Scr.viz,
					     valuemask,&attributes);
    }
  else
    {
      attributes.event_mask = (ButtonPressMask | ButtonReleaseMask |
			       VisibilityChangeMask |
			       KeyPressMask|EnterWindowMask |
			       FocusChangeMask | LeaveWindowMask );

      valuemask = CWEventMask;
      XChangeWindowAttributes(dpy, tmp_win->icon_pixmap_w, valuemask,
			      &attributes);
    }


#ifdef XPM
#ifdef SHAPE
  if (ShapesSupported && IS_ICON_SHAPED(tmp_win))
    {
      XShapeCombineMask(dpy, tmp_win->icon_pixmap_w, ShapeBounding,2, 2,
			tmp_win->icon_maskPixmap, ShapeSet);
    }
#endif
#endif

  if(tmp_win->icon_w != None)
    {
      XSaveContext(dpy, tmp_win->icon_w, FvwmContext, (caddr_t)tmp_win);
      XDefineCursor(dpy, tmp_win->icon_w, Scr.FvwmCursors[DEFAULT]);
      GrabAllWindowKeysAndButtons(dpy, tmp_win->icon_w, Scr.AllBindings,
				  C_ICON, GetUnusedModifiers(),
				  Scr.FvwmCursors[DEFAULT], True);

#ifdef SESSION
      {
        XWindowChanges xwc;
        xwc.sibling = tmp_win->frame;
        xwc.stack_mode = Below;
        XConfigureWindow(dpy, tmp_win->icon_w, CWSibling|CWStackMode, &xwc);
      }
#endif
    }
  if(tmp_win->icon_pixmap_w != None)
    {
      XSaveContext(dpy, tmp_win->icon_pixmap_w, FvwmContext, (caddr_t)tmp_win);
      XDefineCursor(dpy, tmp_win->icon_pixmap_w, Scr.FvwmCursors[DEFAULT]);
      GrabAllWindowKeysAndButtons(dpy, tmp_win->icon_pixmap_w, Scr.AllBindings,
				  C_ICON, GetUnusedModifiers(),
				  Scr.FvwmCursors[DEFAULT], True);

#ifdef SESSION
      {
        XWindowChanges xwc;
        xwc.sibling = tmp_win->frame;
        xwc.stack_mode = Below;
        XConfigureWindow(dpy, tmp_win->icon_pixmap_w, CWSibling|CWStackMode, &xwc);
      }
#endif
  }
  return;
}

/****************************************************************************
 *
 * Draws the icon window
 *
 ****************************************************************************/
void DrawIconWindow(FvwmWindow *Tmp_win)
{
  GC Shadow, Relief;
  Pixel TextColor,BackColor;
  int x;

  if(IS_ICON_SUPPRESSED(Tmp_win))
    return;

  if(Tmp_win->icon_w != None)
    flush_expose (Tmp_win->icon_w);
  if(Tmp_win->icon_pixmap_w != None)
    flush_expose (Tmp_win->icon_pixmap_w);

  if(Scr.Hilite == Tmp_win)
    {
      if(Scr.depth < 2) {
	Relief = Scr.DefaultDecor.HiShadowGC;
	Shadow = Scr.DefaultDecor.HiShadowGC;
	TextColor = Scr.DefaultDecor.HiColors.fore;
	BackColor = Scr.DefaultDecor.HiColors.back;
      } else {
	Relief = GetDecor(Tmp_win,HiReliefGC);
	Shadow = GetDecor(Tmp_win,HiShadowGC);
	TextColor = GetDecor(Tmp_win,HiColors.fore);
	BackColor = GetDecor(Tmp_win,HiColors.back);
      }
    }
  else
    {
      if(Scr.depth < 2)
	{
	  Relief = Scr.StdReliefGC;
	  Shadow = Scr.StdShadowGC;
	}
      else
	{
	  Globalgcv.foreground = Tmp_win->ReliefPixel;
	  Globalgcm = GCForeground;
	  XChangeGC(dpy,Scr.ScratchGC1,Globalgcm,&Globalgcv);
	  Relief = Scr.ScratchGC1;

	  Globalgcv.foreground = Tmp_win->ShadowPixel;
	  XChangeGC(dpy,Scr.ScratchGC2,Globalgcm,&Globalgcv);
	  Shadow = Scr.ScratchGC2;
	}
      TextColor = Tmp_win->TextPixel;
      BackColor = Tmp_win->BackPixel;
    }

  if(Tmp_win->icon_w != None)
    {
      if (IS_ICON_ENTERED(Tmp_win))
	{
	  /* resize the icon name window */
          Tmp_win->icon_w_width = Tmp_win->icon_t_width+6;
          if(Tmp_win->icon_w_width < Tmp_win->icon_p_width)
            Tmp_win->icon_w_width = Tmp_win->icon_p_width;
          Tmp_win->icon_xl_loc = Tmp_win->icon_x_loc -
            (Tmp_win->icon_w_width - Tmp_win->icon_p_width)/2;
          /* start keep label on screen. dje 8/7/97 */
          if (Tmp_win->icon_xl_loc < 0) {  /* if new loc neg (off left edge) */
            Tmp_win->icon_xl_loc = 0;      /* move to edge */
          } else {                         /* if not on left edge */
            /* if (new loc + width) > screen width (off edge on right) */
            if ((Tmp_win->icon_xl_loc + Tmp_win->icon_w_width) >
                Scr.MyDisplayWidth) {      /* off right */
              /* position up against right edge */
              Tmp_win->icon_xl_loc = Scr.MyDisplayWidth-Tmp_win->icon_w_width;
            }
            /* end keep label on screen. dje 8/7/97 */
          }
        }
      else
	{
	  /* resize the icon name window */
          Tmp_win->icon_w_width = Tmp_win->icon_p_width;
          Tmp_win->icon_xl_loc = Tmp_win->icon_x_loc;
        }
    }

  if((IS_ICON_OURS(Tmp_win))&&(Tmp_win->icon_pixmap_w != None))
    XSetWindowBackground(dpy,Tmp_win->icon_pixmap_w,
			 BackColor);
  if(Tmp_win->icon_w != None)
    XSetWindowBackground(dpy,Tmp_win->icon_w,BackColor);

  /* write the icon label */
  NewFontAndColor(Scr.IconFont.font->fid,TextColor,BackColor);

  if(Tmp_win->icon_pixmap_w != None)
    XMoveWindow(dpy,Tmp_win->icon_pixmap_w,Tmp_win->icon_x_loc,
                Tmp_win->icon_y_loc);
  if(Tmp_win->icon_w != None)
    {
      Tmp_win->icon_w_height = ICON_HEIGHT;
      XMoveResizeWindow(dpy, Tmp_win->icon_w, Tmp_win->icon_xl_loc,
                        Tmp_win->icon_y_loc+Tmp_win->icon_p_height,
                        Tmp_win->icon_w_width,ICON_HEIGHT);

      XClearWindow(dpy,Tmp_win->icon_w);
    }

  if((Tmp_win->iconPixmap != None)&&(!(IS_ICON_SHAPED(Tmp_win))))
    RelieveRectangle(dpy,Tmp_win->icon_pixmap_w,0,0,
	             Tmp_win->icon_p_width - 1, Tmp_win->icon_p_height - 1,
	             Relief,Shadow,2);

  /* need to locate the icon pixmap */
  if(Tmp_win->iconPixmap != None)
    {
      if(Tmp_win->iconDepth == Scr.depth)
	{
	  XCopyArea(dpy,Tmp_win->iconPixmap,Tmp_win->icon_pixmap_w,
		    Scr.ScratchGC3,
		    0,0,Tmp_win->icon_p_width-4, Tmp_win->icon_p_height-4,2,2);
	}
      else /* fixme: this only copies one plane, should be min(scr,icon) */
           /* even if fixed the result will probabaly be garbage */
	XCopyPlane(dpy,Tmp_win->iconPixmap,Tmp_win->icon_pixmap_w,
		   Scr.ScratchGC3,0,0,Tmp_win->icon_p_width-4,
		   Tmp_win->icon_p_height-4,2,2,1);
    }

  if(Tmp_win->icon_w != None)
    {
      /* text position */
      x = (Tmp_win->icon_w_width - Tmp_win->icon_t_width)/2;
      if(x<3)x=3;

      XDrawString (dpy, Tmp_win->icon_w, Scr.ScratchGC3, x,
                   Tmp_win->icon_w_height-Scr.IconFont.height+
		   Scr.IconFont.y-3,
                   Tmp_win->icon_name, strlen(Tmp_win->icon_name));
      RelieveRectangle(dpy,Tmp_win->icon_w,0,0,Tmp_win->icon_w_width - 1,
                       ICON_HEIGHT - 1,Relief,Shadow,2);
    }

  if (IS_ICON_ENTERED(Tmp_win))
    {
      if (Tmp_win->icon_w != None)
	{
	  XRaiseWindow (dpy, Tmp_win->icon_w);
	  raisePanFrames ();
	}
    }
  else
    {
      XWindowChanges xwc;
      int mask;
      xwc.sibling = Tmp_win->frame;
      xwc.stack_mode = Below;
      mask = CWSibling|CWStackMode;
      if (Tmp_win->icon_w != None)
	{
	  XConfigureWindow(dpy, Tmp_win->icon_w, mask, &xwc);
	}
      if (Tmp_win->icon_pixmap_w != None)
	{
	  XConfigureWindow(dpy, Tmp_win->icon_pixmap_w, mask, &xwc);
	}
    }
}


/***********************************************************************
 *
 *  Procedure:
 *	RedoIconName - procedure to re-position the icon window and name
 *
 ************************************************************************/
void RedoIconName(FvwmWindow *Tmp_win)
{

  if(IS_ICON_SUPPRESSED(Tmp_win))
    return;

  if (Tmp_win->icon_w == (int)NULL)
    return;

  Tmp_win->icon_t_width = XTextWidth(Scr.IconFont.font,Tmp_win->icon_name,
				     strlen(Tmp_win->icon_name));
  /* clear the icon window, and trigger a re-draw via an expose event */
  if (IS_ICONIFIED(Tmp_win))
    XClearArea(dpy, Tmp_win->icon_w, 0, 0, 0, 0, True);
  return;
}




/************************************************************************
 *
 *  Procedure:
 *	AutoPlace - Find a home for an icon
 *
 ************************************************************************/
void AutoPlaceIcon(FvwmWindow *t)
{
  int tw,th,tx,ty;
  int base_x, base_y;
  int width,height;
  FvwmWindow *test_window;
  Bool loc_ok;
  int real_x=10, real_y=10;
  int new_x, new_y;

  /* New! Put icon in same page as the center of the window */
  /* Not a good idea for StickyIcons. Neither for icons of windows that are
   * visible on the current page. */
  if((IS_ICON_STICKY(t))||(IS_STICKY(t))||(IsWindowOnThisPage(t)))
    {
      base_x = 0;
      base_y = 0;
      /*Also, if its a stickyWindow, put it on the current page! */
      new_x = t->frame_x % Scr.MyDisplayWidth;
      new_y = t->frame_y % Scr.MyDisplayHeight;
      if(new_x < 0)
	new_x += Scr.MyDisplayWidth;
      if(new_y < 0)
	new_y += Scr.MyDisplayHeight;
      SetupFrame(t,new_x,new_y,
		 t->frame_width,t->frame_height,False,False);
      t->Desk = Scr.CurrentDesk;
    }
  else
    {
      base_x=((t->frame_x+Scr.Vx+(t->frame_width>>1))/Scr.MyDisplayWidth)*
	Scr.MyDisplayWidth - Scr.Vx;
      base_y=((t->frame_y+Scr.Vy+(t->frame_height>>1))/Scr.MyDisplayHeight)*
	Scr.MyDisplayHeight - Scr.Vy;
    }
  if(IS_ICON_MOVED(t))
    {
      /* just make sure the icon is on this screen */
      t->icon_x_loc = t->icon_x_loc % Scr.MyDisplayWidth + base_x;
      t->icon_y_loc = t->icon_y_loc % Scr.MyDisplayHeight + base_y;
      if(t->icon_x_loc < 0)
	t->icon_x_loc += Scr.MyDisplayWidth;
      if(t->icon_y_loc < 0)
	t->icon_y_loc += Scr.MyDisplayHeight;
    }
  else if (t->wmhints && t->wmhints->flags & IconPositionHint)
    {
      t->icon_x_loc = t->wmhints->icon_x;
      t->icon_y_loc = t->wmhints->icon_y;
    }
  /* dje 10/12/97:
     Look thru chain of icon boxes assigned to window.
     Add logic for grids and fill direction.
  */
  else {
    /* A place to hold inner and outer loop variables. */
    typedef struct dimension_struct {
      int step;                         /* grid size (may be negative) */
      int start_at;                     /* starting edge */
      int real_start;                   /* on screen starting edge */
      int end_at;                       /* ending edge */
      int base;                         /* base for screen */
      int icon_dimension;               /* height or width */
      int nom_dimension;                /* nonminal height or width */
      int screen_dimension;             /* screen height or width */
    } dimension;
    dimension dim[3];                   /* space for work, 1st, 2nd dimen */
    icon_boxes *icon_boxes_ptr;         /* current icon box */
    int i;                              /* index for inner/outer loop data */

    /* Hopefully this makes the following more readable. */
#define ICONBOX_LFT icon_boxes_ptr->IconBox[0]
#define ICONBOX_TOP icon_boxes_ptr->IconBox[1]
#define ICONBOX_RGT icon_boxes_ptr->IconBox[2]
#define ICONBOX_BOT icon_boxes_ptr->IconBox[3]
#define BOT_FILL icon_boxes_ptr->IconFlags & ICONFILLBOT
#define RGT_FILL icon_boxes_ptr->IconFlags & ICONFILLRGT
#define HRZ_FILL icon_boxes_ptr->IconFlags & ICONFILLHRZ

    /* unnecessary copy of width */
    width = t->icon_p_width;
    /* total height */
    height = t->icon_w_height + t->icon_p_height;
    /* no slot found yet */
    loc_ok = False;

    /* check all boxes in order */
    for(icon_boxes_ptr= t->IconBoxes;   /* init */
        icon_boxes_ptr != NULL; /* until no more boxes */
        icon_boxes_ptr = icon_boxes_ptr->next) { /* all boxes */
      if (loc_ok == True) {
	/* leave for loop */
        break;
      }
      /* y amount */
      dim[1].step = icon_boxes_ptr->IconGrid[1];
      /* init start from */
      dim[1].start_at = ICONBOX_TOP;
      /* init end at */
      dim[1].end_at = ICONBOX_BOT;
      /* save base */
      dim[1].base = base_y;
      /* save dimension */
      dim[1].icon_dimension = height;
      dim[1].screen_dimension = Scr.MyDisplayHeight;
      if (BOT_FILL) {
	/* fill from bottom */
	/* reverse step */
        dim[1].step = 0 - dim[1].step;
      } /* end fill from bottom */

      /* x amount */
      dim[2].step = icon_boxes_ptr->IconGrid[0];
      /* init start from */
      dim[2].start_at = ICONBOX_LFT;
      /* init end at */
      dim[2].end_at = ICONBOX_RGT;
      /* save base */
      dim[2].base   = base_x;
      /* save dimension */
      dim[2].icon_dimension = width;
      dim[2].screen_dimension = Scr.MyDisplayWidth;
      if (RGT_FILL) {
	/* fill from right */
	/* reverse step */
        dim[2].step = 0 - dim[2].step;
      } /* end fill from right */
      for (i=1;i<=2;i++) {
	/* for dimensions 1 and 2 */
        /* If the window is taller than the icon box, ignore the icon height
         * when figuring where to put it. Same goes for the width
         * This should permit reasonably graceful handling of big icons. */
        dim[i].nom_dimension = dim[i].icon_dimension;
        if (dim[i].icon_dimension >= dim[i].end_at - dim[i].start_at) {
          dim[i].nom_dimension = dim[i].end_at - dim[i].start_at - 1;
        }
        if (dim[i].step < 0) {
	  /* if moving backwards */
	  /* save */
          dim[0].start_at = dim[i].start_at;
	  /* swap one */
          dim[i].start_at = dim[i].end_at;
	  /* swap the other */
          dim[i].end_at = dim[0].start_at;
          dim[i].start_at -= dim[i].icon_dimension;
        } /* end moving backwards */
	/* adjust both to base */
        dim[i].start_at += dim[i].base;
        dim[i].end_at += dim[i].base;
      } /* end 2 dimensions */
      if (HRZ_FILL) {
	/* if hrz first */
	/* save */
        memcpy(&dim[0],&dim[1],sizeof(dimension));
	/* switch one */
        memcpy(&dim[1],&dim[2],sizeof(dimension));
	/* switch the other */
        memcpy(&dim[2],&dim[0],sizeof(dimension));
      } /* end horizontal dimension first */
      /* save for reseting inner loop */
      dim[0].start_at = dim[2].start_at;
      while((dim[1].step < 0            /* filling reversed */
             ? (dim[1].start_at + dim[1].icon_dimension - dim[1].nom_dimension
                > dim[1].end_at)        /* check back edge */
             : (dim[1].start_at + dim[1].nom_dimension
                < dim[1].end_at))       /* check front edge */
            && (!loc_ok)) {             /* nothing found yet */
        dim[1].real_start = dim[1].start_at; /* init */
        if (dim[1].start_at + dim[1].icon_dimension >
            dim[1].screen_dimension - 2 + dim[1].base) {
	  /* off screen, move on screen */
          dim[1].real_start = dim[1].screen_dimension
            - dim[1].icon_dimension + dim[1].base;
        } /* end off screen */
        if (dim[1].start_at < dim[1].base) {
	  /* if off other edge, move on screen */
          dim[1].real_start = dim[1].base;
        } /* end off other edge */
	/* reset inner loop */
        dim[2].start_at = dim[0].start_at;
        while((dim[2].step < 0            /* filling reversed */
               ? (dim[2].start_at + dim[2].icon_dimension-dim[2].nom_dimension
                  > dim[2].end_at)        /* check back edge */
               : (dim[2].start_at + dim[2].nom_dimension
                  < dim[2].end_at))       /* check front edge */
              && (!loc_ok)) {             /* nothing found yet */
          dim[2].real_start = dim[2].start_at; /* init */
          if (dim[2].start_at + dim[2].icon_dimension >
              dim[2].screen_dimension - 2 + dim[2].base) {
	    /* if off screen, move on screen */
            dim[2].real_start = dim[2].screen_dimension
              - dim[2].icon_dimension + dim[2].base;
          } /* end off screen */
          if (dim[2].start_at < dim[2].base) {
	    /* if off other edge, move on screen */
	    dim[2].real_start = dim[2].base;
          } /* end off other edge */

          if (HRZ_FILL) {
	    /* hrz first */
	    /* unreverse them */
            real_x = dim[1].real_start;
            real_y = dim[2].real_start;
          } else {
	    /* reverse them */
            real_x = dim[2].real_start;
            real_y = dim[1].real_start;
          }

	  /* this may be a good location */
          loc_ok = True;
          test_window = Scr.FvwmRoot.next;
          while((test_window != (FvwmWindow *)0)
                &&(loc_ok == True)) {
	    /* test overlap */
            if(test_window->Desk == t->Desk) {
              if((IS_ICONIFIED(test_window)) &&
                 (!IS_TRANSIENT(test_window) ||
		  !IS_ICONIFIED_BY_PARENT(test_window)) &&
                 (test_window->icon_w||test_window->icon_pixmap_w) &&
                 (test_window != t)) {
                tw=test_window->icon_p_width;
                th=test_window->icon_p_height+
                  test_window->icon_w_height;
                tx = test_window->icon_x_loc;
                ty = test_window->icon_y_loc;

                if((tx<(real_x+width+3))&&((tx+tw+3) > real_x)&&
                   (ty<(real_y+height+3))&&((ty+th + 3)>real_y)) {
		  /* don't accept this location */
                  loc_ok = False;
                } /* end if icons overlap */
              } /* end if its an icon */
            } /* end if same desk */
            test_window = test_window->next;
          } /* end while icons that may overlap */
	  /* Grid inner value & direction */
          dim[2].start_at += dim[2].step;
        } /* end while room inner dimension */
	/* Grid outer value & direction */
        dim[1].start_at += dim[1].step;
      } /* end while room outer dimension */
    } /* end for all icon boxes, or found space */
    if(loc_ok == False)
      /* If icon never found a home just leave it */
      return;
    t->icon_x_loc = real_x;
    t->icon_y_loc = real_y;

    if(t->icon_pixmap_w)
      XMoveWindow(dpy,t->icon_pixmap_w,t->icon_x_loc, t->icon_y_loc);

    t->icon_w_width = t->icon_p_width;
    t->icon_xl_loc = t->icon_x_loc;

    if (t->icon_w != None)
      XMoveResizeWindow(dpy, t->icon_w, t->icon_xl_loc,
                        t->icon_y_loc+t->icon_p_height,
                        t->icon_w_width,ICON_HEIGHT);
    BroadcastPacket(M_ICON_LOCATION, 7,
                    t->w, t->frame,
                    (unsigned long)t,
                    t->icon_x_loc, t->icon_y_loc,
                    t->icon_w_width, t->icon_w_height+t->icon_p_height);
  }
}


/****************************************************************************
 *
 * Looks for a monochrome icon bitmap file
 *
 ****************************************************************************/
static void GetBitmapFile(FvwmWindow *tmp_win)
{
  char *path = NULL;
  int HotX,HotY;

  path = findImageFile(tmp_win->icon_bitmap_file, NULL, R_OK);

  if(path == NULL)return;
  if(XReadBitmapFile (dpy, Scr.Root,path,
		      (unsigned int *)&tmp_win->icon_p_width,
		      (unsigned int *)&tmp_win->icon_p_height,
		      &tmp_win->iconPixmap,
		      &HotX, &HotY) != BitmapSuccess)
    {
      tmp_win->icon_p_width = 0;
      tmp_win->icon_p_height = 0;
    }

  free(path);
}

/****************************************************************************
 *
 * Looks for a color XPM icon file
 *
 ****************************************************************************/
static void GetXPMFile(FvwmWindow *tmp_win)
{
#ifdef XPM
  XpmAttributes xpm_attributes;
  char *path = NULL;
  XpmImage my_image;
  int rc;

  path = findImageFile(tmp_win->icon_bitmap_file, NULL, R_OK);
  if(path == NULL)return;

  xpm_attributes.visual = PictureVisual;
  xpm_attributes.colormap = PictureCMap;
  xpm_attributes.depth = PictureDepth;
  xpm_attributes.closeness = 40000; /* Allow for "similar" colors */
  xpm_attributes.valuemask = XpmSize | XpmReturnPixels | XpmCloseness
			     | XpmVisual | XpmColormap | XpmDepth;

  rc = XpmReadFileToXpmImage(path, &my_image, NULL);
  if (rc != XpmSuccess) {
    fvwm_msg(ERR,"GetXPMFile","XpmReadFileToXpmImage failed, pixmap %s, rc %d",
           path, rc);
    free(path);
    return;
  }
  free(path);
  color_reduce_pixmap(&my_image,Scr.ColorLimit);
  rc = XpmCreatePixmapFromXpmImage(dpy,Scr.NoFocusWin, &my_image,
                                   &tmp_win->iconPixmap,
                                   &tmp_win->icon_maskPixmap,
                                   &xpm_attributes);
  if (rc != XpmSuccess) {
    fvwm_msg(ERR,"GetXPMFile",
             "XpmCreatePixmapFromXpmImage failed, rc %d\n", rc);
    XpmFreeXpmImage(&my_image);
    return;
  }
  tmp_win->icon_p_width = my_image.width;
  tmp_win->icon_p_height = my_image.height;
  SET_PIXMAP_OURS(tmp_win, 1);
  tmp_win->iconDepth = Scr.depth;

#ifdef SHAPE
  if (ShapesSupported && tmp_win->icon_maskPixmap)
    SET_ICON_SHAPED(tmp_win, 1);
#endif

  XpmFreeXpmImage(&my_image);

#endif /* XPM */
}

/****************************************************************************
 *
 * Looks for an application supplied icon window
 *
 ****************************************************************************/
void GetIconWindow(FvwmWindow *tmp_win)
{
  /* We are guaranteed that wmhints is non-null when calling this
   * routine */
  if(XGetGeometry(dpy,   tmp_win->wmhints->icon_window, &JunkRoot,
		  &JunkX, &JunkY,(unsigned int *)&tmp_win->icon_p_width,
		  (unsigned int *)&tmp_win->icon_p_height,
		  &JunkBW, &JunkDepth)==0)
    {
      fvwm_msg(ERR,"GetIconWindow","Help! Bad Icon Window!");
    }
  tmp_win->icon_p_width += JunkBW<<1;
  tmp_win->icon_p_height += JunkBW<<1;
  /*
   * Now make the new window the icon window for this window,
   * and set it up to work as such (select for key presses
   * and button presses/releases, set up the contexts for it,
   * and define the cursor for it).
   */
  tmp_win->icon_pixmap_w = tmp_win->wmhints->icon_window;
#ifdef SHAPE
  if (ShapesSupported)
  {
    if (tmp_win->wmhints->flags & IconMaskHint)
    {
      SET_ICON_SHAPED(tmp_win, 1);
      tmp_win->icon_maskPixmap = tmp_win->wmhints->icon_mask;
    }
  }
#endif
  /* Make sure that the window is a child of the root window ! */
  /* Olwais screws this up, maybe others do too! */
  XReparentWindow(dpy, tmp_win->icon_pixmap_w, Scr.Root, 0,0);
  SET_ICON_OURS(tmp_win, 0);
}


/****************************************************************************
 *
 * Looks for an application supplied bitmap or pixmap
 *
 ****************************************************************************/
void GetIconBitmap(FvwmWindow *tmp_win)
{
  /* We are guaranteed that wmhints is non-null when calling this
   * routine */
  XGetGeometry(dpy, tmp_win->wmhints->icon_pixmap, &JunkRoot, &JunkX, &JunkY,
	       (unsigned int *)&tmp_win->icon_p_width,
	       (unsigned int *)&tmp_win->icon_p_height, &JunkBW, &JunkDepth);
  tmp_win->iconPixmap = tmp_win->wmhints->icon_pixmap;
  tmp_win->iconDepth = JunkDepth;
#ifdef SHAPE
  if (ShapesSupported)
  {
    if (tmp_win->wmhints->flags & IconMaskHint)
    {
      SET_ICON_SHAPED(tmp_win, 1);
      tmp_win->icon_maskPixmap = tmp_win->wmhints->icon_mask;
    }
  }
#endif
}



/***********************************************************************
 *
 *  Procedure:
 *	DeIconify a window
 *
 ***********************************************************************/
void DeIconify(FvwmWindow *tmp_win)
{
  FvwmWindow *t,*tmp;

  if(!tmp_win)
    return;

  /* AS dje  RaiseWindow(tmp_win); */
  /* now de-iconify transients */
  for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
    {
      if ((t == tmp_win)||
	  ((IS_TRANSIENT(t)) && (t->transientfor == tmp_win->w)))
	{
	  SET_MAPPED(t, 1);
	  SET_ICONIFIED_BY_PARENT(t, 0);
	  if(Scr.Hilite == t)
	    SetBorder (t, False,True,True,None);

          /* AS stuff starts here dje */
          if (t->icon_pixmap_w)
            XUnmapWindow(dpy, t->icon_pixmap_w);
	  if (t->icon_w)
	    XUnmapWindow(dpy, t->icon_w);
          XFlush(dpy);
          if (t == tmp_win)
	    BroadcastPacket(M_DEICONIFY, 11,
                            t->w, t->frame,
                            (unsigned long)t,
                            t->icon_x_loc, t->icon_y_loc,
                            t->icon_p_width, t->icon_p_height+t->icon_w_height,
                            t->frame_x, t->frame_y,
                            t->frame_width, t->frame_height);
          else
	    BroadcastPacket(M_DEICONIFY, 7,
                            t->w, t->frame,
                            (unsigned long)t,
                            t->icon_x_loc, t->icon_y_loc,
                            t->icon_p_width,
			    t->icon_p_height+t->icon_w_height);
          /* End AS */
	  XMapWindow(dpy, t->w);
	  if(t->Desk == Scr.CurrentDesk)
	    {
	      XMapWindow(dpy, t->frame);
	      SET_MAP_PENDING(t, 1);
	    }
	  XMapWindow(dpy, t->Parent);
	  SetMapStateProp(t, NormalState);
	  SET_ICONIFIED(t, 0);
	  SET_ICON_UNMAPPED(t, 0);
	  /* Need to make sure the border is colored correctly,
	   * in case it was stuck or unstuck while iconified. */
	  tmp = Scr.Hilite;
	  Scr.Hilite = t;
	  SetBorder(t,False,True,True,None);
	  Scr.Hilite = tmp;
	  XRaiseWindow(dpy,t->w);
	}
    }

  RaiseWindow(tmp_win); /* moved dje */

  if(HAS_CLICK_FOCUS(tmp_win))
    FocusOn(tmp_win, TRUE, "");

  return;
}


/****************************************************************************
 *
 * Iconifies the selected window
 *
 ****************************************************************************/
void Iconify(FvwmWindow *tmp_win, int def_x, int def_y)
{
  FvwmWindow *t;
  XWindowAttributes winattrs = {0};
  unsigned long eventMask;

  if(!tmp_win)
    return;
  XGetWindowAttributes(dpy, tmp_win->w, &winattrs);
  eventMask = winattrs.your_event_mask;

  if((tmp_win == Scr.Hilite)&&(HAS_CLICK_FOCUS(tmp_win))&&(tmp_win->next))
    {
      SetFocus(tmp_win->next->w,tmp_win->next,1);
    }


  /* iconify transients first */
  for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
    {
      if ((t==tmp_win)||((IS_TRANSIENT(t)) && (t->transientfor == tmp_win->w)))
	{
	  /*
	   * Prevent the receipt of an UnmapNotify, since that would
	   * cause a transition to the Withdrawn state.
	   */
	  SET_MAPPED(t, 0);
	  XSelectInput(dpy, t->w, eventMask & ~StructureNotifyMask);
          XUnmapWindow(dpy, t->w);
          XSelectInput(dpy, t->w, eventMask);
          XUnmapWindow(dpy, t->frame);
	  t->DeIconifyDesk = t->Desk;
	  if (t->icon_w)
	    XUnmapWindow(dpy, t->icon_w);
	  if (t->icon_pixmap_w)
	    XUnmapWindow(dpy, t->icon_pixmap_w);

	  SetMapStateProp(t, IconicState);
	  SetBorder (t, False,False,False,None);
	  if(t != tmp_win)
	    {
	      SET_ICONIFIED(t, 1);
	      SET_ICON_UNMAPPED(t, 1);
	      SET_ICONIFIED_BY_PARENT(t, 1);

	      BroadcastPacket(M_ICONIFY, 7,
                              t->w, t->frame,
                              (unsigned long)t,
                              -10000, -10000,
                              t->icon_w_width,
                              t->icon_w_height+t->icon_p_height);
	      BroadcastConfig(M_CONFIGURE_WINDOW,t);
	    }
	} /* if */
    } /* for */
  if (tmp_win->icon_w == None)
  {
    if(IS_ICON_MOVED(tmp_win))
      CreateIconWindow(tmp_win,tmp_win->icon_x_loc,tmp_win->icon_y_loc);
    else
      CreateIconWindow(tmp_win, def_x, def_y);
  }

  /* if no pixmap we want icon width to change to text width every iconify */
  if( (tmp_win->icon_w != None) && (tmp_win->icon_pixmap_w == None) ) {
    tmp_win->icon_t_width =
      XTextWidth(Scr.IconFont.font,tmp_win->icon_name,
                 strlen(tmp_win->icon_name));
    tmp_win->icon_p_width = tmp_win->icon_t_width+6;
    tmp_win->icon_w_width = tmp_win->icon_p_width;
  }

#ifdef SESSION
  if (!(DO_START_ICONIC(tmp_win) && IS_ICON_MOVED(tmp_win)))
#endif
  AutoPlaceIcon(tmp_win);

  SET_ICONIFIED(tmp_win, 1);
  SET_ICON_UNMAPPED(tmp_win, 0);
  BroadcastPacket(M_ICONIFY, 11,
                  tmp_win->w, tmp_win->frame,
                  (unsigned long)tmp_win,
                  tmp_win->icon_x_loc,
                  tmp_win->icon_y_loc,
                  tmp_win->icon_w_width,
                  tmp_win->icon_w_height+tmp_win->icon_p_height,
                  tmp_win->frame_x, /* next 4 added for Animate module */
                  tmp_win->frame_y,
                  tmp_win->frame_width,
                  tmp_win->frame_height);
  BroadcastConfig(M_CONFIGURE_WINDOW,tmp_win);

#ifdef SESSION
  if (!(DO_START_ICONIC(tmp_win) && IS_ICON_MOVED(tmp_win)))
#endif
  LowerWindow(tmp_win);
  if(tmp_win->Desk == Scr.CurrentDesk)
    {
      if (tmp_win->icon_w != None)
        XMapWindow(dpy, tmp_win->icon_w);

      if(tmp_win->icon_pixmap_w != None)
	XMapWindow(dpy, tmp_win->icon_pixmap_w);
    }
  if(HAS_CLICK_FOCUS(tmp_win) || HAS_SLOPPY_FOCUS(tmp_win))
    {
      if (tmp_win == Scr.Focus)
	{
	  if(Scr.PreviousFocus == Scr.Focus)
	    Scr.PreviousFocus = NULL;
	  if(HAS_CLICK_FOCUS(tmp_win) && tmp_win->next)
	    SetFocus(tmp_win->next->w, tmp_win->next,1);
	  else
	    {
	      SetFocus(Scr.NoFocusWin, NULL,1);
	    }
	}
    }
  return;
}



/****************************************************************************
 *
 * This is used to tell applications which windows on the screen are
 * top level appication windows, and which windows are the icon windows
 * that go with them.
 *
 ****************************************************************************/
void SetMapStateProp(FvwmWindow *tmp_win, int state)
{
  unsigned long data[2];		/* "suggested" by ICCCM version 1 */

  data[0] = (unsigned long) state;
  data[1] = (unsigned long) tmp_win->icon_w;
/*  data[2] = (unsigned long) tmp_win->icon_pixmap_w;*/

  XChangeProperty (dpy, tmp_win->w, _XA_WM_STATE, _XA_WM_STATE, 32,
		   PropModeReplace, (unsigned char *) data, 2);
  return;
}


void iconify_function(F_CMD_ARGS)
{
  int toggle;

  if (DeferExecution(eventp,&w,&tmp_win,&context, SELECT, ButtonRelease))
    return;

  toggle = ParseToggleArgument(action, NULL, -1, 0);
  if (toggle == -1)
  {
    if (GetIntegerArguments(action, NULL, &toggle, 1) > 0)
      {
	if (toggle > 0)
	  toggle = 1;
	else if (toggle < 0)
	  toggle = 0;
	else
	  toggle = -1;
      }
  }
  if (toggle == -1)
    toggle = (IS_ICONIFIED(tmp_win)) ? 0 : 1;

  if (IS_ICONIFIED(tmp_win))
  {
    if (toggle == 0)
      DeIconify(tmp_win);
  }
  else
  {
    if (toggle == 1)
    {
      if(check_if_function_allowed(F_ICONIFY,tmp_win,True,0) == 0)
      {
	XBell(dpy, 0);
	return;
      }
      Iconify(tmp_win,eventp->xbutton.x_root-5,eventp->xbutton.y_root-5);
    }
  }
}
